// Module PI : Peida Interactive     PIDrawer
// Classe de traceurs pouvant tre attachs  une PIBaseWdg
//	      E. Aubourg , R.Ansari     96-98
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include "machdefs.h"
#include <strings.h>
#include <iostream.h>
// #include <typeinfo>
#include "utilgeom.h"
#include "pidrawer.h"
#include "pidrwtools.h"

#include "ucckprot.h"

//++
// Class	PIDrawer
// Lib		PI
// include	pidrawer.h
//
//	Classe pour la cration d'objets, capables de se dessiner
//	dans une composante graphique ("PIBaseWdg").
//--
//++
// Links	Voir
// PIBaseWdg
//--

//++
// Titre	Constructeurs et mthodes
//--

//++
// PIDawer()
//	Constructeur.
//--

PIDrawer::PIDrawer()
{
  mFgDeleteOnDetach = false;
  mBWdg = NULL; 
  mGrUC = NULL;
  xW0 = yW0 = 0; xWd = yWd = 100; 
  aXdir = false;  // Vrai si Axe X de Droite vers Gauche 
  aYdir = false;  // Vrai si Axe Y de Haut vers Bas
  aXlog = aYlog = false;   // Vrai -> echelle axe logarithmique
  SetLimits(-1, 1, -1, 1, kAxeDirLtoR, kAxeDirDownUp);
  limitsFixed = 0;
  SetAxesFlags(kAxesNone);   // Pas de trace d'axes par defaut 
  SetAxesAutoFontSize();

  mDndfg = false;  // Pour controle de l'appel de Detach() si delete

  mFgHighLight = false;  // Flag de control d'affichage en mode HighLight


  // Nom par defaut : Le nom de la classe
  //  mName = typeid(*this).name();
  mName = "PIDrawer";
  // Par defaut, pas de fenetre de controle specifique
  mFgSpecContWind = false;
}

PIDrawer::~PIDrawer()
{
  // Desactivation totale de la fenetre de controle specialise 
  // ---> parametre d'appel PIBaseWdgGen* wdg=NULL
  DeactivateControlWindow(NULL);

  list<DrwBWId>::iterator it;
  mDndfg = true;
  for(it = mBWdgList.begin(); it != mBWdgList.end(); it++)
         (*it).wdg->RemoveDrawer((*it).id);
  if (mGrUC)  delete mGrUC;
}

//++
// Titre	Les axes
//--
//++
// void  SetLimits(double xmin, double xmax, double ymin, double ymax, -
//                 int axrl=kAxeDirSame, int ayud=kAxeDirSame)
//	Dfinit les limites du systme de coordonnes.
//|	kAxeDirSame , kAxeDirAuto
//|	kAxeDirLtoR , kAxeDirRtoL  (Axe X)
//|	kAxeDirDownUp , kAxeDirUpDown (Axe Y)	
// void  SetLogScale(bool logx, bool logy) 
//	Dfinition d'chelle logarithmique pour les axes X,Y.
// void  GetAxesConfig(int& xa, int& ya) 
//	Renvoie la configuration des axes.
// bool  isLogScaleX()   isLogScaleY() 
//	Renvoie "true" si chelle logarithmique pour axe X , Y
// double XMin()  XMax()
//	Limites de l'axe X
// double YMin()  YMax()
//	Limites de l'axe Y
// void  SetAxesFlags(unsigned int flags=kAxesNone)
//	Attributs de trac d'axes
//|	kStdAxes , kBoxAxes , kTicks , kIntTicks , kExtTicks 
//|	kMajTicks , kMinTicks , kLabels , kGridOn
//|	kAxesDflt , kAxesNone
// unsigned int GetAxesFlags()
//	Renvoie les attributs de trac d'axes  
// void  DrawAxes(PIGraphicUC* g)
//	Mthode de trac des axes
//--

void
PIDrawer::SetLimits(double xmin, double xmax, double ymin, double ymax,
                    int axrl, int ayud)
{
  if (axrl == kAxeDirAuto) {
    double ff;
    if (xmax < xmin)  { axrl = kAxeDirRtoL;   ff=xmin;  xmin=xmax;  xmax=ff; }
    else  axrl = kAxeDirLtoR;
  }
  if (ayud == kAxeDirAuto) {
    double ff;
    if (ymax < ymin)  { ayud = kAxeDirUpDown;   ff=ymin;  ymin=ymax;  ymax=ff; }
    else  ayud = kAxeDirDownUp;
  }
//  if (xmax <= xmin || ymax <= ymin) {  
//    cerr << "PIDrawer::SetLimits() Error - xmax <= xmin  || ymax <= ymin ! " << endl;
//    return; 
//    }
  
  if (!UC_CheckFinite(xmin, xmax) || !UC_CheckFinite(ymin, ymax)) {
    cerr << "PIDrawer::SetLimits() Error - Non finite value for x/ymin-max ! " << endl;
    return; 
    }

  if (axrl == kAxeDirLtoR)  aXdir = false;
  else if (axrl == kAxeDirRtoL)  aXdir = true;
  if (ayud == kAxeDirDownUp)  aYdir = false;
  else if (ayud == kAxeDirUpDown)  aYdir = true;
  //  cerr << " DBG-SetLimits xmax-xmin= " << xmax-xmin << " xmin,max=" 
  //     << xmin << "," << xmax << endl;
  UC_CheckMinMaxDiff(xmin, xmax);
  //cerr << " DBG-SetLimits ymax-ymin= " << ymax-ymin << " ymin,max=" 
  //     << ymin << "," << ymax << endl;
  UC_CheckMinMaxDiff(ymin, ymax);
  xMin = xmin;
  xMax = xmax;
  yMin = ymin;
  yMax = ymax;
  aXFlg = axrl;  aYFlg = ayud;
  limitsFixed = 1;
}


void
PIDrawer::GetAxesConfig(int& xa, int& ya)
{
xa = (aXdir) ? kAxeDirRtoL : kAxeDirLtoR;
ya = (aYdir) ? kAxeDirUpDown : kAxeDirDownUp;
}

//++
//  void  Draw(PIGraphicUC* g, double xmin, double ymin, double xmax, double ymax)
//	Mthode qui est appel lorsquer l'objet PIDrawer doit rafraichir la zone
//	dfinie par "xmin-xmax" , "ymin-ymax" dans le systme de coordonnes du
//	PIDrawer.
//  void  Refresh()
//	Rafrachit le dessin du PIDrawer sur tous les PIBaseWdg auxquel il est attach.
//  void  UpdateLimits()
//	Cette mthode doit calculer les limites (X/Y Min-Max) prferes par l'objet
//	et doit appeler "SetLimits()". L'implementation par dfaut ne fait rien.
//  void  HighLight(bool fgh)
//      Rafraichit le dessin du PIDrawer sur tous les PIBaseWdg auxquel il est attach,
//	en mode "HighLight" si "fgh==true", en mode normal sinon.
//  void  AppendTextInfo(string& info, double xmin, double ymin, double xmax, double ymax)
//	Mthode qui met  jour la chane "info", avec les informations (textuelles) de 
//	la zone dfinie par "xmin-xmax" , "ymin-ymax".
//--

void
PIDrawer::UpdateLimits()
{
// Ne fait rien !
  return;
}

void
PIDrawer::Refresh()
{
  list<DrwBWId>::iterator it;
  for(it = mBWdgList.begin(); it != mBWdgList.end(); it++)
         (*it).wdg->CallDrawer((*it).id);

}
void
PIDrawer::HighLight(bool fgh)
{
  mFgHighLight = fgh;
  Refresh();
  mFgHighLight = false;
}

//++
// Titre	Les attributs graphiques
//--
//++
// void  SetColAtt(PIColors fg=PI_NotDefColor, PIColors bg=PI_NotDefColor)
//	Modifie la couleur d'avant-plan et de fond par dfaut
// void  SetLineAtt(PILineAtt lat=PI_NotDefLineAtt)
//	Modifie l'attribut type de ligne
// void  SetFontAtt(PIFontSize fsz=PI_NotDefFontSize, PIFontAtt fat=PI_NotDefFontAtt)
//	Selection taille et attribut de fonte 
// void  SetFont(PIFontName fn, PIFontSize fsz=PI_NotDefFontSize, PIFontAtt fat=PI_NotDefFontAtt)
//	Selection de type, taile et attribut de fonte
// void  SetMarkerAtt(int sz=-1, PIMarker mrk=PI_NotDefMarker)
//	L'attribut type et taille de marker
// void  SetColMapId(CMapId cid=CMAP_OTHER, bool rev=false)
//	L'attribut choix de la table de couleurs, et son flag d'inversion
// void  SelGraAtt(PIGraphicUC* g)
//	Modifie les attributs graphiques de "g"  partir des attributs courant
//	de l'objet "PIDrawer"
// 
// PIColors  GetFgColAtt()
//	Renvoie l'attribut de couleur d'avant-plan
// PIColors  GetBgColAtt()
//	Renvoie l'attribut de couleur d'arrire-plan
// PILineAtt  GetLineAtt()
//	Renvoie l'attribut de ligne
// PIFontSize GetFontSz()
//	Renvoie l'attribut de taille de fonte
// PIFontAtt  GetFontAtt()
//	Renvoie l'attribut de type de fonte
// int  GetMarkerSz()
//	Renvoie l'attribut de taille de marker
// PIMarker GetMarker()
//	Renvoie l'attribut de type de marker
// CMapId GetColMapId()
//	Renvoie l'attribut de type de table de couleur
// CMapId GetColMapId(bool & rev)
//	Renvoie l'attribut de type de table de couleur, 
//	et le flag d'inversion de la table de couleur
//--


void 
PIDrawer::SelGraAtt(PIGraphicUC* g)
{
  PIColors fcol = GetGraphicAtt().GetFgColor(); 
  if (fcol != PI_NotDefColor)    g->SelForeground(fcol);
  PIColors bcol = GetGraphicAtt().GetBgColor(); 
  if (bcol != PI_NotDefColor)    g->SelBackground(bcol);
  if (GetGraphicAtt().GetLineAtt() != PI_NotDefLineAtt)  
    g->SelLine(GetGraphicAtt().GetLineAtt());
  if ( (GetGraphicAtt().GetFontName() != PI_DefaultFont) ||
       (GetGraphicAtt().GetFontAtt() != PI_NotDefFontAtt) )
    g->SelFont(GetGraphicAtt().GetFont());
  if (GetGraphicAtt().GetMarker() != PI_NotDefMarker) 
    g->SelMarker(GetGraphicAtt().GetMarkerSize(), GetGraphicAtt().GetMarker());
}


void
PIDrawer::Attach(PIBaseWdgGen* wdg, int id)
{
  if (!wdg) return;
  DrwBWId bwi;
  bwi.id = id;  bwi.wdg = wdg;
  mBWdgList.push_back(bwi);
  return;
}

void
PIDrawer::Detach(PIBaseWdgGen* wdg, int id)
{
  if (mDndfg) return;
  if (!wdg) return;

  // Desactivation de la fenetre de controle specialisee pour la 
  // base-widget duquel on se detache 
  DeactivateControlWindow(wdg);

  list<DrwBWId>::iterator it;
  for(it = mBWdgList.begin(); it != mBWdgList.end(); it++)
     if ( ((*it).id == id) && ((*it).wdg == wdg) ) { mBWdgList.erase(it);  break; }
  if (mFgDeleteOnDetach && (mBWdgList.size() == 0))  delete this;
  return;
}

void
PIDrawer::ShowControlWindow(PIBaseWdgGen* wdg)
{
  // On affiche par defaut la fenetre de gestion des attributs graphiques
   PIDrwTools::ShowPIDrwTools(wdg);
}

//--------------------------------------------------------------
// Methode de desactivation de la fenetre de controle specialise
// Cette methode est appellee dans deux cas :
// 1/ lors de la destruction du drawer
//    Dans ce cas, le parametre PIBaseWdgGen* wdg = NULL
// 2/ Lorsque le drawer est detache d'un PIBaseWdg 
//    Dans ce cas, le parametre PIBaseWdgGen* wdg pointe vers le 
//    PIBaseWdg duquel le drawer est detache
// Evidemment, "this" pointe toujours sur le drawer qui est 
// en cours de destruction ou de detachement
//--------------------------------------------------------------
void
PIDrawer::DeactivateControlWindow(PIBaseWdgGen* wdg)
{
  // Par defaut, il n'y a pas de fenetre de controle specialise
  // Donc, on ne fait rien 
}

int
PIDrawer::DecodeOptionString(vector<string> & opt, bool rmdecop)
{
  if (opt.size() < 1)  return(0);  
  PIGraphicAtt gratt;
  int ndec = gratt.DecodeAttStrings(opt, rmdecop);
  UpdateGraphicAtt(gratt);
  return(ndec);
}

void
PIDrawer::GetOptionsHelpInfo(string& info)
{
  info += " ------- PIDrawer options help info ---------- \n";
  info += ">> Colors: defcol black white grey red blue green yellow \n";
  info += "           magenta cyan turquoise navyblue orange siennared purple \n";
  info += "           limegreen gold violet violetred blueviolet darkviolet \n";
  info += ">> Lines:  defline normalline thinline thickline dashedline thindashedline \n";
  info += "           thickdashedline dottedline thindottedline thickdottedline \n";
  info += ">> Font Att: deffontatt normalfont boldfont italicfont bolditalicfont  \n";
  info += "             smallfont smallboldfont smallitalicfont smallbolditalicfont \n";
  info += "             bigfont bigboldfont bigitalicfont bigbolditalicfont \n";
  info += "             hugefont  hugeboldfont hugeitalicfont hugebolditalicfont \n";
  info += ">> Font Names: deffont courierfont helveticafont timesfont symbolfont  \n";
  info += ">> Marker: dotmarker<S>  plusmarker<S>  crossmarker<S> circlemarker <S> \n";
  info += "           fcirclemarker<S> boxmarker<S> fboxmarker<S> trianglemarker<S> \n";
  info += "           ftrianglemarker<S>  starmarker<S>  fstarmarker<S> \n";
  info += "   with <S> = 1 3 5 7 9 , Example fboxmarker5 , plusmarker9 ... \n";
  info += ">> ColorTables: defcmap  grey32  invgrey32  colrj32  colbr32 \n";
  info += "                grey128  invgrey128  colrj128  colbr128 \n";
  info += "                midas_pastel midas_heat midas_rainbow3 midas_bluered\n";
  info += "                midas_bluewhite midas_redwhite \n";
  info += "                rainbow16 \n";
  info += "   revcmap : This flag reverses ColorMap indexing \n";
  return;
}

PIGraphicUC*
PIDrawer::SetDrwWdg(PIBaseWdgGen* drw, int x0, int y0, int dx, int dy, PIGraphicGen* g)
{
  mBWdg = drw;
  xW0 = x0;   yW0 = y0;
  xWd = dx;   yWd = dy;
  if (mGrUC)  delete mGrUC;
  mGrUC = new PIGraphicUC(g, x0, y0, dx, dy);
  mGrUC->SetUCS(xMin, xMax, yMin, yMax, aXFlg, aYFlg);
  if (aXlog || aYlog) mGrUC->SetLogScale(aXlog, aYlog);
  if (mFgHighLight) mGrUC->SelGOMode(PI_GOInvert); // $CHECK$ A changer , Reza 07/2001
  return(mGrUC);
}

void
PIDrawer::CalcTicks()
{
  int ntick_x = (aXlog) ? 6 : 10;
  BestTicks(xMin,xMax,ntick_x,xMajTickStep,xFirstMajTick);
  xMinTickStep = xMajTickStep/5;
  xFirstMinTick = floor(xMin / xMinTickStep) * xMinTickStep;
  if(xFirstMinTick < xMin) xFirstMinTick += xMinTickStep;

  int ntick_y = (aYlog) ? 6 : 12;
  BestTicks(yMin,yMax,ntick_y,yMajTickStep,yFirstMajTick);
  yMinTickStep = yMajTickStep/5;
  yFirstMinTick = floor(yMin / yMinTickStep) * yMinTickStep;
  if(yFirstMinTick < yMin) yFirstMinTick += yMinTickStep;

  yMajTickLen = (xMax-xMin)/100;
  yMinTickLen = (xMax-xMin)/250;
  xMajTickLen = (yMax-yMin)/100;
  xMinTickLen = (yMax-yMin)/250;
}

void
PIDrawer::SetAxesFlags(unsigned int flags)
{
  axesFlags = flags;
  if (axesFlags & (kIntTicks | kExtTicks | kMajTicks | kMinTicks))
    axesFlags |= kTicks;
  if ((axesFlags & (kTicks | kIntTicks | kExtTicks)) == kTicks)
    axesFlags |= kIntTicks | kExtTicks;
  if ((axesFlags & (kTicks | kMajTicks | kMinTicks)) == kTicks)
    axesFlags |= kMajTicks;
}


void
PIDrawer::DrawAxes(PIGraphicUC* g)
{
  g->NoClip();
  if (GetGraphicAtt().GetLineAtt() == PI_NotDefLineAtt)  g->SelLine(PI_ThinLine);
  else g->SelLine(GetGraphicAtt().GetLineAtt());
  if (GetGraphicAtt().GetColor() == PI_NotDefColor) g->SelForeground(PI_Black);
  else g->SelForeground(GetGraphicAtt().GetColor());
  //  On calcule les ticks 
  CalcTicks();

  unsigned int flags = axesFlags;

  if (flags & kStdAxes) {

    // Les axes
    
    g->DrawLine(xMin, (yMin+yMax)/2., xMax, (yMin+yMax)/2.);
    g->DrawLine((xMin+xMax)/2., yMin, (xMin+xMax)/2., yMax);

    // La grille en pointilles

    if (flags & kGridOn) DrawGrid(g);
  
    // Les ticks majeurs
  
    if (flags & kMajTicks) {
        DrawHTicks(g, (yMin+yMax)/2., xMajTickLen, xMajTickLen, xFirstMajTick, xMajTickStep);
        DrawVTicks(g, (xMin+xMax)/2., yMajTickLen, yMajTickLen, yFirstMajTick, yMajTickStep);
    }
    
    // Les ticks mineurs
  
    if (flags & kMinTicks) {
        DrawHTicks(g, (yMin+yMax)/2., xMinTickLen, xMinTickLen, xFirstMinTick, xMinTickStep);
        DrawVTicks(g, (xMin+xMax)/2., yMinTickLen, yMinTickLen, yFirstMinTick, yMinTickStep);
    }
    
    // Les labels
    
    if (flags & kLabels) {
      if (!aYdir)
        DrawHLabels(g, (yMin+yMax)/2.-xMajTickLen*2, xFirstMajTick, xMajTickStep,PI_VerticalTop);
      else 
        DrawHLabels(g, (yMin+yMax)/2.+xMajTickLen*2, xFirstMajTick, xMajTickStep,PI_VerticalTop);
      if (!aXdir)
        DrawVLabels(g, (xMin+xMax)/2.-yMajTickLen*2, yFirstMajTick, yMajTickStep,PI_HorizontalRight);
      else 
        DrawVLabels(g, (xMin+xMax)/2.+yMajTickLen*2, yFirstMajTick, yMajTickStep,PI_HorizontalRight);
    }
    
  }
  
  if (flags & kBoxAxes) {
  
    // La boite
    
    g->DrawLine(xMin, yMin, xMax, yMin);
    g->DrawLine(xMax, yMin, xMax, yMax);
    g->DrawLine(xMax, yMax, xMin, yMax);
    g->DrawLine(xMin, yMax, xMin, yMin);

    // Longueur des ticks
    
    double extXMajTickLen = flags&kExtTicks ? xMajTickLen : 0;
    double intXMajTickLen = flags&kIntTicks ? xMajTickLen : 0;
    double extXMinTickLen = flags&kExtTicks ? xMinTickLen : 0;
    double intXMinTickLen = flags&kIntTicks ? xMinTickLen : 0;
    double extYMajTickLen = flags&kExtTicks ? yMajTickLen : 0;
    double intYMajTickLen = flags&kIntTicks ? yMajTickLen : 0;
    double extYMinTickLen = flags&kExtTicks ? yMinTickLen : 0;
    double intYMinTickLen = flags&kIntTicks ? yMinTickLen : 0;

    // La grille en pointilles

    if (flags & kGridOn) DrawGrid(g);

    // Les ticks majeurs
  
    if (flags & kMajTicks) {
        DrawHTicks(g, yMin, intXMajTickLen, extXMajTickLen, xFirstMajTick, xMajTickStep);
        DrawHTicks(g, yMax, extXMajTickLen, intXMajTickLen, xFirstMajTick, xMajTickStep);
        DrawVTicks(g, xMin, extYMajTickLen, intYMajTickLen, yFirstMajTick, yMajTickStep);
        DrawVTicks(g, xMax, intYMajTickLen, extYMajTickLen, yFirstMajTick, yMajTickStep);
    }
    
    // Les ticks mineurs
  
    if (flags & kMinTicks) {
        DrawHTicks(g, yMin, intXMinTickLen, extXMinTickLen, xFirstMinTick, xMinTickStep);
        DrawHTicks(g, yMax, extXMinTickLen, intXMinTickLen, xFirstMinTick, xMinTickStep);
        DrawVTicks(g, xMin, extYMinTickLen, intYMinTickLen, yFirstMinTick, yMinTickStep);
        DrawVTicks(g, xMax, intYMinTickLen, extYMinTickLen, yFirstMinTick, yMinTickStep);
    }


    // Les labels
    
    if (flags & kLabels) {
      if (!aYdir) {
        DrawHLabels(g, g->DeltaUCY(yMin, -xMajTickLen*2), xFirstMajTick, xMajTickStep,PI_VerticalTop);
        //DrawHLabels(g, g->DeltaUCY(yMax,  xMajTickLen*2), xFirstMajTick, xMajTickStep,PI_VerticalBottom);
      }
      else {
        //DrawHLabels(g, g->DeltaUCY(yMin, -xMajTickLen*2), xFirstMajTick, xMajTickStep,PI_VerticalBottom);
        DrawHLabels(g, g->DeltaUCY(yMax,  xMajTickLen*2), xFirstMajTick, xMajTickStep,PI_VerticalTop);
      }
      if (!aXdir) {
        DrawVLabels(g, g->DeltaUCX(xMin, -yMajTickLen*2), yFirstMajTick, yMajTickStep,PI_HorizontalRight);
        //DrawVLabels(g, g->DeltaUCX(xMax,  yMajTickLen*2), yFirstMajTick, yMajTickStep,PI_HorizontalLeft);
      }
      else {
        //DrawVLabels(g, g->DeltaUCX(xMin, -yMajTickLen*2), yFirstMajTick, yMajTickStep,PI_HorizontalLeft);
        DrawVLabels(g, g->DeltaUCX(xMax,  yMajTickLen*2), yFirstMajTick, yMajTickStep,PI_HorizontalRight);
      }
    }
  }
  g->Clip();
}

  

void
PIDrawer::Draw(PIGraphicUC* g, double /*xmin*/, double /*ymin*/, double /*xmax*/, double /*ymax*/)
{
}

void
PIDrawer::AppendTextInfo(string& info, double /*xmin*/, double /*ymin*/, double /*xmax*/, double /*ymax*/)
{
}

void
PIDrawer::GetClickInfo(string& info,double x,double y,double x0,double y0,bool fgdiff)
{
}

void
PIDrawer::DrawHTicks(PIGraphicUC* g, double y, double tickUp, double tickDown, double xBeg, double xStep)
{
  for (double x=xBeg; x<=xMax+xStep/10.; x += xStep)
    g->DrawLine(x, g->DeltaUCY(y, -tickDown), x, g->DeltaUCY(y, tickUp));
}

void
PIDrawer::DrawVTicks(PIGraphicUC* g, double x, double tickLeft, double tickRight, double yBeg, double yStep)
{
  for (double y=yBeg; y<=yMax; y += yStep)
    g->DrawLine(g->DeltaUCX(x, -tickLeft), y, g->DeltaUCX(x,tickRight), y);
}

void
PIDrawer::DrawHLabels(PIGraphicUC* g, double y, double xBeg, double xStep, unsigned long just)
{
  double fsz = xMajTickLen*3.5;
  if(axesAFSz) g->SelFontSz(fsz);

  // Choix du bon format pour les labels des axes;
  char label[64]; string format; double fac=1.; bool p10=false;
  int npuiss = BonFormatAxes(xBeg,xMax,xStep,format,2,1);
  if(npuiss<-2 || npuiss>3) {p10 = true; fac=pow(10.,(double)npuiss);}
  else BonFormatAxes(xBeg,xMax,xStep,format,0,1);

  double xOffset = 0;
  double xlastlabelfin = xBeg - 2.*(xMax-xBeg);
  for(double x=xBeg; x<=xMax+xStep/10.; x+=xStep) {
    //Attention erreur d'arrondi x->0 (on code 5.1698e-26 au lieu de 0)
    double xx = (fabs(x/xStep)<1.e-5) ? 0.: x;
    sprintf(label,format.c_str(),xx/fac);
    for(int kk=0;kk<64;kk++) if(label[kk]==' ') {label[kk]='\0'; break;}
    double largeur = g->CalcStringWidth(label);
    if(aXdir) xOffset = largeur/2; else xOffset=-largeur/2;
    if(x+xOffset > xlastlabelfin) {
      g->DrawString(x,y,label,PI_HorizontalCenter|just);
      xlastlabelfin = x + xOffset + 1.1*largeur;
    }
  }

  if(p10) {
    PIGrCoord asc,desc;
    double h = g->GetFontHeight(asc,desc);
    if((aYdir && (just&PI_VerticalBottom)) || (!aYdir && (just&PI_VerticalTop)) ) h = -h; 
    double xm = (aXdir)? xMin: xMax;
    double ym = g->DeltaUCY(y,1.5*h);
    sprintf(label,"%d",npuiss);
    g->DrawCompString(xm,ym,"x 10",label,NULL,PI_HorizontalCenter|just);
  }

}

void
PIDrawer::DrawVLabels(PIGraphicUC* g, double x, double yBeg, double yStep, unsigned long just)
{
  double fsz = xMajTickLen*3.5;
  if(axesAFSz) g->SelFontSz(fsz);

  // Choix du bon format pour les labels des axes;
  char label[64]; string format; double fac=1.; bool p10=false;
  int npuiss = BonFormatAxes(yBeg,yMax,yStep,format,2,1);
  if(npuiss<-2 || npuiss>3) {p10 = true; fac=pow(10.,(double)npuiss);}
  else BonFormatAxes(yBeg,yMax,yStep,format,0,1);

  for(double y=yBeg; y<=yMax; y += yStep) {
    double yy = (fabs(y/yStep)<1.e-5) ? 0.: y;
    sprintf(label,format.c_str(),yy/fac);
    for(int kk=0;kk<64;kk++) if(label[kk]==' ') {label[kk]='\0'; break;}
    g->DrawString(x,y,label,PI_VerticalCenter|just);
  }

  if(p10) {
    PIGrCoord asc,desc;
    double h = g->GetFontHeight(asc,desc);
    if(aYdir) h = -h; 
    double ym = (aYdir)? yMin: yMax; ym = g->DeltaUCY(ym,h);
    sprintf(label,"%d",npuiss);
    g->DrawCompString(x,ym,"x 10",label,NULL,PI_VerticalBottom|just);
  }

}

void
PIDrawer::DrawGrid(PIGraphicUC* g)
{
  PILineAtt savlineatt = g->GetLineAtt();
//  g->SelLine(PI_ThinDashedLine);
  g->SelLine(PI_ThinDottedLine);

  for (double x=xFirstMajTick; x<=xMax; x += xMajTickStep)
    g->DrawLine(x, yMin, x, yMax);

  for (double y=yFirstMajTick; y<=yMax; y += yMajTickStep)
    g->DrawLine(xMin, y, xMax, y);

  g->SelLine(savlineatt);
}

///////////////////////////////////////////////////////////////////////////
/* --Methode-Static-- */
void PIDrawer::BestTicks(double xmin,double xmax,int nticks
                        ,double& steptick,double& xfirsttick)
// *** Calcul de l'intervalle entre les ticks et de la valeur du premier tick
{
double d=xmax-xmin; if(d<1.e-100) d=1.e-100; //if (d<1.e-39) d=1.e-39;
double ld  = log10(d);
double fld = floor( ((ld<0.)? -ld: ld) );
double del,del0;
if(ld>=0.) {fld-=1.; del0=del=pow(10.,fld);}
  else     {fld+=2.; del0=del=pow(10.,-fld);}
// *** Intervalle entre les ticks
// xmin  xmax    d       ld       fld -->fld  del0
// 1     1500    1499    3.17     3      2    10^2
// 1     9500    9499    3.98     3      2    10^2
// 1     1.005   0.005   -2.3     2      4    10^-4
// 1     1.995   0.995   -0.0022  0      2    10^-2
// - Et recherche de la valeur del={del0,2*del0,...,20*del0}
// telle que "nticks*del" soit le plus petit nombre ">=d"
// Par exemple, si nticks=10:
// 1-/ pour le 2sd cas ou d=9499 et del0=100 :
// del        =    100    200    500    1000    2000
// nticks*del =   1000   2000   5000   10000   20000
// d/del      =   94.9   47.4   18.9 | 9.499   4.749  ==> majt = 1000
// 2-/ pour le 3ieme cas ou d=5e-3 et del0=1e-4 :
// del        =  1e-4   2e-4   5e-4  1e-3   2e-3
// nticks*del =  1e-3   2e-3   5e-3  1e-2   2e-2
// d/del      =  50     25   | 10    5      2.5       ==> majt = 5e-4
int k=0;
double fac[4] = {2.,5.,10.,20.};
while(d/del>(double)nticks && k<4 ) {del=fac[k]*del0; k++;}
steptick=del;
//***  Valeur du premier tick
xfirsttick = floor(xmin/steptick); if(xmin<0.) xfirsttick+=1.;
xfirsttick *= steptick;
if(xfirsttick<xmin) xfirsttick += steptick;
}

/* --Methode-Static-- */
int PIDrawer::BonFormatAxes(double xmin,double xmax,double xstep
                           ,string& format,int typf,int add_digit)
// *** Calcul format optimal pour ecrire les labels numeriques des axes:
// ---- Input
// . xmin,xmax : limites du plot sur l'axe considere.
// . xstep : distance entre les ticks.
// . add_digit : nombre de digits a ajouter au nombre de digits minimum.
// . typf : type de format en sortie
//        0 : format optimum %-nn.mme ou %-nn.mmf selon valeurs
//        1 : format %-nn.mme
//        2 : format %-nn.mmf pour imprimer x/10^npuiss
//            tel que x/10^npuiss soit entre 0 et 10
// ---- Output
// . format : le format d'impression
// ---- Return:
// Si typ=0 ou 1 
//   "ndig" : nombre de digits necessaires pour distinguer
//            les valeurs xmin+k*dx (<=xmax)
// Si typ=2
//   "npuiss" : tel que x/10^npuiss soit entre 0 et 10
//              Dans ce cas le format est celui qui imprime x/10^npuiss
{
 format = "%-5g";  // format par default
 if(xmin>=xmax) {if(typf==2) return 0; else return -1;}

 if(xstep<=0. || xstep>xmax-xmin) xstep = xmax-xmin;

 double axmin=fabs(xmin), axmax=fabs(xmax);
 if(axmin>axmax) swap(axmin,axmax);
 double l10amax = log10(axmax), l10xstep = log10(xstep);
 int il10amax = int(floor(l10amax));

 // choix du type de format
 char ftype = 'e';
 int npuiss = 0;
 if(typf==2) {
   npuiss = il10amax;
   if(npuiss<-300 || npuiss>300) {
     ftype='e'; npuiss=0;
   } else {
     // On recalcule les valeurs de decision pour  axmax/10^npuiss, xstep/10^npuiss
     l10amax -= (double)npuiss; l10xstep -=  (double)npuiss;
     il10amax = int(floor(l10amax));
     ftype = 'f';
   }
 } else if(typf==1) {
   ftype='e';
 } else {
   ftype='e';
   // On evite d'ecrire +a.bbbe+ccc -> format %f
   // Ex: 1.2345e+2 -> 123.45   /  -1.2345e+2 -> -123.45
   //     1.2345e-1 -> 0.12345  /  -1.2345e-1 -> -0.12345
   if((axmin>=1e-4 || axmin==0.) && axmax<1e4) ftype='f';
 }

 //printf("BonFormatAxes[npuiss=%d]: xmin=%-21.14e xmax=%-21.14e\n",npuiss,xmin,xmax);
 //printf("        xstep=%-21.14e log10(xstep/10^%d)=%g\n",xstep,npuiss,l10xstep);
 //printf("        axmax=%-21.14e log10(axmax/10^%d)=%g diff=%g\n"
 //      ,axmax,npuiss,l10amax,l10amax-l10xstep);

 // Nombre de digits necessaires pour ecrire axmax et xstep
 int ndig = il10amax  -  int(floor(l10xstep));
 if(ndig<0) ndig *= -1;  ndig += 1;
 //printf("ndig=%d",ndig);
 
 // Add more digits (or suppress digits)
 ndig += add_digit; if(ndig<0) ndig=0;
 //printf(" + %d  ==> ndig=%d\n",add_digit,ndig);

 // Calcul du bon format
 char str[16];
 if(ftype=='f') { // Calcul du format %-nn.mmf
   int mm=-1, nn;
   if(il10amax<0) { // +0.12345 +0.0012345 +0.0012345 ...
     mm = ndig - il10amax - 1; nn = mm+3;
   } else {         // +1.2345 +12.345 +123.45 ...
     mm = ndig - il10amax - 1; nn = ndig+2;
   }
   //printf("format %%f : mm=%d  nn=%d\n",mm,nn);
   if(mm<0.) mm=0; if(nn<mm+3) nn=mm+3;
   sprintf(str,"%%-%d.%df",nn,mm);
 } else if(ftype=='e') { // Calcul du format %-nn.mme
   //   +d.<--ddd-->e+123
   //   1 2         34567  ==> nn=ndig+7   mm=ndig-1
   sprintf(str,"%%-%d.%de",ndig+7,ndig-1);
 }

 format = str;
 //printf("format=[%s]\n",format.c_str());

 if(typf==2) return npuiss;
 return ndig;
}
