// ArchTOIPipe           (C)     CEA/DAPNIA/SPP IN2P3/LAL
//                               Eric Aubourg
//                               Christophe Magneville
//                               Reza Ansari

/*   Traitement simple des donnees Archeops KS1-3   - Reza Juin 2002

----------------   Exemple d'appel  ---------------------
csh> aksj02 -start 104385384 -end 104399964 -range -500,500 \
            -intoi boloMuV_26 -wtoi 8192 -wclean 512,5 \
	    inputbolo.fits filt.fits xx.ppf
*/



#include "machdefs.h"
#include <math.h>
#include <unistd.h>

#include "toimanager.h"
#include "cgt.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "simtoipr.h"
#include "simoffset.h"
#include "simcleaner.h"
#include "nooppr.h"
#include "timing.h"
#include "histinit.h"
#include "psighand.h"
#include <stdexcept>

void Usage(bool fgerr)
{
  cout << endl;
  if (fgerr) {
    cout << " aksj02 : Argument Error ! aksj02 -h for usage " << endl; 
    exit(1);
  }
  else {
    cout << "\n Usage : aksj02 [-intoi toiname] [-start snb] [-end sne] \n" 
	 << "         [-dbg] [-wtoi sz] [-wclean wsz,nbw] [-range min,max] \n"
	 << "         [-soff wsz,nptfit,degpol] [-soffnt PPFName] \n"
	 << "         [-bgalcut bmin,bmax] [-bgal toiname] [-bgalfile pointFitsName] \n"
	 << "         [-soff2 wsz,nptfit,degpol] [-soff2nt PPFName] \n"
	 << "         [-bgal2cut bmin,bmax] \n"
	 << "         [-gfilt wsz,sigma] \n" 
	 << "         [-nooutflg] [-wrtms] [-wrtifo] [-prstat] [-useseqbuff] \n"
	 << "         inFitsName outFitsName \n" 
	 << "   -dbg : sets TOISeqBuffered debug level to 1 \n"
	 << "   -start snb : sets the start sample num \n"
	 << "   -end sne : sets the end sample num \n"
	 << "   -range min,max : sets the acceptable range for intoi \n"
	 << "              default= -16000,16000\n"
	 << "   -intoi toiName : select input TOI name (def bolo)\n"
	 << "   -wtoi sz : sets TOISeqBuff buffer size (def= 8192)\n"
	 << "   -wclean wsz,nbw : sets cleaner window parameters (256,5) \n"
	 << "   -soff wsz,nptfit,degpol : set SimpleOffsetEstimator parameters\n"
	 << "   -soffnt PPFName : Writes out SimpleOffsetEstimator NTuple \n"
         << "   -bgalcut bmin,bmax: Galactic latitude cut (in degree) for SimpleOffsetEstimator\n"
         << "   -bgal toiname: Galactic latitude toiname \n"
         << "   -bgalfile pointFitsName: Pointing FITS file name (for gal. latitude) \n"
	 << "   -soff2 wsz,nptfit,degpol : set SimpleOffsetEstimator_2 parameters\n"
	 << "   -soff2nt PPFName : Writes out SimpleOffsetEstimator_2 NTuple \n"
         << "   -bgal2cut bmin,bmax: Galactic latitude cut (in degree) for SimpleOffsetEstimator_2\n"
	 << "   -gfilt wsz,sigma : Activate gaussian filter (par= window size, sigma) \n"
      //	 << "   -wfft sz : Activate Fourier filter and sets its width \n"
      //	 << "   -keepfft n : Keeps n spectra (if Fourier filter activated) \n" 
      //	 << "   -killfreq bf,nharm,sigf : kills nharm frequency, basefreq=bf \n"
      //         << "              with a gaussian with width=sigf \n"
	 << "   -nooutflg : Don't write flags to output FITS \n"
	 << "   -wrtms : Write mean/sigma TOI's from cleaner \n"
	 << "   -wrtifo : Write incopie flagged  TOI's from cleaner/ offset from SimpleOffsetEstimator \n"
	 << "   -prstat : PrintStat with ProcSampleCounter \n"
	 << "   -useseqbuff : Use TOISeqBuffered TOI's (default: TOISegmented) \n"
	 << endl;
  }
  if (fgerr) exit(1);
}

int main(int narg, char** arg) {

  if ((narg > 1) && (strcmp(arg[1],"-h") == 0) ) Usage(false);

  cout << "aksj02 starting - Decoding arguments " << " narg=" << narg << endl;

  bool fgdbg = false;
  bool fgsetstart = false;

  bool fgprstat = false;
  bool fgsegmented = true;

  int dbglev = 0;

  int wtoi = 8192;
  int wdegli = 512;
  
  int wfft = 4096;
  int keepfft = 0;
  int nmax = 10;
  int istart = 0;
  int iend = 0;

  double range_min = -16000;
  double range_max = 16000.;

  // cleaner parameters
  double clean_wsz = 256;
  double clean_nbw = 5;

  // SimpleOffsetEstimator parameters
  int soff_mwsz = 256;
  int soff_nptfit = 5;
  int soff_degpol = 2;
  bool soff_nt = false;
  string soff_nt_ppfname = "";

  // SimpleOffsetEstimator (2) parameters
  bool soff2_dofg = false;
  int soff2_mwsz = 256;
  int soff2_nptfit = 5;
  int soff2_degpol = 2;
  bool soff2_nt = false;
  string soff2_nt_ppfname = "";

  // Gaussian filter parameters
  bool gfilt_fg = false;
  int gfilt_wsz = 16;
  double gfilt_sigma = 2.;

  // Coordinate file for galactic cut
  string pointfile = "pointgal.fits";
  double bmin = -10;
  double bmax = 10.;
  bool bgalcut = false;
  string bgaltoi = "bgal";
  bool bgalfile = false;
  double bmin2 = -10;
  double bmax2 = 10.;
  bool bgal2cut = false;

  // File names 
  string infile;
  string outfile;
  string outppfname;
  string intoi = "bolo";

  bool fg_wrtflag = true;
  bool fg_wrtms = false;
  bool fg_wrtifo = false;


  bool fg_f_filt = false;
  bool fg_killfreq = false;
  int bf_killfreq = 1;
  int nharm_killfreq = 1;
  double sigf_killfreq = 1.;

  if (narg < 4) Usage(true);
  int ko=1;
  // decoding arguments 
  for(int ia=1; ia<narg; ia++) {
    if (strcmp(arg[ia],"-start") == 0) {
      if (ia == narg-1) Usage(true);  // -start est suivi d'un argument 
      istart = atoi(arg[ia+1]); ia++;
      fgsetstart = true;
    }    
    else if (strcmp(arg[ia],"-end") == 0) {
      if (ia == narg-1) Usage(true);  
      iend = atoi(arg[ia+1]); ia++;
    }    
    else if (strcmp(arg[ia],"-wtoi") == 0) {
      if (ia == narg-1) Usage(true);  
      wtoi = atoi(arg[ia+1]); ia++;
    }    
    else if (strcmp(arg[ia],"-wclean") == 0) {
      if (ia == narg-1) Usage(true);  
      wdegli = atoi(arg[ia+1]); ia++;
      sscanf(arg[ia+1],"%d,%d",  &clean_wsz, &clean_nbw);
    }    
    else if (strcmp(arg[ia],"-soff") == 0) {
      if (ia == narg-1) Usage(true);
      sscanf(arg[ia+1],"%d,%d,%d",  &soff_mwsz, &soff_nptfit, &soff_degpol);
      ia++;
    }
    else if (strcmp(arg[ia],"-soffnt") == 0) { 
      soff_nt = true;
      soff_nt_ppfname = arg[ia+1];  ia++;
    }
    else if (strcmp(arg[ia],"-soff2") == 0) {
      if (ia == narg-1) Usage(true);
      soff2_dofg = true;
      sscanf(arg[ia+1],"%d,%d,%d",  &soff2_mwsz, &soff2_nptfit, &soff2_degpol);
      ia++;
    }
    else if (strcmp(arg[ia],"-soff2nt") == 0) { 
      soff2_nt = true;
      soff2_nt_ppfname = arg[ia+1];  ia++;
    }
    else if (strcmp(arg[ia],"-gfilt") == 0) {
      if (ia == narg-1) Usage(true);
      sscanf(arg[ia+1],"%d,%lf", &gfilt_wsz, &gfilt_sigma);
      gfilt_fg = true;
      ia++;
    }    
    else if (strcmp(arg[ia],"-bgalcut") == 0) {
      if (ia == narg-1) Usage(true);
      sscanf(arg[ia+1],"%lf,%lf", &bmin, &bmax);
      bgalcut = true;
      ia++;
    }
    else if (strcmp(arg[ia],"-bgal2cut") == 0) {
      if (ia == narg-1) Usage(true);
      sscanf(arg[ia+1],"%lf,%lf", &bmin2, &bmax2);
      bgal2cut = true;
      ia++;
    }
    else if (strcmp(arg[ia],"-bgalfile") == 0) {
      if (ia == narg-1) Usage(true);
      pointfile = arg[ia+1];
      bgalfile = true;
      ia++;
    }
    else if (strcmp(arg[ia],"-bgal") == 0) {
      if (ia == narg-1) Usage(true);
      bgaltoi = arg[ia+1];
      ia++;
    }
    else if (strcmp(arg[ia],"-range") == 0) {
      if (ia == narg-1) Usage(true);
      sscanf(arg[ia+1],"%lf,%lf",&range_min, &range_max);
      ia++;
    }    
    else if (strcmp(arg[ia],"-intoi") == 0) {
      if (ia == narg-1) Usage(true);  
      intoi = arg[ia+1]; ia++;
    }    
    else if (strcmp(arg[ia],"-dbg") == 0)  fgdbg = true;
    else if (strcmp(arg[ia],"-nooutflg") == 0)  fg_wrtflag = false;
    else if (strcmp(arg[ia],"-wrtms") == 0)  fg_wrtms = true;
    else if (strcmp(arg[ia],"-wrtifo") == 0)  fg_wrtifo = true;

    else if (strcmp(arg[ia],"-prstat") == 0)  fgprstat = true;
    else if (strcmp(arg[ia],"-useseqbuff") == 0)  fgsegmented = false;

    else { ko = ia; break; }  // Debut des noms
  }

  if (iend < istart) iend = istart+wtoi*(nmax+5);
  if ((narg-ko) < 2)  Usage(true); 
  infile = arg[ko];
  outfile = arg[ko+1];
  //  outppfname = arg[ko+2];

  cout << " Initializing SOPHYA ... " << endl;
  SophyaInit();
  SophyaConfigureSignalhandling(true);

  InitTim();
  if ((soff2_dofg == true) && (wtoi < 32768))      wtoi = 32768;
  if (!bgalfile)  bgalcut = bgal2cut = false;

  cout << ">>>> aksj02: Infile= " << infile << " outFile=" 
       << outfile << endl; 
  cout << ">>>> Window Size WTOI= " << wtoi << " WDEGLI= " << wdegli
       << "  iStart= " << istart << " iEnd= " << iend << endl;
  cout << ">>>> InTOIName= " << intoi << endl;

  try {
    TOIManager* mgr = TOIManager::getManager();
  
    //  mgr->setRequestedSample(11680920,11710584);
    //  mgr->setRequestedSample(104121000, 104946120);
    if (fgsetstart) 
      mgr->setRequestedSample(istart, iend);

    cout << "> Creating FITSTOIReader object - InFile=" << infile << endl;
    FITSTOIReader r(infile); 

    cout << "> Creating SimpleCleaner() " << endl;
    SimpleCleaner cleaner(clean_wsz, clean_nbw);
    cout << " Setting Range for cleaner: " << range_min 
	 << " - " << range_max << endl;
    cleaner.SetRange(range_min, range_max);

    cout << "> Creating SimpleOffsetEstimator(" << soff_mwsz << "," << soff_nptfit 
	 << "," << soff_degpol << ")" << endl;
    SimpleOffsetEstimator offes(soff_mwsz, soff_nptfit, soff_degpol);
    offes.SavePolyNTuple(soff_nt, soff_nt_ppfname);

    SimpleOffsetEstimator* offes2 = NULL;
    if (soff2_dofg) {
      cout << "> Creating 2nd SimpleOffsetEstimator(" << soff2_mwsz << "," << soff2_nptfit 
	   << "," << soff2_degpol << ")" << endl;
      offes2 = new SimpleOffsetEstimator(soff2_mwsz, soff2_nptfit, soff2_degpol);
      offes2->SavePolyNTuple(soff2_nt, soff2_nt_ppfname);
    }

    SimpleFilter *filt = NULL;
    if (gfilt_fg) {
      cout << "> Creating a GaussianFilter SimpleFilter Object " << endl;
      double G_sigma = gfilt_sigma;
      double G_a = 1./(G_sigma*sqrt(M_PI*2.));
      filt = new SimpleFilter(gfilt_wsz, SimpleFilter::GaussFilter, G_a, G_sigma);
    }
         
    FITSTOIReader* rbgal = NULL;
    if (bgalfile) {  // if Galactic cut 
      cout << "> Creating bgal FITSTOIReader object - InFile=" << pointfile 
	   << " bgaltoiname= " << bgaltoi << endl;
      rbgal = new FITSTOIReader(pointfile); 
    }
    
    if (bgalfile && bgalcut) { 
      cout << "  offes.SetBGalCut( " << bmin << "," << bmax << ")" << endl;
      offes.SetBGalCut(bmin, bmax);      
    }
    if (soff2_dofg && bgalfile && bgal2cut) { 
      cout << "  offes2.SetBGalCut( " << bmin2 << "," << bmax2 << ")" << endl;
      offes2->SetBGalCut(bmin, bmax);      
    }

    cout << "> Creating FITSTOIWriter OutFitsName= " << outfile << endl;
    FITSTOIWriter w(outfile);

    CGT plombier(fgsegmented, wtoi);
    plombier.SetDebugLevel(dbglev);

    cout << "> Connecting Processors through plombier ... " << endl;
    string inname = "in";
    string outname = "out";
    plombier.Connect(r, intoi, cleaner, inname);
    plombier.Connect(cleaner, "out", offes, "in");

    if (soff2_dofg && bgalcut && bgal2cut ) {
      inname = "bgal";
      outname = "bgalcopie";
      plombier.Connect(*rbgal, bgaltoi, offes, inname);      
      plombier.Connect(offes, outname, *offes2, inname);      
    }
    else if (soff2_dofg && bgal2cut) {
      inname = "bgal";
      plombier.Connect(*rbgal, bgaltoi, *offes2, inname);      
    }
    else if (bgalcut) {
      inname = "bgal";
      plombier.Connect(*rbgal, bgaltoi, offes, inname);      
    }

    if (soff2_dofg) {
      inname = "in";
      outname = "out";
      plombier.Connect(offes, outname, *offes2, inname);      
    }

    if (gfilt_fg) {
      if (soff2_dofg)
	plombier.Connect(*offes2, "out", *filt, "in");
      else 
	plombier.Connect(offes, "out", *filt, "in");
      plombier.Connect(*filt, "out", w, "outfcln", "", 0, fg_wrtflag);
    }
    else {
      if (soff2_dofg)
	plombier.Connect(*offes2, "out", w, "outcln", "", 0, fg_wrtflag);
      else 
	plombier.Connect(offes, "out", w, "outcln", "", 0, fg_wrtflag);
    }

    if (fg_wrtifo) {
      plombier.Connect(offes, "incopie", w, "inflg", "", 0, fg_wrtflag);
      plombier.Connect(offes, "offset", w, "offset");
      if (soff2_dofg)
	plombier.Connect(*offes2, "offset", w, "offset2");
    }
    if (fg_wrtms) {
      plombier.Connect(cleaner, "mean", w, "mean");
      plombier.Connect(cleaner, "sigma", w, "sigma");
    }

    cout << "> Plombier status before start" << endl;
    cout << plombier ; 

    cout << cleaner;
    cout << offes;
    if (soff2_dofg)  cout << *offes2;
    cout << w;

    PrtTim("starting processors");
    plombier.Start();


    // ------------------- Impression continu de stat ------------------------ 
    if (fgprstat) {
      ProcSampleCounter<SimpleOffsetEstimator> stats(offes);
      stats.InfoMessage() = "aksj02/Info";
      stats.PrintStats();
    }
    // -----------------------------------------------------------------------

    cout << cleaner;
    cout << offes;
    if (soff2_dofg)  cout << *offes2;
    if (gfilt_fg)  cout << *filt;
    cout << w;

    mgr->joinAll();
    PrtTim("End threads");
    if (bgalcut)  delete rbgal;
    if (gfilt_fg)  delete filt;
    if (soff2_dofg)  delete offes2;
  }
  catch (PThrowable & exc) {
    cerr << "\n aksj02: Catched Exception \n" << (string)typeid(exc).name() 
	 << " - Msg= " << exc.Msg() << endl;
  }
  catch (const std::exception & sex) {
    cerr << "\n aksj02: Catched std::exception \n" 
	 << (string)typeid(sex).name() << endl; 
  }
  catch (...) {
    cerr << "\n aksj02: some other exception was caught ! " << endl;
  }

  return(0);
}
