// ArchTOIPipe           (C)     CEA/DAPNIA/SPP IN2P3/LAL
//                               Eric Aubourg
//                               Christophe Magneville
//                               Reza Ansari

/*   Calcul de statistique simples sur TOI  

----------------   Exemple d'appel  ---------------------
csh> toistat -start 104385384 -end 104399964 -range -500,500 \
            -intoi boloMuV_26 -wtoi 8192 -wclean 512,5 \
	    inputbolo.fits filt.fits xx.ppf
*/



#include "machdefs.h"
#include <math.h>
#include <unistd.h>

#include "toimanager.h"
#include "cgt.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "simtoipr.h"
#include "simoffset.h"
#include "simcleaner.h"
#include "nooppr.h"
#include "timing.h"
#include "histinit.h"
#include "ntuple.h"
#include "fitsntuple.h"

#include "psighand.h"
#include <stdexcept>

void Usage(bool fgerr)
{
  cout << endl;
  if (fgerr) {
    cout << " toistat : Argument Error ! toistat -h for usage " << endl; 
    exit(1);
  }
  else {
    cout << " toistat : produce a stat NTuple (mean,sigma) from a TOI" << endl;
    cout << "\n Usage : toistat [-intoi toiname] \n " 
	 << "         [-start snb] [-end sne] [-snimplicit] \n" 
	 << "         [-wtoi sz] [-wclean wsz,nbw] [-range min,max] [-cleannsig nsig] \n"
	 << "         [-sepflg sepFlagFile] [-outppf] [-noprstat] [-useseqbuff] \n"
	 << "         inFitsName outFileName \n" 
	 << "   -start snb : sets the start sample num \n"
	 << "   -end sne : sets the end sample num \n"
	 << "   -snimplicit : sampleNum are implicit in fits files (def=no) \n"
	 << "   -range min,max : sets the acceptable range for intoi \n"
	 << "              default= -16000,16000\n"
	 << "   -intoi toiName : select input TOI name (def bolo)\n"
	 << "   -sepflg sepFlagFileName: sets separate flag file (Level2)\n" 
	 << "   -wtoi sz : sets TOISeqBuff buffer size (def= 8192)\n"
	 << "   -wclean wsz,nbw : sets cleaner window parameters (256,5) \n"
         << "   -cleannsig nsig : Sets cleanner ThrNSig (default = 999999.) \n"
	 << "   -outppf : Write the NTuple in PPF format (default: FITS) \n"
	 << "   -noprstat : DO NOT PrintStat with ProcSampleCounter \n"
	 << "   -useseqbuff : Use TOISeqBuffered TOI's (default: TOISegmented) \n"
	 << endl;
  }
  if (fgerr) exit(1); 
}

int main(int narg, char** arg) {

  if ((narg > 1) && (strcmp(arg[1],"-h") == 0) ) Usage(false);

  cout << "toistat starting - Decoding arguments " << " narg=" << narg << endl;

  bool fgsetstart = false;

  bool fgprstat = true;
  bool fgsegmented = true;
  bool fgoutppf = false;
  bool snimplicit = false;

  int dbglev = 0;

  int wtoi = 8192;

  int nmax = 10;
  int istart = 0;
  int iend = 0;

  double range_min = -16000;
  double range_max = 16000.;

  // cleaner parameters
  int clean_wsz = 256;
  int clean_nbw = 5;
  double clean_nsig = 999999.;

// Fichier de flag separe / Level2 / Reza 18/6/2002
  string sepflagfile;   
  bool sepflg = false;  

  // File names 
  string infile;
  string outfile;
  string intoi = "bolo"; 

  if (narg < 3) Usage(true);
  int ko=1;
  // decoding arguments 
  for(int ia=1; ia<narg; ia++) {
    if (strcmp(arg[ia],"-start") == 0) {
      if (ia == narg-1) Usage(true);  // -start est suivi d'un argument 
      istart = atoi(arg[ia+1]); ia++;
      fgsetstart = true;
    }    
    else if (strcmp(arg[ia],"-end") == 0) {
      if (ia == narg-1) Usage(true);  
      iend = atoi(arg[ia+1]); ia++;
    }    
    else if (strcmp(arg[ia],"-snimplicit") == 0)  snimplicit = true;
    else if (strcmp(arg[ia],"-wtoi") == 0) {
      if (ia == narg-1) Usage(true);  
      wtoi = atoi(arg[ia+1]); ia++;
    }    
    else if (strcmp(arg[ia],"-wclean") == 0) {
      if (ia == narg-1) Usage(true);  
      sscanf(arg[ia+1],"%d,%d",  &clean_wsz, &clean_nbw);
    }    
    else if (strcmp(arg[ia],"-cleannsig") == 0) {
      if (ia == narg-1) Usage(true);  
      clean_nsig = atof(arg[ia+1]); ia++;
    }    
    else if (strcmp(arg[ia],"-range") == 0) {
      if (ia == narg-1) Usage(true);
      sscanf(arg[ia+1],"%lf,%lf",&range_min, &range_max);
      ia++;
    }    
    else if (strcmp(arg[ia],"-intoi") == 0) {
      if (ia == narg-1) Usage(true);  
      intoi = arg[ia+1]; ia++;
    }    
    else if (strcmp(arg[ia],"-sepflg") == 0) {
      if (ia == narg-1) Usage(true);  
      sepflagfile = arg[ia+1]; 
      sepflg = true; ia++;
    }    
    else if (strcmp(arg[ia],"-outppf") == 0)  fgoutppf = true;

    else if (strcmp(arg[ia],"-noprstat") == 0)  fgprstat = false;
    else if (strcmp(arg[ia],"-prstat") == 0)  fgprstat = true;
    else if (strcmp(arg[ia],"-useseqbuff") == 0)  fgsegmented = false;

    else { ko = ia; break; }  // Debut des noms
  }

  if (iend < istart) iend = istart+wtoi*(nmax+5);
  if ((narg-ko) < 2)  Usage(true); 
  infile = arg[ko];
  outfile = arg[ko+1];
  //  outppfname = arg[ko+2];

  cout << " Initializing SOPHYA ... " << endl;
  SophyaInit();
  //  SophyaConfigureSignalhandling(true);

  InitTim();

  cout << ">>>> toistat: Infile= " << infile << " outFile=" 
       << outfile << endl; 
  cout << "  iStart= " << istart << " iEnd= " << iend << endl;
  cout << ">>>> InTOIName= " << intoi << endl;

  try {
    TOIManager* mgr = TOIManager::getManager();
  
    //  mgr->setRequestedSample(11680920,11710584);
    //  mgr->setRequestedSample(104121000, 104946120);
    if (fgsetstart) 
      mgr->setRequestedSample(istart, iend);

    cout << "> Creating FITSTOIReader object - InFile=" << infile << endl;
    FITSTOIReader r(infile); 
    if (snimplicit) {
      cout << " Setting Implicit SN flag for reader: r.setImplicitSN()" << endl;
      r.setImplicitSN();
    }
    if (sepflg) {
      cout << " Setting separate flag file for InTOI_bolo File=" << sepflagfile 
	   << " (Flags=FlgToiSpike, FlgToiSource)" << endl;
      vector<FlagToiDef> flgcol;
      flgcol.push_back(FlgToiSpike);
      flgcol.push_back(FlgToiSource);
      r.setFlagFile(sepflagfile, flgcol);
    }

    cout << "> Creating SimpleCleaner() " << endl;
    SimpleCleaner cleaner(clean_wsz, clean_nbw);
    cout << " Setting Range for cleaner: " << range_min 
	 << " - " << range_max << endl;
    cleaner.SetRange(range_min, range_max);
    cleaner.SetCleanForMeanThrNSig(clean_nsig);
    cleaner.FillMeanSigNTuple(true);

    CGT plombier(fgsegmented, wtoi);
    //    plombier.SetDebugLevel(dbglev);

    cout << "> Connecting Processors through plombier ... " << endl;
    string inname = "in";
    string outname = "out";
    plombier.Connect(r, intoi, cleaner, inname);

    cout << "> Plombier status before start" << endl;
    cout << plombier ; 

    cout << cleaner;

    PrtTim("starting processors");
    plombier.Start();


    // ------------------- Impression continu de stat ------------------------ 
    if (fgprstat) {
      ProcSampleCounter<SimpleCleaner> stats(cleaner);
      stats.InfoMessage() = "toistat/Info";
      stats.PrintStats();
    }
    // -----------------------------------------------------------------------

    cout << cleaner;

    mgr->joinAll();
    PrtTim("End threads");
    cleaner.GetMeanSigNTuple().Info()["TOIFILE"] = infile;
    cleaner.GetMeanSigNTuple().Info()["TOINAME"] = intoi;
  
    if (fgoutppf) {
      cout << " \n --------------------------------------------------------- " << endl;
      cout << " Saving Stats (mean-sigma) NTuple to PPF file " << outfile << endl;
      POutPersist pos(outfile);
      pos << cleaner.GetMeanSigNTuple();
      cout << " \n --------------------------------------------------------- " << endl;
    }
    else {
      cout << " \n --------------------------------------------------------- " << endl;
      cout << " Saving Stats (mean-sigma NTuple to FITS file " << outfile << endl;
      FitsOutFile  fos(outfile, FitsFile::clear);
      fos << cleaner.GetMeanSigNTuple();
      cout << " \n --------------------------------------------------------- " << endl;      
    }
  }
  catch (PThrowable & exc) {
    cerr << "\n toistat: Catched Exception \n" << (string)typeid(exc).name() 
	 << " - Msg= " << exc.Msg() << endl;
  }
  catch (const std::exception & sex) {
    cerr << "\n toistat: Catched std::exception \n" 
	 << (string)typeid(sex).name() << endl; 
  }
  catch (...) {
    cerr << "\n toistat: some other exception was caught ! " << endl;
  }

  return(0);
}
