/* Interface Fits pour bolometre    cmv 29/09/2001 */
#ifndef FBOLOREAD_H_SEEN
#define FBOLOREAD_H_SEEN

#include "machdefs.h"
#include <iostream>
#include <string.h>
#include <string>

#include "anydataobj.h"
#include "tvector.h"
#include "fabtcolread.h"

#include <vector>

namespace SOPHYA {

//! Class for defining a bolometer by connecting columns out FITS files
class FitsBoloRead : public AnyDataObj {
public:
  enum {ColSNum=0, ColBolo=1, ColFlag=2, ColAlpha=3, ColDelta=4};

  FitsBoloRead(long blen=100,long bsens=1);
  virtual ~FitsBoloRead();

  //! Define the Alpha column connection (see addcol).
  inline int SetAlpha(const char *label,const char* fname="",int ihdu=0)
              {return addcol(ColAlpha,label,fname,ihdu);}
  //! Define the Delta column connection (see addcol).
  inline int SetDelta(const char *label,const char* fname="",int ihdu=0)
              {return addcol(ColDelta,label,fname,ihdu);}
  //! Define the Bolo column connection (see addcol).
  inline int SetBolo(const char *label,const char* fname="",int ihdu=0)
              {return addcol(ColBolo,label,fname,ihdu);}
  //! Define the Flag column connection (see addcol).
  inline int SetFlag(const char *label,const char* fname="",int ihdu=0)
              {return addcol(ColFlag,label,fname,ihdu);}
  //! Define the Sample Number column connection (see addcol).
  inline int SetSNum(const char *label,const char* fname="",int ihdu=0)
              {return addcol(ColSNum,label,fname,ihdu);}
  //! Add a new column connection (see addcol).
  inline int AddCol(const char *label,const char* fname,int ihdu=0)
              {return addcol(-1,label,fname,ihdu);}

  //! return Alpha value for row "n" into double
  inline double GetAlpha(long n,bool usebuffer=true)
                {return GetCol(ColAlpha,n,usebuffer);}
  //! return Delta value for row "n" into double
  inline double GetDelta(long n,bool usebuffer=true)
                {return GetCol(ColDelta,n,usebuffer);}
  //! return Bolo value for row "n" into double
  inline double GetBolo(long n,bool usebuffer=true)
                {return GetCol(ColBolo,n,usebuffer);}
  //! return Flag value for row "n" into double
  inline double GetFlag(long n,bool usebuffer=true)
         {if(!IsFlag()) return 0.; return GetCol(ColFlag,n,usebuffer);}
  //! return SNum value for row "n" into double
  inline double GetSNum(long n,bool usebuffer=true)
                {return GetCol(ColSNum,n,usebuffer);}
  //! return column "col" value for row "n" into double
  inline double GetCol(int col,long n,bool usebuffer=true) {
    if(col<0 || col>=(int)mFABT.size())
      throw ParmError("FitsBoloRead::GetCol: bad column number\n");
    if(mFABT[col]==NULL)
      throw NullPtrError("FitsBoloRead::GetCol: column not connected\n");
    return mFABT[col]->Read(n,usebuffer);
  }

  //! return Alpha values for rows [n1,n2] into TVector
  inline long GetAlpha(long n1,long n2,TVector<double>& data)
              {return GetCol(ColAlpha,n1,n2,data);}
  inline long GetAlpha(long n1,long n2,TVector<float>& data)
              {return GetCol(ColAlpha,n1,n2,data);}
  inline long GetAlpha(long n1,long n2,TVector<int_4>& data)
              {return GetCol(ColAlpha,n1,n2,data);}

  //! return Delta values for rows [n1,n2] into TVector
  inline long GetDelta(long n1,long n2,TVector<double>& data)
              {return GetCol(ColDelta,n1,n2,data);}
  inline long GetDelta(long n1,long n2,TVector<float>& data)
              {return GetCol(ColDelta,n1,n2,data);}
  inline long GetDelta(long n1,long n2,TVector<int_4>& data)
              {return GetCol(ColDelta,n1,n2,data);}

  //! return Bolo values for rows [n1,n2] into TVector
  inline long GetBolo(long n1,long n2,TVector<double>& data)
              {return GetCol(ColBolo,n1,n2,data);}
  inline long GetBolo(long n1,long n2,TVector<float>& data)
              {return GetCol(ColBolo,n1,n2,data);}
  inline long GetBolo(long n1,long n2,TVector<int_4>& data)
              {return GetCol(ColBolo,n1,n2,data);}

  //! return Flag values for rows [n1,n2] into TVector
  inline long GetFlag(long n1,long n2,TVector<double>& data)
              {if(!IsFlag()) return 0; return GetCol(ColFlag,n1,n2,data);}
  inline long GetFlag(long n1,long n2,TVector<float>& data)
              {if(!IsFlag()) return 0; return GetCol(ColFlag,n1,n2,data);}
  inline long GetFlag(long n1,long n2,TVector<int_4>& data)
              {if(!IsFlag()) return 0; return GetCol(ColFlag,n1,n2,data);}

  //! return Sample Number values for rows [n1,n2] into TVector
  inline long GetSNum(long n1,long n2,TVector<double>& data)
              {return GetCol(ColSNum,n1,n2,data);}
  inline long GetSNum(long n1,long n2,TVector<float>& data)
              {return GetCol(ColSNum,n1,n2,data);}
  inline long GetSNum(long n1,long n2,TVector<int_4>& data)
              {return GetCol(ColSNum,n1,n2,data);}

  //! return column "col" values for rows [n1,n2] into TVector
  inline long GetCol(int col,long n1,long n2,TVector<double>& data) {
    if(col<0 || col>=(int)mFABT.size()) return 0; if(mFABT[col]==NULL) return 0;
    return mFABT[col]->Read(n1,n2,data);
  }
  inline long GetCol(int col,long n1,long n2,TVector<float>& data) {
    if(col<0 || col>=(int)mFABT.size()) return 0; if(mFABT[col]==NULL) return 0;
    return mFABT[col]->Read(n1,n2,data);
  }
  inline long GetCol(int col,long n1,long n2,TVector<int_4>& data) {
    if(col<0 || col>=(int)mFABT.size()) return 0; if(mFABT[col]==NULL) return 0;
    return mFABT[col]->Read(n1,n2,data);
  }

  //! return the value of the first sampleNum
  inline double ReadFirstSNum(void) {return GetSNum(0,false);}
  //! return the value of the last sampleNum
  inline double ReadLastSNum(void)  {return GetSNum(NBline-1,false);}

  //! Is Alpha connected ?
  inline bool IsAlpha(void)  {return IsCol(ColAlpha);}
  //! Is Delta connected ?
  inline bool IsDelta(void)  {return IsCol(ColDelta);}
  //! Is Bolo connected ?
  inline bool IsBolo(void)   {return IsCol(ColBolo);}
  //! Is Flag connected ?
  inline bool IsFlag(void)   {return IsCol(ColFlag);}
  //! Is SNum connected ?
  inline bool IsSNum(void)   {return IsCol(ColSNum);}
  //! Is column col connected ?
  inline bool IsCol(int col) {if(mFABT[col]) return true; return false;}

  //! Set debug level
  inline void SetDebug(int lp=0) {DbgLevel = (unsigned short) lp;}
  //! Set the number of rows for that class choosing column "col"
  void SetNbLine(int col=-1);
  //! Get the number of rows to be read
  inline long GetNbLine(void) const {return NBline;}
  //! Get the number of columns in the FITS HDU to be read
  inline int  GetNbCol(void) const {return mFABT.size();}
  //! Get the read requested buffer length
  inline long GetBLen(void) const {return BuffLen;}
  //! Get the read buffer direction
  inline long GetBSens(void) const {return BuffSens;}
  //! Print to os
  virtual void Print(ostream& os,int lp=1) const;
  //! Print to stdout
  inline  void Print(int lp=1) const {Print(cout,lp);}

  //! Get the FitsABTColRead pointer for the Alpha
  inline FitsABTColRead* GetAlphaReader(void) {return GetColReader(ColAlpha);}
  //! Get the FitsABTColRead pointer for the Delta
  inline FitsABTColRead* GetDeltaReader(void) {return GetColReader(ColDelta);}
  //! Get the FitsABTColRead pointer for the Bolo
  inline FitsABTColRead* GetBoloReader(void) {return GetColReader(ColBolo);}
  //! Get the FitsABTColRead pointer for the Flag
  inline FitsABTColRead* GetFlagReader(void) {return GetColReader(ColFlag);}
  //! Get the FitsABTColRead pointer for the SNum
  inline FitsABTColRead* GetSNumReader(void) {return GetColReader(ColSNum);}
  //! Get the FitsABTColRead pointer for the column "col"
  inline FitsABTColRead* GetColReader(int col)
    {if(col<0 || col>=(int)mFABT.size()) return NULL; return mFABT[col];}

protected:
  //! Protected method. Do not use directly but read the doc.
  int addcol(int col,const char *label,const char* fname,int ihdu);
  void Gess_If_Not_Define(int col);

  long NBline;
  unsigned short DbgLevel;
  long BuffLen, BuffSens;

  vector<string> mFName;
  vector<string> mLabel;
  vector<int> mHDU;
  vector<FitsABTColRead *> mFABT;
};

} // namespace SOPHYA
#endif    /* FBOLOREAD_H_SEEN */
