// This may look like C code, but it is really -*- C++ -*-

// ArchTOIPipe           (C)     CEA/DAPNIA/SPP IN2P3/LAL
//                               Eric Aubourg
//                               Christophe Magneville
//                               Reza Ansari
// $Id: toisegment.h,v 1.21 2003-11-14 12:34:55 aubourg Exp $

#ifndef TOISEGMENT_H
#define TOISEGMENT_H

#include "config.h"
#include <iostream.h>
#include <vector>
#include <set>

#include "toi.h"

// ------------ TOISegmented ------------------------------
// Classe de TOI pour echantillonage regulier, avec buffer 
// segmente pour optimiser le multithread et limiter les
// verrous.
// Il faut que les fournisseurs fassent arriver les donnees
// par samplenum croissant et continu.
// --------------------------------------------------------

class TOISegmented : public TOIRegular {
 public:
  TOISegmented(int bufsz=1024, int maxseg=20);
  TOISegmented(string nm, int bufsz=1024, int maxseg=20);
  TOISegmented(char* cnm, int bufsz=1024, int maxseg=20);
  ~TOISegmented();

  virtual double        getData(long i);
  virtual void          getData(long i, double& data,  uint_8& flag);
  virtual void          getData(long i, int n, double* data, uint_8* flg=0);
  virtual void          putData(long i, double  value, uint_8  flag=0);
  virtual void          putData(long i, int n, double const* val, uint_8 const* flg=0);
  virtual void          wontNeedBefore(long i);
  virtual void          putDone();
  virtual void          addConsumer(TOIProcessor*);

  // Methodes ignorees car on reimplemente les methodes de base
  virtual DataStatus    isDataAvail(long iStart, long iEnd);
  virtual DataStatus    isDataAvail(long i);
  virtual DataStatus isDataAvailNL(long i);
  virtual DataStatus    isDataAvailNL(long iStart, long iEnd); // abstract
  virtual void          waitForData(long iStart, long iEnd);
  virtual void          waitForData(long i);
  virtual void          waitForAnyData();
  virtual long           nextDataAvail(long iAfter); // abstract
  virtual bool          hasSomeData(); // abstract
  virtual void          doGetData(long i, double& data, uint_8& flag); // abs
  virtual void          doPutData(long i, double value, uint_8 flag=0); // abs

  
 protected:
  class BufferSegment;
  class BufferView;
  class MasterView;

  MasterView* master;


  class BufferSegment {
  public:
    BufferSegment(int sz);
    ~BufferSegment();
    static const int NEW = 0;
    static const int WRITE = 1;     // single-thread write access
    static const int COMMITTED = 2; // multiple read without lock are ok
    
    int getStatus() {return status;}
    void incRefCount();
    void decRefCount();
    int  getRefCount();
    
    void putData(long sn, double data, uint_8 flag);
    void putData(long sn, int n, double const* data, uint_8 const* flag);
    inline double getData(long sn);
    inline uint_8 getFlag(long sn);
    void getData(long sn, int n, double* data, uint_8* flag);

    bool isPastEnd(long sn) {
      return sn >= sn0+bufferSize;
    }
    
  private:
    void checkCommitted() {
      if (status != COMMITTED) { 
	cerr << "TOISegment: Read on not committed buffer segment" << endl;
	throw(ForbiddenError("TOISegment: Read on not committed buffer segment"));
      }
    }

    void checkInRange(long sn) {
      if (sn < sn0 || sn >= sn0+bufferSize) {
	cerr << "TOISegment: out of range access in buffer segment " << sn << " not in " 
	     << sn0 << " - " << sn0+bufferSize << endl;
	throw(RangeCheckError("TOISegment: out of range access in buffer segment"));
      }
    }


    int status;      // NEW, WRITE, COMMITTED
    int bufferSize;  
    long sn0;         // Samplenum du premier echantillon

    int refcount;    // Nombre de vues qui utilisent 
    pthread_mutex_t refcount_mutex; // Pour modification refcount

    double* data;
    uint_8* flags;

    friend class BufferView;
    friend class MasterView;
  };
  
  // Master view, gere le lock, et l'ecriture
  class MasterView {
  public:
    MasterView(int bufsz=256, int maxseg=20, string nm="");
    ~MasterView();

    void putData(long sn, double data, uint_8 flag);
    double getData(long sn);
    uint_8 getFlag(long sn);
    void   getData(long i, int n, double* data, uint_8* flg);
    void   putData(long i, int n, double const* val, uint_8 const* flg);
    BufferView* getView(); // thread-specific
    void putDone();

  protected:

    friend class BufferView;
    friend class TOISegmented;
    string name;
    void signalWaitingViews(); // views are waiting on read
    void signalWrite();        // we are waiting on write
    void nextSegment();
    void waitForCleaning();
    BufferView* createView();
    void updateView(BufferView*); // called on reader thread of the view
    
    BufferSegment*   currentSegment;

    int maxSegments;
    int segmentSize;
    long sn0;                         // First sn in first segment
    vector<BufferSegment*> segments; // Committed segments
    int nConsumers;
    
    pthread_mutex_t  views_mutex; // lock for master buffer list access
    pthread_cond_t   write_wait_condv; // waiting for cleaning (on writer thread)
    pthread_key_t    buffer_key; // thread-specific buffer view
    static void BufferDestroy(void *);

    pthread_mutex_t  read_wait_mutex;
    pthread_cond_t   read_wait_condv;

    bool   waitingOnWrite; // wait on writer thread
    int    waitingViews;

    set<BufferView*>  allViews;

    void checkDeadLock();
  };


  // per-thread read-only view of a buffer set
  class BufferView {
  public:
    BufferView(MasterView*);
    ~BufferView();

    double getData(long sn);
    uint_8 getFlag(long sn);    
    void   getData(long i, int n, double* data, uint_8* flg);

    void wontNeedBefore(long sn);
    
  protected:
    void wait(long sn);  // Passe en attente d'un nouveau segment -- lecture
    void sync();  // recupere les nouveaux segments, resync avec master
    void ensure(long sn);

    bool waiting;
    long  waitingFor;

    friend class MasterView;
    MasterView* master;
    vector<BufferSegment*> segments; // Committed
    long sn0; 
    int segmentSize;
    long firstNeeded;
  };  

  
  
};

/***********************************/
/* Inline methods -- BufferSegment */
/***********************************/

double TOISegmented::BufferSegment::getData(long sn) {
  checkCommitted();
  checkInRange(sn);
  return data[sn-sn0];
}

uint_8 TOISegmented::BufferSegment::getFlag(long sn) {
  checkCommitted();
  checkInRange(sn);
  return flags[sn-sn0];
}

#endif
