#include "machdefs.h"
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "histinit.h"
#include "histerr.h"
#include "perrors.h"

/*!
  \class SOPHYA::HistoErr
  \ingroup HiStats
  Classe d'histogrammes 1D avec erreurs donnees par l'utilisateur
*/

/********* Methode *********/
/*! Constructeur par defaut */
HistoErr::HistoErr(void)
: Histo(), mNData(NULL)
{
}

/********* Methode *********/
/*! Constructeur d'un histo de nBin bins allant de xMin a xMax */
HistoErr::HistoErr(r_8 xMin, r_8 xMax, int_4 nBin)
: Histo(xMin,xMax,nBin), mNData((nBin>0) ? new uint_4[nBin] : NULL)
{
 this->Errors();
 if(mNData) memset(mNData,0,mBins*sizeof(uint_4));
}

/********* Methode *********/
/*! Constructeur par copie */
HistoErr::HistoErr(const HistoErr& H)
: Histo(H), mNData((H.mBins>0)? new uint_4[H.mBins] : NULL)
{
 if(mBins>0) memcpy(mNData,H.mNData,mBins*sizeof(uint_4));
}

/********* Methode *********/
/*! Destructeur */
HistoErr::~HistoErr(void)
{
 if(mNData) {delete [] mNData; mNData=NULL;}
}

/********* Methode *********/
/*!
  Remise a zero
*/
void HistoErr::Zero(void)
{
 if(mNData != NULL) memset(mNData,0,mBins*sizeof(uint_4));
 Histo::Zero();
}

/********* Methode *********/
/*!
 Addition du contenu de l'histo pour abscisse x poids w et l'erreur e
*/
void HistoErr::Add(r_8 x, r_8 w, r_8 e)
{
 int_4 numBin = (int_4)floor((x-mMin)/binWidth);
 if(numBin<0) mUnder += w;
 else if(numBin>=mBins) mOver += w;
 else {
   mData[numBin] += w; mNData[numBin]++; mErr2[numBin] += e*e;
   nHist += w; nEntries++;
 }
}

/********* Methode *********/
/*!
 Addition du contenu de l'histo pour le bin numBin poids w et l'erreur e
*/
void HistoErr::AddBin(int_4 numBin, r_8 w, r_8 e)
{
 if(numBin<0) mUnder += w;
 else if(numBin>=mBins) mOver += w;
 else {
   mData[numBin] += w; mNData[numBin]++; mErr2[numBin] += e*e;
   nHist += w; nEntries++;
 }
}

/*!
  Remplissage d'un tableau avec les nombres d'entrees dans le bin de l'histo
*/
void HistoErr::GetNBin(TVector<int_4>& v) const
{
v.Realloc(mBins);
for(int_4 i=0;i<mBins;i++) v(i) = mNData[i];
return;
}

/********* Methode *********/
/*!
  Remplissage du nombre d'entrees dans les bins de l'histo avec les valeurs d'un vecteur
*/
void HistoErr::PutNBin(TVector<int_4> &v)
{
uint_4 n = (v.NElts()<(uint_4) mBins) ? v.NElts(): (uint_4) mBins;
if(n>0) for(uint_4 i=0;i<n;i++) mNData[i] = v(i);
return;
}

///////////////////////////////////////////////////////////
// --------------------------------------------------------
//   Les objets delegues pour la gestion de persistance 
// --------------------------------------------------------
///////////////////////////////////////////////////////////

DECL_TEMP_SPEC  /* equivalent a template <> , pour SGI-CC en particulier */
void ObjFileIO<HistoErr>::ReadSelf(PInPersist& is)
{
char strg[256];

if(dobj==NULL) dobj=new HistoErr;
  else         dobj->Delete();

// Lecture entete
is.GetLine(strg, 255);

// Lecture des donnees HistoErr
is.Get(dobj->mBins);
is.GetLine(strg, 255);
is.Get(dobj->mNData, dobj->mBins);

// Lecture de l'histogramme
is >> (Histo&)(*dobj);

return;
}

DECL_TEMP_SPEC  /* equivalent a template <> , pour SGI-CC en particulier */
void ObjFileIO<HistoErr>::WriteSelf(POutPersist& os) const
{
if(dobj == NULL)   return;
char strg[256];

sprintf(strg,"HistErr");
os.PutLine(strg);

// Ecriture des valeurs
os.Put(dobj->mBins);

// Ecriture des donnees HistoErr nombre d entree par bin
sprintf(strg,"Tableau du nombre d entree par bin");
os.PutLine(strg);
os.Put(dobj->mNData, dobj->mBins);

// Ecriture de l'histogramme
os << (Histo&)(*dobj);

return;
}

#ifdef __CXX_PRAGMA_TEMPLATES__
#pragma define_template ObjFileIO<HistoErr>
#endif

#if defined(ANSI_TEMPLATES) || defined(GNU_TEMPLATES)
template class ObjFileIO<HistoErr>;
#endif
