// Gestionnaire de compilation-linker C++ - R. Ansari 10/2000
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include "sopnamsp.h"
#include "cxxcmplnk.h"
#include <iostream>

/*!
  \class SOPHYA::CxxCompilerLinker
  \ingroup SysTools
  This classes handles the compilation of a C++ source code and 
  building of a shared library.
  The present version has been adapted for different compilers and 
  systems:
  - Linux-g++
  - Linux-KCC
  - MacOS X/Darwin (Apple OS X 10.2 and 10.3)
  - OSF-cxx : (HP/Compaq/Digital OSF-Tru64 and cxx c++ compiler)
  - SGI-CC : Silicon Graphics system and C++ compiler

  \sa SOPHYA::PDynLinkMgr

  \code
  #include "cxxcmplnk.h"
  CxxCompilerLinker cxx;
  string name = "toto.cc";
  string oname = "toto.o";
  string soname = "toto.so";
  int rc;
  // compiling file 
  rc = cxx.Compile(name, oname);
  // linking and building the shared object 
  rc = cxx.BuildSO(oname, soname);
  \endcode
*/

static char * gcxx_opt = 
"-O -Wall -Wpointer-arith -Wmissing-prototypes -Wsynth -fdollars-in-identifiers";
static char * KCC_opt = 
"-O --exceptions --rtti  --auto_instantiation --one_instantiation_per_object -D__KCC__";
static char * cxx_opt = 
"-O -no_implicit_include ";
static char * SGICC_opt =  "-O -prelink -D__SGICC__ -LANG:std";

/* --Methode-- */
/*!
  Constructor
  \param fglibsophya : if \c true libsophya.so is used when linking
  \param fglibextsophya : if \c true libextsophya.so is used when linking
  \param fglibpi : if \c true libPI.so is used when linking
*/
CxxCompilerLinker::CxxCompilerLinker(bool fglibsophya, bool fglibextsophya, bool fglibpi)
  : verbose(false)
{
  
  string syscomp = "";

#if defined(OSF1) 
  cppFlags += "-DOSF1" ; 
  syscomp = "OSF1-";
#elif defined(Linux)
  cppFlags += "-DLinux" ;
  syscomp = "Linux-"; 
#elif defined(SunOS)
  cppFlags += "-DSunOS" ;
  syscomp = "SunOS-";  
#elif defined(IRIX64)
  cppFlags += "-DIRIX64" ;
  syscomp = "IRIX64-";  
#elif defined(AIX)
  cppFlags += "-DAIX" ;
  syscomp = "AIX-";  
#elif defined(HPUX)
  cppFlags += "-DHPUX" ;
  syscomp = "HPUX-";  
#elif defined(Darwin)
  cppFlags += "-DDarwin";
  syscomp = "Darwin-";
#endif


#if defined( __GNUG__ )
  compCmd = "g++ ";
  compOptions = gcxx_opt;
  linkOptions = "-O -shared";
  syscomp += "g++/"; 
#elif defined( __DECCXX )
  compCmd = "cxx ";
  compOptions = cxx_opt;
  linkOptions = compOptions + "-shared";
  syscomp += "cxx/"; 
#elif defined( __KCC__ )
  compCmd = "KCC ";
  compOptions = KCC_opt;
  linkOptions = KCC_opt;
  syscomp += "KCC/";  
#elif defined( __SGICC__ )
  compCmd = "CC ";
  compOptions = SGICC_opt;
  linkOptions = "-shared -O -LANG:std";
  syscomp += "CC/";  
#ifdef SGI_ARCH64
  compOptions += " -64 -DSGI_ARCH64 ";
  linkOptions += " -64 "; 
#endif
#endif

#ifndef Darwin
  linkCmd = compCmd;
#else
  linkCmd = "c++ -bundle ";
  linkOptions = "-lSystem -lm";
#endif

  cppFlags += " -I. ";
  compOptions += " -c ";

  string dpcbase;
  char* varenv=NULL;
  varenv=getenv("SOPHYABASEREP");
  if (varenv) {
    dpcbase = varenv; 
    if (dpcbase[dpcbase.length()-1] != '/')  dpcbase += '/';
    cppFlags += ( " -I" + dpcbase + "Include/ ");
    linkOptions += " -L" + dpcbase + syscomp + "ShLibs/";
    if (fglibsophya) linkOptions += " -lsophya ";
    if (fglibextsophya) linkOptions += " -lextsophya ";
    if (fglibpi) linkOptions += " -lPI ";
  }
  linkOptions += " -lm ";

  string extlib;
  varenv=getenv("EXTLIBDIR");
  if (varenv) {
    extlib = varenv; 
    if (extlib[extlib.length()-1] != '/')  extlib += '/';
    cppFlags += ( " -I" + extlib +"Include/ ");
  }

  if ( (varenv=getenv("TMPDIR")) != NULL )  { 
    tmpDir = varenv; 
    if (tmpDir[tmpDir.length()-1] != '/') tmpDir += '/';
  }
}

/* --Methode-- */
CxxCompilerLinker::~CxxCompilerLinker()
{
}

/* --Methode-- */
/*!
  Compiles the file \c name using the C++ compiler driver and produces 
  the output object file \c oname. If no output name is specified,
  a default output file name is made from the input name, with the 
  suffix .o , the in temporary directory.
  \param name : input C++ source file
  \param oname : output object file 
*/
int CxxCompilerLinker::Compile(string const & name, string & oname)
{
  if (oname.length() < 1) {
    size_t l,p,q;
    l = name.length();
    p = name.rfind('/');
    if (p >= l)  p = 0;
    else p++;
    q = name.rfind('.');
    if ((q < l) && (q > p))  oname = tmpDir + name.substr(p, q-p) + ".o";
    else oname = tmpDir + name.substr(p) + ".o";
  }
  string cmd;
  cmd = compCmd + cppFlags + compOptions + "-o " + oname + " " + name ; 
  if (verbose)
    cout << "CxxCompilerLinker::Compile() - Executing \n" << cmd << endl; 
  int rc = system(cmd.c_str());
  if (rc != 0)   
    cerr << "CxxCompilerLinker::Compile() Error Rc(" << cmd <<")= "<< rc << endl; 
  
  return(rc);
}

/* --Methode-- */
/*!
  Creates a shared library from the object file \c oname.
  If no output name \c soname is specified,
  a default output file name is made from the object name, with the 
  suffix .so , in the temporary directory.
  \param oname : input object file 
  \param soname : shared library name
*/
int CxxCompilerLinker::BuildSO(string const & oname, string & soname)
{
  //  char * soext = ".dylib"; if defined(Darwin) - pas necessaire Reza 02/2002
  char * soext = ".so";

  if (soname.length() < 1) {
    size_t l,p,q;
    l = oname.length();
    p = oname.rfind('/');
    if (p >= l)  p = 0;
    else p++;
    q = oname.rfind('.');
    if ((q < l) && (q > p))  soname = tmpDir + oname.substr(p, q-p) + soext;
    else soname = tmpDir + oname.substr(p) + soext;
  }
  string cmd;
  cmd = linkCmd + " " + oname + " " + linkOptions + " -o " + soname + " " ; 
  if (verbose)
    cout << "CxxCompilerLinker::BuildSO() - Executing \n" << cmd << endl; 
  int rc = system(cmd.c_str());
  if (rc != 0)   
    cerr << "CxxCompilerLinker::BuildSO() Error Rc(" << cmd <<")= "<< rc << endl; 
  
  return(rc);
}

