/* code to handle constellation boundaries and figures */

/*
Constellation boundaries:

 Primary reference:

  METHOD TO DETERMINE THE CONSTELLATION IN WHICH A POSITION IS LOCATED

        Recently, Mr. Barry N. Rappaport of New  Mexico State University
  transcribed  the constellation  boundaries as  fixed  by the IAU  into
  machine-readable form.  These have  been  transcribed  by Dr. Nancy G.
  Roman to make it possible  to determine by  computer the constellation
  in which a position is located.

 NSSDC catalog description:
 6042   AN     Catalog of Constellation Boundary Data (Delporte, E. 1930, 
               Cambridge Univ. Press)
               Comment(s): includes constellation identification software 
               (ADC 1987; see Roman, N.G. 1987, Publ. Astron. Soc. Pacific 
               99, 695); 23 description, 118 software, 358 data records. 
               3 files: 23x80, 118x80, 358x29 

 Further adapted for xephem by:

    Craig Counterman: conversion from original F77 to C
    Elwood Downey:    incorporation into xephem
    Ernie Wright:     additional speed and time improvments

Constellation figures:

    Chris Marriott:   original figures
    Elwood Downey:    incorporated into xephem
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "astro.h"

/*
======================================================================
Ernie Wright  2 Mar 94

Find the constellation for a given position.

First C version by Craig Counterman and Elwood Downey.  Based on a
FORTRAN program by Nancy G. Roman (Roman, N.G. 1987, Publ. Astron.
Soc. Pacific 99, 695).  IAU constellation boundaries transcribed into
machine-readable form by Barry N. Rappaport, New Mexico State Univ.
======================================================================
*/

#define NBOUNDS 357

/* constellation ids */
#define And  0
#define Ant  1
#define Aps  2
#define Aql  3
#define Aqr  4
#define Ara  5
#define Ari  6
#define Aur  7
#define Boo  8
#define CMa  9
#define CMi 10
#define CVn 11
#define Cae 12
#define Cam 13
#define Cap 14
#define Car 15
#define Cas 16
#define Cen 17
#define Cep 18
#define Cet 19
#define Cha 20
#define Cir 21
#define Cnc 22
#define Col 23
#define Com 24
#define CrA 25
#define CrB 26
#define Crt 27
#define Cru 28
#define Crv 29
#define Cyg 30
#define Del 31
#define Dor 32
#define Dra 33
#define Equ 34
#define Eri 35
#define For 36
#define Gem 37
#define Gru 38
#define Her 39
#define Hor 40
#define Hya 41
#define Hyi 42
#define Ind 43
#define LMi 44
#define Lac 45
#define Leo 46
#define Lep 47
#define Lib 48
#define Lup 49
#define Lyn 50
#define Lyr 51
#define Men 52
#define Mic 53
#define Mon 54
#define Mus 55
#define Nor 56
#define Oct 57
#define Oph 58
#define Ori 59
#define Pav 60
#define Peg 61
#define Per 62
#define Phe 63
#define Pic 64
#define PsA 65
#define Psc 66
#define Pup 67
#define Pyx 68
#define Ret 69
#define Scl 70
#define Sco 71
#define Sct 72
#define Se1 73
#define Sex 74
#define Sge 75
#define Sgr 76
#define Tau 77
#define Tel 78
#define TrA 79
#define Tri 80
#define Tuc 81
#define UMa 82
#define UMi 83
#define Vel 84
#define Vir 85
#define Vol 86
#define Vul 87
#define Se2 88

static char *cns_namemap[ NCNS ] = {
   /*  0 */   "And: Andromeda",
   /*  1 */   "Ant: Antlia",
   /*  2 */   "Aps: Apus",
   /*  3 */   "Aql: Aquila",
   /*  4 */   "Aqr: Aquarius",
   /*  5 */   "Ara: Ara",
   /*  6 */   "Ari: Aries",
   /*  7 */   "Aur: Auriga",
   /*  8 */   "Boo: Bootes",
   /*  9 */   "CMa: Canis Major",
   /* 10 */   "CMi: Canis Minor",
   /* 11 */   "CVn: Canes Venatici",
   /* 12 */   "Cae: Caelum",
   /* 13 */   "Cam: Camelopardalis",
   /* 14 */   "Cap: Capricornus",
   /* 15 */   "Car: Carina",
   /* 16 */   "Cas: Cassiopeia",
   /* 17 */   "Cen: Centaurus",
   /* 18 */   "Cep: Cepheus",
   /* 19 */   "Cet: Cetus",
   /* 20 */   "Cha: Chamaeleon",
   /* 21 */   "Cir: Circinus",
   /* 22 */   "Cnc: Cancer",
   /* 23 */   "Col: Columba",
   /* 24 */   "Com: Coma Berenices",
   /* 25 */   "CrA: Corona Australis",
   /* 26 */   "CrB: Corona Borealis",
   /* 27 */   "Crt: Crater",
   /* 28 */   "Cru: Crux",
   /* 29 */   "Crv: Corvus",
   /* 30 */   "Cyg: Cygnus",
   /* 31 */   "Del: Delphinus",
   /* 32 */   "Dor: Dorado",
   /* 33 */   "Dra: Draco",
   /* 34 */   "Equ: Equuleus",
   /* 35 */   "Eri: Eridanus",
   /* 36 */   "For: Fornax",
   /* 37 */   "Gem: Gemini",
   /* 38 */   "Gru: Grus",
   /* 39 */   "Her: Hercules",
   /* 40 */   "Hor: Horologium",
   /* 41 */   "Hya: Hydra",
   /* 42 */   "Hyi: Hydrus",
   /* 43 */   "Ind: Indus",
   /* 44 */   "LMi: Leo Minor",
   /* 45 */   "Lac: Lacerta",
   /* 46 */   "Leo: Leo",
   /* 47 */   "Lep: Lepus",
   /* 48 */   "Lib: Libra",
   /* 49 */   "Lup: Lupus",
   /* 50 */   "Lyn: Lynx",
   /* 51 */   "Lyr: Lyra",
   /* 52 */   "Men: Mensa",
   /* 53 */   "Mic: Microscopium",
   /* 54 */   "Mon: Monoceros",
   /* 55 */   "Mus: Musca",
   /* 56 */   "Nor: Norma",
   /* 57 */   "Oct: Octans",
   /* 58 */   "Oph: Ophiuchus",
   /* 59 */   "Ori: Orion",
   /* 60 */   "Pav: Pavo",
   /* 61 */   "Peg: Pegasus",
   /* 62 */   "Per: Perseus",
   /* 63 */   "Phe: Phoenix",
   /* 64 */   "Pic: Pictor",
   /* 65 */   "PsA: Piscis Austrinus",
   /* 66 */   "Psc: Pisces",
   /* 67 */   "Pup: Puppis",
   /* 68 */   "Pyx: Pyxis",
   /* 69 */   "Ret: Reticulum",
   /* 70 */   "Scl: Sculptor",
   /* 71 */   "Sco: Scorpius",
   /* 72 */   "Sct: Scutum",
   /* 73 */   "Se1: Serpens Caput",
   /* 74 */   "Sex: Sextans",
   /* 75 */   "Sge: Sagitta",
   /* 76 */   "Sgr: Sagittarius",
   /* 77 */   "Tau: Taurus",
   /* 78 */   "Tel: Telescopium",
   /* 79 */   "TrA: Triangulum Australe",
   /* 80 */   "Tri: Triangulum",
   /* 81 */   "Tuc: Tucana",
   /* 82 */   "UMa: Ursa Major",
   /* 83 */   "UMi: Ursa Minor",
   /* 84 */   "Vel: Vela",
   /* 85 */   "Vir: Virgo",
   /* 86 */   "Vol: Volans",
   /* 87 */   "Vul: Vulpecula",
   /* 88 */   "Se2: Serpens Cauda",
};

static struct {
   unsigned short lower_ra;      /* hours * 1800 */
   unsigned short upper_ra;      /* hours * 1800 */
   short          lower_dec;     /* degrees * 60 */
   short          index;
} cbound[ NBOUNDS ] = {
   {     0, 43200,  5280, UMi },
   { 14400, 26100,  5190, UMi },
   { 37800, 41400,  5170, UMi },
   { 32400, 37800,  5160, UMi },
   {     0, 14400,  5100, Cep },
   { 16500, 19200,  4920, Cam },
   {     0,  9000,  4800, Cep },
   { 19200, 26100,  4800, Cam },
   { 31500, 32400,  4800, UMi },
   { 36300, 37800,  4800, Dra },
   {     0,  6315,  4620, Cep },
   { 20700, 24450,  4620, Cam },
   { 29760, 31500,  4500, UMi },
   { 36300, 37200,  4500, Cep },
   { 14340, 16500,  4410, Cam },
   { 16500, 20400,  4410, Dra },
   { 23400, 29760,  4200, UMi },
   {  5580,  6150,  4080, Cas },
   { 36750, 37200,  4020, Dra },
   { 20400, 21600,  3990, Dra },
   {     0,   600,  3960, Cep },
   { 25200, 28200,  3960, UMi },
   { 42450, 43200,  3960, Cep },
   { 21600, 24300,  3840, Dra },
   { 24300, 25950,  3780, Dra },
   { 41700, 42450,  3780, Cep },
   { 10980, 12600,  3720, Cam },
   { 36000, 36750,  3690, Dra },
   { 36966, 37080,  3655, Cep },
   { 12600, 14340,  3600, Cam },
   { 14340, 15150,  3600, UMa },
   { 35580, 36000,  3570, Dra },
   { 36000, 36966,  3570, Cep },
   { 41160, 41700,  3545, Cep },
   {     0,  4380,  3510, Cas },
   { 34950, 35580,  3480, Dra },
   {  3060,  3435,  3450, Cas },
   {  4380,  5580,  3420, Cas },
   {  5580,  5700,  3420, Cam },
   { 40170, 41160,  3375, Cep },
   {  9000, 10980,  3360, Cam },
   { 25260, 25950,  3330, UMa },
   { 25950, 34950,  3330, Dra },
   {  5700,  6000,  3300, Cam },
   { 39840, 40170,  3300, Cep },
   { 37080, 39540,  3290, Cep },
   {     0,  3060,  3240, Cas },
   { 10980, 11700,  3240, Lyn },
   { 21750, 24300,  3180, UMa },
   { 27450, 28350,  3180, Dra },
   { 39540, 39840,  3165, Cep },
   {  6000,  9000,  3150, Cam },
   { 41160, 42000,  3150, Cas },
   { 28350, 30600,  3090, Dra },
   {  3675,  4530,  3030, Per },
   { 30600, 32820,  3030, Dra },
   {     0,  2460,  3000, Cas },
   {  2460,  3000,  3000, Per },
   { 11700, 12240,  3000, Lyn },
   { 42000, 43200,  3000, Cas },
   { 24300, 25260,  2910, UMa },
   {     0,  2010,  2880, Cas },
   { 42450, 43200,  2880, Cas },
   { 32715, 32820,  2850, Her },
   { 32820, 34350,  2850, Dra },
   { 34350, 34500,  2850, Cyg },
   {  3000,  3675,  2820, Per },
   { 15150, 16500,  2820, UMa },
   {   300,  1560,  2760, Cas },
   { 21600, 21750,  2700, UMa },
   { 12240, 13260,  2670, Lyn },
   { 39435, 39540,  2640, Cyg },
   { 39375, 39435,  2625, Cyg },
   { 34500, 34920,  2610, Cyg },
   { 16500, 18300,  2520, UMa },
   { 18300, 19410,  2400, UMa },
   { 27780, 28350,  2400, Boo },
   { 28350, 29400,  2400, Her },
   { 16650, 17250,  2385, Lyn },
   {     0,  4530,  2205, And },
   {  4530,  4620,  2205, Per },
   { 34845, 34920,  2190, Lyr },
   {  8100,  8445,  2160, Per },
   { 39120, 39375,  2160, Cyg },
   { 39375, 39600,  2160, Lac },
   { 11760, 13260,  2130, Aur },
   { 13260, 13950,  2130, Lyn },
   {     0,  3600,  2100, And },
   { 39600, 41070,  2100, Lac },
   { 41070, 41160,  2070, Lac },
   { 41160, 42300,  2070, And },
   {  4620,  4890,  2040, Per },
   { 19410, 19800,  2040, UMa },
   { 21600, 22200,  2040, CVn },
   { 13950, 16650,  2010, Lyn },
   { 16650, 17790,  2010, LMi },
   {  1290,  2535,  1980, And },
   { 27330, 27780,  1980, Boo },
   { 42300, 42750,  1925, And },
   { 22200, 23850,  1920, CVn },
   { 42750, 43200,  1880, And },
   { 25125, 25260,  1845, CVn },
   {  4350,  4890,  1840, Tri },
   {  4890,  8100,  1840, Per },
   {  8100,  8550,  1800, Aur },
   { 32715, 34845,  1800, Lyr },
   { 19800, 21600,  1740, UMa },
   { 35400, 37650,  1740, Cyg },
   {  8550, 10590,  1710, Aur },
   { 17790, 18900,  1710, LMi },
   { 23850, 25125,  1710, CVn },
   {     0,   120,  1680, And },
   {  2535,  3000,  1680, Tri },
   { 10590, 11760,  1680, Aur },
   { 14190, 14400,  1680, Gem },
   { 37650, 39120,  1680, Cyg },
   { 34665, 35400,  1650, Cyg },
   {  3450,  4350,  1635, Tri },
   { 29100, 29400,  1620, CrB },
   { 27150, 27330,  1560, Boo },
   { 27330, 29100,  1560, CrB },
   { 33060, 33960,  1560, Lyr },
   { 19350, 19800,  1530, LMi },
   { 33960, 34665,  1530, Lyr },
   {  3000,  3450,  1500, Tri },
   {  1290,  1530,  1425, Psc },
   { 18900, 19350,  1410, LMi },
   { 38250, 38550,  1410, Vul },
   { 10260, 10590,  1370, Tau },
   {   120,   255,  1320, And },
   { 28650, 28860,  1320, Se1 },
   { 10590, 11190,  1290, Gem },
   { 35700, 36450,  1275, Vul },
   { 33960, 34650,  1265, Vul },
   {   255,  1530,  1260, And },
   { 36450, 37020,  1230, Vul },
   { 14055, 14190,  1200, Gem },
   { 37020, 38250,  1170, Vul },
   { 34650, 35700,  1150, Vul },
   {  5910,  6060,  1140, Ari },
   { 33960, 34200,  1110, Sge },
   { 10260, 10380,  1080, Ori },
   { 11190, 11355,  1050, Gem },
   { 34200, 35700,   970, Sge },
   {  8940,  9600,   960, Tau },
   { 28650, 28950,   960, Her },
   { 35700, 36450,   945, Sge },
   {  8310,  8940,   930, Tau },
   {  9600, 10080,   930, Tau },
   { 23100, 24300,   900, Com },
   { 31050, 32850,   860, Her },
   { 21360, 23100,   840, Com },
   { 13500, 14055,   810, Gem },
   { 30150, 31050,   770, Her },
   {     0,   255,   750, Peg },
   { 10080, 10380,   750, Tau },
   { 12600, 13500,   750, Gem },
   { 38010, 38400,   750, Peg },
   { 11355, 12480,   720, Gem },
   { 32850, 33960,   720, Her },
   { 37575, 37890,   710, Del },
   { 37890, 38010,   710, Peg },
   { 20730, 21360,   660, Leo },
   { 11235, 11355,   600, Ori },
   { 12480, 12600,   600, Gem },
   { 14055, 14265,   600, Cnc },
   { 42900, 43200,   600, Peg },
   {  3000,  5910,   595, Ari },
   { 36255, 36540,   510, Del },
   { 24300, 27150,   480, Boo },
   { 40950, 42900,   450, Peg },
   { 14265, 16650,   420, Cnc },
   { 16650, 19350,   420, Leo },
   { 32850, 33592,   375, Oph },
   { 33592, 33960,   375, Aql },
   { 37500, 37575,   360, Del },
   { 12600, 12630,   330, CMi },
   { 32850, 33165,   270, Se2 },
   { 28950, 30150,   240, Her },
   { 32850, 33165,   180, Oph },
   { 38640, 39000,   165, Peg },
   {     0,  3600,   120, Psc },
   { 33450, 33960,   120, Se2 },
   { 36540, 37500,   120, Del },
   { 37500, 38400,   120, Equ },
   { 38400, 38640,   120, Peg },
   { 39600, 40950,   120, Peg },
   { 39000, 39600,   105, Peg },
   { 12630, 12960,    90, CMi },
   {  6450,  8310,     0, Tau },
   {  8310,  8400,     0, Ori },
   { 12960, 14550,     0, CMi },
   { 26400, 27150,     0, Vir },
   { 32100, 32850,     0, Oph },
   {  4770,  5910,  -105, Cet },
   {  5910,  6450,  -105, Tau },
   { 27150, 29280,  -195, Se1 },
   {  8400,  9150,  -240, Ori },
   { 10500, 11235,  -240, Ori },
   { 32100, 32340,  -240, Se2 },
   { 32850, 33450,  -240, Se2 },
   { 33450, 33960,  -240, Aql },
   { 40950, 42900,  -240, Psc },
   { 19350, 20730,  -360, Leo },
   { 20730, 21300,  -360, Vir },
   {     0,   600,  -420, Psc },
   { 42900, 43200,  -420, Psc },
   { 25650, 26400,  -480, Vir },
   { 28650, 29280,  -480, Oph },
   { 36000, 36960,  -540, Aql },
   { 38400, 39360,  -540, Aqr },
   { 30900, 32340,  -600, Oph },
   { 10500, 14550,  -660, Mon },
   {  8850,  9150,  -660, Eri },
   {  9150, 10500,  -660, Ori },
   { 14550, 15060,  -660, Hya },
   { 17250, 19350,  -660, Sex },
   { 21300, 23100,  -660, Vir },
   { 31650, 31800,  -700, Oph },
   { 33960, 36000,  -722, Aql },
   {  8700,  8850,  -870, Eri },
   { 36960, 38400,  -900, Aqr },
   { 30900, 32850,  -960, Se2 },
   { 32850, 33960,  -960, Sct },
   { 15060, 15450, -1020, Hya },
   { 29280, 29475, -1095, Oph },
   { 15450, 16350, -1140, Hya },
   { 19350, 19500, -1140, Crt },
   { 29280, 29475, -1155, Sco },
   { 28200, 28650, -1200, Lib },
   { 22650, 23100, -1320, Crv },
   { 23100, 25650, -1320, Vir },
   { 16350, 17550, -1440, Hya },
   {  3000,  4770, -1463, Cet },
   {  4770,  6750, -1463, Eri },
   { 19500, 21300, -1470, Crt },
   { 21300, 22650, -1470, Crv },
   { 25650, 26850, -1470, Lib },
   { 29280, 30150, -1475, Oph },
   {     0,  3000, -1530, Cet },
   { 38400, 39360, -1530, Cap },
   { 39360, 42900, -1530, Aqr },
   { 42900, 43200, -1530, Cet },
   { 17550, 18450, -1590, Hya },
   {  8460,  8700, -1635, Eri },
   {  8700, 11010, -1635, Lep },
   { 36000, 38400, -1680, Cap },
   { 18450, 19050, -1750, Hya },
   { 22650, 26850, -1770, Hya },
   { 26850, 28200, -1770, Lib },
   { 28200, 28800, -1770, Sco },
   {  8250,  8460, -1800, Eri },
   { 30150, 31680, -1800, Oph },
   { 31680, 32100, -1800, Sgr },
   { 19050, 19500, -1870, Hya },
   { 11010, 13260, -1980, CMa },
   { 22050, 22650, -1980, Hya },
   { 19500, 22050, -2100, Hya },
   {  6300,  6750, -2160, For },
   { 15060, 16860, -2205, Pyx },
   {  7680,  8250, -2220, Eri },
   { 32100, 34500, -2220, Sgr },
   { 38400, 41400, -2220, PsA },
   { 41400, 42000, -2220, Scl },
   {  5400,  6300, -2375, For },
   { 16860, 19800, -2385, Ant },
   {     0,  3000, -2400, Scl },
   {  3000,  5400, -2400, For },
   {  6960,  7680, -2400, Eri },
   { 42000, 43200, -2400, Scl },
   { 25500, 26850, -2520, Cen },
   { 28200, 28800, -2520, Lup },
   { 28800, 29557, -2520, Sco },
   {  8700,  9000, -2580, Cae },
   {  9000, 11850, -2580, Col },
   { 14400, 15060, -2580, Pup },
   {  6150,  6960, -2640, Eri },
   { 29557, 32100, -2730, Sco },
   { 32100, 34500, -2730, CrA },
   { 34500, 36600, -2730, Sgr },
   { 36600, 38400, -2730, Mic },
   {  5400,  6150, -2760, Eri },
   {  8100,  8700, -2790, Cae },
   { 27600, 28200, -2880, Lup },
   {     0,  4200, -2890, Phe },
   {  4800,  5400, -2940, Eri },
   {  7350,  7680, -2940, Hor },
   {  7680,  8100, -2940, Cae },
   { 38400, 39600, -3000, Gru },
   { 10800, 14400, -3045, Pup },
   { 14400, 14700, -3045, Vel },
   {  4350,  4800, -3060, Eri },
   {  6900,  7350, -3060, Hor },
   {     0,  3300, -3090, Phe },
   { 10800, 11100, -3150, Car },
   { 14700, 15210, -3180, Vel },
   {  6300,  6900, -3190, Hor },
   {  6900,  7200, -3190, Dor },
   {     0,  2850, -3210, Phe },
   {  3900,  4350, -3240, Eri },
   {  8100,  9000, -3240, Pic },
   { 27090, 27600, -3240, Lup },
   { 15210, 15900, -3270, Vel },
   { 11100, 11700, -3300, Car },
   { 21300, 23100, -3300, Cen },
   { 25500, 27090, -3300, Lup },
   { 27090, 27600, -3300, Nor },
   {  7200,  7800, -3390, Dor },
   { 15900, 19800, -3390, Vel },
   { 19800, 20250, -3390, Cen },
   { 31500, 32400, -3420, Ara },
   { 32400, 36600, -3420, Tel },
   { 39600, 42000, -3420, Gru },
   {  5760,  6300, -3450, Hor },
   {  9000,  9900, -3450, Pic },
   { 11700, 12300, -3480, Car },
   {     0,  2400, -3510, Phe },
   {  2400,  3900, -3510, Eri },
   { 42000, 43200, -3510, Phe },
   {  7800,  8250, -3540, Dor },
   { 27600, 29557, -3600, Nor },
   { 36600, 38400, -3600, Ind },
   {  9900, 10800, -3660, Pic },
   { 27300, 27600, -3660, Cir },
   { 29557, 29850, -3660, Ara },
   { 26850, 27300, -3815, Cir },
   { 29850, 30150, -3815, Ara },
   { 10800, 12300, -3840, Pic },
   { 12300, 16260, -3840, Car },
   { 20250, 21300, -3840, Cen },
   { 21300, 23100, -3840, Cru },
   { 23100, 26160, -3840, Cen },
   { 24300, 24600, -3900, Cir },
   { 30150, 30300, -3900, Ara },
   {  3900,  5760, -4050, Hor },
   {  5760,  8250, -4050, Ret },
   { 26550, 26850, -4050, Cir },
   { 30300, 31500, -4050, Ara },
   { 31500, 32400, -4050, Pav },
   { 39600, 42000, -4050, Tuc },
   {  8250, 11850, -4200, Dor },
   { 24600, 26550, -4200, Cir },
   { 26550, 30600, -4200, TrA },
   {     0,  2400, -4500, Tuc },
   {  6300,  8250, -4500, Hyi },
   { 11850, 16260, -4500, Vol },
   { 16260, 20250, -4500, Car },
   { 20250, 24600, -4500, Mus },
   { 32400, 38400, -4500, Pav },
   { 38400, 42000, -4500, Ind },
   { 42000, 43200, -4500, Tuc },
   {  1350,  2400, -4560, Tuc },
   {     0,  6300, -4950, Hyi },
   { 13800, 24600, -4950, Cha },
   { 24600, 32400, -4950, Aps },
   {  6300, 13800, -5100, Men },
   {     0, 43200, -5400, Oct }
};

static short start[] = {
   355, 352, 343, 340, 332, 320,
   303, 288, 277, 266, 257, 251,
   239, 229, 221, 211, 203, 189,
   177, 163, 149, 136, 124, 104,
    87,  75,  69,  54,  43,  29,
    23,  16,  12,   6,   4,   0,   0
};


/*
======================================================================
constellation_pick()

Do a constellation pick from RA and Dec.

INPUTS
   r        right ascension, radians
   d        declination, radians
   e        epoch to which r and d precessed, as an mjd

RESULTS
   Returns an index for the constellation region that the coordinates
   belong to, or -1 if no constellation pick can be found.

The constellation is identified by linear search.  We look for a
member of cbound[] whose lower_dec is less than the declination of the
pick point and whose lower_ra and upper_ra bracket the pick point's
right ascension.  The data for that cbound[] member describes a sub-
region belonging to the picked constellation.

In geometric terms, the search amounts to starting at the north pole
and traveling south on a line of constant right ascension through the
pick point.  At every declination where a cbound[] member lives, we
wake up, and if (1) the pick point is now behind us, and (2) the
cbound[] segment is crossing our path, we know we've hit the edge of
a constellation subregion that the pick point belongs to.

The cbound[] right ascension and declination values are scaled into
integers; this conserves storage and makes comparisons faster.  The
start[] array, which gives starting points in cbound[] for different
declinations in 5-degree increments, further speeds the search by
skipping early parts of the list for which we know the search can't
succeed--geometrically, we start no more than 5 degrees north of the
pick point, rather than at the north pole.

The data in cbound[] are for epoch 1875.
======================================================================
*/

int
cns_pick(double r, double d, double e)
{
   double Mjd;
   unsigned short ra;
   short de, i;

   cal_mjd( 1, 1.0, 1875, &Mjd );
   precess( e, Mjd, &r, &d );
   ra = ( unsigned short )( radhr( r ) * 1800 );
   de = ( short )( raddeg( d ) * 60 );
   if (d < 0.0) --de;

   i = ( de + 5400 ) / 300;
   if ( i < 0 || i > 36 ) return -1;
   i = start[ i ];

   for ( ; i < NBOUNDS; i++ )
      if ( cbound[ i ].lower_dec <= de &&
           cbound[ i ].upper_ra   > ra &&
           cbound[ i ].lower_ra  <= ra ) break;

   return ( i == NBOUNDS ) ? -1 : ( int ) cbound[ i ].index;
}

/* given a constellation id (as from cns_pick()), return pointer to static
 * storage containg its name in the form "AAA: Name".
 * return "???: ???" if id is invalid.
 */
char *
cns_name (int id)
{
	if (id < 0 || id >= NCNS)
	    return ("???: ???");
	return (cns_namemap[id]);
}

/* return cns_namemap index matching first three chars in abbrev[], else -1.
 */
int
cns_id (char *abbrev)
{
	int i;

	for (i = 0; i < NCNS; i++)
	    if (strncmp (abbrev, cns_namemap[i], 3) == 0)
		return (i);
	return (-1);
}

/* edges of constant ra */
static struct {
   unsigned short ra;	/* hours * 1800 */
   short dec0, dec1;	/* degrees * 60 */
} ra_edges[] = {
    {     0, -4950, -4500 },
    {     0,   600,   750 },
    {     0,  1680,  1879 },
    {   120,  1320,  1680 },
    {   255,   750,  1260 },
    {   255,  1260,  1320 },
    {   300,  2760,  2880 },
    {   599,  -420,     0 },
    {   599,     0,   120 },
    {   599,  3960,  4620 },
    {  1290,  1425,  1980 },
    {  1350, -4500, -4560 },
    {  1530,  1260,  1425 },
    {  1560,  2760,  2880 },
    {  2010,  2880,  3000 },
    {  2399, -4560, -3510 },
    {  2399, -3510, -3210 },
    {  2460,  3000,  3240 },
    {  2534,  1680,  1980 },
    {  2534,  1980,  2100 },
    {  2849, -3210, -3090 },
    {  3000, -2400, -1530 },
    {  3000, -1530, -1462 },
    {  3000,   595,  1500 },
    {  3000,  1500,  1680 },
    {  3000,  2820,  3000 },
    {  3060,  3240,  3450 },
    {  3299, -3090, -2890 },
    {  3434,  3450,  3510 },
    {  3450,  1500,  1635 },
    {  3600,   120,   595 },
    {  3600,  2100,  2205 },
    {  3675,  2820,  3030 },
    {  3900, -4050, -3510 },
    {  3900, -3510, -3240 },
    {  4199, -2890, -2400 },
    {  4350, -3240, -3060 },
    {  4350,  1635,  1840 },
    {  4379,  3420,  3510 },
    {  4530,  2205,  3030 },
    {  4620,  2040,  2205 },
    {  4770, -1462,  -105 },
    {  4800, -3060, -2940 },
    {  4890,  1840,  2040 },
    {  5400, -2940, -2760 },
    {  5400, -2400, -2374 },
    {  5580,  3420,  4080 },
    {  5700,  3300,  3420 },
    {  5760, -3450, -4050 },
    {  5909,  -105,     0 },
    {  5909,     0,   595 },
    {  5909,   595,  1140 },
    {  5999,  3150,  3300 },
    {  6060,  1140,  1840 },
    {  6150, -2760, -2640 },
    {  6150,  4080,  4620 },
    {  6300, -5100, -4950 },
    {  6300, -4950, -4500 },
    {  6300, -3190, -3450 },
    {  6300, -2374, -2160 },
    {  6314,  4620,  4800 },
    {  6449,  -105,     0 },
    {  6750, -2160, -1462 },
    {  6899, -3060, -3190 },
    {  6960, -2640, -2400 },
    {  7200, -3390, -3190 },
    {  7349, -2940, -3060 },
    {  7680, -2400, -2940 },
    {  7680, -2220, -2400 },
    {  7799, -3540, -3390 },
    {  8100, -3240, -2940 },
    {  8100, -2940, -2790 },
    {  8100,  1800,  1840 },
    {  8100,  1840,  2160 },
    {  8249, -4500, -4200 },
    {  8249, -4050, -4200 },
    {  8249, -4050, -3540 },
    {  8249, -1800, -2220 },
    {  8310,     0,   930 },
    {  8400,  -240,     0 },
    {  8445,  2160,  3150 },
    {  8460, -1635, -1800 },
    {  8550,  1710,  1800 },
    {  8699, -2790, -2580 },
    {  8699,  -870, -1635 },
    {  8850,  -660,  -870 },
    {  8940,   930,   960 },
    {  9000, -3450, -3240 },
    {  9000, -2580, -1635 },
    {  9000,  3150,  3360 },
    {  9000,  4800,  5100 },
    {  9149,  -660,  -240 },
    {  9599,   930,   960 },
    {  9900, -3660, -3450 },
    { 10080,   750,   930 },
    { 10260,  1080,  1369 },
    { 10380,   750,  1080 },
    { 10499,  -660,  -240 },
    { 10589,  1290,  1369 },
    { 10589,  1369,  1680 },
    { 10589,  1680,  1710 },
    { 10800, -3840, -3660 },
    { 10800, -3150, -3045 },
    { 10800, -3045, -2580 },
    { 10980,  3240,  3360 },
    { 10980,  3360,  3720 },
    { 11010, -1980, -1635 },
    { 11010, -1635,  -660 },
    { 11100, -3300, -3150 },
    { 11190,  1050,  1290 },
    { 11235,  -240,     0 },
    { 11235,     0,   600 },
    { 11354,   600,   720 },
    { 11354,   720,  1050 },
    { 11700, -3480, -3300 },
    { 11700,  3000,  3240 },
    { 11759,  1680,  2130 },
    { 11849, -4200, -4500 },
    { 11849, -4200, -3840 },
    { 11849, -1980, -2580 },
    { 12240,  2670,  3000 },
    { 12299, -3840, -3480 },
    { 12479,   600,   720 },
    { 12600,   330,   600 },
    { 12600,   600,   750 },
    { 12600,  3600,  3720 },
    { 12630,    90,   330 },
    { 12960,     0,    90 },
    { 13260,  -660, -1980 },
    { 13260,  2130,  2670 },
    { 13500,   750,   810 },
    { 13800, -5100, -4950 },
    { 13800, -4500, -4950 },
    { 13950,  2010,  2130 },
    { 14054,   600,   810 },
    { 14054,   810,  1200 },
    { 14189,  1200,  1680 },
    { 14265,   420,   600 },
    { 14340,  3600,  4410 },
    { 14400, -3045, -2580 },
    { 14400,  1680,  2010 },
    { 14400,  5100,  5190 },
    { 14400,  5280,  5190 },
    { 14549,  -660,     0 },
    { 14549,     0,   420 },
    { 14700, -3180, -3045 },
    { 15060, -2580, -2205 },
    { 15060, -2205, -1020 },
    { 15060, -1020,  -660 },
    { 15150,  2820,  3600 },
    { 15209, -3270, -3180 },
    { 15449, -1140, -1020 },
    { 15899, -3390, -3270 },
    { 16259, -4500, -3840 },
    { 16349, -1440, -1140 },
    { 16500,  2520,  2820 },
    { 16500,  4410,  4920 },
    { 16650,   420,  2010 },
    { 16650,  2010,  2385 },
    { 16860, -2385, -2205 },
    { 16860, -1440, -2205 },
    { 17249,  -660,     0 },
    { 17249,     0,   420 },
    { 17249,  2385,  2520 },
    { 17550, -1590, -1440 },
    { 17789,  1710,  2010 },
    { 18300,  2400,  2520 },
    { 18450, -1750, -1590 },
    { 18900,  1410,  1710 },
    { 19049, -1870, -1750 },
    { 19200,  4800,  4920 },
    { 19350,  -660, -1140 },
    { 19350,  -360,  -660 },
    { 19350,  -360,     0 },
    { 19350,     0,   420 },
    { 19350,  1410,  1530 },
    { 19409,  2040,  2400 },
    { 19499, -2100, -1870 },
    { 19499, -1140, -1470 },
    { 19800, -3390, -2385 },
    { 19800, -2385, -2100 },
    { 19800,  1530,  1740 },
    { 19800,  1740,  2040 },
    { 20250, -4500, -3840 },
    { 20250, -3840, -3390 },
    { 20399,  3990,  4410 },
    { 20700,  4620,  4800 },
    { 20730,  -360,     0 },
    { 20730,     0,   660 },
    { 21299, -3840, -3300 },
    { 21299, -1470,  -660 },
    { 21299,  -660,  -360 },
    { 21360,   660,   840 },
    { 21360,   840,  1740 },
    { 21600,  1740,  2040 },
    { 21600,  2040,  2700 },
    { 21600,  3840,  3990 },
    { 21749,  2700,  3180 },
    { 22050, -1980, -2100 },
    { 22199,  1920,  2040 },
    { 22649, -1770, -1980 },
    { 22649, -1470, -1320 },
    { 23099, -3840, -3300 },
    { 23099, -1320,  -660 },
    { 23099,   840,   900 },
    { 23400,  4620,  4200 },
    { 23850,  1710,  1920 },
    { 24300, -3900, -3840 },
    { 24300,   480,   900 },
    { 24300,   900,  1710 },
    { 24300,  2910,  3180 },
    { 24300,  3780,  3840 },
    { 24449,  4800,  4620 },
    { 24600, -4950, -4500 },
    { 24600, -4500, -4200 },
    { 24600, -4200, -3900 },
    { 25124,  1710,  1845 },
    { 25200,  4200,  3960 },
    { 25259,  1845,  2910 },
    { 25259,  2910,  3330 },
    { 25500, -3300, -2520 },
    { 25650, -1320, -1470 },
    { 25650,  -480, -1320 },
    { 25950,  3330,  3780 },
    { 26100,  5190,  4800 },
    { 26159, -3840, -3300 },
    { 26400,     0,  -480 },
    { 26550, -4200, -4050 },
    { 26850, -4050, -3814 },
    { 26850, -2520, -1770 },
    { 26850, -1470, -1770 },
    { 27090, -3300, -3240 },
    { 27149,  -195,     0 },
    { 27149,     0,   480 },
    { 27149,   480,  1560 },
    { 27300, -3814, -3660 },
    { 27329,  1560,  1980 },
    { 27450,  3180,  3330 },
    { 27599, -3660, -3600 },
    { 27599, -3600, -3300 },
    { 27599, -3240, -2880 },
    { 27779,  1980,  2400 },
    { 28200, -2880, -2520 },
    { 28200, -1770, -1200 },
    { 28200,  3960,  4200 },
    { 28350,  2400,  3090 },
    { 28350,  3090,  3180 },
    { 28650, -1200,  -480 },
    { 28650,  -480,  -195 },
    { 28650,   960,  1320 },
    { 28800, -2520, -1770 },
    { 28859,  1320,  1560 },
    { 28949,   240,   960 },
    { 29100,  1560,  1620 },
    { 29280, -1474, -1155 },
    { 29280, -1095,  -480 },
    { 29280,  -195,     0 },
    { 29280,     0,   240 },
    { 29399,  1620,  2400 },
    { 29475, -1155, -1095 },
    { 29557, -3660, -3600 },
    { 29557, -3600, -2730 },
    { 29557, -2730, -2520 },
    { 29759,  4200,  4500 },
    { 29849, -3814, -3660 },
    { 30150, -3900, -3814 },
    { 30150, -1800, -1474 },
    { 30150,   240,   769 },
    { 30299, -4050, -3900 },
    { 30600, -4050, -4200 },
    { 30600,  3030,  3090 },
    { 30900,  -960,  -600 },
    { 31050,   769,   859 },
    { 31500, -3420, -4050 },
    { 31500,  4500,  4800 },
    { 31649,  -700,  -600 },
    { 31680,  -960, -1800 },
    { 31800,  -700,  -600 },
    { 32099, -2730, -2220 },
    { 32099, -1800, -2220 },
    { 32099,  -240,     0 },
    { 32340,  -600,  -240 },
    { 32400, -4950, -4500 },
    { 32400, -4500, -4050 },
    { 32400, -2730, -3420 },
    { 32400,  4800,  5160 },
    { 32715,  1800,  2850 },
    { 32819,  2850,  3030 },
    { 32850,  -240,  -960 },
    { 32850,     0,   180 },
    { 32850,   270,   375 },
    { 32850,   720,   859 },
    { 33060,  1560,  1800 },
    { 33165,   180,   270 },
    { 33449,  -240,     0 },
    { 33449,     0,   120 },
    { 33591,   375,   720 },
    { 33960,  -960,  -721 },
    { 33960,  -721,  -240 },
    { 33960,   120,   375 },
    { 33960,   720,  1110 },
    { 33960,  1110,  1264 },
    { 33960,  1264,  1530 },
    { 33960,  1530,  1560 },
    { 34200,   970,  1110 },
    { 34349,  2850,  3330 },
    { 34500, -2220, -2730 },
    { 34500,  2610,  2850 },
    { 34650,  1150,  1264 },
    { 34664,  1530,  1650 },
    { 34664,  1650,  1800 },
    { 34844,  1800,  2190 },
    { 34920,  2190,  2610 },
    { 34950,  3330,  3480 },
    { 35400,  1650,  1740 },
    { 35580,  3480,  3570 },
    { 35699,   945,   970 },
    { 35699,  1150,  1275 },
    { 36000, -1680,  -721 },
    { 36000,  -540,  -721 },
    { 36000,  3570,  3690 },
    { 36255,   510,   945 },
    { 36300,  4500,  4800 },
    { 36450,   945,  1230 },
    { 36450,  1230,  1275 },
    { 36540,   120,   510 },
    { 36599, -3600, -3420 },
    { 36599, -3420, -2730 },
    { 36599, -2730, -1680 },
    { 36750,  3690,  4020 },
    { 36959,  -900,  -540 },
    { 36959,  -540,     0 },
    { 36959,     0,   120 },
    { 36966,  3570,  3655 },
    { 37020,  1170,  1230 },
    { 37080,  3289,  3655 },
    { 37200,  4020,  4500 },
    { 37499,   120,   360 },
    { 37575,   360,   709 },
    { 37650,  1680,  1740 },
    { 37800,  4800,  5160 },
    { 37800,  5160,  5169 },
    { 37890,   709,  1170 },
    { 38010,   709,   750 },
    { 38250,  1170,  1410 },
    { 38399, -4500, -3600 },
    { 38399, -3000, -2730 },
    { 38399, -2220, -2730 },
    { 38399, -1680, -2220 },
    { 38399, -1680, -1530 },
    { 38399,  -540,  -900 },
    { 38399,   120,   750 },
    { 38550,  1410,  1680 },
    { 38640,   120,   165 },
    { 39000,   105,   165 },
    { 39119,  1680,  2160 },
    { 39360, -1530,  -540 },
    { 39375,  2160,  2625 },
    { 39434,  2625,  2640 },
    { 39540,  2640,  3165 },
    { 39540,  3165,  3289 },
    { 39600, -4050, -3420 },
    { 39600, -3420, -3000 },
    { 39600,   105,   120 },
    { 39600,  2100,  2160 },
    { 39839,  3165,  3300 },
    { 40170,  3300,  3375 },
    { 40950,  -240,     0 },
    { 40950,     0,   120 },
    { 40950,   120,   450 },
    { 41070,  2070,  2100 },
    { 41160,  2070,  3150 },
    { 41160,  3150,  3375 },
    { 41160,  3375,  3544 },
    { 41400, -2220, -1530 },
    { 41400,  5169,  5280 },
    { 41700,  3544,  3780 },
    { 41999, -4500, -4050 },
    { 41999, -3510, -3420 },
    { 41999, -3420, -2400 },
    { 41999, -2400, -2220 },
    { 41999,  3000,  3150 },
    { 42300,  1924,  2070 },
    { 42449,  2880,  3000 },
    { 42449,  3780,  3960 },
    { 42750,  1879,  1924 },
    { 42899, -1530,  -420 },
    { 42899,  -420,  -240 },
    { 42899,   450,   600 },
};

#define	NRA	((int)(sizeof(ra_edges)/sizeof(ra_edges[0])))

/* edges of constant dec */
static struct {
   short dec;			/* degrees * 60 */
   unsigned short ra0, ra1;	/* hours * 1800 */
} dec_edges[] = {
    { -5100,  6300, 13800 },
    { -4950,     0,  6300 },
    { -4950, 13800, 24600 },
    { -4950, 24600, 32400 },
    { -4560,  1350,  2399 },
    { -4500,     0,  1350 },
    { -4500,  6300,  8249 },
    { -4500, 11849, 13800 },
    { -4500, 13800, 16259 },
    { -4500, 16259, 20250 },
    { -4500, 20250, 24600 },
    { -4500, 32400, 38399 },
    { -4500, 38399, 41999 },
    { -4500, 41999,     0 },
    { -4200,  8249, 11849 },
    { -4200, 24600, 26550 },
    { -4200, 30600, 26550 },
    { -4050,  3900,  5760 },
    { -4050,  5760,  8249 },
    { -4050, 26550, 26850 },
    { -4050, 30299, 30600 },
    { -4050, 31500, 30600 },
    { -4050, 31500, 32400 },
    { -4050, 39600, 41999 },
    { -3900, 24300, 24600 },
    { -3900, 30150, 30299 },
    { -3840, 10800, 11849 },
    { -3840, 11849, 12299 },
    { -3840, 12299, 16259 },
    { -3840, 20250, 21299 },
    { -3840, 21299, 23099 },
    { -3840, 23099, 24300 },
    { -3840, 24300, 26159 },
    { -3814, 26850, 27300 },
    { -3814, 29849, 30150 },
    { -3660,  9900, 10800 },
    { -3660, 27300, 27599 },
    { -3660, 29557, 29849 },
    { -3600, 27599, 29557 },
    { -3600, 36599, 38399 },
    { -3540,  7799,  8249 },
    { -3510,  2399,  3900 },
    { -3510,  2399, 41999 },
    { -3510,  3900,  2399 },
    { -3510, 41999,  2399 },
    { -3480, 11700, 12299 },
    { -3450,  6300,  5760 },
    { -3450,  9000,  9900 },
    { -3420, 32400, 31500 },
    { -3420, 32400, 36599 },
    { -3420, 39600, 41999 },
    { -3390,  7200,  7799 },
    { -3390, 15899, 19800 },
    { -3390, 19800, 20250 },
    { -3300, 11100, 11700 },
    { -3300, 21299, 23099 },
    { -3300, 25500, 26159 },
    { -3300, 26159, 27090 },
    { -3300, 27090, 27599 },
    { -3270, 15209, 15899 },
    { -3240,  3900,  4350 },
    { -3240,  8100,  9000 },
    { -3240, 27090, 27599 },
    { -3210,  2399,  2849 },
    { -3190,  6899,  6300 },
    { -3190,  6899,  7200 },
    { -3180, 14700, 15209 },
    { -3150, 10800, 11100 },
    { -3090,  2849,  3299 },
    { -3060,  4350,  4800 },
    { -3060,  7349,  6899 },
    { -3045, 10800, 14400 },
    { -3045, 14400, 14700 },
    { -3000, 38399, 39600 },
    { -2940,  4800,  5400 },
    { -2940,  7680,  7349 },
    { -2940,  7680,  8100 },
    { -2890,  3299,  4199 },
    { -2880, 27599, 28200 },
    { -2790,  8100,  8699 },
    { -2760,  5400,  6150 },
    { -2730, 29557, 32099 },
    { -2730, 32099, 32400 },
    { -2730, 34500, 32400 },
    { -2730, 34500, 36599 },
    { -2730, 38399, 36599 },
    { -2640,  6150,  6960 },
    { -2580,  8699,  9000 },
    { -2580, 10800,  9000 },
    { -2580, 11849, 10800 },
    { -2580, 14400, 15060 },
    { -2520, 25500, 26850 },
    { -2520, 28200, 28800 },
    { -2520, 29557, 28800 },
    { -2400,  3000, 41999 },
    { -2400,  4199,  3000 },
    { -2400,  4199,  5400 },
    { -2400,  6960,  7680 },
    { -2385, 16860, 19800 },
    { -2374,  5400,  6300 },
    { -2220,  8249,  7680 },
    { -2220, 32099, 34500 },
    { -2220, 38399, 41400 },
    { -2220, 41999, 41400 },
    { -2205, 16860, 15060 },
    { -2160,  6300,  6750 },
    { -2100, 19800, 19499 },
    { -2100, 22050, 19800 },
    { -1980, 11010, 11849 },
    { -1980, 13260, 11849 },
    { -1980, 22649, 22050 },
    { -1870, 19499, 19049 },
    { -1800,  8460,  8249 },
    { -1800, 30150, 31680 },
    { -1800, 31680, 32099 },
    { -1770, 26850, 22649 },
    { -1770, 26850, 28200 },
    { -1770, 28800, 28200 },
    { -1750, 19049, 18450 },
    { -1680, 36000, 36599 },
    { -1680, 36599, 38399 },
    { -1635,  8699,  8460 },
    { -1635,  8699,  9000 },
    { -1635,  9000, 11010 },
    { -1590, 18450, 17550 },
    { -1530,  3000, 42899 },
    { -1530, 38399, 39360 },
    { -1530, 41400, 39360 },
    { -1530, 41400, 42899 },
    { -1530, 42899,  3000 },
    { -1530, 42899, 41400 },
    { -1474, 29280, 30150 },
    { -1470, 19499, 21299 },
    { -1470, 21299, 22649 },
    { -1470, 25650, 26850 },
    { -1462,  3000,  4770 },
    { -1462,  4770,  6750 },
    { -1440, 16349, 16860 },
    { -1440, 17550, 16860 },
    { -1320, 22649, 23099 },
    { -1320, 23099, 25650 },
    { -1200, 28200, 28650 },
    { -1155, 29280, 29475 },
    { -1140, 15449, 16349 },
    { -1140, 19350, 19499 },
    { -1095, 29280, 29475 },
    { -1020, 15060, 15449 },
    {  -960, 30900, 31680 },
    {  -960, 32850, 31680 },
    {  -960, 32850, 33960 },
    {  -900, 38399, 36959 },
    {  -870,  8850,  8699 },
    {  -721, 36000, 33960 },
    {  -700, 31649, 31800 },
    {  -660,  9149,  8850 },
    {  -660, 10499,  9149 },
    {  -660, 11010, 10499 },
    {  -660, 11010, 13260 },
    {  -660, 14549, 13260 },
    {  -660, 15060, 14549 },
    {  -660, 17249, 19350 },
    {  -660, 23099, 21299 },
    {  -600, 30900, 31649 },
    {  -600, 31800, 32340 },
    {  -540, 36959, 36000 },
    {  -540, 39360, 38399 },
    {  -480, 26400, 25650 },
    {  -480, 28650, 29280 },
    {  -420,   599, 42899 },
    {  -360, 20730, 19350 },
    {  -360, 21299, 20730 },
    {  -240,  8400,  9149 },
    {  -240, 10499, 11235 },
    {  -240, 32099, 32340 },
    {  -240, 33449, 32850 },
    {  -240, 33960, 33449 },
    {  -240, 40950, 42899 },
    {  -195, 28650, 27149 },
    {  -195, 28650, 29280 },
    {  -105,  4770,  5909 },
    {  -105,  5909,  6449 },
    {     0,  6449,  8310 },
    {     0,  8310,  8400 },
    {     0, 12960, 14549 },
    {     0, 27149, 26400 },
    {     0, 32099, 32850 },
    {    90, 12630, 12960 },
    {   105, 39000, 39600 },
    {   120,   599,  3600 },
    {   120, 33449, 33960 },
    {   120, 36540, 36959 },
    {   120, 36959, 37499 },
    {   120, 37499, 38399 },
    {   120, 38399, 38640 },
    {   120, 39600, 40950 },
    {   165, 38640, 39000 },
    {   180, 32850, 33165 },
    {   240, 28949, 29280 },
    {   240, 29280, 30150 },
    {   270, 33165, 32850 },
    {   330, 12600, 12630 },
    {   360, 37499, 37575 },
    {   375, 32850, 33591 },
    {   375, 33591, 33960 },
    {   420, 14265, 14549 },
    {   420, 14549, 16650 },
    {   420, 16650, 17249 },
    {   420, 17249, 19350 },
    {   450, 40950, 42899 },
    {   480, 24300, 27149 },
    {   510, 36255, 36540 },
    {   595,  3000,  3600 },
    {   595,  3600,  5909 },
    {   600,     0, 42899 },
    {   600, 11235, 11354 },
    {   600, 12479, 12600 },
    {   600, 14054, 14265 },
    {   600, 42899,     0 },
    {   660, 20730, 21360 },
    {   709, 37575, 37890 },
    {   709, 37890, 38010 },
    {   720, 11354, 12479 },
    {   720, 32850, 33591 },
    {   720, 33591, 33960 },
    {   750,     0,   255 },
    {   750, 10080, 10380 },
    {   750, 12600, 13500 },
    {   750, 38010, 38399 },
    {   769, 30150, 31050 },
    {   810, 13500, 14054 },
    {   840, 21360, 23099 },
    {   859, 31050, 32850 },
    {   900, 23099, 24300 },
    {   930,  8310,  8940 },
    {   930,  9599, 10080 },
    {   945, 35699, 36255 },
    {   945, 36255, 36450 },
    {   960,  8940,  9599 },
    {   960, 28650, 28949 },
    {   970, 34200, 35699 },
    {  1050, 11190, 11354 },
    {  1080, 10260, 10380 },
    {  1110, 33960, 34200 },
    {  1140,  5909,  6060 },
    {  1150, 34650, 35699 },
    {  1170, 37020, 37890 },
    {  1170, 37890, 38250 },
    {  1200, 14054, 14189 },
    {  1230, 36450, 37020 },
    {  1260,   255,  1530 },
    {  1264, 33960, 34650 },
    {  1275, 35699, 36450 },
    {  1290, 10589, 11190 },
    {  1320,   120,   255 },
    {  1320, 28650, 28859 },
    {  1369, 10260, 10589 },
    {  1410, 18900, 19350 },
    {  1410, 38250, 38550 },
    {  1425,  1290,  1530 },
    {  1500,  3000,  3450 },
    {  1530, 19350, 19800 },
    {  1530, 33960, 34664 },
    {  1560, 27149, 27329 },
    {  1560, 27329, 28859 },
    {  1560, 28859, 29100 },
    {  1560, 33060, 33960 },
    {  1620, 29100, 29399 },
    {  1635,  3450,  4350 },
    {  1650, 34664, 35400 },
    {  1680,     0,   120 },
    {  1680,  2534,  3000 },
    {  1680, 10589, 11759 },
    {  1680, 14189, 14400 },
    {  1680, 37650, 38550 },
    {  1680, 38550, 39119 },
    {  1710,  8550, 10589 },
    {  1710, 17789, 18900 },
    {  1710, 23850, 24300 },
    {  1710, 24300, 25124 },
    {  1740, 19800, 21360 },
    {  1740, 21360, 21600 },
    {  1740, 35400, 37650 },
    {  1800,  8100,  8550 },
    {  1800, 32715, 33060 },
    {  1800, 34664, 34844 },
    {  1840,  4350,  4890 },
    {  1840,  4890,  6060 },
    {  1840,  6060,  8100 },
    {  1845, 25124, 25259 },
    {  1879,     0, 42750 },
    {  1920, 22199, 23850 },
    {  1924, 42300, 42750 },
    {  1980,  1290,  2534 },
    {  1980, 27329, 27779 },
    {  2010, 13950, 14400 },
    {  2010, 14400, 16650 },
    {  2010, 16650, 17789 },
    {  2040,  4620,  4890 },
    {  2040, 19409, 19800 },
    {  2040, 21600, 22199 },
    {  2070, 41070, 41160 },
    {  2070, 41160, 42300 },
    {  2100,  2534,  3600 },
    {  2100, 39600, 41070 },
    {  2130, 11759, 13260 },
    {  2130, 13260, 13950 },
    {  2160,  8100,  8445 },
    {  2160, 39119, 39375 },
    {  2160, 39375, 39600 },
    {  2190, 34844, 34920 },
    {  2205,  3600,  4530 },
    {  2205,  4530,  4620 },
    {  2385, 16650, 17249 },
    {  2400, 18300, 19409 },
    {  2400, 27779, 28350 },
    {  2400, 28350, 29399 },
    {  2520, 16500, 17249 },
    {  2520, 17249, 18300 },
    {  2610, 34500, 34920 },
    {  2625, 39375, 39434 },
    {  2640, 39434, 39540 },
    {  2670, 12240, 13260 },
    {  2700, 21600, 21749 },
    {  2760,   300,  1560 },
    {  2820,  3000,  3675 },
    {  2820, 15150, 16500 },
    {  2850, 32715, 32819 },
    {  2850, 32819, 34349 },
    {  2850, 34349, 34500 },
    {  2880,   300, 42449 },
    {  2880,  1560,  2010 },
    {  2880, 42449,   300 },
    {  2910, 24300, 25259 },
    {  3000,  2010,  2460 },
    {  3000,  2460,  3000 },
    {  3000, 11700, 12240 },
    {  3000, 41999, 42449 },
    {  3030,  3675,  4530 },
    {  3030, 30600, 32819 },
    {  3090, 28350, 30600 },
    {  3150,  5999,  8445 },
    {  3150,  8445,  9000 },
    {  3150, 41160, 41999 },
    {  3165, 39540, 39839 },
    {  3180, 21749, 24300 },
    {  3180, 27450, 28350 },
    {  3240,  2460,  3060 },
    {  3240, 10980, 11700 },
    {  3289, 37080, 39540 },
    {  3300,  5700,  5999 },
    {  3300, 39839, 40170 },
    {  3330, 25259, 25950 },
    {  3330, 25950, 27450 },
    {  3330, 34349, 34950 },
    {  3360,  9000, 10980 },
    {  3375, 40170, 41160 },
    {  3420,  4379,  5580 },
    {  3420,  5580,  5700 },
    {  3450,  3060,  3434 },
    {  3480, 34950, 35580 },
    {  3510,  3434,  4379 },
    {  3544, 41160, 41700 },
    {  3570, 35580, 36000 },
    {  3570, 36000, 36966 },
    {  3600, 12600, 14340 },
    {  3600, 14340, 15150 },
    {  3655, 36966, 37080 },
    {  3690, 36000, 36750 },
    {  3720, 10980, 12600 },
    {  3780, 24300, 25950 },
    {  3780, 41700, 42449 },
    {  3840, 21600, 24300 },
    {  3960,   599, 42449 },
    {  3960, 25200, 28200 },
    {  3960, 42449,   599 },
    {  3990, 20399, 21600 },
    {  4020, 36750, 37200 },
    {  4080,  5580,  6150 },
    {  4200, 23400, 25200 },
    {  4200, 28200, 29759 },
    {  4410, 14340, 16500 },
    {  4410, 16500, 20399 },
    {  4500, 29759, 31500 },
    {  4500, 36300, 37200 },
    {  4620,   599,  6150 },
    {  4620,  6150,  6314 },
    {  4620, 20700, 23400 },
    {  4620, 24449, 23400 },
    {  4800,  6314,  9000 },
    {  4800, 19200, 20700 },
    {  4800, 26100, 24449 },
    {  4800, 31500, 32400 },
    {  4800, 36300, 37800 },
    {  4920, 16500, 19200 },
    {  5100,  9000, 14400 },
    {  5160, 32400, 37800 },
    {  5169, 37800, 41400 },
    {  5190, 14400, 26100 },
    {  5280,  6300, 14400 },
    {  5280, 41400,  6300 },
};

#define	NDEC	((int)(sizeof(dec_edges)/sizeof(dec_edges[0])))

/* given an epoch, give caller a list of all constellation edges.
 * return count if ok, else -1.
 * N.B. caller should *not* free what we return because we cache it here.
 */
int
cns_edges (double e, double **ra0p, double **dec0p, double **ra1p,
double **dec1p)
{
#define	NEDGES	(NRA+NDEC)
	static double *ra0, *dec0, *ra1, *dec1;
	static double laste = -12345.6;		/* any bogus value */
	double mjd0;
	int i, n;

	/* if same epoch just return the same list */
	if (e == laste) {
	    *ra0p = ra0;
	    *dec0p = dec0;
	    *ra1p = ra1;
	    *dec1p = dec1;
	    return (NEDGES);
	}

	/* get space for arrays, first time only */
	if (!ra0) {
	    ra0 = (double *)malloc (NEDGES * sizeof(double));
	    if (!ra0)
		return (-1);
	    dec0 = (double *)malloc (NEDGES * sizeof(double));
	    if (!dec0) {
		free ((void *)ra0);
		return (-1);
	    }
	    ra1 = (double *)malloc (NEDGES * sizeof(double));
	    if (!ra1) {
		free ((void *)ra0);
		free ((void *)dec0);
		return (-1);
	    }
	    dec1 = (double *)malloc (NEDGES * sizeof(double));
	    if (!dec1) {
		free ((void *)ra0);
		free ((void *)dec0);
		free ((void *)ra1);
		return (-1);
	    }
	}

	/* prepare for precession from 1875 */
	cal_mjd (1, 1.0, 1875, &mjd0);

	/* build the constant-ra edge lists */
	n = 0;
	for (i = 0; i < NRA; i++) {
	    ra0[n] = ra1[n] = hrrad((double)ra_edges[i].ra/1800.0);
	    dec0[n] = degrad((double)ra_edges[i].dec0/60.0);
	    dec1[n] = degrad((double)ra_edges[i].dec1/60.0);
	    precess (mjd0, e, &ra0[n], &dec0[n]);
	    precess (mjd0, e, &ra1[n], &dec1[n]);
	    n++;
	}

	/* add the constant-dec edge lists */
	for (i = 0; i < NDEC; i++) {
	    ra0[n] = hrrad((double)dec_edges[i].ra0/1800.0);
	    ra1[n] = hrrad((double)dec_edges[i].ra1/1800.0);
	    dec0[n] = dec1[n] = degrad((double)dec_edges[i].dec/60.0);
	    precess (mjd0, e, &ra0[n], &dec0[n]);
	    precess (mjd0, e, &ra1[n], &dec1[n]);
	    n++;
	}

	/* sanity check the count */
	if (n != NEDGES) {
	    printf ("cns_edges(): n=%d NEDGES=%ld\n", n, (long)NEDGES);
	    abort();
	}
	
	/* ok */
	*ra0p = ra0;
	*dec0p = dec0;
	*ra1p = ra1;
	*dec1p = dec1;
	laste = e;
	return (NEDGES);
}

/* given an ra, dec and epoch return the list of constellation ids which
 * *may* fall within the given radius of said location.
 * return the number of ids.
 * ids[] need be no larger than 89.
 */
/* ARGSUSED */
int
cns_list (double ra, double dec, double e, double rad, int ids[])
{
	int i;

	/* TODO: this! */
	for (i = 0; i < NCNS; i++)
	    ids[i] = i;
	return (NCNS);
}

/* epoch 2000 RA/Dec of constellation figure end-points.
 * drawcodes: 0=move to; 1=draw to; 2=draw to dashed; 3=end
 */
typedef struct {
    short drawcode;
    unsigned short ra;	/* hours * 1800 */
    short dec;		/* degrees * 60 */
} ConFig;

/* Andromeda */
static ConFig _Andromeda_figs[] = {
    { 0,  3717,  2539},		/* move gamma 1 */
    { 1,  2091,  2137},		/* draw beta */
    { 1,  1179,  1851},		/* draw delta */
    { 1,   251,  1745},		/* draw alpha */
    { 0,  1716,  1405},		/* move eta */
    { 1,  1420,  1456},		/* draw zeta */
    { 1,  1156,  1758},		/* draw epsilon */
    { 1,  1179,  1851},		/* draw delta */
    { 1,  1106,  2023},		/* draw pi */
    { 1,   512,  2320},		/* draw theta */
    { 1, 42544,  2596},		/* draw iota */
    { 1, 42612,  2660},		/* draw kappa */
    { 1, 42526,  2787},		/* draw lambda */
    { 0, 42544,  2596},		/* move iota */
    { 1, 41457,  2539},		/* draw omicron */
    { 0,  1106,  2023},		/* move pi */
    { 1,  2091,  2137},		/* draw beta */
    { 1,  1702,  2309},		/* draw mu */
    { 1,  1494,  2464},		/* draw nu */
    { 1,  2085,  2834},		/* draw phi */
    { 1,  2939,  2917},		/* draw 51 */
    {-1,     0,     0},
};

/* Antlia */
static ConFig _Antlia_figs[] = {
    { 0, 17077, -2157},		/* move epsilon */
    { 2, 18814, -1864},		/* dotted alpha */
    { 2, 19701, -2228},		/* dotted iota */
    {-1,     0,     0},
};

/* Apus */
static ConFig _Apus_figs[] = {
    { 0, 26635, -4742},		/* move alpha */
    { 2, 29803, -4733},		/* dotted gamma */
    { 2, 30092, -4651},		/* dotted beta */
    { 2, 29410, -4721},		/* dotted delta 1 */
    { 2, 29803, -4733},		/* dotted gamma */
    {-1,     0,     0},
};

/* Aquarius */
static ConFig _Aquarius_figs[] = {
    { 0, 37430,  -569},		/* move epsilon */
    { 1, 38746,  -334},		/* draw beta */
    { 1, 39773,   -19},		/* draw alpha */
    { 1, 40249,   -83},		/* draw gamma */
    { 1, 40464,    -1},		/* draw zeta 1 */
    { 1, 40358,    82},		/* draw pi */
    { 1, 39773,   -19},		/* draw alpha */
    { 0, 40464,    -1},		/* move zeta 1 */
    { 1, 40660,    -7},		/* draw eta */
    { 0, 38746,  -334},		/* move beta */
    { 1, 39793,  -832},		/* draw iota */
    { 0, 39773,   -19},		/* move alpha */
    { 1, 40105,  -467},		/* draw theta */
    { 1, 41178,  -454},		/* draw lambda */
    { 1, 41829,  -362},		/* draw phi */
    { 1, 41937,  -550},		/* draw psi 2 */
    { 1, 41239,  -949},		/* draw delta */
    { 1, 41087,  -815},		/* draw tau 2 */
    { 1, 41178,  -454},		/* draw lambda */
    { 0, 42089, -1206},		/* move 98 */
    { 1, 41937,  -550},		/* draw psi 2 */
    { 1, 41683, -1270},		/* draw 88 */
    {-1,     0,     0},
};

/* Aquila */
static ConFig _Aquila_figs[] = {
    { 0, 35859,   384},		/* move beta */
    { 1, 35723,   532},		/* draw alpha */
    { 1, 35587,   636},		/* draw gamma */
    { 1, 34964,   186},		/* draw delta */
    { 1, 34387,  -292},		/* draw lambda */
    { 1, 34362,   831},		/* draw zeta */
    { 1, 34964,   186},		/* draw delta */
    { 1, 35774,    60},		/* draw eta */
    { 1, 36339,   -49},		/* draw theta */
    { 1, 35301,   -77},		/* draw iota */
    { 1, 34387,  -292},		/* draw lambda */
    {-1,     0,     0},
};

/* Ara */
static ConFig _Ara_figs[] = {
    { 0, 31811, -2964},		/* move lambda */
    { 1, 31555, -2992},		/* draw alpha */
    { 1, 31359, -3331},		/* draw beta */
    { 1, 31361, -3382},		/* draw gamma */
    { 1, 31532, -3641},		/* draw delta */
    { 1, 30293, -3542},		/* draw eta */
    { 1, 30558, -3359},		/* draw zeta */
    { 1, 30587, -3189},		/* draw epsilon 1 */
    { 1, 31555, -2992},		/* draw alpha */
    {-1,     0,     0},
};

/* Aries */
static ConFig _Aries_figs[] = {
    { 0,  3405,  1157},		/* move gamma */
    { 1,  3439,  1248},		/* draw beta */
    { 1,  3815,  1407},		/* draw alpha */
    { 1,  5099,  1635},		/* draw 41 */
    {-1,     0,     0},
};

/* Auriga */
static ConFig _Auriga_figs[] = {
    { 0,  9500,  2759},		/* move capella */
    { 1, 10785,  2696},		/* draw beta */
    { 1, 10791,  2232},		/* draw theta */
    { 1,  9788,  1716},		/* draw beta tau */
    { 1,  8909,  1989},		/* draw iota */
    { 1,  9500,  2759},		/* draw capella */
    { 1, 10785,  3257},		/* draw delta */
    { 1, 10785,  2696},		/* draw beta */
    { 0,  9500,  2759},		/* move capella */
    { 1,  9059,  2629},		/* draw epsilon */
    { 1,  9074,  2464},		/* draw zeta */
    {-1,     0,     0},
};

/* Bootes */
static ConFig _Bootes_figs[] = {
    { 0, 26434,   823},		/* move zeta */
    { 1, 25669,  1150},		/* draw arcturus */
    { 1, 26549,  1624},		/* draw epsilon */
    { 1, 27465,  1998},		/* draw delta */
    { 1, 27058,  2423},		/* draw beta */
    { 1, 26162,  2298},		/* draw gamma */
    { 1, 26154,  1822},		/* draw rho */
    { 1, 25669,  1150},		/* draw arcturus */
    { 1, 25040,  1103},		/* draw eta */
    { 1, 24817,  1047},		/* draw tau */
    { 0, 26162,  2298},		/* move gamma */
    { 1, 25691,  2765},		/* draw lambda */
    { 1, 25604,  3107},		/* draw kappa 2 */
    { 1, 25955,  3111},		/* draw theta */
    { 1, 25691,  2765},		/* draw lambda */
    {-1,     0,     0},
};

/* Caelum */
static ConFig _Caelum_figs[] = {
    { 0,  9132, -2129},		/* move gamma */
    { 2,  8461, -2228},		/* dotted beta */
    { 2,  8416, -2511},		/* dotted alpha */
    { 2,  8125, -2697},		/* dotted delta */
    {-1,     0,     0},
};

/* Camelopardalis */
static ConFig _Camelopardalis_figs[] = {
    { 0,  8918,  3225},		/* move 7 */
    { 2,  9102,  3626},		/* dotted beta */
    { 2,  8821,  3980},		/* dotted alpha */
    { 2,  6910,  4279},		/* dotted gamma */
    { 2,  6885,  3931},		/* dotted 3:49:31 65:31:34 */
    { 2,  6272,  3596},		/* dotted 6272 3596 */
    { 0,  8821,  3980},		/* move alpha */
    { 2, 11365,  4159},		/* dotted 6:18:51 69:19:11 */
    { 2, 12602,  4618},		/* dotted 7:00:04 76:58:39 */
    {-1,     0,     0},
};

/* Cancer */
static ConFig _Cancer_figs[] = {
    { 0, 15800,  1725},		/* move iota1 */
    { 2, 15698,  1288},		/* dotted gamma */
    { 2, 15740,  1089},		/* dotted delta */
    { 2, 16154,   711},		/* dotted alpha */
    { 0, 15740,  1089},		/* move delta */
    { 2, 14895,   551},		/* dotted beta */
    {-1,     0,     0},
};

/* Canes Venatici */
static ConFig _Canes_Venatici_figs[] = {
    { 0, 23280,  2299},		/* move alpha2 */
    { 2, 22612,  2481},		/* dotted beta */
    {-1,     0,     0},
};

/* Canis Major */
static ConFig _Canis_Major_figs[] = {
    { 0, 13322, -1758},		/* move eta */
    { 1, 12851, -1583},		/* draw delta */
    { 1, 12690, -1430},		/* draw omicron2 */
    { 1, 12154, -1002},		/* draw alpha */
    { 1, 11481, -1077},		/* draw beta */
    { 1, 11900, -1155},		/* draw nu2 */
    { 1, 12423, -1451},		/* draw omicron1 */
    { 1, 12558, -1738},		/* draw epsilon */
    { 1, 12651, -1676},		/* draw sigma */
    { 1, 12851, -1583},		/* draw delta */
    { 0, 12154, -1002},		/* move alpha */
    { 2, 12484, -1023},		/* dotted iota */
    { 2, 12712,  -938},		/* dotted gamma */
    { 2, 12483,  -842},		/* dotted mu */
    { 2, 12484, -1023},		/* dotted iota */
    {-1,     0,     0},
};

/* Canis Minor */
static ConFig _Canis_Minor_figs[] = {
    { 0, 13779,   313},		/* move alpha */
    { 1, 13414,   497},		/* draw beta */
    {-1,     0,     0},
};

/* Capricornus */
static ConFig _Capricornus_figs[] = {
    { 0, 36529,  -750},		/* move alpha1 */
    { 1, 36630,  -886},		/* draw beta */
    { 1, 37382, -1516},		/* draw psi */
    { 1, 37554, -1615},		/* draw omega */
    { 1, 38013, -1500},		/* draw 24 */
    { 1, 38600, -1344},		/* draw zeta */
    { 1, 38912, -1167},		/* draw epsilon */
    { 1, 39211,  -967},		/* draw delta */
    { 1, 39002,  -999},		/* draw gamma */
    { 1, 38467, -1010},		/* draw iota */
    { 1, 37978, -1033},		/* draw theta */
    { 1, 36529,  -750},		/* draw alpha1 */
    {-1,     0,     0},
};

/* Carina */
static ConFig _Carina_figs[] = {
    { 0, 11518, -3161},		/* move canopus */
    { 1, 16596, -4183},		/* draw beta */
    { 1, 18412, -4202},		/* draw omega */
    { 1, 19288, -3863},		/* draw theta */
    { 1, 19996, -3745},		/* draw 11:06:32 -62:25:27 */
    { 1, 20178, -3619},		/* draw 11:12:36 -60:19:03 */
    { 1, 20057, -3538},		/* draw 11:08:35 -58:58:30 */
    { 1, 19605, -3531},		/* draw 10:53:30 -58:51:12 */
    { 1, 19351, -3581},		/* draw eta */
    { 1, 18960, -3701},		/* draw 10:32:01 -61:41:07 */
    { 1, 19288, -3863},		/* draw theta */
    { 0, 18960, -3701},		/* move 10:32:01 -61:41:07 */
    { 1, 18512, -3679},		/* draw 10:17:05 -61:19:56 */
    { 1, 16712, -3556},		/* draw iota */
    { 1, 15741, -3282},		/* draw delta vel */
    { 0, 16712, -3556},		/* move iota */
    { 1, 15075, -3570},		/* draw epsilon */
    { 1, 14303, -3178},		/* draw chi */
    { 1, 14686, -2840},		/* draw gamma2 vel */
    {-1,     0,     0},
};

/* Cassiopeia */
static ConFig _Cassiopeia_figs[] = {
    { 0,   275,  3548},		/* move beta */
    { 1,  1215,  3392},		/* draw alpha */
    { 1,  1701,  3643},		/* draw gamma */
    { 1,  2574,  3614},		/* draw delta */
    { 1,  3431,  3820},		/* draw epsilon */
    {-1,     0,     0},
};

/* Centaurus */
static ConFig _Centaurus_figs[] = {
    { 0, 26387, -3650},		/* move alpha1 */
    { 1, 25314, -3622},		/* draw hadar (agena) */
    { 1, 24596, -3207},		/* draw epsilon */
    { 1, 25066, -2837},		/* draw zeta */
    { 1, 25251, -2736},		/* draw upsilon2 */
    { 1, 25160, -2688},		/* draw upsilon1 */
    { 1, 25148, -2526},		/* draw phi */
    { 1, 26265, -2529},		/* draw eta */
    { 1, 26974, -2526},		/* draw kappa */
    { 0, 25148, -2526},		/* move phi */
    { 1, 25381, -2470},		/* draw chi */
    { 1, 25816, -2273},		/* draw psi */
    { 1, 25400, -2182},		/* draw theta */
    { 1, 24885, -2501},		/* draw nu */
    { 1, 24331, -2364},		/* draw 24331 -2364 */
    { 1, 24017, -2202},		/* draw iota */
    { 1, 23203, -2410},		/* draw 23203 -2410 */
    { 0, 24885, -2501},		/* move nu */
    { 1, 24888, -2548},		/* draw mu */
    { 1, 25066, -2837},		/* draw zeta */
    { 1, 22845, -2937},		/* draw gamma */
    { 1, 24596, -3207},		/* draw epsilon */
    { 0, 22845, -2937},		/* move gamma */
    { 1, 22441, -3013},		/* draw sigma */
    { 1, 21949, -3142},		/* draw rho */
    { 1, 21195, -3670},		/* draw 21195 -3670 */
    { 1, 20873, -3781},		/* draw lambda */
    { 0, 22441, -3013},		/* move sigma */
    { 1, 21850, -3043},		/* draw delta */
    { 1, 20430, -3269},		/* draw pi */
    {-1,     0,     0},
};

/* Cepheus */
static ConFig _Cepheus_figs[] = {
    { 0, 38357,  3755},		/* move alpha */
    { 1, 38659,  4233},		/* draw beta */
    { 1, 42580,  4657},		/* draw gamma */
    { 1, 41090,  3972},		/* draw iota */
    { 1, 42580,  4657},		/* draw gamma */
    { 0, 41090,  3972},		/* move iota */
    { 1, 39925,  3492},		/* draw zeta */
    { 1, 39163,  3667},		/* draw nu */
    { 1, 39105,  3526},		/* draw mu */
    { 1, 38357,  3755},		/* draw alpha */
    { 1, 37358,  3710},		/* draw eta */
    { 1, 36887,  3779},		/* draw theta */
    {-1,     0,     0},
};

/* Cetus */
static ConFig _Cetus_figs[] = {
    { 0,  4899,   194},		/* move gamma */
    { 1,  5468,   245},		/* draw alpha */
    { 1,  5391,   534},		/* draw lambda */
    { 1,  4948,   606},		/* draw mu */
    { 1,  4444,   507},		/* draw xi2 */
    { 1,  4676,   335},		/* draw nu */
    { 1,  4899,   194},		/* draw gamma */
    { 1,  4784,    19},		/* draw delta */
    { 1,  4180,  -178},		/* draw mira */
    { 1,  3343,  -620},		/* draw zeta */
    { 1,  2520,  -491},		/* draw theta */
    { 1,  2057,  -610},		/* draw eta */
    { 1,   582,  -529},		/* draw iota */
    { 1,  1307, -1079},		/* draw beta */
    { 1,  3122,  -956},		/* draw tau */
    { 1,  3343,  -620},		/* draw zeta */
    {-1,     0,     0},
};

/* Chamaeleon */
static ConFig _Chamaeleon_figs[] = {
    { 0, 14955, -4615},		/* move alpha */
    { 1, 15019, -4649},		/* draw theta */
    { 1, 19064, -4716},		/* draw gamma */
    { 1, 21588, -4693},		/* draw epsilon */
    { 1, 22150, -4758},		/* draw beta */
    { 1, 19357, -4828},		/* draw delta1 */
    { 1, 19064, -4716},		/* draw gamma */
    {-1,     0,     0},
};

/* Circinus */
static ConFig _Circinus_figs[] = {
    { 0, 27525, -3528},		/* move beta */
    { 1, 26475, -3898},		/* draw alpha */
    { 1, 27701, -3559},		/* draw gamma */
    {-1,     0,     0},
};

/* Columba */
static ConFig _Columba_figs[] = {
    { 0,  9936, -2128},		/* move epsilon */
    { 1, 10189, -2044},		/* draw alpha */
    { 1, 10528, -2146},		/* draw beta */
    { 1, 10726, -2117},		/* draw gamma */
    { 1, 11463, -2006},		/* draw delta */
    { 0, 10528, -2146},		/* move beta */
    { 1, 10774, -2568},		/* draw eta */
    {-1,     0,     0},
};

/* Coma Berenices */
static ConFig _Coma_Berenices_figs[] = {
    { 0, 23699,  1051},		/* move alpha */
    { 2, 23756,  1672},		/* dotted beta */
    { 2, 22408,  1696},		/* dotted gamma */
    {-1,     0,     0},
};

/* Corona Australis */
static ConFig _Corona_Australis_figs[] = {
    { 0, 34392, -2223},		/* move gamma */
    { 1, 34484, -2274},		/* draw alpha */
    { 1, 34500, -2360},		/* draw beta */
    { 1, 34450, -2429},		/* draw delta */
    { 1, 33405, -2538},		/* draw theta */
    {-1,     0,     0},
};

/* Corona Borealis */
static ConFig _Corona_Borealis_figs[] = {
    { 0, 27987,  1881},		/* move theta */
    { 1, 27834,  1746},		/* draw beta */
    { 1, 28040,  1602},		/* draw alpha */
    { 1, 28282,  1577},		/* draw gamma */
    { 1, 28487,  1564},		/* draw delta */
    { 1, 28727,  1612},		/* draw epsilon */
    { 1, 28843,  1791},		/* draw iota */
    {-1,     0,     0},
};

/* Corvus */
static ConFig _Corvus_figs[] = {
    { 0, 21852, -1483},		/* move alpha */
    { 1, 21903, -1357},		/* draw epsilon */
    { 1, 22631, -1403},		/* draw beta */
    { 1, 22495,  -990},		/* draw delta */
    { 1, 22074, -1052},		/* draw gamma */
    { 1, 21903, -1357},		/* draw epsilon */
    {-1,     0,     0},
};

/* Crater */
static ConFig _Crater_figs[] = {
    { 0, 21480, -1029},		/* move eta */
    { 1, 21142, -1101},		/* draw zeta */
    { 1, 20546, -1061},		/* draw gamma */
    { 1, 20500, -1126},		/* draw lambda */
    { 1, 20149, -1369},		/* draw beta */
    { 1, 19793, -1097},		/* draw alpha */
    { 1, 20380,  -886},		/* draw delta */
    { 1, 20546, -1061},		/* draw gamma */
    { 0, 20380,  -886},		/* move delta */
    { 1, 20538,  -651},		/* draw epsilon */
    { 1, 20900,  -588},		/* draw theta */
    {-1,     0,     0},
};

/* Crux */
static ConFig _Crux_figs[] = {
    { 0, 22397, -3785},		/* move alpha1 */
    { 1, 22534, -3426},		/* draw gamma */
    { 0, 23031, -3581},		/* move beta */
    { 1, 22054, -3524},		/* draw delta */
    {-1,     0,     0},
};

/* Cygnus */
static ConFig _Cygnus_figs[] = {
    { 0, 35121,  1677},		/* move beta1 */
    { 1, 35716,  1974},		/* draw chi */
    { 1, 35889,  2105},		/* draw eta */
    { 1, 36666,  2415},		/* draw gamma */
    { 1, 37386,  2038},		/* draw epsilon */
    { 2, 38188,  1813},		/* dotted zeta */
    { 2, 37715,  2470},		/* dotted nu */
    { 2, 37242,  2716},		/* dotted deneb */
    { 1, 36666,  2415},		/* draw gamma */
    { 1, 35549,  2707},		/* draw delta */
    { 2, 35091,  3103},		/* dotted iota */
    { 2, 34713,  3202},		/* dotted kappa */
    { 0, 35091,  3103},		/* move iota */
    { 2, 36464,  2862},		/* dotted 32 */
    { 2, 37242,  2716},		/* dotted deneb */
    {-1,     0,     0},
};

/* Delphinus */
static ConFig _Delphinus_figs[] = {
    { 0, 36996,   678},		/* move epsilon */
    { 1, 37018,   781},		/* draw eta */
    { 1, 37126,   875},		/* draw beta */
    { 1, 37303,   904},		/* draw delta */
    { 1, 37399,   967},		/* draw gamma1 */
    { 1, 37189,   954},		/* draw alpha */
    { 1, 37126,   875},		/* draw beta */
    {-1,     0,     0},
};

/* Dorado */
static ConFig _Dorado_figs[] = {
    { 0,  7680, -3089},		/* move gamma */
    { 1,  8219, -3302},		/* draw alpha */
    { 1, 10008, -3749},		/* draw beta */
    { 1, 10343, -3944},		/* draw delta */
    { 1, 10623, -3785},		/* draw 5:54:06 -63:05:23 */
    { 1, 10008, -3749},		/* draw beta */
    { 1,  9165, -3448},		/* draw zeta */
    { 1,  8219, -3302},		/* draw alpha */
    {-1,     0,     0},
};

/* Draco */
static ConFig _Draco_figs[] = {
    { 0, 20742,  4159},		/* move lambda */
    { 1, 22604,  4187},		/* draw kappa */
    { 1, 25331,  3862},		/* draw alpha */
    { 1, 27747,  3537},		/* draw iota */
    { 1, 28856,  3513},		/* draw theta */
    { 1, 29519,  3690},		/* draw eta */
    { 1, 30863,  3942},		/* draw zeta */
    { 1, 33022,  4280},		/* draw phi */
    { 1, 33031,  4363},		/* draw chi */
    { 0, 33022,  4280},		/* move phi */
    { 1, 34576,  4059},		/* draw delta */
    { 1, 35645,  4216},		/* draw epsilon */
    { 0, 34576,  4059},		/* move delta */
    { 1, 32205,  3412},		/* draw xi */
    { 1, 31565,  3311},		/* draw nu1 */
    { 1, 31513,  3138},		/* draw beta */
    { 1, 32298,  3089},		/* draw gamma */
    { 1, 32205,  3412},		/* draw xi */
    {-1,     0,     0},
};

/* Equuleus */
static ConFig _Equuleus_figs[] = {
    { 0, 38110,   607},		/* move gamma */
    { 2, 38234,   600},		/* dotted delta */
    { 2, 38274,   314},		/* dotted alpha */
    {-1,     0,     0},
};

/* Eridanus */
static ConFig _Eridanus_figs[] = {
    { 0,  9235,  -305},		/* move beta */
    { 1,  8786,  -327},		/* draw omega */
    { 1,  8565,  -195},		/* draw mu */
    { 1,  8289,  -201},		/* draw nu */
    { 1,  7555,  -410},		/* draw omicron1 */
    { 1,  7140,  -810},		/* draw gamma */
    { 1,  6784,  -726},		/* draw pi */
    { 1,  6697,  -585},		/* draw delta */
    { 1,  6387,  -567},		/* draw epsilon */
    { 1,  5292,  -533},		/* draw eta */
    { 1,  4923,  -831},		/* draw cet pi */
    { 1,  4953, -1114},		/* draw tau1 */
    { 1,  5471, -1417},		/* draw tau3 */
    { 1,  5985, -1305},		/* draw tau4 */
    { 1,  6413, -1297},		/* draw tau5 */
    { 1,  6805, -1394},		/* draw tau6 */
    { 1,  7011, -1476},		/* draw tau8 */
    { 1,  7197, -1440},		/* draw tau9 */
    { 1,  8205, -1786},		/* draw upsilon1 */
    { 1,  8266, -1833},		/* draw upsilon2 */
    { 1,  7921, -2041},		/* draw 43 */
    { 1,  7736, -2027},		/* draw upsilon4 */
    { 1,  6883, -2172},		/* draw 3:49:27 -36:12:01 */
    { 1,  6685, -2238},		/* draw 3:42:50 -37:18:49 */
    { 1,  6513, -2416},		/* draw 3:37:06 -40:16:29 */
    { 1,  5998, -2584},		/* draw 3:19:56 -43:04:11 */
    { 1,  5348, -2418},		/* draw theta2 */
    { 1,  4820, -2391},		/* draw iota */
    { 1,  4794, -2573},		/* draw 4794 -2573 */
    { 1,  4409, -2862},		/* draw kappa */
    { 1,  4095, -3090},		/* draw phi */
    { 1,  3478, -3096},		/* draw chi */
    { 1,  2931, -3434},		/* draw achernar */
    {-1,     0,     0},
};

/* Fornax */
static ConFig _Fornax_figs[] = {
    { 0,  5762, -1739},		/* move alpha */
    { 1,  5072, -1944},		/* draw beta */
    { 1,  3734, -1757},		/* draw nu */
    {-1,     0,     0},
};

/* Gemini */
static ConFig _Gemini_figs[] = {
    { 0, 10923,  1395},		/* move gem 1 */
    { 1, 11246,  1350},		/* draw gem eta */
    { 1, 11488,  1350},		/* draw gem mu */
    { 1, 12117,  1507},		/* draw gem epsilon */
    { 1, 12934,  1814},		/* draw gem tau */
    { 1, 13473,  1907},		/* draw gem rho */
    { 1, 13638,  1913},		/* draw castor */
    { 0, 13959,  1681},		/* move pollux */
    { 1, 13677,  1613},		/* draw gem upsilon */
    { 1, 13203,  1318},		/* draw gem delta */
    { 1, 12723,  1234},		/* draw gem zeta */
    { 1, 11931,   983},		/* draw gem gamma */
    { 0, 12383,  2037},		/* move gem theta */
    { 1, 12934,  1814},		/* draw gem tau */
    { 1, 13336,  1658},		/* draw gem 59 */
    { 1, 13677,  1613},		/* draw gem upsilon */
    { 1, 13933,  1463},		/* draw gem kappa */
    { 0, 13203,  1318},		/* move gem delta */
    { 1, 13142,   992},		/* draw gem lambda */
    { 1, 13001,   969},		/* draw gem 51 */
    { 1, 13001,   969},		/* draw gem 51 */
    { 1, 12158,   773},		/* draw gem xi */
    { 0, 12117,  1507},		/* move gem epsilon */
    { 1, 11668,  1212},		/* draw gem nu */
    {-1,     0,     0},
};

/* Grus */
static ConFig _Grus_figs[] = {
    { 0, 39417, -2241},		/* move gamma */
    { 1, 39783, -2372},		/* draw lambda */
    { 1, 40478, -2609},		/* draw delta1 */
    { 1, 40880, -2813},		/* draw beta */
    { 1, 41056, -3079},		/* draw epsilon */
    { 1, 41426, -3165},		/* draw zeta */
    { 0, 40880, -2813},		/* move beta */
    { 1, 39847, -2817},		/* draw alpha */
    { 1, 40478, -2609},		/* draw delta1 */
    {-1,     0,     0},
};

/* Hercules */
static ConFig _Hercules_figs[] = {
    { 0, 28580,  2547},		/* move chi */
    { 1, 29063,  2696},		/* draw phi */
    { 1, 29392,  2778},		/* draw tau */
    { 1, 29823,  2546},		/* draw sigma */
    { 1, 30086,  2335},		/* draw eta */
    { 1, 30038,  1896},		/* draw zeta */
    { 1, 29706,  1289},		/* draw beta */
    { 1, 29457,  1149},		/* draw gamma */
    { 1, 29562,   842},		/* draw omega */
    { 1, 29778,   689},		/* draw 29 */
    { 0, 29706,  1289},		/* move beta */
    { 1, 31039,   863},		/* draw alpha */
    { 1, 31050,  1490},		/* draw delta */
    { 1, 30608,  1855},		/* draw epsilon */
    { 1, 30038,  1896},		/* draw zeta */
    { 0, 31050,  1490},		/* move delta */
    { 1, 31522,  1566},		/* draw lambda */
    { 1, 31993,  1663},		/* draw mu */
    { 1, 32332,  1754},		/* draw xi */
    { 1, 32626,  1725},		/* draw omicron */
    { 0, 30608,  1855},		/* move epsilon */
    { 1, 31051,  2208},		/* draw pi */
    { 1, 30086,  2335},		/* draw eta */
    { 0, 31051,  2208},		/* move pi */
    { 1, 32287,  2235},		/* draw theta */
    { 1, 31783,  2760},		/* draw iota */
    {-1,     0,     0},
};

/* Horologium */
static ConFig _Horologium_figs[] = {
    { 0,  7620, -2537},		/* move alpha */
    { 1,  4876, -3048},		/* draw iota */
    { 1,  4722, -3152},		/* draw eta */
    { 1,  4819, -3273},		/* draw zeta */
    { 1,  5508, -3584},		/* draw mu */
    { 1,  5363, -3844},		/* draw beta */
    {-1,     0,     0},
};

/* Hydra */
static ConFig _Hydra_figs[] = {
    { 0, 26708, -1677},		/* move 58 */
    { 1, 25391, -1600},		/* draw pi */
    { 1, 23967, -1390},		/* draw gamma */
    { 1, 21387, -2034},		/* draw beta */
    { 1, 20790, -1911},		/* draw xi */
    { 1, 20149, -1369},		/* draw crt beta */
    { 1, 19793, -1097},		/* draw crt alpha */
    { 1, 19488,  -971},		/* draw nu */
    { 1, 18782, -1010},		/* draw mu */
    { 1, 18317,  -741},		/* draw lambda */
    { 1, 18153,  -783},		/* draw upsilon2 */
    { 1, 17744,  -890},		/* draw upsilon1 */
    { 1, 17027,  -519},		/* draw alpha */
    { 1, 17395,   -68},		/* draw iota */
    { 1, 16630,   138},		/* draw theta */
    { 1, 16061,   356},		/* draw zeta */
    { 1, 15803,   385},		/* draw epsilon */
    { 1, 15529,   342},		/* draw delta */
    { 1, 15562,   200},		/* draw sigma */
    { 1, 15696,   203},		/* draw eta */
    { 1, 15853,   350},		/* draw rho */
    { 1, 15803,   385},		/* draw epsilon */
    {-1,     0,     0},
};

/* Hydrus */
static ConFig _Hydrus_figs[] = {
    { 0,  6817, -4454},		/* move gamma */
    { 1,   772, -4635},		/* draw beta */
    { 1,  3563, -3694},		/* draw alpha */
    { 1,  6817, -4454},		/* draw gamma */
    {-1,     0,     0},
};

/* Indus */
static ConFig _Indus_figs[] = {
    { 0, 37644, -3507},		/* move beta */
    { 2, 37321, -3115},		/* dotted eta */
    { 2, 37127, -2837},		/* dotted alpha */
    { 2, 38396, -3206},		/* dotted theta */
    { 2, 39537, -3299},		/* dotted delta */
    {-1,     0,     0},
};

/* Lacerta */
static ConFig _Lacerta_figs[] = {
    { 0, 40079,  2264},		/* move 1 */
    { 2, 40016,  2382},		/* dotted 22:13:53 39:42:53 */
    { 2, 40514,  2587},		/* dotted 6 */
    { 2, 40815,  2656},		/* dotted 11 */
    { 2, 40485,  2862},		/* dotted 5 */
    { 2, 40538,  3016},		/* dotted alpha */
    { 2, 40306,  3133},		/* dotted beta */
    { 2, 40335,  2968},		/* dotted 4 */
    { 2, 40485,  2862},		/* dotted 5 */
    { 2, 40230,  2792},		/* dotted 2 */
    { 2, 40514,  2587},		/* dotted 6 */
    {-1,     0,     0},
};

/* Leo */
static ConFig _Leo_figs[] = {
    { 0, 18251,   718},		/* move regulus */
    { 1, 18220,  1005},		/* draw eta */
    { 1, 18599,  1190},		/* draw gamma1 */
    { 1, 18500,  1405},		/* draw zeta */
    { 1, 17782,  1560},		/* draw mu */
    { 1, 16939,  1570},		/* draw kappa */
    { 1, 17151,  1378},		/* draw lambda */
    { 1, 17575,  1426},		/* draw epsilon */
    { 1, 17782,  1560},		/* draw mu */
    { 0, 17575,  1426},		/* move epsilon */
    { 1, 18220,  1005},		/* draw eta */
    { 1, 20227,   925},		/* draw theta */
    { 1, 20517,   631},		/* draw iota */
    { 1, 20434,   361},		/* draw sigma */
    { 0, 20227,   925},		/* move theta */
    { 1, 21271,   874},		/* draw beta */
    { 1, 20223,  1231},		/* draw delta */
    { 1, 20227,   925},		/* draw theta */
    { 0, 20223,  1231},		/* move delta */
    { 1, 18599,  1190},		/* draw gamma1 */
    {-1,     0,     0},
};

/* Leo Minor */
static ConFig _Leo_Minor_figs[] = {
    { 0, 17226,  2183},		/* move 10 */
    { 2, 18222,  2114},		/* dotted 21 */
    { 2, 18777,  2027},		/* dotted 30 */
    { 2, 19599,  2052},		/* dotted 46 */
    { 2, 18836,  2202},		/* dotted beta */
    { 2, 18222,  2114},		/* dotted 21 */
    {-1,     0,     0},
};

/* Lepus */
static ConFig _Lepus_figs[] = {
    { 0,  9396,  -776},		/* move kappa */
    { 1,  9387,  -972},		/* draw mu */
    { 1,  9587,  -790},		/* draw lambda */
    { 0,  9981, -1069},		/* move alpha */
    { 1,  9847, -1245},		/* draw beta */
    { 1, 10333, -1346},		/* draw gamma */
    { 1, 10539, -1252},		/* draw delta */
    { 1, 10984,  -896},		/* draw theta */
    { 1, 10692,  -850},		/* draw eta */
    { 1, 10408,  -889},		/* draw zeta */
    { 1,  9981, -1069},		/* draw alpha */
    { 1,  9387,  -972},		/* draw mu */
    { 1,  9163, -1342},		/* draw epsilon */
    { 1,  9847, -1245},		/* draw beta */
    {-1,     0,     0},
};

/* Libra */
static ConFig _Libra_figs[] = {
    { 0, 27122, -1516},		/* move sigma */
    { 1, 26726,  -962},		/* draw alpha2 */
    { 1, 27510,  -562},		/* draw beta */
    { 1, 28065,  -887},		/* draw gamma */
    { 1, 26726,  -962},		/* draw alpha2 */
    { 0, 28065,  -887},		/* move gamma */
    { 1, 28110, -1688},		/* draw upsilon */
    { 1, 28159, -1786},		/* draw tau */
    {-1,     0,     0},
};

/* Lupus */
static ConFig _Lupus_figs[] = {
    { 0, 26955, -2588},		/* move beta */
    { 1, 27641, -2438},		/* draw delta */
    { 1, 28054, -2470},		/* draw gamma */
    { 1, 28803, -2303},		/* draw eta */
    { 1, 28528, -2017},		/* draw chi */
    { 1, 27654, -2175},		/* draw phi1 */
    { 1, 28803, -2303},		/* draw eta */
    { 1, 27368, -3125},		/* draw zeta */
    { 1, 26457, -2843},		/* draw alpha */
    { 0, 27368, -3125},		/* move zeta */
    { 1, 27556, -2872},		/* draw mu */
    { 1, 27680, -2681},		/* draw epsilon */
    { 1, 28054, -2470},		/* draw gamma */
    {-1,     0,     0},
};

/* Lynx */
static ConFig _Lynx_figs[] = {
    { 0, 16831,  2063},		/* move alpha */
    { 2, 16765,  2208},		/* dotted 38 */
    { 2, 16396,  2307},		/* dotted 9:06:32 38:27:08 */
    { 2, 16219,  2506},		/* dotted 9:00:38 41:46:58 */
    { 2, 15085,  2591},		/* dotted 31 */
    { 2, 13401,  2952},		/* dotted 21 */
    { 2, 12518,  3505},		/* dotted 15 */
    { 2, 11388,  3540},		/* dotted 2 */
    {-1,     0,     0},
};

/* Lyra */
static ConFig _Lyra_figs[] = {
    { 0, 33743,  2256},		/* move zeta1 */
    { 1, 33730,  2380},		/* draw epsilon1 */
    { 1, 33508,  2327},		/* draw vega */
    { 1, 33743,  2256},		/* draw zeta1 */
    { 1, 33902,  2001},		/* draw beta */
    { 1, 34168,  1961},		/* draw gamma */
    { 1, 34035,  2213},		/* draw delta2 */
    { 1, 33743,  2256},		/* draw zeta1 */
    {-1,     0,     0},
};

/* Mensa */
static ConFig _Mensa_figs[] = {
    { 0, 11107, -4485},		/* move alpha */
    { 2,  9956, -4580},		/* dotted gamma */
    { 2,  8855, -4496},		/* dotted eta */
    { 2,  9081, -4278},		/* dotted beta */
    {-1,     0,     0},
};

/* Microscopium */
static ConFig _Microscopium_figs[] = {
    { 0, 38422, -2448},		/* move theta1 */
    { 2, 38338, -1930},		/* dotted epsilon */
    { 2, 37838, -1935},		/* dotted gamma */
    { 2, 37499, -2026},		/* dotted alpha */
    {-1,     0,     0},
};

/* Monoceros */
static ConFig _Monoceros_figs[] = {
    { 0, 12029,   593},		/* move 15 */
    { 2, 11787,   439},		/* dotted 13 */
    { 2, 11513,   275},		/* dotted epsilon */
    { 2, 12235,   144},		/* dotted 18 */
    { 2, 11787,   439},		/* dotted 13 */
    { 0, 12235,   144},		/* move 18 */
    { 2, 12955,   -29},		/* dotted delta */
    { 2, 11664,  -422},		/* dotted beta */
    { 1, 11245,  -376},		/* draw gamma */
    { 0, 12955,   -29},		/* move delta */
    { 2, 13837,  -573},		/* dotted alpha */
    { 2, 14657,  -179},		/* dotted zeta */
    {-1,     0,     0},
};

/* Musca */
static ConFig _Musca_figs[] = {
    { 0, 21168, -4003},		/* move lambda */
    { 1, 22127, -4077},		/* draw epsilon */
    { 1, 22715, -4148},		/* draw alpha */
    { 1, 22574, -4327},		/* draw gamma */
    { 1, 23468, -4292},		/* draw delta */
    { 1, 22988, -4086},		/* draw beta */
    {-1,     0,     0},
};

/* Norma */
static ConFig _Norma_figs[] = {
    { 0, 29615, -2853},		/* move epsilon */
    { 1, 29310, -3004},		/* draw gamma1 */
    { 1, 29395, -3009},		/* draw gamma2 */
    { 1, 28896, -2953},		/* draw eta */
    { 1, 28994, -2710},		/* draw delta */
    { 1, 29615, -2853},		/* draw epsilon */
    {-1,     0,     0},
};

/* Octans */
static ConFig _Octans_figs[] = {
    { 0, 26007, -5020},		/* move delta */
    { 1, 39044, -4643},		/* draw nu */
    { 1, 40200, -4826},		/* draw epsilon */
    { 1, 40981, -4882},		/* draw beta */
    { 1, 26007, -5020},		/* draw delta */
    {-1,     0,     0},
};

/* Ophiuchus */
static ConFig _Ophiuchus_figs[] = {
    { 0, 31420, -1792},		/* move 45 */
    { 1, 31260, -1499},		/* draw theta */
    { 1, 30911,  -943},		/* draw eta */
    { 1, 29914,  -634},		/* draw zeta */
    { 1, 29734,  -996},		/* draw phi */
    { 1, 29610, -1107},		/* draw chi */
    { 1, 29523, -1202},		/* draw psi */
    { 1, 29567, -1406},		/* draw rho */
    { 0, 29914,  -634},		/* move zeta */
    { 1, 29634,  -502},		/* draw upsilon */
    { 1, 29349,  -281},		/* draw epsilon */
    { 1, 29230,  -221},		/* draw delta */
    { 1, 29727,   119},		/* draw lambda */
    { 1, 30530,   562},		/* draw kappa */
    { 1, 29914,  -634},		/* draw zeta */
    { 0, 30530,   562},		/* move kappa */
    { 1, 31648,   753},		/* draw alpha */
    { 1, 31904,   274},		/* draw beta */
    { 1, 30911,  -943},		/* draw eta */
    { 0, 31904,   274},		/* move beta */
    { 1, 32036,   162},		/* draw gamma */
    { 1, 32370,  -586},		/* draw nu */
    {-1,     0,     0},
};

/* Orion */
static ConFig _Orion_figs[] = {
    { 0, 11158,   852},		/* move xi */
    { 1, 11027,   886},		/* draw nu */
    { 1, 10631,  1216},		/* draw chi1 */
    { 1, 10917,  1208},		/* draw chi2 */
    { 1, 11158,   852},		/* draw xi */
    { 1, 10871,   578},		/* draw mu */
    { 1, 10655,   444},		/* draw betelgeuse */
    { 1,  9753,   380},		/* draw gamma */
    { 1, 10054,   596},		/* draw lambda */
    { 1, 10655,   444},		/* draw betelgeuse */
    { 1, 10222,  -116},		/* draw zeta */
    { 1, 10432,  -580},		/* draw kappa */
    { 0, 10222,  -116},		/* move zeta */
    { 1, 10086,   -72},		/* draw epsilon */
    { 1,  9960,   -17},		/* draw delta */
    { 1,  9734,  -143},		/* draw eta */
    { 1,  9436,  -492},		/* draw rigel */
    { 0,  9960,   -17},		/* move delta */
    { 1,  9753,   380},		/* draw gamma */
    { 1,  8695,   417},		/* draw pi3 */
    { 1,  8736,   336},		/* draw pi4 */
    { 1,  8827,   146},		/* draw pi5 */
    { 1,  8956,   102},		/* draw pi6 */
    { 0,  8695,   417},		/* move pi3 */
    { 1,  8718,   534},		/* draw pi2 */
    { 1,  8846,   609},		/* draw pi1 */
    { 1,  8891,   810},		/* draw omicron2 */
    { 1,  9137,   924},		/* draw 11 */
    {-1,     0,     0},
};

/* Pavo */
static ConFig _Pavo_figs[] = {
    { 0, 36017, -4374},		/* move epsilon */
    { 1, 36261, -3970},		/* draw delta */
    { 1, 33691, -4285},		/* draw zeta */
    { 0, 31971, -3883},		/* move eta */
    { 1, 32657, -3820},		/* draw pi */
    { 1, 33096, -3689},		/* draw xi */
    { 1, 33966, -3731},		/* draw lambda */
    { 1, 36261, -3970},		/* draw delta */
    { 1, 36769, -3404},		/* draw alpha */
    { 1, 38593, -3921},		/* draw gamma */
    { 1, 37348, -3972},		/* draw beta */
    { 1, 36261, -3970},		/* draw delta */
    { 1, 34108, -4034},		/* draw kappa */
    { 1, 32657, -3820},		/* draw pi */
    {-1,     0,     0},
};

/* Pegasus */
static ConFig _Pegasus_figs[] = {
    { 0, 39125,   592},		/* move epsilon */
    { 1, 39906,   371},		/* draw theta */
    { 1, 40843,   649},		/* draw zeta */
    { 1, 41542,   912},		/* draw alpha */
    { 1,   397,   911},		/* draw gamma */
    { 1,   251,  1745},		/* draw and alpha */
    { 1, 41513,  1684},		/* draw beta */
    { 1, 41542,   912},		/* draw alpha */
    { 0, 39899,  1990},		/* move pi2 */
    { 1, 40890,  1813},		/* draw eta */
    { 1, 41513,  1684},		/* draw beta */
    { 1, 41100,  1476},		/* draw mu */
    { 1, 40995,  1413},		/* draw lambda */
    { 1, 39810,  1520},		/* draw iota */
    { 1, 39139,  1538},		/* draw kappa */
    {-1,     0,     0},
};

/* Perseus */
static ConFig _Perseus_figs[] = {
    { 0,  3109,  3041},		/* move phi */
    { 1,  4926,  2953},		/* draw theta */
    { 1,  5672,  2976},		/* draw iota */
    { 1,  5684,  2691},		/* draw kappa */
    { 1,  5645,  2457},		/* draw algol */
    { 1,  5555,  2330},		/* draw rho */
    { 0,  5645,  2457},		/* move algol */
    { 1,  7135,  2400},		/* draw epsilon */
    { 1,  7168,  2147},		/* draw xi */
    { 1,  7023,  1913},		/* draw zeta */
    { 1,  6729,  1937},		/* draw omicron */
    { 0,  7135,  2400},		/* move epsilon */
    { 1,  6687,  2867},		/* draw delta */
    { 1,  7459,  2862},		/* draw 48 */
    { 1,  7646,  2904},		/* draw mu */
    { 1,  7747,  3017},		/* draw 4:18:15 50:17:44 */
    { 1,  7397,  3021},		/* draw lambda */
    { 0,  6687,  2867},		/* move delta */
    { 1,  6129,  2991},		/* draw alpha */
    { 1,  5543,  3210},		/* draw gamma */
    { 1,  5120,  3353},		/* draw eta */
    { 1,  5227,  3165},		/* draw tau */
    { 1,  5672,  2976},		/* draw iota */
    { 1,  6129,  2991},		/* draw alpha */
    {-1,     0,     0},
};

/* Phoenix */
static ConFig _Phoenix_figs[] = {
    { 0,  1982, -2803},		/* move beta */
    { 1,   282, -2744},		/* draw epsilon */
    { 1,   788, -2538},		/* draw alpha */
    { 1,  1982, -2803},		/* draw beta */
    { 1,  2650, -2599},		/* draw gamma */
    { 1,  2737, -2944},		/* draw delta */
    { 1,  2051, -3314},		/* draw zeta */
    { 1,  1982, -2803},		/* draw beta */
    {-1,     0,     0},
};

/* Pictor */
static ConFig _Pictor_figs[] = {
    { 0, 12245, -3716},		/* move alpha */
    { 2, 10494, -3370},		/* dotted gamma */
    { 2, 10418, -3063},		/* dotted beta */
    {-1,     0,     0},
};

/* Pisces */
static ConFig _Pisces_figs[] = {
    { 0,  1970,  1288},		/* move psi1 */
    { 1,  2384,  1635},		/* draw upsilon */
    { 1,  2149,  1805},		/* draw tau */
    { 1,  1970,  1288},		/* draw psi1 */
    { 1,  2143,  1262},		/* draw chi */
    { 1,  2744,   920},		/* draw eta */
    { 1,  3161,   549},		/* draw omicron */
    { 1,  3661,   165},		/* draw alpha */
    { 1,  3406,   191},		/* draw xi */
    { 1,  3042,   329},		/* draw nu */
    { 1,  2705,   368},		/* draw mu */
    { 1,  1888,   473},		/* draw epsilon */
    { 1,  1460,   455},		/* draw delta */
    { 1, 43179,   411},		/* draw omega */
    { 1, 42598,   337},		/* draw iota */
    { 1, 42239,   382},		/* draw theta */
    { 1, 41914,   196},		/* draw gamma */
    { 1, 42208,    75},		/* draw kappa */
    { 1, 42661,   106},		/* draw lambda */
    { 1, 42598,   337},		/* draw iota */
    {-1,     0,     0},
};

/* Piscis Austrinus */
static ConFig _Piscis_Austrinus_figs[] = {
    { 0, 41329, -1777},		/* move fomalhaut */
    { 1, 40819, -1622},		/* draw epsilon */
    { 1, 39851, -1979},		/* draw mu */
    { 1, 39232, -1853},		/* draw theta */
    { 1, 39148, -1981},		/* draw iota */
    { 1, 39851, -1979},		/* draw mu */
    { 1, 40545, -1940},		/* draw beta */
    { 1, 41175, -1972},		/* draw gamma */
    { 1, 41278, -1952},		/* draw delta */
    { 1, 41329, -1777},		/* draw fomalhaut */
    {-1,     0,     0},
};

/* Puppis */
static ConFig _Puppis_figs[] = {
    { 0, 11518, -3161},		/* move canopus */
    { 1, 11932, -2591},		/* draw nu */
    { 1, 13114, -2225},		/* draw pi */
    { 1, 13906, -1704},		/* draw 1 */
    { 1, 13914, -1737},		/* draw 3 */
    { 1, 14078, -1491},		/* draw xi */
    { 1, 13764, -1608},		/* draw 7:38:49 -26:48:06 */
    { 1, 13906, -1704},		/* draw 1 */
    { 0, 14078, -1491},		/* move xi */
    { 1, 14626, -1458},		/* draw rho */
    { 1, 14507, -2400},		/* draw zeta */
    { 1, 14686, -2840},		/* draw vel gamma2 */
    {-1,     0,     0},
};

/* Pyxis */
static ConFig _Pyxis_figs[] = {
    { 0, 15915, -1662},		/* move gamma */
    { 1, 15707, -1991},		/* draw alpha */
    { 1, 15603, -2118},		/* draw beta */
    { 1, 14507, -2400},		/* draw pup zeta */
    {-1,     0,     0},
};

/* Reticulum */
static ConFig _Reticulum_figs[] = {
    { 0,  7632, -3748},		/* move alpha */
    { 1,  6726, -3888},		/* draw beta */
    { 1,  7162, -3684},		/* draw delta */
    { 1,  7694, -3558},		/* draw epsilon */
    { 1,  7632, -3748},		/* draw alpha */
    {-1,     0,     0},
};

/* Sagitta */
static ConFig _Sagitta_figs[] = {
    { 0, 35402,  1080},		/* move alpha */
    { 1, 35621,  1112},		/* draw delta */
    { 1, 35962,  1169},		/* draw gamma */
    { 0, 35621,  1112},		/* move delta */
    { 1, 35431,  1048},		/* draw beta */
    {-1,     0,     0},
};

/* Sagittarius */
static ConFig _Sagittarius_figs[] = {
    { 0, 32928, -2205},		/* move eta */
    { 2, 33125, -2063},		/* dotted epsilon */
    { 1, 32574, -1825},		/* draw gamma */
    { 1, 33029, -1789},		/* draw delta */
    { 1, 33125, -2063},		/* draw epsilon */
    { 1, 34278, -1792},		/* draw zeta */
    { 1, 34408, -1660},		/* draw tau */
    { 1, 34057, -1577},		/* draw sigma */
    { 1, 33769, -1619},		/* draw phi */
    { 1, 34278, -1792},		/* draw zeta */
    { 0, 33239, -1525},		/* move lambda */
    { 1, 33769, -1619},		/* draw phi */
    { 1, 33029, -1789},		/* draw delta */
    { 1, 33239, -1525},		/* draw lambda */
    { 2, 32812, -1263},		/* dotted mu */
    { 0, 34850, -1070},		/* move rho1 */
    { 2, 34492, -1261},		/* dotted pi */
    { 2, 34340, -1304},		/* dotted omicron */
    { 2, 34131, -1266},		/* dotted xi2 */
    { 2, 34492, -1261},		/* dotted pi */
    {-1,     0,     0},
};

/* Scorpius */
static ConFig _Scorpius_figs[] = {
    { 0, 32095, -2222},		/* move 17:49:52 -37:02:36 */
    { 1, 31608, -2226},		/* draw lambda */
    { 1, 31522, -2237},		/* draw upsilon */
    { 1, 31874, -2341},		/* draw kappa */
    { 1, 32027, -2407},		/* draw iota1 */
    { 1, 31719, -2579},		/* draw theta */
    { 1, 30964, -2594},		/* draw eta */
    { 1, 30437, -2541},		/* draw zeta 2 */
    { 1, 30356, -2282},		/* draw mu 1 */
    { 1, 30304, -2057},		/* draw epsilon */
    { 1, 29876, -1692},		/* draw tau */
    { 1, 29682, -1585},		/* draw antares */
    { 1, 29435, -1535},		/* draw sigma */
    { 1, 28810, -1357},		/* draw delta */
    { 1, 28765, -1566},		/* draw pi */
    { 1, 28706, -1752},		/* draw rho */
    { 0, 28810, -1357},		/* move delta */
    { 1, 28963, -1188},		/* draw beta1 */
    { 1, 29159, -1166},		/* draw nu */
    {-1,     0,     0},
};

/* Sculptor */
static ConFig _Sculptor_figs[] = {
    { 0,  1758, -1761},		/* move alpha */
    { 1, 42867, -1687},		/* draw delta */
    { 1, 41964, -1951},		/* draw gamma */
    { 1, 42389, -2269},		/* draw beta */
    {-1,     0,     0},
};

/* Scutum */
static ConFig _Scutum_figs[] = {
    { 0, 33456,  -494},		/* move alpha */
    { 1, 33815,  -284},		/* draw beta */
    { 1, 33668,  -543},		/* draw delta */
    { 1, 33275,  -873},		/* draw gamma */
    { 1, 33456,  -494},		/* draw alpha */
    {-1,     0,     0},
};

/* Serpens Caput */
static ConFig _Serpens_Caput_figs[] = {
    { 0, 28385,   925},		/* move ser beta */
    { 1, 28693,   939},		/* draw ser gamma */
    { 1, 28462,  1088},		/* draw ser kappa */
    { 1, 28246,  1180},		/* draw ser iota */
    { 1, 28385,   925},		/* draw ser beta */
    { 1, 28044,   632},		/* draw ser delta */
    { 1, 28328,   385},		/* draw ser alpha */
    { 1, 28524,   268},		/* draw ser epsilon */
    { 1, 28488,  -205},		/* draw ser mu */
    { 1, 29230,  -221},		/* draw oph delta */
    {-1,     0,     0},
};

/* Serpens Cauda */
static ConFig _Serpens_Cauda_figs[] = {
    { 0, 34086,   252},		/* move ser theta1 */
    { 1, 33039,  -173},		/* draw ser eta */
    { 1, 32370,  -586},		/* draw oph nu */
    { 1, 31727,  -923},		/* draw ser xi */
    { 1, 30911,  -943},		/* draw oph eta */
    {-1,     0,     0},
};

/* Sextans */
static ConFig _Sextans_figs[] = {
    { 0, 18884,  -164},		/* move delta */
    { 2, 18908,   -38},		/* dotted beta */
    { 2, 18238,   -22},		/* dotted alpha */
    { 2, 17775,  -486},		/* dotted gamma */
    {-1,     0,     0},
};

/* Taurus */
static ConFig _Taurus_figs[] = {
    { 0,  6506,    24},		/* move 10 */
    { 1,  6144,   541},		/* draw omicron */
    { 1,  6215,   583},		/* draw xi */
    { 1,  7294,   359},		/* draw nu */
    { 0,  6215,   583},		/* move xi */
    { 1,  7220,   749},		/* draw lambda */
    { 1,  7793,   937},		/* draw gamma */
    { 1,  8057,   957},		/* draw theta1 */
    { 1,  8277,   990},		/* draw aldebaran */
    { 1, 10129,  1268},		/* draw zeta */
    { 0,  7793,   937},		/* move gamma */
    { 1,  7888,  1052},		/* draw delta1 */
    { 1,  7964,  1075},		/* draw delta3 */
    { 1,  8058,  1150},		/* draw epsilon */
    { 1,  9788,  1716},		/* draw beta */
    {-1,     0,     0},		
};

/* Telescopium */
static ConFig _Telescopium_figs[] = {
    { 0, 32736, -2757},		/* move epsilon */
    { 1, 33209, -2758},		/* draw alpha */
    { 1, 33264, -2944},		/* draw zeta */
    {-1,     0,     0},
};

/* Triangulum */
static ConFig _Triangulum_figs[] = {
    { 0,  3392,  1774},		/* move alpha */
    { 1,  3886,  2099},		/* draw beta */
    { 1,  4119,  2030},		/* draw gamma */
    { 1,  3392,  1774},		/* draw alpha */
    {-1,     0,     0},
};

/* Triangulum Australe */
static ConFig _Triangulum_Australe_figs[] = {
    { 0, 30259, -4141},		/* move alpha */
    { 1, 28654, -3805},		/* draw beta */
    { 1, 28101, -3979},		/* draw epsilon */
    { 1, 27567, -4120},		/* draw gamma */
    { 1, 30259, -4141},		/* draw alpha */
    {-1,     0,     0},
};

/* Tucana */
static ConFig _Tucana_figs[] = {
    { 0, 40420, -3897},		/* move delta */
    { 1, 40155, -3615},		/* draw alpha */
    { 1, 41922, -3494},		/* draw gamma */
    { 1,   946, -3777},		/* draw beta1 */
    { 1,   602, -3892},		/* draw zeta */
    { 1, 43197, -3934},		/* draw epsilon */
    { 1, 40420, -3897},		/* draw delta */
    {-1,     0,     0},
};

/* Ursa Major */
static ConFig _Ursa_Major_figs[] = {
    { 0, 24826,  2958},		/* move eta */
    { 1, 24117,  3295},		/* draw mizar a */
    { 1, 23220,  3357},		/* draw epsilon */
    { 1, 22062,  3421},		/* draw delta */
    { 1, 21414,  3221},		/* draw gamma */
    { 1, 19855,  3382},		/* draw beta */
    { 1, 19911,  3705},		/* draw alpha */
    { 1, 22062,  3421},		/* draw delta */
    { 0, 21414,  3221},		/* move gamma */
    { 2, 21181,  2866},		/* dotted chi */
    { 2, 20354,  1985},		/* dotted nu */
    { 2, 20345,  1891},		/* dotted xi */
    { 0, 21181,  2866},		/* move chi */
    { 2, 20089,  2669},		/* dotted psi */
    { 2, 18669,  2489},		/* dotted mu */
    { 2, 18512,  2574},		/* dotted lambda */
    { 0, 19855,  3382},		/* move beta */
    { 2, 17729,  3542},		/* dotted upsilon */
    { 2, 15307,  3643},		/* dotted omicron */
    { 2, 17145,  3783},		/* dotted 23 */
    { 2, 19911,  3705},		/* dotted alpha */
    { 0, 17145,  3783},		/* move 23 */
    { 2, 17729,  3542},		/* dotted upsilon */
    { 2, 17185,  3100},		/* dotted theta */
    { 2, 16308,  2829},		/* dotted kappa */
    { 2, 16176,  2882},		/* dotted iota */
    {-1,     0,     0},
};

/* Ursa Minor */
static ConFig _Ursa_Minor_figs[] = {
    { 0,  4554,  5355},		/* move polaris */
    { 1, 31566,  5195},		/* draw umi delta */
    { 1, 30179,  4922},		/* draw umi epsilon */
    { 1, 28321,  4667},		/* draw umi zeta */
    { 1, 26721,  4449},		/* draw umi beta */
    { 1, 27621,  4310},		/* draw umi gamma */
    { 1, 29325,  4545},		/* draw umi eta */
    { 1, 28321,  4667},		/* draw umi zeta */
    {-1,     0,     0},
};

/* Vela */
static ConFig _Vela_figs[] = {
    { 0, 15741, -3282},		/* move delta */
    { 1, 16863, -3300},		/* draw kappa */
    { 1, 17905, -3274},		/* draw phi */
    { 1, 19403, -2965},		/* draw mu */
    { 1, 18442, -2527},		/* draw 10:14:44 -42:07:20 */
    { 1, 17121, -2428},		/* draw psi */
    { 1, 16439, -2605},		/* draw lambda */
    { 1, 14684, -2840},		/* draw gamma1 */
    { 1, 15741, -3282},		/* draw delta */
    {-1,     0,     0},
};

/* Virgo */
static ConFig _Virgo_figs[] = {
    { 0, 26491,  -339},		/* move mu */
    { 1, 25680,  -360},		/* draw iota */
    { 1, 24155,  -669},		/* draw spica */
    { 1, 22849,   -86},		/* draw gamma */
    { 1, 22197,   -40},		/* draw eta */
    { 1, 21320,   105},		/* draw beta */
    { 1, 21175,   391},		/* draw nu */
    { 1, 21756,   523},		/* draw omicron */
    { 1, 22849,   -86},		/* draw gamma */
    { 1, 23268,   203},		/* draw delta */
    { 1, 23465,   657},		/* draw epsilon */
    { 0, 23268,   203},		/* move delta */
    { 1, 24440,   -35},		/* draw zeta */
    { 1, 24155,  -669},		/* draw spica */
    { 0, 24440,   -35},		/* move zeta */
    { 1, 25249,    92},		/* draw tau */
    { 1, 25680,  -360},		/* draw iota */
    { 0, 25249,    92},		/* move tau */
    { 1, 26587,   113},		/* draw 109 */
    {-1,     0,     0},
};

/* Volans */
static ConFig _Volans_figs[] = {
    { 0, 16273, -3983},		/* move alpha */
    { 1, 14994, -4290},		/* draw kappa1 */
    { 1, 13854, -4356},		/* draw zeta */
    { 1, 12861, -4229},		/* draw gamma1 */
    { 1, 13104, -4077},		/* draw delta */
    { 1, 14637, -4117},		/* draw epsilon */
    { 1, 15172, -3968},		/* draw beta */
    { 1, 16273, -3983},		/* draw alpha */
    {-1,     0,     0},
};

/* Vulpecula */
static ConFig _Vulpecula_figs[] = {
    { 0, 35803,  1444},		/* move 13 */
    { 2, 35061,  1479},		/* dotted alpha */
    {-1,     0,     0},
};


static ConFig *figmap[NCNS] = {
    _Andromeda_figs,
    _Antlia_figs,
    _Apus_figs,
    _Aquila_figs,
    _Aquarius_figs,
    _Ara_figs,
    _Aries_figs,
    _Auriga_figs,
    _Bootes_figs,
    _Canis_Major_figs,
    _Canis_Minor_figs,
    _Canes_Venatici_figs,
    _Caelum_figs,
    _Camelopardalis_figs,
    _Capricornus_figs,
    _Carina_figs,
    _Cassiopeia_figs,
    _Centaurus_figs,
    _Cepheus_figs,
    _Cetus_figs,
    _Chamaeleon_figs,
    _Circinus_figs,
    _Cancer_figs,
    _Columba_figs,
    _Coma_Berenices_figs,
    _Corona_Australis_figs,
    _Corona_Borealis_figs,
    _Crater_figs,
    _Crux_figs,
    _Corvus_figs,
    _Cygnus_figs,
    _Delphinus_figs,
    _Dorado_figs,
    _Draco_figs,
    _Equuleus_figs,
    _Eridanus_figs,
    _Fornax_figs,
    _Gemini_figs,
    _Grus_figs,
    _Hercules_figs,
    _Horologium_figs,
    _Hydra_figs,
    _Hydrus_figs,
    _Indus_figs,
    _Leo_Minor_figs,
    _Lacerta_figs,
    _Leo_figs,
    _Lepus_figs,
    _Libra_figs,
    _Lupus_figs,
    _Lynx_figs,
    _Lyra_figs,
    _Mensa_figs,
    _Microscopium_figs,
    _Monoceros_figs,
    _Musca_figs,
    _Norma_figs,
    _Octans_figs,
    _Ophiuchus_figs,
    _Orion_figs,
    _Pavo_figs,
    _Pegasus_figs,
    _Perseus_figs,
    _Phoenix_figs,
    _Pictor_figs,
    _Piscis_Austrinus_figs,
    _Pisces_figs,
    _Puppis_figs,
    _Pyxis_figs,
    _Reticulum_figs,
    _Sculptor_figs,
    _Scorpius_figs,
    _Scutum_figs,
    _Serpens_Caput_figs,
    _Sextans_figs,
    _Sagitta_figs,
    _Sagittarius_figs,
    _Taurus_figs,
    _Telescopium_figs,
    _Triangulum_Australe_figs,
    _Triangulum_figs,
    _Tucana_figs,
    _Ursa_Major_figs,
    _Ursa_Minor_figs,
    _Vela_figs,
    _Virgo_figs,
    _Volans_figs,
    _Vulpecula_figs,
    _Serpens_Cauda_figs,
};

/* given a constellation id and epoch, return arrays of ra[] and dec[]
 *   end-points precessed to the desired epoch that, if connected, will form the
 *   given constellation figure.
 * dcodes is 0 if the coord is a "move-to", 1 if a "draw-to" or 2 if a "draw-to
 *   as dotted-line".
 * return the total number of tripples or -1 if id is bogus.
 * the arrays need be no larger than 35 entries.
 */
int
cns_figure (int id, double e, double ra[], double dec[], int dcodes[])
{
	ConFig *cfp;

	if (id < 0 || id >= NCNS)
	    return (-1);

	for (cfp = figmap[id]; cfp->drawcode >= 0; cfp++) {
	    *ra = hrrad((double)cfp->ra/1800.0);
	    *dec = degrad((double)cfp->dec/60.0);
	    precess (J2000, e, ra, dec);
	    ra++;
	    dec++;
	    *dcodes++ = cfp->drawcode;
	}

	return (cfp - figmap[id]);
}

/*
    Following figures changed to those created by H.A. Rey
    (incorporated by Michael Mills):
*/

/* Andromeda */
static ConFig _Andromeda_rey_figs[] = {
  { 0,  3717,  2539}, /* gamma-1 */
  { 1,  2904,  2484}, /* upsilon */
  { 1,  2092,  2137}, /* Mirach */ 
  { 1,  1180,  1851}, /* delta */ 
  { 1,   251,  1745}, /* Alpheratz */
  { 0,  2092,  2137}, /* Mirach */
  { 1,  1702,  2309}, /* mu */
  { 1,  1494,  2464}, /* nu */
  { 1,  2085,  2834}, /* phi */
  { 1,  2940,  2917}, /* 51 */
  { 0,  1716,  1405}, /* eta */
  { 1,  1420,  1456}, /* zeta */
  { 1,  1156,  1758}, /* epsilon */
  { 1,  1180,  1851}, /* delta */
  { 1,  1106,  2023}, /* pi */
  { 1,   513,  2320}, /* theta */ 
  { 1, 42544,  2596}, /* iota */
  { 1, 42612,  2660}, /* kappa */
  { 1, 42527,  2787}, /* lambda */
  { 0, 42544,  2596}, /* iota */
  { 1, 41457,  2539}, /* omicron */
  { 0,  1106,  2023}, /* pi */
  { 1,  2092,  2137}, /* Mirach */ 
  {-1,     0,     0},
};

/* Antlia - no change */

/* Apus */
static ConFig _Apus_rey_figs[] = {
    { 0, 30092, -4651},
    { 1, 29803, -4733},
    { 1, 29410, -4721},
    { 1, 26636, -4742},
    { 0, 30092, -4651}, /* Aps Beta */
    { 1, 29410, -4721}, /* Aps Delta1 */
    {-1,     0,     0},
};

/* Aquarius */
static ConFig _Aquarius_rey_figs[] = {
  {0, 37430, -569}, /* Aqr Epsilon-2 */
  {1, 37580, -539}, /* Aqr Mu-6 */
  {1, 38088, -682}, /* Aqr Nu-13 */
  {1, 38747, -334}, /* Sadalsud */
  {1, 39774, -19}, /* Sadalmelik */
  {1, 40250, -83}, /* Sadachbia */
  {1, 40661, -7}, /* Aqr Eta-62 */
  {1, 40358, 83}, /* Aqr Pi-52 */
  {1, 39774, -19}, /* Sadalmelik */
  {0, 39793, -832}, /* Aqr Iota-33 */
  {1, 38747, -334}, /* Sadalsud */
  {0, 39774, -19}, /* Sadalmelik */
  {1, 40105, -467}, /* Aqr Theta-43 */
  {1, 41178, -454}, /* Aqr Lambda-73 */
  {1, 41830, -362}, /* Aqr Phi-90 */
  {1, 41877, -545}, /* Aqr Psi1-91 */
  {1, 41240, -949}, /* Skat */
  {1, 41088, -815}, /* Aqr Tau2-71 */
  {1, 41178, -454}, /* Aqr Lambda-73 */
  {0, 41178, -454}, /* Aqr Lambda-73 */
  {1, 41877, -545}, /* Aqr Psi1-91 */
  {1, 42089, -1206}, /* Aqr 98 */
  {1, 42181, -1238}, /* Aqr 99 */
  {1, 42398, -1254}, /* Aqr 101 */
  {0, 41877, -545}, /* Aqr Psi1-91 */
  {1, 41683, -1270}, /* Aqr 88 */
  {1, 41697, -1347}, /* Aqr 89 */
  {1, 41600, -1424}, /* Aqr 86 */
  {-1,     0,     0},
};

/* Aquila */
static ConFig _Aquila_rey_figs[] = {
  {0, 35859, 384}, /* Aql Beta-60 */
  {1, 35724, 532}, /* Altair */
  {1, 35588, 637}, /* Tarazed */
  {1, 34965, 187}, /* Aql Delta-30 */
  {1, 34387, -292}, /* Aql Lambda-16 */
  {1, 34250, -344}, /* Aql 12 */
  {0, 36339, -49}, /* Aql Theta-65 */
  {1, 35774, 60}, /* Aql Eta-55 */
  {1, 34965, 187}, /* Aql Delta-30 */
  {1, 34362, 832}, /* Aql Zeta-17 */
  {1, 34189, 904}, /* Aql Epsilo-13 */
  {-1,     0,     0},
};

/* Ara */
static ConFig _Ara_rey_figs[] = {
  {0, 31555, -2992}, /* Ara Alpha */
  {1, 30588, -3189}, /* Ara Epsilon1 */
  {1, 30559, -3359}, /* Ara Zeta */
  {1, 30294, -3542}, /* Ara Eta */
  {1, 31533, -3641}, /* Ara Delta */
  {1, 32599, -3005}, /* Ara Theta */
  {1, 31555, -2992}, /* Ara Alpha */
  {-1,     0,     0},
};

/* Aries */
static ConFig _Aries_rey_figs[] = {
  { 0,  5749,  1183}, /* delta */
  { 1,  5376,  1280}, /* epsilon */
  { 1,  5099,  1635}, /* 41 */
  { 1,  5037,  1754}, /* 39 */
  { 1,  4903,  1662}, /* 35 */
  { 1,  3815,  1407}, /* Hamal */
  { 1,  3439,  1248}, /* Sheratan */
  { 1,  3537,  1415}, /* lambda */
  { 1,  3815,  1407}, /* Hamal */
  { 1,  3984,  1272}, /* eta */
  { 0,  5376,  1280}, /* epsilon */  
  { 1,  3984,  1272}, /* eta */
  { 1,  3406,  1157}, /* gamma */
  { 0,  5099,  1635}, /* 41 */
  { 1,  4903,  1662}, /* 35 */
  {-1,     0,     0},
};

/* Auriga */
static ConFig _Auriga_rey_figs[] = {
  { 0, 10786,  3257}, /* delta */
  { 1, 10786,  2696}, /* Menkalinan */ 
  { 1, 10791,  2232}, /* theta */ 
  { 1,  8910,  1989}, /* iota */
  { 1,  9195,  2474}, /* eta */
  { 1,  9074,  2464}, /* zeta */
  { 1,  9059,  2629}, /* epsilon */
  { 1,  9500,  2759}, /* Capella */
  { 1,  9195,  2474}, /* eta */
  { 1, 10786,  2696}, /* Menkalinan */
  { 0, 10786,  3257}, /* delta */
  { 1,  9500,  2759}, /* Capella */
  {-1,     0,     0},
};

/* Bootes */
static ConFig _Bootes_rey_figs[] = {
    { 0, 24818,  1047}, /* Tau */
    { 1, 24884,   947}, /* Upsilon */
    { 1, 25040,  1103}, /* Muphrid */
    { 1, 25670,  1150}, /* Arcturus */
    { 1, 26422,   985}, /* Pi */
    { 1, 26549,  1624}, /* Pulcherrima */
    { 1, 26240,   1784}, /* Sigma */
    { 1, 26155,  1822}, /* Rho */
    { 1, 26162,  2298}, /* Gamma  */
    { 1, 27058,  2423}, /* Nekkar */
    { 1, 27465,  1998}, /* Delta  */
    { 0, 25670,  1150}, /* Arcturus */
    { 1, 26549,  1624}, /* Pulcherrima */
    { 1, 27465,  1998}, /* Delta  */
    { 2, 27734,  2242}, /* Mu */
    { 2, 27058,  2423}, /* Nekkar */
    { 0, 26162,  2298}, /* Gamma  */
    { 1, 25691,  2765}, /* Lambda */
    { 1, 25604,  3107}, /* Kappa2 */
    { 1, 25956,  3111}, /* Theta */
    { 1, 25691,  2765}, /* Lambda */
    {-1,     0,     0}, 
};

/* Caelum - no change */

/* Camelopardalis */
static ConFig _Camelopardalis_rey_figs[] = {
  {0,  8919, 3225}, /* Cam 7 */
  {1,  9103, 3627}, /* Cam Beta-10 */
  {1,  8822, 3981}, /* Cam Alpha-9 */
  {1,  6911, 4280}, /* Cam Gamma */
  {1,  6886, 3932}, /* From GSC */
  {1,  6272, 3596}, /* From GSC */
  {0,  8822, 3981}, /* Cam Alpha-9 */
  {1, 12602, 4619}, /*  */
  {1, 12016, 4305}, /*  */
  {0,  9103, 3627}, /* Cam Beta-10 */
  {1,  6886, 3932}, /* From GSC */
  {-1,     0,     0},
};

/* Cancer */
static ConFig _Cancer_rey_figs[] = {
  {0, 16432, 640}, /* Cnc Kappa-76 */
  {1, 15741, 1089}, /* Cnc Delta-47 */
  {1, 14895, 551}, /* Cnc Beta-17 */
  {1, 14633, 1295}, /* Cnc Mu2-10 */
  {1, 15699, 1288}, /* Cnc Gamma-43 */
  {1, 15741, 1089}, /* Cnc Delta-47 */
  {0, 15699, 1288}, /* Cnc Gamma-43 */
  {1, 15801, 1726}, /* Cnc Iota-48 */
  {-1,     0,     0},
};

/* Canes Venatici - no change */

/* Canis Major */
static ConFig _Canis_Major_rey_figs[] = {
  {0, 13323, -1758}, /* CMa Eta-31 */
  {1, 13044, -1606}, /* CMa Omega-28 */
  {1, 12852, -1583}, /* CMa Delta-25 */
  {1, 12691, -1430}, /* CMa Omicr2-24 */
  {1, 12154, -1002}, /* Sirius */
  {1, 12484, -1023}, /* CMa Iota-20 */
  {1, 12713, -938}, /* CMa Gamma-23 */
  {1, 12426, -722}, /* CMa Theta-14 */
  {1, 12484, -1023}, /* CMa Iota-20 */
  {0, 12852, -1583}, /* CMa Delta-25 */
  {1, 12652, -1676}, /* CMa Sigma-22 */
  {1, 12559, -1738}, /* Adara */
  {1, 12295, -1950}, /* CMa Kappa-13 */
  {0, 11409, -1803}, /* CMa Zeta-1 */
  {1, 12559, -1738}, /* Adara */
  {1, 12424, -1451}, /* CMa Omicr1-16 */
  {1, 11901, -1155}, /* CMa Nu2-7 */
  {1, 12154, -1002}, /* Sirius */
  {0 , 11756, -1405}, /* CMa Xi1-4 */
  {1, 11901, -1155}, /* CMa Nu2-7 */
  {1, 11481, -1077}, /* Mirzam */
  {-1,     0,     0},
};

/* Capricornus */
static ConFig _Capricornus_rey_figs[] = {
  {0, 36542, -752}, /* Cap Alpha2-6 */
  {1, 36630, -886}, /* Cap Beta-9 */
  {1, 36866, -1068}, /* Cap Rho-11 */
  {1, 37383, -1516}, /* Cap Psi-16 */
  {1, 37555, -1615}, /* Cap Omega-18 */
  {0, 36866, -1068}, /* Cap Rho-11 */
  {1, 37932, -1191}, /* Cap Eta-22 */
  {0, 38014, -1500}, /* Cap 24 */
  {1, 37932, -1191}, /* Cap Eta-22 */
  {1, 37978, -1033}, /* Cap Theta-23 */
  {1, 38600, -1344}, /* Cap Zeta-34 */
  {1, 38662, -1308}, /* Cap 36 */
  {1, 38467, -1010}, /* Cap Iota-32 */
  {0, 39211, -967}, /* Deneb Algiedi */
  {1, 39003, -999}, /* Cap Gamma-40 */
  {1, 38467, -1010}, /* Cap Iota-32 */
  {1, 37978, -1033}, /* Cap Theta-23 */
  {1, 36630, -886}, /* Cap Beta-9 */
  {-1,     0,     0},
};

/* Carina */
static ConFig _Carina_rey_figs[] = {
  {0, 11519, -3161}, /* Canopus */
  {1, 16596, -4183}, /* Car Beta */
  {1, 18412, -4202}, /* Car Omega */
  {1, 19289, -3863}, /* Car Theta */
  /* Figurehead follows, made up of stars not present in YBS */
  {1, 18961, -3701}, /* GSC star */
  {1, 19306, -3634},
  {1, 19601, -3534},
  {1, 19003, -3491},
  {1, 18836, -3524},
  {1, 18512, -3680},
  {1, 16713, -3556}, /* Car Iota */
  {1, 15075, -3570}, /* Car Epsilon */
  {1, 14303, -3178}, /* Car Chi */
  {1, 14686, -2840}, /* Vel Gamma2 */
  {0, 19601, -3534}, 
  {1, 20058, -3539}, /* "Nose" of figurehead */
  {0, 15075, -3570}, /* Car Epsilon */
  {2, 16863, -3300}, /* Vel Kappa */
  {0, 16713, -3556}, /* Car Iota */
  {2, 15741, -3282}, /* Vel Delta */
  {-1,     0,     0},
};

/* Cassiopeia */
static ConFig _Cassiopeia_rey_figs[] = {
  {0,  275, 3549}, /* Caph */
  {1, 1215, 3392}, /* Schedar */
  {1, 1701, 3643}, /* Cas Gamma-27 */
  {1, 2575, 3614}, /* Cas Delta-37 */
  {1, 3432, 3820}, /* Cas Epsilo-45 */
  {1, 3703, 4345}, /* Cas 50 */
  {-1,     0,     0},
};

/* Centaurus */
static ConFig _Centaurus_rey_figs[] = {
  {0, 20873, -3781}, /* Cen Lambda */
  {1, 21950, -3142}, /* Cen Rho */
  {1, 22441, -3013}, /* Cen Sigma */
  {1, 21851, -3043}, /* Cen Delta */
  {1, 20430, -3269}, /* Cen Pi */
  {0, 22441, -3013}, /* Cen Sigma */
  {1, 22846, -2937}, /* Cen Gamma */
  {1, 24597, -3207}, /* Cen Epsilon */
  {1, 25066, -2837}, /* Cen Zeta */
  {1, 22846, -2937}, /* Cen Gamma */
  {0, 26388, -3650}, /* Rigil Kentaurus */
  {1, 24597, -3207}, /* Cen Epsilon */
  {1, 25315, -3622}, /* Hadar (Agena) */
  {0, 25066, -2837}, /* Cen Zeta */
  {1, 25252, -2736}, /* Cen Upsilon2 */
  {1, 25160, -2688}, /* Cen Upsilon1 */
  {1, 25148, -2526}, /* Cen Phi */
  {1, 24885, -2501}, /* Cen Nu */
  {1, 24889, -2548}, /* Cen Mu */
  {1, 25066, -2837}, /* Cen Zeta */
  {0, 24885, -2501}, /* Cen Nu */
  {1, 25401, -2182}, /* Cen Theta-5 */
  {1, 25817, -2273}, /* Cen Psi */
  {1, 25891, -2371}, /* From GSC 7809-2783 */
  {1, 25381, -2470}, /* Cen Chi */
  {1, 25148, -2526}, /* Cen Phi */
  {1, 26265, -2529}, /* Cen Eta */
  {1, 26975, -2526}, /* Cen Kappa */
  {1, 26459, -2268}, /* From GSC */
  {1, 26510, -2110}, /* From GSC */
  {0, 24885, -2501}, /* Cen Nu */
  {1, 24331, -2364}, /* From GSC */
  {1, 24018, -2202}, /* Cen Iota */
  {1, 24771, -1982}, /* Cen 1 */
  {0, 24771, -1982}, /* Cen 1 */
  {1, 24883, -2067}, /* Cen 2 */
  {0, 24771, -1982}, /* Cen 1 */
  {1, 24955, -1979}, /* Cen 3 */
  {0, 24771, -1982}, /* Cen 1 */
  {1, 24996, -1915}, /* Cen 4 */
  {-1,     0,     0},
};

/* Cepheus */
static ConFig _Cepheus_rey_figs[] = {
  {0, 36887, 3780}, /* Cep Theta-2 */
  {1, 37359, 3710}, /* Cep Eta-3 */
  {1, 38357, 3755}, /* Alderamin */
  {1, 38660, 4234}, /* Alphirk */
  {1, 42580, 4658}, /* Alrai */
  {1, 41090, 3972}, /* Cep Iota-32 */
  {1, 40475, 3505}, /* Cep Delta-27 */
  {1, 39926, 3492}, /* Cep Zeta-21 */
  {1, 40051, 3423}, /* Cep Epsilo-23 */
  {1, 39105, 3527}, /* Cep Mu */
  {1, 38357, 3755}, /* Alderamin */
  {0, 38660, 4234}, /* Alphirk */
  {1, 41090, 3972}, /* Cep Iota-32 */
  {0, 39926, 3492}, /* Cep Zeta-21 */
  {1, 39163, 3667}, /* Cep Nu-10 */
  {-1,     0,     0}, 
};

/* Cetus */
static ConFig _Cetus_rey_figs[] = {
  {0, 583, -529}, /* Cet Iota-8 */
  {1, 2521, -491}, /* Cet Theta-45 */
  {1, 4180, -178}, /* Mira */
  {1, 4785, 20}, /* Cet Delta-82 */
  {1, 4899, 194}, /* Cet Gamma-86 */
  {1, 4676, 336}, /* Cet Nu-78 */
  {1, 4445, 508}, /* Cet Xi2-73 */
  {1, 4948, 607}, /* Cet Mu-87 */
  {1, 5391, 534}, /* Cet Lambda-91 */
  {1, 5468, 245}, /* Mekab */
  {1, 4899, 194}, /* Cet Gamma-86 */
  {0, 4785, 20}, /* Cet Delta-82 */
  {1, 3600, -1264}, /* Cet Upsilo-59 */
  {1, 1308, -1079}, /* Diphda */
  {1, 583, -529}, /* Cet Iota-8 */
  {0, 1308, -1079}, /* Diphda */
  {1, 3122, -956}, /* Cet Tau-52 */
  {-1,     0,     0},
};

/* Chamaeleon */
static ConFig _Chamaeleon_rey_figs[] = {
  {0, 14956, -4615}, /* Cha Alpha */
  {1, 19064, -4716}, /* Cha Gamma */
  {1, 21589, -4693}, /* Cha Epsilon */
  {1, 22150, -4758}, /* Cha Beta */
  {1, 19373, -4832}, /* Cha Delta2 */
  {1, 19064, -4716}, /* Cha Gamma */
  {-1,     0,     0},
};

/* Circinus - no change */

/* Columba */
static ConFig _Columba_rey_figs[] = {
    { 0,  9936, -2128},
    { 1, 10189, -2044},
    { 1, 10529, -2146},
    { 1, 10726, -2117},
    { 1, 11296, -2108},
    { 1, 11463, -2006},
    { 0, 10529, -2146},
    { 1, 10774, -2568},
    { 0, 10774, -2568}, /* Col Eta */
    { 1, 10726, -2117}, /* Col Gamma */
    { 1, 11463, -2006}, /* Col Delta */
    {-1,     0,     0},
};

/* Coma Berenices */
static ConFig _Coma_Berenices_rey_figs[] = {
    { 0, 23699,  1051}, /* Alpha */
    { 1, 23756,  1672}, /* Beta */
    { 1, 22408,  1696}, /* Gamma */
    { 0, 23756,  1672}, /* Beta */
    { 2, 22090,  1436}, /* Com 7 */
    { 0, 23756,  1672}, /* Beta */
    { 2, 22275,  1550}, /* Com 12 */
    { 0, 23756,  1672}, /* Beta */
    { 2, 22392,  1636}, /* Com 14 */
    {-1,     0,     0},
};

/* Corona Australis - no change */

/* Corona Borealis - no change */

/* Corvus */
static ConFig _Corvus_rey_figs[] = {
  {0, 21852, -1483}, /* Crv Alpha-1 */
  {1, 21904, -1357}, /* Crv Epsilon-2 */
  {1, 22217, -1332}, /* Crv Zeta-5 */
  {1, 22632, -1403}, /* Crv Beta-9 */
  {0, 22217, -1332}, /* Crv Zeta-5 */
  {1, 22496, -990}, /* Crv Delta-7 */
  {1, 22562, -971}, /* Crv Eta-8 */
  {0, 22496, -990}, /* Crv Delta-7 */
  {1, 22074, -1052}, /* Crv Gamma-4 */
  {1, 21904, -1357}, /* Crv Epsilon-2 */
  {-1,     0,     0},
};

/* Crater */
static ConFig _Crater_rey_figs[] = {
  {0, 19793, -1097}, /* Crt Alpha-7 */
  {1, 20380, -886}, /* Crt Delta-12 */
  {1, 20546, -1061}, /* Crt Gamma-15 */
  {1, 21143, -1101}, /* Crt Zeta-27 */
  {1, 21480, -1029}, /* Crt Eta-30 */
  {1, 20900, -588}, /* Crt Theta-21 */
  {1, 20538, -651}, /* Crt Epsilo-14 */
  {1, 20380, -886}, /* Crt Delta-12 */
  {0, 20546, -1061}, /* Crt Gamma-15 */
  {1, 20150, -1369}, /* Crt Beta-11 */
  {-1,     0,     0},
};

/* Crux - no change */

/* Cygnus */
static ConFig _Cygnus_rey_figs[] = {
    { 0, 34713,  3202}, /* Kappa */
    { 2, 35549,  2707}, /* Delta */
    { 1, 36667,  2415}, /* Gamma */
    { 1, 37386,  2038}, /* Epsilon */
    { 2, 38188,  1813}, /* Zeta  */
    /*   { 2, 39124,  1724},*/ /* 78    */
    { 2, 37715,  2470}, /* Nu    */
    { 2, 37243,  2716}, /* Deneb */
    { 2, 36464,  2863}, /* 32    */
    { 2, 35091,  3103}, /* Iota  */
    { 2, 34713,  3202}, /* Kappa */
    /* { 0, 38188,  1813}, */ /* Zeta  */
    /* { 2, 35293,  3013}, */ /* Theta */
    /* { 2, 38337,  2093}, */ /* Upsilon */
    { 0, 38243,  2282}, /* Tau   */
    { 2, 38322,  2363}, /* Sigma  */
    { 2, 37715,  2470}, /* Nu    */
    /* { 2, 38819,  2735}, */ /* Rho   */
    /* { 2, 39204,  2958}, */ /* Pi    */
    { 0, 37243,  2716}, /* Deneb */
    { 2, 37948,  2636}, /* Xi    */
    { 0, 37243,  2716}, /* Deneb */
    { 1, 36667,  2415}, /* Gamma */
    { 1, 35889,  2105}, /* Eta */
    { 1, 35381,  1809}, /* Phi */
    { 1, 35121,  1677}, /* Albireo */
    {-1,     0,     0}, /*  */
};

/* Delphinus */
static ConFig _Delphinus_rey_figs[] = {
  {0, 37135, 683}, /* Del Iota-5 */
  {1, 36996, 678}, /* Del Epsilon-2 */
  {1, 37174, 605}, /* Del Kappa-7 */
  {1, 37135, 683}, /* Del Iota-5 */
  {1, 37019, 782}, /* Del Eta-3 */
  {1, 37127, 876}, /* Rotanev */
  {1, 37189, 955}, /* Svalocin */
  {1, 37400, 967}, /* Del Gamma2-12 */
  {1, 37304, 904}, /* Del Delta-11 */
  {1, 37162, 799}, /* Del Theta-8 */
  {1, 37135, 683}, /* Del Iota-5 */
  {0, 37127, 876}, /* Rotanev */
  {1, 37059, 880}, /* Del Zeta-4 */
  {-1,     0,     0},
};

/* Dorado */
static ConFig _Dorado_rey_figs[] = {
    { 0,  7681, -3089},
    { 1,  8220, -3302},
    { 1,  9165, -3448},
    { 1, 10009, -3749},
    { 1, 10343, -3944},
    { 0, 10009, -3749},
    { 1, 10623, -3785},
    { 1, 10343, -3944}, /* Back to Dor Delta */
    { 0, 10009, -3749}, /* Dor Beta */
    { 1,  8220, -3302}, /* Dor Alpha */
    {-1,     0,     0},
};

/* Draco */
static ConFig _Draco_rey_figs[] = {
  {0, 20742, 4160}, /* Dra Lambda-1 */
  {1, 22605, 4187}, /* Dra Kappa-5 */
  {1, 25332, 3863}, /* Thuban */
  {1, 27748, 3538}, /* Dra Iota-12 */
  {1, 28857, 3514}, /* Dra Theta-13 */
  {1, 29520, 3691}, /* Dra Eta-14 */
  {1, 30864, 3943}, /* Dra Zeta-22 */
  {1, 33023, 4280}, /* Dra Phi-43 */
  {1, 34577, 4060}, /* Dra Delta-57 */
  {1, 32206, 3412}, /* Dra Xi-32 */
  {1, 31568, 3310}, /* Dra Nu2-25 */
  {1, 31513, 3138}, /* Alwaid */
  {1, 32298, 3089}, /* Rastaban */
  {1, 32206, 3412}, /* Dra Xi-32 */
  {0, 33023, 4280}, /* Dra Phi-43 */
  {1, 33032, 4364}, /* Dra Chi-44 */
  {0, 34577, 4060}, /* Dra Delta-57 */
  {1, 35645, 4216}, /* Dra Epsilo-63 */
  {-1,     0,     0},
};

/* Equuleus */
static ConFig _Equuleus_rey_figs[] = {
    { 0, 38110,   607},
    { 1, 38234,   600},
    /*    { 1, 38487,   408}, */
    { 1, 38274,   314},
    {-1,     0,     0},
};

/* Eridanus */
static ConFig _Eridanus_rey_figs[] = {
  {0, 8345, -858}, /* Eri 53 */
  {1, 9236, -305}, /* Cursa */
  {1, 8565, -195}, /* Eri Mu-57 */
  {1, 8290, -201}, /* Eri Nu-48 */
  {1, 7556, -410}, /* Eri Omicr1-38 */
  {1, 7658, -459}, /* Eri Omicr2-40 */
  {1, 7141, -810}, /* Zaurak */
  {1, 6697, -585}, /* Eri Delta-23 */
  {1, 6388, -567}, /* Eri Epsilo-18 */
  {1, 5293, -533}, /* Eri Eta-3 */
  {1, 5472, -1417}, /* Eri Tau3-11 */
  {1, 5986, -1305}, /* Eri Tau4-16 */
  {1, 6414, -1297}, /* Eri Tau5-19 */
  {1, 6805, -1394}, /* Eri Tau6-27 */
  {1, 8267, -1833}, /* Eri Upsil2-52 */
  {1, 7921, -2041}, /* Eri 43 */
  {1, 7737, -2027}, /* Eri Upsil4-41 */
  {1, 7737, -2027}, /* Eri Upsil4-41 */
  {1, 6227, -2140}, /* Fornax Chi2 */
  {1, 5348, -2418}, /* Eri Theta1 */
  {1, 5207, -2306}, /* For Psi */
  {1, 4820, -2391}, /* Eri Iota */
  {1, 4410, -2862}, /* Eri Kappa */
  {1, 4095, -3090}, /* Eri Phi */
  {1, 3479, -3096}, /* Eri Chi */
  {1, 2931, -3434}, /* Achernar */
  {-1,     0,     0},
};

/* Gemini */
static ConFig _Gemini_rey_figs[] = {
  { 0, 10923,  1395}, /* 1 */ 
  { 1, 11246,  1350}, /* eta */
  { 1, 11489,  1350}, /* mu */
  { 1, 12118,  1507}, /* Mebsuta */
  { 1, 12934,  1814}, /* tau */
  { 1, 13473,  1907}, /* rho */
  { 1, 13638,  1913}, /* Castor */
  { 0, 13899,  1733}, /* sigma */
  { 0, 13959,  1681}, /* Pollux */ 
  { 1, 13678,  1613}, /* upsilon */
  { 1, 13203,  1318}, /* Wasat */
  { 1, 12723,  1234}, /* zeta */
  { 1, 11931,   983}, /* gamma */
  { 0, 11669,  1212}, /* nu */
  { 1, 12118,  1507}, /* Mebsuta */
  { 0, 11931,   983}, /* gamma */
  { 0, 13203,  1318}, /* Wasat */
  { 1, 13143,   992}, /* lambda */
  { 1, 12159,   773}, /* Xi */
  { 0, 13933,  1463}, /* kappa */
  { 1, 13678,  1613}, /* upsilon */
  { 1, 13372,  1667}, /* iota */
  { 1, 12934,  1814}, /* tau */
  { 1, 12383,  2037}, /* theta */
  {-1,     0,     0},
};

/* Grus */
static ConFig _Grus_rey_figs[] = {
  {0, 39418, -2241}, /* Gru Gamma */
  {1, 39783, -2372}, /* Gru Lambda */
  {1, 39847, -2817}, /* Gru Alpha */
  {1, 40880, -2813}, /* Gru Beta */
  {1, 41711, -2714}, /* Gru Iota */
  {1, 41606, -2611}, /* Gru Theta */
  {1, 40478, -2609}, /* Gru Delta1 */
  {1, 39847, -2817}, /* Gru Alpha */
  {0, 41057, -3079}, /* Gru Epsilon */
  {1, 40880, -2813}, /* Gru Beta */
  {1, 41426, -3165}, /* Gru Zeta */
  {-1,     0,     0},
};

/* Hercules */
static ConFig _Hercules_rey_figs[] = {
  { 0, 29457,  1149}, /* gamma */
  { 1, 29706,  1289}, /* Kornephoros */
  { 1, 30608,  1855}, /* epsilon */
  { 1, 31051,  2208}, /* pi */
  { 1, 31310,  2228}, /* rho */
  { 1, 32287,  2235}, /* theta */
  { 1, 31784,  2760}, /* iota */
  { 1, 29823,  2546}, /* Sigma */
  { 1, 29063,  2696}, /* Phi */
  { 1, 29392,  2779}, /* Tau */
  { 1, 31784,  2760}, /* iota */
  { 0, 31994,  1663}, /* mu */ 
  { 1, 31521,  1566}, /* lambda */
  { 1, 31050,  1490}, /* delta */
  { 1, 31039,   863}, /* alpha */
  { 0, 33810,  1090}, /* 111 */
  { 1, 33770,  1232}, /* 110 */
  { 1, 33111,  1306}, /* 109 */
  { 1, 32663,  1248}, /* 102 */
  { 1, 32445,  1296}, /*  95 */
  { 1, 32626,  1725}, /* omicron */
  { 1, 32333,  1754}, /* xi */ 
  { 1, 31051,  2208}, /* pi */
  { 1, 30087,  2335}, /* eta */
  { 1, 30038,  1896}, /* zeta */
  { 1, 30608,  1855}, /* epsilon */
  { 1, 31994,  1663}, /* mu */ 
  { 1, 32333,  1754}, /* xi */ 

  {-1,     0,     0}, 
};

/* Horologium - no change */

/* Hydra */
static ConFig _Hydra_rey_figs[] = {
  {0, 16062, 357}, /* Hya Zeta-16 */
  {1, 15803, 385}, /* Hya Epsilo-11 */
  {1, 15530, 342}, /* Hya Delta-4 */
  {1, 15563, 200}, /* Hya Sigma-5 */
  {1, 15697, 204}, /* Hya Eta-7 */
  {1, 16062, 357}, /* Hya Zeta-16 */
  {1, 16631, 139}, /* Hya Theta-22 */
  {1, 17396, -68}, /* Hya Iota-35 */
  {1, 17028, -519}, /* Hya Alpha-30 */
  {1, 18318, -741}, /* Hya Lambda-41 */
  {1, 18783, -1010}, /* Hya Mu-42 */
  {1, 19489, -971}, /* Hya Nu */
  {1, 19793, -1097}, /* Crt Alpha-7 */
  {1, 20150, -1369}, /* Crt Beta-11 */
  {1, 20790, -1911}, /* Hya Xi */
  {1, 21387, -2034}, /* Hya Beta */
  {1, 23968, -1390}, /* Hya Gamma-46 */
  {1, 25391, -1600}, /* Hya Pi-49 */
  {-1,     0,     0},
};

/* Hydrus */
static ConFig _Hydrus_rey_figs[] = {
    { 0,   773, -4635},
    { 1,  6817, -4454},
    { 1,  4788, -4096},
    { 1,  4252, -4119},
    { 1,  3448, -4058},
    { 1,  3563, -3694},
    {-1,     0,     0},
};

/* Indus */
static ConFig _Indus_rey_figs[] = {
  {0, 37127, -2837}, /* Ind Alpha */
  {1, 37321, -3115}, /* Ind Eta */
  {1, 37644, -3507}, /* Ind Beta */
  {1, 39538, -3299}, /* Ind Delta */
  {0, 39701, -3407}, /* Ind Epsilon */
  {1, 37644, -3507}, /* Ind Beta */
  {-1,     0,     0},
};

/* Lacerta */
static ConFig _Lacerta_rey_figs[] = {
  {0, 40079, 2265}, /* Lac 1 */
  {1, 40515, 2587}, /* Lac 6 */
  {1, 40815, 2657}, /* Lac 11 */
  {1, 40486, 2862}, /* Lac 5 */
  {1, 40336, 2969}, /* Lac 4 */
  {1, 40307, 3134}, /* Lac Beta-3 */
  {1, 40539, 3017}, /* Lac Alpha-7 */
  {1, 40486, 2862}, /* Lac 5 */
  {1, 40231, 2792}, /* Lac 2 */
  {1, 40515, 2587}, /* Lac 6 */
  {-1,     0,     0},
  /* Old Figure Follows
   *    { 0, 40079,  2264},
   *    { 1, 40514,  2587},
   *    { 1, 40486,  2862},
   *    { 1, 40335,  2968},
   *    { 1, 40307,  3133},
   *    { 1, 40538,  3016},
   *    { 1, 40486,  2862}, */

};

/* Leo */
static ConFig _Leo_rey_figs[] = {
    { 0, 16939,  1570},
    { 1, 17151,  1378},
    { 1, 17575,  1426},
    { 1, 17783,  1560},
    { 1, 16939,  1570}, /* Finish going around head */
    { 0, 17783,  1560},
    { 1, 18500,  1405},
    { 1, 18599,  1190},
    { 1, 18220,  1005},
    { 1, 18251,   718}, /* Regulus */
    { 0, 17575,  1426}, /* Draw  */
    { 1, 18220,  1005}, /* Front of neck */
    { 1, 17434,   593}, /* and forward front leg */
    { 0, 18220,  1005}, /* eta */
    /*{ 0, 18251,   718},  Regulus */ 
    { 1, 20227,   925}, /* theta */
    { 1, 20223,  1231}, /* Zozca */
    { 0, 21272,   874}, /* Denebola */
    { 1, 20223,  1231}, /* Zozca */
    { 1, 19870,  1210}, /* 60 */
    { 1, 18599,  1190}, /* Gamma */
    { 0, 20227,   925}, /* theta */
    { 1, 20517,   631}, /* iota */
    { 1, 20434,   361}, /* sigma */
    { 0, 20227,   925}, /* theta */
    { 1, 18984,   558},
    {-1,     0,     0},
};

/* Leo Minor */
static ConFig _Leo_Minor_rey_figs[] = {
  {0, 18223, 2115}, /* LMi 21 */
  {1, 18777, 2028}, /* LMi 30 */
  {1, 19599, 2053}, /* LMi 46 */
  {1, 18837, 2202}, /* LMi Beta-31 */
  {1, 18223, 2115}, /* LMi 21 */
  {1, 17227, 2184}, /* LMi 10 */
  {-1,     0,     0},
};

/* Lepus */
static ConFig _Lepus_rey_figs[] = {
  {0, 9397, -776}, /* Lep Kappa-4 */
  {1, 9388, -972}, /* Lep Mu-5 */
  {1, 9164, -1342}, /* Lep Epsilon-2 */
  {1, 9847, -1245}, /* Lep Beta-9 */
  {1, 9982, -1069}, /* Arneb */
  {1, 10409, -889}, /* Lep Zeta-14 */
  {1, 10692, -850}, /* Lep Eta-16 */
  {1, 10540, -1252}, /* Lep Delta-15 */
  {1, 10334, -1346}, /* Lep Gamma-13 */
  {1, 9847, -1245}, /* Lep Beta-9 */
  {0, 9982, -1069}, /* Arneb */
  {1, 9388, -972}, /* Lep Mu-5 */
  {1, 9587, -790}, /* Lep Lambda-6 */
  {-1,     0,     0},
};

/* Libra */
static ConFig _Libra_rey_figs[] = {
  {0, 28160, -1786}, /* Lib Tau-40 */
  {1, 28111, -1688}, /* Lib Upsilo-39 */
  {1, 28066, -887}, /* Zuben el Hakrabi */
  {1, 27510, -562}, /* Zuben el Chamali */
  {1, 26726, -962}, /* Lib Alpha2-9 */
  {1, 27122, -1516}, /* Lib Sigma-20 */
  {0, 28066, -887}, /* Zuben el Hakrabi */
  {1, 26726, -962}, /* Lib Alpha2-9 */
  {-1,     0,     0},
};

/* Lupus */
static ConFig _Lupus_rey_figs[] = {
  {0, 28998, -2208}, /* Lup Theta */
  {1, 28529, -2017}, /* Lup Chi-5 */
  {1, 28193, -2064}, /* Lup Psi1-3 */
  {1, 27654, -2175}, /* Lup Phi1 */
  {1, 28804, -2303}, /* Lup Eta */
  {0, 27654, -2175}, /* Lup Phi1 */
  {1, 26956, -2588}, /* Lup Beta */
  {1, 26458, -2843}, /* Lup Alpha */
  {1, 26337, -2965}, /* Lup Rho */
  {1, 27154, -2823}, /* Lup Pi */
  {1, 27265, -2716}, /* Lup Lambda */
  {1, 27641, -2438}, /* Lup Delta */
  {1, 27742, -2382}, /* Lup Upsilon */
  {1, 27695, -2211}, /* Lup Phi2 */
  {1, 27654, -2175}, /* Lup Phi1 */
  {0, 26337, -2965}, /* Lup Rho */
  {1, 26178, -3027}, /* Lup Sigma */
  {0, 27154, -2823}, /* Lup Pi */
  {1, 27358, -2924}, /* Lup Kappa1 */
  {1, 27369, -3125}, /* Lup Zeta */
  {0, 27265, -2716}, /* Lup Lambda */
  {1, 27556, -2872}, /* Lup Mu */
  {1, 27680, -2681}, /* Lup Epsilon */
  {0, 27742, -2382}, /* Lup Upsilon */
  {1, 28054, -2470}, /* Lup Gamma */
  {1, 28142, -2554}, /* Lup Omega */
  {1, 28236, -2680}, /*  */
  {0, 25985, -2722}, /* Lup Tau2 */
  {1, 26458, -2843}, /* Lup Alpha */
  {1, 25782, -2763}, /* Lup Iota */
  {-1,     0,     0},
};

/* Lynx - no change */

/* Lyra */
static ConFig _Lyra_rey_figs[] = {
  {0, 33508, 2327}, /* Vega */
  {1, 32996, 2164}, /* Lyr Kappa-1 */
  {1, 33902, 2002}, /* Sheliak */
  {1, 34168, 1961}, /* Sulaphat */
  {1, 34691, 2288}, /* Lyr Theta-21 */
  {1, 34613, 2349}, /* Lyr Eta-20 */
  {1, 34060, 2637}, /* Lyr 13 */
  {1, 33730, 2380}, /* Lyr Epsilo1-4 */
  {2, 33508, 2327}, /* Vega */
  {0, 33743, 2256}, /* Lyr Zeta1-6 */
  {1, 33902, 2002}, /* Sheliak */
  {0, 34012, 2218}, /* Lyr Delta1-11 */
  {1, 34168, 1961}, /* Sulaphat */
  {-1,     0,     0},
    /* Old Figure follows
     * { 0, 34060,  2636}, 
     *  { 1, 34613,  2348},
     *  { 1, 34691,  2288},
     *  { 1, 33508,  2327},
     *  { 1, 33730,  2380},
     *  { 0, 33508,  2327},
     *  { 1, 33743,  2256},
     *  { 1, 33902,  2001},
     *  { 1, 34168,  1961},
     *  { 1, 34012,  2218},
     *  { 1, 33743,  2256},
     *  { 0, 33508,  2327},
     *  { 1, 32996,  2163}, */
};

/* Mensa - no change */

/* Microscopium - no change */

/* Monoceros */
static ConFig _Monoceros_rey_figs[] = {
  {0, 12029, 594}, /* Mon 15 */
  {1, 11787, 440}, /* Mon 13 */
  {1, 11513, 276}, /* Mon Epsilon-8 */
  {1, 12236, 145}, /* Mon 18 */
  {1, 11787, 440}, /* Mon 13 */
  {0, 12236, 145}, /* Mon 18 */
  {1, 12956, -29}, /* Mon Delta-22 */
  {1, 11665, -422}, /* Mon Beta-11 */
  {1, 11246, -376}, /* Mon Gamma-5 */
  {0, 12956, -29}, /* Mon Delta-22 */
  {1, 14437, -83}, /* Mon 28 */
  {1, 14658, -179}, /* Mon Zeta-29 */
  {0, 14437, -83}, /* Mon 28 */
  {1, 13837, -573}, /* Mon Alpha-26 */
  {-1,     0,     0},
  /* Old figure follows
   *    { 0, 14657,  -179},
   *    { 1, 13837,  -573},
   *    { 1, 12956,   -29},
   *    { 1, 11664,  -421},
   *    { 1, 11245,  -376},
   *    { 1, 11513,   275},
   *    { 1, 11787,   439}, */
};

/* Musca */
static ConFig _Musca_rey_figs[] = {
  {0, 21168, -4003}, /* Mus Lambda */
  {1, 21247, -4008}, /* Mus Mu */
  {1, 22127, -4077}, /* Mus Epsilon */
  {1, 22716, -4148}, /* Mus Alpha */
  {1, 23468, -4292}, /* Mus Delta */
  {1, 22574, -4327}, /* Mus Gamma */
  {1, 22716, -4148}, /* Mus Alpha */
  {1, 22988, -4086}, /* Mus Beta */
  {-1,     0,     0},
};

/* Norma */
static ConFig _Norma_rey_figs[] = {
  {0, 28995, -2710}, /* Nor Delta */
  {1, 28896, -2953}, /* Nor Eta */
  {1, 29395, -3009}, /* Nor Gamma2 */
  {1, 29616, -2853}, /* Nor Epsilon */
  {1, 28995, -2710}, /* Nor Delta */
  {-1,     0,     0},
};

/* Octans */
static ConFig _Octans_rey_figs[] = {
   { 0, 26007, -5020},
   { 1, 39044, -4643},
   /*   { 1, 40200, -4826}, */
   { 1, 40981, -4882},
   { 1, 26007, -5020},
   {-1,     0,     0},
};

/* Ophiuchus */
static ConFig _Ophiuchus_rey_figs[] = {
    { 0, 31301, -1688}, /* 43    */
    { 1, 31060, -1596}, /* 36    */
    { 1, 31260, -1499}, /* Theta */ 
    { 1, 31391, -1450}, /* 44    */
    { 1, 31230, -1267}, /* Xi    */
    { 1, 30911,  -943}, /* Eta   */
    { 1, 29915,  -634}, /* Zeta  */
    { 1, 29734,  -996}, /* Phi   */
    { 1, 29610, -1107}, /* Chi   */
    { 1, 29523, -1202}, /* Psi   */
    { 1, 29764, -1288}, /* Omega */
    { 0, 30911,  -943}, /* Eta   */
    { 1, 31904,   274}, /* Kelb al Rai */
    { 1, 31647,   753}, /* Ras Alhague */
    { 1, 30530,   562}, /* Kappa */
    { 1, 29915,  -634}, /* Zeta  */
    { 0, 32419,   175}, /* 67    */
    { 1, 32036,   162}, /* Gamma */
    { 1, 31904,   274}, /* Kelb al Rai */
    { 1, 30530,   562}, /* Kappa */
    { 1, 30419,   609}, /* Iota  */
    { 1, 29727,   119}, /* Lambda */
    { 1, 29230,  -221}, /* Delta */
    {-1,     0,     0},
};

/* Orion */
static ConFig _Orion_rey_figs[] = {
  {0, 10433, -580}, /* Ori Kappa-53 */
  {1, 10223, -116}, /* Ori Zeta-50 */
  {1, 10655, 444}, /* Betelgeuse */
  {1, 9924, 357}, /* Ori 32 */
  {1, 9754, 381}, /* Bellatrix */
  {1, 9960, -17}, /* Mintaka */
  {1, 9734, -143}, /* Ori Eta-28 */
  {1, 9436, -492}, /* Rigel */
  {0, 10223, -116}, /* Ori Zeta-50 */
  {1, 10086, -72}, /* Alnilam */
  {1, 9960, -17}, /* Mintaka */
  {0, 9924, 357}, /* Ori 32 */
  {1, 10054, 596}, /* Ori Lambda-39 */
  {0, 9754, 381}, /* Bellatrix */
  {1, 8695, 418}, /* Ori Pi3-1 */
  {0, 8956, 103}, /* Ori Pi6-10 */
  {1, 8828, 146}, /* Ori Pi5-8 */
  {1, 8736, 336}, /* Ori Pi4-3 */
  {1, 8695, 418}, /* Ori Pi3-1 */
  {1, 8718, 534}, /* Ori Pi2-2 */
  {1, 8847, 609}, /* Ori Pi1-7 */
  {1, 8891, 811}, /* Ori Omicro2-9 */
  {0, 10655, 444}, /* Betelgeuse */
  {1, 10872, 579}, /* Ori Mu-61 */
  {1, 11158, 853}, /* Ori Xi-70 */
  {1, 11027, 886}, /* Ori Nu-67 */
  {1, 10631, 1217}, /* Ori Chi1-54 */
  {1, 10918, 1208}, /* Ori Chi2-62 */
  {1, 11158, 853}, /* Ori Xi-70 */
  {-1,     0,     0},
};

/* Pavo */
static ConFig _Pavo_rey_figs[] = {
  {0, 38593, -3921}, /* Pav Gamma */
  {1, 36769, -3404}, /* Pav Alpha */
  {1, 36262, -3970}, /* Pav Delta */
  {1, 36018, -4374}, /* Pav Epsilon */
  {1, 33691, -4285}, /* Pav Zeta */
  {1, 34109, -4034}, /* Pav Kappa */
  {1, 36262, -3970}, /* Pav Delta */
  {1, 37349, -3972}, /* Pav Beta */
  {1, 38593, -3921}, /* Pav Gamma */
  {0, 34109, -4034}, /* Pav Kappa */
  {1, 33967, -3731}, /* Pav Lambda */
  {1, 32657, -3820}, /* Pav Pi */
  {1, 31972, -3883}, /* Pav Eta */
  {0, 32657, -3820}, /* Pav Pi */
  {1, 33097, -3689}, /* Pav Xi */
  {1, 33967, -3731}, /* Pav Lambda */
  {-1,     0,     0},
};

/* Pegasus */
static ConFig _Pegasus_rey_figs[] = {
  { 0, 39125,   592}, /* Eniph */
  { 1, 40844,   649}, /* Homam */
  { 0, 39906,   371}, /* theta */
  { 1, 40844,   649}, /* Homam */
  { 1, 41001,   730}, /* xi */
  { 1, 41543,   912}, /* Markab */
  { 1,   397,   911}, /* Algenib */
  { 2,   251,  1745}, /* Alpheratz */
  { 2, 41513,  1684}, /* beta */
  { 1, 41543,   912}, /* Markab */
  { 0,   397,   911}, /* Algenib */
  { 1, 41513,  1684}, /* beta */
  { 0, 38462,  1188}, /* 1 */
  { 1, 39139,  1538}, /* kappa */
  { 1, 39810,  1520}, /* iota */
  { 1, 40996,  1413}, /* lambda */
  { 1, 41100,  1476}, /* mu */
  { 1, 40890,  1813}, /* eta */
  { 1, 39876,  1990}, /* pi */
  { 0, 39135,  1041}, /* 9 */
  { 1, 39810,  1520}, /* iota */
  { 0, 39810,  1520}, /* iota */
  { 1, 41001,   730}, /* xi */
  { 0, 41543,   912}, /* Markab */
  { 1, 40996,  1413}, /* lambda */
  {-1,     0,     0},
};

/* Perseus */
static ConFig _Perseus_rey_figs[] = {
  { 0,  7397,  3021}, /* lambda */
  { 1,  7646,  2904}, /* mu */
  { 1,  7460,  2862}, /* 48 */
  { 1,  6687,  2867}, /* delta */ 
  { 1,  6494,  2891}, /* psi */
  { 1,  6129,  2991}, /* Mirfak */
  { 1,  5544,  3210}, /* gamma */
  { 1,  5121,  3353}, /* eta */
  { 1,  5227,  3165}, /* tau */
  { 1,  5672,  2976}, /* iota */
  { 0,  5544,  3210}, /* gamma */
  { 1,  5227,  3165}, /* tau */
  { 0,  6129,  2991}, /* Mirfak */
  { 1,  6317,  2879}, /* sigma */
  { 1,  6756,  2554}, /* nu */
  { 1,  7135,  2400}, /* epsilon */
  { 1,  7169,  2147}, /* xi */
  { 1,  7024,  1913}, /* zeta */
  { 1,  6729,  1937}, /* omicron */
  { 0,  6129,  2991}, /* Mirfak */
  { 1,  5672,  2976}, /* iota */
  { 1,  4925,  2953}, /* theta */
  { 1,  3109,  3041}, /* phi */
  { 0,  5672,  2976}, /* iota */
  { 1,  5685,  2691}, /* kappa */
  { 1,  5645,  2457}, /* Algol */
  { 1,  5555,  2330}, /* rho */
  { 1,  5117,  2299}, /* 16 */
  { 0,  5645,  2457}, /* Algol */
  { 1,  7135,  2400}, /* epsilon */
  {-1,     0,     0},
};

/* Phoenix */
static ConFig _Phoenix_rey_figs[] = {
  {0, 2052, -3314}, /* Phe Zeta */
  {1, 1983, -2803}, /* Phe Beta */
  {1, 1240, -2765}, /* Phe Mu */
  {1,  786, -2620}, /* Phe Kappa */
  {1,  789, -2538}, /* Phe Alpha */
  {1,  282, -2744}, /* Phe Epsilon */
  {1,  786, -2620}, /* Phe Kappa */
  {0, 2738, -2944}, /* Phe Delta */
  {1, 1983, -2803}, /* Phe Beta */
  {1, 2651, -2599}, /* Phe Gamma */
  {1, 1240, -2765}, /* Phe Mu */
  {0, 1983, -2803}, /* Phe Beta */
  {1, 1301, -3447}, /* Phe Eta */
  {1, 1240, -2765}, /* Phe Mu */
  {-1,     0,     0},
};

/* Pictor */
static ConFig _Pictor_rey_figs[] = {
  {0, 12246, -3716}, /* Pic Alpha */
  {1, 10495, -3370}, /* Pic Gamma */
  {1, 10419, -3063}, /* Pic Beta */
  {1, 12246, -3716}, /* Pic Alpha */
  {-1,     0,     0},
};

/* Pisces */
static ConFig _Pisces_rey_figs[] = {
  {0, 42599, 338}, /* Psc Iota-17 */
  {1, 42239, 383}, /* Psc Theta-10 */
  {1, 42010, 323}, /* Psc 7 */
  {1, 41915, 197}, /* Psc Gamma-6 */
  {1, 42217, 67}, /* Psc 9 */
  {1, 42661, 107}, /* Psc Lambda-18 */
  {1, 42599, 338}, /* Psc Iota-17 */
  {1, 43179, 412}, /* Psc Omega-28 */
  {1, 1461, 455}, /* Psc Delta-63 */
  {1, 1888, 473}, /* Psc Epsilo-71 */
  {1, 2213, 455}, /* Psc Zeta-86 */
  {1, 2706, 369}, /* Psc Mu-98 */
  {1, 3043, 329}, /* Psc Nu-106 */
  {1, 3661, 166}, /* Kaitain */
  {1, 3162, 549}, /* Psc Omicr-110 */
  {1, 2745, 921}, /* Psc Eta-99 */
  {1, 2588, 1150}, /* Psc Rho-93 */
  {1, 2212, 1475}, /* Psc Phi-85 */
  {1, 2384, 1636}, /* Psc Upsilo-90 */
  {1, 2150, 1805}, /* Psc Tau-83 */
  {1, 2212, 1475}, /* Psc Phi-85 */
  {-1,     0,     0},
};

/* Piscis Austrinus */
static ConFig _Piscis_Austrinus_rey_figs[] = {
  {0, 39148, -1981}, /* PsA Iota-9 */
  {1, 39852, -1979}, /* PsA Mu-14 */
  {1, 40545, -1940}, /* PsA Beta-17 */
  {1, 41176, -1972}, /* PsA Gamma-22 */
  {1, 41278, -1952}, /* PsA Delta-23 */
  {1, 41330, -1777}, /* Fomalhaut */
  {1, 40820, -1622}, /* PsA Epsilo-18 */
  {1, 39852, -1979}, /* PsA Mu-14 */
  {1, 39232, -1853}, /* PsA Theta-10 */
  {1, 39148, -1981}, /* PsA Iota-9 */
  {-1,     0,     0},
};

/* Puppis */
static ConFig _Puppis_rey_figs[] = {
  {0, 11519, -3161}, /* Canopus */
  {1, 11933, -2591}, /* Pup Nu */
  {1, 13114, -2225}, /* Pup Pi */
  {1, 13661, -1702}, /* GSC star */
  {1, 13765, -1608}, /* 'Nother GSC star */
  {1, 13749, -1522}, /* GSC star */
  {1, 14079, -1492}, /* Pup Xi-7 */
  {1, 14626, -1458}, /* Pup Rho-15 */
  {1, 14508, -2400}, /* Pup Zeta */
  {1, 14686, -2840}, /* Vel Gamma2 */
  {0, 13661, -1702}, /* GSC Star */
  {1, 13914, -1737}, /* Pup 3 */
  {1, 14043, -1556}, /* Pup Omicron */
  {1, 14079, -1492}, /* Pup Xi-7 */
  {-1,     0,     0},
};

/* Pyxis */
static ConFig _Pyxis_rey_figs[] = {
  { 0, 14508, -2400}, /* Pup Zeta */
  { 1, 15603, -2118},
  { 1, 15708, -1991},
  { 1, 15916, -1662},
  {-1,     0,     0},
};

/* Reticulum - no change */

/* Sagitta */
static ConFig _Sagitta_rey_figs[] = {
  /*    { 0, 36154,  1199}, */  /*  Sge Eta-16 */
    { 0, 35962,  1169},
    { 1, 35621,  1112},
    { 1, 35431,  1048},
    { 0, 35621,  1112},
    { 1, 35403,  1080},
    {-1,     0,     0},
};

/* Sagittarius */
static ConFig _Sagittarius_rey_figs[] = {
  {0, 32929, -2205}, /* Sgr Eta */
  {1, 33125, -2063}, /* Kaus Australis */
  {1, 33030, -1789}, /* Sgr Delta-19 */
  {1, 33239, -1525}, /* Sgr Lambda-22 */
  {1, 32813, -1263}, /* Sgr Mu-13 */
  {1, 32574, -1825}, /* Sgr Gamma2-10 */
  {1, 32929, -2205}, /* Sgr Eta */
  {0, 32574, -1825}, /* Sgr Gamma2-10 */
  {1, 33030, -1789}, /* Sgr Delta-19 */
  {1, 33770, -1619}, /* Sgr Phi-27 */
  {1, 34058, -1577}, /* Sgr Sigma-34 */
  {1, 34132, -1266}, /* Sgr Xi2-37 */
  {1, 34493, -1261}, /* Sgr Pi-41 */
  {1, 34850, -1070}, /* Sgr Rho1-44 */
  {1, 34852, -957}, /* Sgr Upsilo-46 */
  {0, 34493, -1261}, /* Sgr Pi-41 */
  {1, 34058, -1577}, /* Sgr Sigma-34 */
  {1, 34408, -1660}, /* Sgr Tau-40 */
  {1, 35992, -2116}, /* Sgr Theta1 */
  {1, 35858, -2512}, /* Sgr Iota */
  {0, 33770, -1619}, /* Sgr Phi-27 */
  {1, 34278, -1792}, /* Sgr Zeta-38 */
  {1, 34917, -2436}, /* Sgr Alpha */
  {1, 34879, -2667}, /* Sgr Beta1 */
  {0, 34917, -2436}, /* Sgr Alpha */
  {1, 35992, -2116}, /* Sgr Theta1 */
  {0 , 34278, -1792}, /* Sgr Zeta-38 */
  {1, 34408, -1660}, /* Sgr Tau-40 */
  {-1,     0,     0},
};

/* Scorpius */
static ConFig _Scorpius_rey_figs[] = {
  {0, 28810, -1357}, /* Sco Delta-7 */
  {1, 28963, -1188}, /* Sco Beta1-8 */
  {1, 29159, -1166}, /* Sco Nu-14 */
  {1, 29436, -1535}, /* Sco Sigma-20 */
  {1, 29169, -1675}, /* Sco 13 */
  {1, 28707, -1752}, /* Sco Rho-5 */
  {1, 28766, -1566}, /* Sco Pi-6 */
  {0, 29436, -1535}, /* Sco Sigma-20 */
  {1, 29682, -1585}, /* Antares */
  {1, 29877, -1692}, /* Sco Tau-23 */
  {1, 30305, -2057}, /* Sco Epsilo-26 */
  {1, 30356, -2282}, /* Sco Mu1 */
  {1, 30438, -2541}, /* Sco Zeta2 */
  {1, 30965, -2594}, /* Sco Eta */
  {1, 31720, -2579}, /* Sco Theta */
  {1, 32028, -2407}, /* Sco Iota1 */
  {1, 31875, -2341}, /* Sco Kappa */
  {1, 31523, -2237}, /* Sco Upsilo-34 */
  {1, 31608, -2226}, /* Shaula */
  { 1, 32095, -2222}, /* NGC 6441 (GC) ? */
  {-1,     0,     0},
  /* Orginal figure Follows
     *    { 0, 28963, -1188},
     *    { 1, 29004, -1240},
     *  { 1, 29159, -1166},
     *  { 1, 28930,  -682},
     *  { 1, 28963, -1188},
     *  { 1, 28810, -1357},
     *  { 1, 28765, -1566},
     *  { 1, 28706, -1752},
     *  { 0, 28810, -1357},
     *  { 1, 29435, -1535},
     *  { 1, 29681, -1585},
     *  { 1, 29876, -1692},
     *  { 1, 30305, -2057},
     *  { 1, 29741, -2082},
     *  { 1, 29891, -2115},
     *  { 1, 30305, -2057},
     *  { 1, 30356, -2282},
     *  { 1, 30437, -2541},
     *  { 1, 30964, -2594},
     *  { 1, 31719, -2579},
     *  { 1, 32027, -2407},
     *  { 1, 31874, -2341},
     *  { 1, 31608, -2226},
     *  { 0, 32027, -2407}, */
};

/* Sculptor - no change */

/* Scutum */
static ConFig _Scutum_rey_figs[] = {
  {0, 33456, -494}, /* Sct Alpha */
  {1, 33276, -873}, /* Sct Gamma */
  {1, 33668, -543}, /* Sct Delta */
  {1, 33815, -284}, /* Sct Beta */
  {1, 33456, -494}, /* Sct Alpha */
  {-1,     0,     0},
};

/* Serpens Caput */
static ConFig _Serpens_Caput_rey_figs[] = {
    { 0, 29634,  -502},  /* Upsilon Oph */
    { 1, 29350,  -282},  /* Epsilon Oph */
    { 1, 29230,  -221}, /* Delta Oph*/
    { 1, 28488,  -205}, /* Mu */
    { 1, 28509,   131}, /* Omega */
    { 1, 28524,   268}, /* Epsilon */
    { 1, 28328,   385}, /* Unulkalhay */
    { 1, 28407,   441}, /* Lambda */
    { 1, 28044,   632}, /* Delta */
    { 1, 28385,   925}, /* Beta  */
    { 1, 28693,   939}, /* Gamma */
    { 1, 28462,  1088}, /* Kappa */
    { 1, 28246,  1180}, /* Iota */
    { 1, 28385,   925}, /* Beta */
    {-1,     0,     0},
};

/* Sextans - no change */

/* Taurus */
static ConFig _Taurus_rey_figs[] = {
  { 0,  9789,  1716}, /* Alnath */
  { 1,  8467,  1377}, /* tau */
  { 1,  7989,  1368}, /* upsilon */
  { 1,  8058,  1150}, /* epsilon */
  { 1,  7964,  1075}, /* delta-3 */
  { 1,  7888,  1052}, /* delta-1 */
  { 1,  7794,   937}, /* gamma */
  { 1,  8060,   952}, /* theta-2 */
  { 1,  8277,   990}, /* Aldebaran */
  { 1, 10129,  1268}, /* zeta */
  { 1,  8467,  1377}, /* tau */
  { 0,  7794,   937}, /* gamma */
  { 1,  7220,   749}, /* lambda */
  { 1,  6215,   583}, /* xi-2 */
  { 1,  6326,   776}, /* 5 */
  { 0,  6215,   583}, /* xi-2 */
  { 1,  6144,   541}, /* omicron-1 */
  { 1,  6506,    24}, /* 10 */
  { 0,  6215,   583}, /* xi-2 */
  { 1,  7294,   359}, /* nu */
  { 0,  7665,   533}, /* mu */
  { 1,  7794,   937}, /* gamma */
  { 1,  8344,   750}, /* 90 */
  { 1,  8269,   609}, /* 88 */ 
  { 0,  7989,  1368}, /* upsilon */
  { 1, 6810,   1447}, /* Pleides */ 
  {-1,     0,     0},
};

/* Telescopium */
static ConFig _Telescopium_rey_figs[] = {
  /*    { 0, 32737, -2757}, */
    { 0, 33209, -2758},
    { 1, 33265, -2944},
    {-1,     0,     0},
};

/* Triangulum */
static ConFig _Triangulum_rey_figs[] = {
  {0, 3392, 1775}, /* Tri Alpha-2 */
  {1, 4119, 2031}, /* Tri Gamma-9 */
  {1, 3886, 2099}, /* Tri Beta-4 */
  {1, 3392, 1775}, /* Tri Alpha-2 */
  {-1,     0,     0},
};

/* Triangulum Australe - no change */

/* Tucana */
static ConFig _Tucana_rey_figs[] = {
  /*    { 0, 40420, -3897}, */
    { 0, 40155, -3615},
    { 1, 41923, -3494},
    { 1, 43197, -3934},
    { 1,   602, -3892},
    { 1,   946, -3777},
    { 1, 41923, -3494},
    {-1,     0,     0},
};

/* Ursa Major */
static ConFig _Ursa_Major_rey_figs[] = {
    { 0, 24826,  2958}, /* Benetnasch */
    { 1, 24118,  3295}, /* Mizar  */
    { 1, 23221,  3357}, /* Alioth */
    { 1, 22063,  3421}, /* Megrez */
    { 1, 21415,  3221}, /* Phecda */
    { 1, 19855,  3382}, /* Merak  */
    { 1, 19912,  3705}, /* Dubhe  */
    { 0, 24826,  2958}, /* Benetnasch */
    { 2, 21181,  2866}, /* Chi */
    { 2, 20090,  2669}, /* Psi */
    { 2, 20354,  1985}, /* Nu */
    { 0, 20345,  1891}, /* Xi */
    { 2, 18670,  2489}, /* Mu */
    { 2, 16309,  2829}, /* Kappa */
    { 2, 16176,  2882}, /* Talitha */
    { 2, 15308,  3643}, /* Omicron */
    { 2, 17146,  3783}, /* 23 */
    { 2, 19912,  3705}, /* Dubhe */
    { 2, 22063,  3421}, /* Megrez */
    { 0, 17146,  3783}, /* 23 */
    { 2, 17729,  3542}, /* Upsilon */
    { 2, 17185,  3100}, /* Theta */
    { 2, 18513,  2574}, /* Lambda */
    {-1,     0,     0},
};

/* Ursa Minor - no change */

/* Vela - no change */

/* Virgo */
static ConFig _Virgo_rey_figs[] = {
    { 0, 21756,   523},
    { 1, 21176,   391},
    { 1, 21321,   105},
    { 1, 22197,   -40},
    { 1, 22850,   -86},
    { 1, 23268,   203},
    { 1, 23465,   657},
    { 0, 23268,   203},
    { 1, 24441,   -35},
    { 1, 24156,  -669},
    { 1, 23698,  -332},
    { 1, 22850,   -86},
    { 0, 24441,   -35},
    { 1, 25249,    92},
    { 1, 26587,   113},
    { 0, 24156,  -669},
    { 1, 25587,  -616},
    { 1, 25680,  -360},
    { 1, 26492,  -339},
    /* Only need to add two segments to generate Rey's figure: */
    {0, 21756, 524}, /* Vir Omicron-9 */
    {1, 22850, -86}, /* Vir Gamma-29 */
    {0, 25249, 93}, /* Vir Tau-93 */
    {1, 25680, -360}, /* Vir Iota-99 */
    {-1,     0,     0},
};

/* Volans - no change */

/* Vulpecula */
static ConFig _Vulpecula_rey_figs[] = {
    { 0, 35804,  1444},
    { 1, 35061,  1479},
    /*    { 1, 34686,  1283}, */
    {-1,     0,     0},
};

/* Serpens Cauda */
static ConFig _Serpens_Cauda_rey_figs[] = {
    { 0, 34086,   252}, /* Theta  */
    { 1, 32620,   573}, /* Oph 72 */
    { 1, 32563,   149}, /* Oph 70 */
    { 1, 33039,  -173}, /* Eta    */
    { 1, 32371,  -586}, /* Oph Nu */
    { 1, 31842,  -772}, /* Omicron */
    { 1, 31727,  -923}, /* Xi     */
    { 1, 31225,  -770}, /* Nu     */ 
    {-1,     0,     0},
};


static ConFig *reyfigmap[NCNS] = {
    _Andromeda_rey_figs,
    _Antlia_figs,
    _Apus_rey_figs,
    _Aquila_rey_figs,
    _Aquarius_rey_figs,
    _Ara_rey_figs,
    _Aries_rey_figs,
    _Auriga_rey_figs,
    _Bootes_rey_figs,
    _Canis_Major_rey_figs,
    _Canis_Minor_figs,
    _Canes_Venatici_figs,
    _Caelum_figs,
    _Camelopardalis_rey_figs,
    _Capricornus_rey_figs,
    _Carina_rey_figs,
    _Cassiopeia_rey_figs,
    _Centaurus_rey_figs,
    _Cepheus_rey_figs,
    _Cetus_rey_figs,
    _Chamaeleon_rey_figs,
    _Circinus_figs,
    _Cancer_rey_figs,
    _Columba_rey_figs,
    _Coma_Berenices_rey_figs,
    _Corona_Australis_figs,
    _Corona_Borealis_figs,
    _Crater_rey_figs,
    _Crux_figs,
    _Corvus_rey_figs,
    _Cygnus_rey_figs,
    _Delphinus_rey_figs,
    _Dorado_rey_figs,
    _Draco_rey_figs,
    _Equuleus_rey_figs,
    _Eridanus_rey_figs,
    _Fornax_figs,
    _Gemini_rey_figs,
    _Grus_rey_figs,
    _Hercules_rey_figs,
    _Horologium_figs,
    _Hydra_rey_figs,
    _Hydrus_rey_figs,
    _Indus_rey_figs,
    _Leo_Minor_rey_figs,
    _Lacerta_rey_figs,
    _Leo_rey_figs,
    _Lepus_rey_figs,
    _Libra_rey_figs,
    _Lupus_rey_figs,
    _Lynx_figs,
    _Lyra_rey_figs,
    _Mensa_figs,
    _Microscopium_figs,
    _Monoceros_rey_figs,
    _Musca_rey_figs,
    _Norma_rey_figs,
    _Octans_rey_figs,
    _Ophiuchus_rey_figs,
    _Orion_rey_figs,
    _Pavo_rey_figs,
    _Pegasus_rey_figs,
    _Perseus_rey_figs,
    _Phoenix_rey_figs,
    _Pictor_rey_figs,
    _Piscis_Austrinus_rey_figs,
    _Pisces_rey_figs,
    _Puppis_rey_figs,
    _Pyxis_rey_figs,
    _Reticulum_figs,
    _Sculptor_figs,
    _Scorpius_rey_figs,
    _Scutum_rey_figs,
    _Serpens_Caput_rey_figs,
    _Sextans_figs,
    _Sagitta_rey_figs,
    _Sagittarius_rey_figs,
    _Taurus_rey_figs,
    _Telescopium_rey_figs,
    _Triangulum_Australe_figs,
    _Triangulum_rey_figs,
    _Tucana_rey_figs,
    _Ursa_Major_rey_figs,
    _Ursa_Minor_figs,
    _Vela_figs,
    _Virgo_rey_figs,
    _Volans_figs,
    _Vulpecula_rey_figs,
    _Serpens_Cauda_rey_figs,
};

/* given a constellation id and epoch, return arrays of ra[] and dec[]
 *   end-points precessed to the desired epoch that, if connected, will form the
 *   given constellation figure.
 * dcodes is 0 if the coord is a "move-to", 1 if a "draw-to" or 2 if a "draw-to
 *   as dotted-line".
 * return the total number of tripples or -1 if id is bogus.
 * the arrays need be no larger than 33 entries.
 */
int
cns_reyfigure (int id, double e, double ra[], double dec[], int dcodes[])
{
	ConFig *cfp;

	if (id < 0 || id >= NCNS)
	    return (-1);

	for (cfp = reyfigmap[id]; cfp->drawcode >= 0; cfp++) {
	    *ra = hrrad((double)cfp->ra/1800.0);
	    *dec = degrad((double)cfp->dec/60.0);
	    precess (J2000, e, ra, dec);
	    ra++;
	    dec++;
	    *dcodes++ = cfp->drawcode;
	}

	return (cfp - reyfigmap[id]);
}

/* For RCS Only -- Do Not Edit */
static char *rcsid[2] = {(char *)rcsid, "@(#) $RCSfile: constel.c,v $ $Date: 2005-01-17 10:13:04 $ $Revision: 1.2 $ $Name: not supported by cvs2svn $"};
