// Peida Interactive   -   PI        R. Ansari 97-99
// Traceur (Drawer) pour NTupleInterface  
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <math.h>
#include "sopnamsp.h"
#include "pintuple.h"

//++
// Class	PINTuple
// Lib		PIext
// include	pintuple.h
//
//	Classe de traceur 2D (dans un plan)  partir des donnes
//	d'un objet implmentant l'interface *NTupleInterface*.
//	Les objets "PINTuple" peuvent tracer des signes (markers)
//	ventuellement avec des barres d'erreur et une tiquette
//	pour chaque point. Si un nom de colonne poids est spcifi,
//	la taille et/ou la couleur des signes (markers) sera fonction
//	de la valeur de poids.
//--
//++
// Links	Parents
// PIDrawer
//--
//++
// Links	Voir aussi
// NTupleInterface
// PINTuple3D
//--

//++
// Titre	Constructeur
//--
//++
// PINTuple(NTupleInterface* nt, bool ad)
//	Constructeur. Si "ad == true", l'objet "nt" est dtruit par 
//	le destructeur de l'objet "PINTuple"
//	Note : "nt" doit tre cr par new dans ce cas.
//--
 
/* --Methode-- */
PINTuple::PINTuple(NTupleInterface* nt, bool ad)
: PIDrawer()
{
  mNT = nt;
  mAdDO = ad; 
  SetStats(true);
  SetStatPosOffset();
  ConnectPoints(false);
  UseSizeScale(true, 5);
  UseColorScale(true);
  SelectXY(NULL, NULL);
  SelectWt(NULL);
  SelectErrBar();
  SelectLabel(NULL);
  SetName("NTupleDrw");
  NptDraw = 0;

}

PINTuple::~PINTuple()
{
  if (mAdDO && mNT)  delete mNT;
}

//++
// Titre	Mthodes
//--
//++
// void  SelectXY(const char* px, const char* py)
//	Choix des noms de colonnes X,Y dfinissant les coordonnes des points. 
//	Ces deux colonnes doivent tre spcifies pour obtenir un trac.
// void  SelectErrBar(const char* erbx=NULL, const char* erby=NULL)
//	Choix des noms de colonnes pour le trac des barres d'erreur.
// void  SelectWt(const char* pw=NULL, int nbins=10)
//	Choix du nom de colonne poids. Dans ce cas, la taille du signe
//	(marker) sera proportionnel  la valeur de cette colonne pour 
//	chaque point.
// void  SelectLabel(const char* plabel=NULL)
//	Choix du nom de colonne correspondant  l'tiquette.
//--

/* --Methode-- */
void  PINTuple::SelectXY(const char* px, const char* py)
{
string name;
if (mNT == NULL)  xK = yK = -1;
if (px == NULL) xK = -1;
else { name = px; xK = mNT->ColumnIndex(name); }
if (py == NULL) yK = -1;
else { name = py; yK = mNT->ColumnIndex(name); }
}

/* --Methode-- */
void  PINTuple::SelectWt(const char* pw)
{
if (pw == NULL) wK = -1;  
else { string name = pw;   wK = mNT->ColumnIndex(name);  }
 
if (wK >= 0) mNT->GetMinMax(wK, wMin, wMax);
else  { wMin = 0.; wMax = 1.; }
}

/* --Methode-- */
void  PINTuple::SelectLabel(const char* plabel)
{
if (plabel == NULL) lK = -1;
else {  string name = plabel;  lK = mNT->ColumnIndex(name);  }
}

/* --Methode-- */
void  PINTuple::SelectErrBar(const char* erbx, const char* erby)
{
string name;
if (mNT == NULL)  xebK = yebK = -1;
if (erbx == NULL) xebK = -1;
else { name = erbx;  xebK = mNT->ColumnIndex(name); }
if (erby == NULL) yebK = -1;
else { name = erby;  yebK = mNT->ColumnIndex(name); }
}


/* --Methode-- */
void PINTuple::UpdateLimits()
{
  if (!mNT) return;
  if (mNT->NbLines() <= 0)  return;
  if ( (xK < 0) || (yK < 0) )   return;

  // Commencer par trouver nos limites
  double xmin, xmax, ymin, ymax;
  xmin = ymin = 9.e19;
  xmax = ymax = -9.e19;
  mNT->GetMinMax(xK, xmin, xmax);
  mNT->GetMinMax(yK, ymin, ymax);
  PIAxes::ReSizeMinMax(isLogScaleX(),xmin,xmax);
  PIAxes::ReSizeMinMax(isLogScaleY(),ymin,ymax);
  SetLimits(xmin,xmax,ymin,ymax);
//  SetAxesFlags(kBoxAxes | kExtTicks | kLabels);  Ne pas faire - Reza 11/99
}


/* --Methode-- */
#define NMXMULTP_LOCAL  30   // Pour multipoint sans new
void PINTuple::Draw(PIGraphicUC* g, double xmin, double ymin, double xmax, double ymax)
{
double xp,yp,xer,yer,wp;
double xl,yl;
long nok;
int npolyg;

if (!mNT) return;
if (axesFlags != kAxesNone) DrawAxes(g);
if ( (xK < 0) || (yK < 0) )  return;
if (GetGraphicAtt().GetLineAtt() == PI_NotDefLineAtt)  g->SelLine(PI_ThinLine);

//  Pour tracer des markers avec taille fonction de Wt (poids)
double dw = (wMax-wMin)/nWbins;
if (dw < 1.e-19) dw = 1.e-19;

//  Pour tracer des markers avec couleur en fonction de Wt (poids)
PIColorMap * cmap = NULL;
double dwc = 1.;
int nwc = 1;
bool revcmap;
CMapId mcmapid = GetGraphicAtt().GetColMapId(revcmap);
if( colorScale && (wK >= 0) && (mcmapid != CMAP_OTHER) ) {
  cmap = new PIColorMap(mcmapid);
  cmap->ReverseColorIndex(revcmap);
  nwc = cmap->NCol();
  dwc = (wMax-wMin)/(double)nwc;
}

int msz,sz;

PIMarker mmrk = GetGraphicAtt().GetMarker();
PIMarker mrk;
if (wK >= 0)  mrk = (mmrk != PI_NotDefMarker) ? mmrk : PI_CircleMarker;
else   mrk = (mmrk != PI_NotDefMarker) ? mmrk : PI_DotMarker;
msz = GetGraphicAtt().GetMarkerSz(); 
if (msz < 1) msz = 1;
g->SelMarker(msz, mrk);

PIGrCoord uxmin, uxmax, uymin, uymax;
g->GetGrSpace(uxmin, uxmax, uymin, uymax);
double xmin2 = uxmin;
double ymin2 = uymin;
double xmax2 = uxmax;
double ymax2 = uymax;

nok = 0;  
xp = yp = xl = yl = 0;
PIGrCoord xpolyg[NMXMULTP_LOCAL], ypolyg[NMXMULTP_LOCAL];
npolyg = 0;
NptDraw = 0;
for (long i=0; i<(long)mNT->NbLines(); i++) {
  xl = xp;  yl = yp;
  xp = mNT->GetCell(i, xK);
  yp = mNT->GetCell(i, yK);
  if ( (xp < xmin2) || (xp > xmax2) || (yp < ymin2) || (yp > ymax2) )  continue;
  nok++;
  if ( (xp < xmin) || (xp > xmax) || (yp < ymin) || (yp > ymax) )  continue;

  NptDraw++;
// Taille - couleur de marker en fonction du poids
  if (wK >= 0) wp = mNT->GetCell(i, wK);
  if (mrkSzScale && (wK >= 0)) {  // Changement de taille
    sz = (int)((wp-wMin)/dw);
    if (sz < 0) sz = 0;
    if (sz > nWbins)  sz = nWbins;
    sz += msz;
    if (sz < 2)  g->SelMarker(sz, PI_DotMarker);
    else g->SelMarker(sz, mrk);
  }
// Couleur du marker en fonction du poids
  if( colorScale && (wK >= 0) && cmap ) {
    int cid = (int)((wp-wMin)/dwc);
    if (cid < 0) cid = 0;
    if (cid >= nwc)  cid = nwc-1;
    g->SelForeground(*cmap, cid);
  }

  // Trace d'une ligne reliant les points
  if( connectPts ) {
    if((npolyg==0) && (i>0)) {xpolyg[0]=xl; ypolyg[0]=yl; npolyg=1;}
    if(npolyg<NMXMULTP_LOCAL)
      {xpolyg[npolyg]=xp; ypolyg[npolyg]=yp; npolyg++;}
    if(npolyg==NMXMULTP_LOCAL)
      {g->DrawPolygon(xpolyg,ypolyg,npolyg,false); npolyg=0;}
  }

  // Trace des erreurs selon X et Y
  if ( xebK >= 0 ) {
    xer = mNT->GetCell(i, xebK);
    if(xer>0.) g->DrawLine(xp-xer, yp, xp+xer, yp);
  }
  if ( yebK >= 0 ) {
    yer = mNT->GetCell(i, yebK);
    if(yer>0.) g->DrawLine(xp, yp-yer, xp, yp+yer);
  }

  // Trace du marker
  if ((wK >= 0)||(lK < 0)||(mmrk != PI_NotDefMarker))  g->DrawMarker(xp, yp); 

  // Trace eventuel du label
  if (lK >= 0) g->DrawString(xp, yp, mNT->GetCelltoString(i, lK).c_str());

}

// Fin du trace d'une ligne reliant les points  si necessaire
if( connectPts && npolyg>1 )
  {g->DrawPolygon(xpolyg,ypolyg,npolyg,false); npolyg=0;}

if (stats) { // Trace de stats 
  g->SelFontSz((YMax() - YMin())/30);  
  // La hauteur de la cellule
  PIGrCoord a,d;
  double cH = (double)g->GetFontHeight(a,d);
  double cellHeight = 1.2 * cH;
  // Les labels et leurs longueurs -> largeur de la cellule
  char label[64];
  sprintf(label, "N=%ld (/%ld)", (long)nok, (long)mNT->NbLines());
  double cellWidth =   1.1 * (double)g->CalcStringWidth(label);
  double xu, yu, cw;
  double ofpx = spoX*(XMax()-XMin());
  double ofpy = spoY*(YMax()-YMin());
  // Les limites du cadre
  xu = g->DeltaUCX(XMax(), - cellWidth);
  yu = g->DeltaUCY(YMax(), - cellHeight);
  double recw = XMax()-xu;
  double rech = YMax()-yu;
  xu += ofpx;  yu += ofpy;
  g->DrawBox(xu, yu, recw, rech);
  // L'ecriture des labels (attention aux inversions possibles des axes!)
  cw = (g->isAxeXDirRtoL()) ? -0.05*cellWidth : -0.95*cellWidth;
  xu = g->DeltaUCX(XMax(),cw);
  cw = (g->isAxeYDirUpDown()) ? -0.1*cH : -1.1*cH;
  yu = g->DeltaUCY(YMax(),cw);
  xu += ofpx;  yu += ofpy;
  g->DrawString(xu,yu,label);
}

if (cmap) delete cmap;
return;
}
#undef NMXMULTP_LOCAL

/* --Methode-- */
void PINTuple::AppendTextInfo(string& info, double xmin, double ymin, double xmax, double ymax)
{
if (!mNT) return;
if ( (xK < 0) || (yK < 0) )  return;

long ncnt = 0;
double xp,yp;
char buff[128];
sprintf(buff,"PINTuple: NLines= %ld  NCol= %ld \n", (long)mNT->NbLines(),  
	(long)mNT->NbColumns());
info += buff;
info += mNT->LineHeaderToString();
for(long i=0; i<(long)mNT->NbLines(); i++) {
  xp = mNT->GetCell(i, xK);
  yp = mNT->GetCell(i, yK);
  if ( (xp < xmin) || (xp > xmax) || (yp < ymin) || (yp > ymax) )  continue;
  ncnt++;
  if (ncnt > 101) continue;
  info += mNT->LineToString(i);
  }
if (ncnt >= 101) info += " .... \n";
sprintf(buff," %ld points inside selected region \n", (long)ncnt);
info += buff;
// printf("PINTuple::AppendTextInfo()-DBG %g %g %g %g - %d\n", xmin, ymin, xmax, ymax, ncnt); 
return;
}

/*  La methode DecodeOptionString permet de decoder un ensemble d'options 
    et de parametre d'affichage specifie sous forme d'un vecteur de string.
    Si rmdecopt == true, les options decodees sont supprimees du vecteur 
    de string fourni en entree - ce qui permet l'enchainement eventuel 
    de plusieurs decodages de string.
    Les options peuvent etre sous forme de flag : "stat" "nostat" 
    ou plus complexes, par exemple "dynamic=-3,3"
    Rc: La methode renvoie le nombre d'options decodees 
*/

/* --Methode-- */
int PINTuple::DecodeOptionString(vector<string> & opt, bool rmdecopt)
{
  int optsz1 = opt.size();
  if(optsz1<1)  return(0);  
  // On appelle d'abord le decodage de la classe PIDrawer de laquelle
  // on herite. (Pas obligatoire) on decode donc ici les attributs de
  // couleur, fontes ...
  int ndec1 = PIDrawer::DecodeOptionString(opt, rmdecopt);
  if(optsz1-ndec1<1) return(ndec1);  // si tout a ete decode

  vector<string> udopt;  // On gardera ici les options non decodees
  unsigned int k = 0;
  int ndec = opt.size();
  for( k=0; k<opt.size(); k++ ) {
    string opts = opt[k];
    if(opts=="sta" || opts=="stat" || opts=="stats") SetStats(true);
    else if(  opts=="nsta"   || opts=="nstat"
           || opts=="nostat" || opts=="nostats") SetStats(false);
    else if(opts.substr(0,11) == "statposoff=") {
      float xo=0., yo=0.;
      sscanf(opts.substr(11).c_str(),"%g,%g",&xo, &yo);
      SetStatPosOffset(xo, yo);
    }
    else if (opts == "connectpoints") ConnectPoints(true);
    else if (opts == "noconnectpoints") ConnectPoints(false);
    else if (opts == "colorscale") UseColorScale(true);
    else if (opts == "nocolorscale") UseColorScale(false);
    else if (opts == "sizescale") UseSizeScale(true);
    else if (opts == "nosizescale") UseSizeScale(false);
    else if (opts.substr(0,10) == "sizescale=") { 
      int nbn = atoi(opts.substr(10).c_str());
      UseSizeScale(true, nbn);
    }

    else {
      // Si option non decode
      ndec--;
      // S'il faut supprimer les options decodees
      if (rmdecopt)  udopt.push_back(opts);
    }
  }  
  // S'il faut supprimer les options decodees, on remplace l'argument opt
  // par le vecteur des options non decodees.
  if (rmdecopt)  opt = udopt;
  return(ndec+ndec1);  
}

int PINTuple::OptionToString(vector<string> & opt) const
{
 PIDrawer::OptionToString(opt);

 char str[256]; 

 if(stats) opt.push_back("stat"); else opt.push_back("nstat");

 sprintf(str,"statposoff=%g,%g",spoX,spoY); opt.push_back(str);

 if(connectPts) opt.push_back("connectpoints");
   else opt.push_back("noconnectpoints");

 if(colorScale) opt.push_back("colorscale");
   else opt.push_back("nocolorscale");

 if(mrkSzScale) {
   if(nWbins>0) {sprintf(str,"sizescale=%d",nWbins); opt.push_back(str);}
     else opt.push_back("sizescale");
 } else opt.push_back("nosizescale");

 return 1;
}

/* La methode GetOptionsHelpInfo(string& info) renvoie une chaine 
   avec la description des options comprises par ce drawer  
   Note: Il est preferable de ne pas initialiser la chaine 
   string info au depart, afin de permettre de mettre bout a 
   bout les aides de differents Drawer */

/* --Methode-- */
void PINTuple::GetOptionsHelpInfo(string& info)
{
info += " ---- PINTuple options help info : \n" ;
info += "  sta,stat,stats:            activate   statistic display\n";
info += "  nsta,nstat,nostat,nostats: deactivate statistic display\n";
info += "  statposoff=OffsetX,OffsetY : Position offset for Stats drawing \n";
info += "      as a fraction of total size \n";
info += "  connectpoints: The points are connected by a line \n";
info += "  noconnectpoints (this is the default) \n";
info += "  colorscale/nocolorscale (Use color scale for weight) \n";
info += "  sizescale/sizescale=nbins/nosizescale (Use marker size for weight) \n";
info += "   (and usual color/line/marker/... attribute decoding)  \n";
// On recupere ensuite la chaine info de la classe de base
PIDrawer::GetOptionsHelpInfo(info);
return;
}


/* --Methode-- */
double PINTuple::GetDistanceToPoint(double x, double y)
{
 if(!mNT) return 1.e+9;
 if( xK<0 || yK<0 ) return 1.e+9;

 const int nessai = 100;
 long inc = (NptDraw>nessai) ? (long)(NptDraw/nessai)+1 : 1;

 double dist = -1.e+18;
 long n = 0;
 for(long i=0; i<(long)mNT->NbLines(); i++) {
   double xp=mNT->GetCell(i,xK);
   if(xp<XMin() || xp>XMax()) continue;
   double yp=mNT->GetCell(i,yK);
   if(yp<YMin() || yp>YMax()) continue;
   if(n%inc==0) {
     xp = (xp-x)/(XMax()-XMin())/0.5;
     yp = (yp-y)/(YMax()-YMin())/0.5;
     xp = xp*xp+yp*yp;
     if(dist<0. || xp<dist) dist = xp;
   }
   n++;
 }
 dist=sqrt(fabs(dist));
 //cout<<"PINTuple: xlim="<<XMin()<<","<<XMax()<<" ylim="<<YMin()<<","<<YMax()
 //    <<" NbLines="<<mNT->NbLines()<<" inc="<<inc<<endl;
 //cout<<"....d="<<dist<<" x="<<x<<" y="<<y<<" NptDraw="<<NptDraw<<endl;

 return dist;
}
