#include "datatable.h"
#include "sopnamsp.h"
#include "strutil.h"
#include "pexceptions.h"
#include "fiosegdb.h"


/*!
   \class SOPHYA::DataTable
   \ingroup HiStats
   This class can be used to organize data in table (row-column) form.
   Each column holds homogeneous data (same data type), while different
   columns can be used for different data types (integer, float, string ...)
   \sa SOPHYA::MuTyV
   \sa SOPHYA::BaseDataTable
   \sa SOPHYA::ObjFileIO<DataTable>

   \code
   #include "datatable.h"
   // ...
   DataTable dt(64);
   dt.AddFloatColumn("X0_f");
   dt.AddFloatColumn("X1_f");
   dt.AddDoubleColumn("X0X0pX1X1_d");
   double x[5];
   for(int i=0; i<63; i++) {
     x[0] = (i%9)-4.;  x[1] = (i/9)-3.;  x[2] = x[0]*x[0]+x[1]*x[1];
     dt.AddLine(x);
   }
   // Printing table info
   cout << dt ;
   // Saving object into a PPF file
   POutPersist po("dtable.ppf");
   po << dt ;
   \endcode
*/


//! Default constructor - optional specification of block (or segment) size
DataTable::DataTable(sa_size_t segsz)
  : BaseDataTable(segsz)
{
}

//! copy constructor - shares the data
DataTable::DataTable(DataTable const & a, bool share)
  : BaseDataTable(a.SegmentSize())
{
  if (share) Share(a);
  else Clone(a);
  mNEnt = a.mNEnt;
  mNSeg = a.mNSeg;
  if (a.mInfo)  mInfo = new DVList(*(a.mInfo));
}

void DataTable::Share(DataTable const & a)
{
  // On copie la structure de table 
  CopyStructure(a);
  // Et on partage les donnees des colonnes 
  for (size_t kk=0; kk<mNames.size(); kk++) {
    sa_size_t sk = mNames[kk].ser;
    sa_size_t ska = a.mNames[kk].ser;
    switch (mNames[kk].type) {
    case IntegerField :
      mICols[sk].Share(a.mICols[ska]); 
    break;
    case LongField :
      mLCols[sk].Share(a.mLCols[ska]); 
      break;
    case FloatField :
      mFCols[sk].Share(a.mFCols[ska]); 
      break;
    case DoubleField :
      mDCols[sk].Share(a.mDCols[ska]); 
      break;
    case StringField :
      mSCols[sk].Share(a.mSCols[ska]); 
      break;
    default:
      throw ForbiddenError("DataTable::Share() : unknown column type ");
    break;
    }
  }  
}

void DataTable::Clone(DataTable const & a)
{
  // On copie la structure de table 
  CopyStructure(a);
  // Et on partage les donnees des colonnes 
  for (size_t kk=0; kk<mNames.size(); kk++) {
    sa_size_t sk = mNames[kk].ser;
    sa_size_t ska = a.mNames[kk].ser;
    switch (mNames[kk].type) {
    case IntegerField :
      mICols[sk].Clone(a.mICols[ska], true); 
    break;
    case LongField :
      mLCols[sk].Clone(a.mLCols[ska], true); 
      break;
    case FloatField :
      mFCols[sk].Clone(a.mFCols[ska], true); 
      break;
    case DoubleField :
      mDCols[sk].Clone(a.mDCols[ska], true); 
      break;
    case StringField :
      mSCols[sk].Clone(a.mSCols[ska], true); 
      break;
    default:
      throw ForbiddenError("DataTable::Clone() : unknown column type ");
    break;
    }
  }  
}

void DataTable::Clear()
{
  if ( (NVar() == 0) && (NEntry() == 0)) return;
  mNEnt = 0;
  mNSeg = 0;
  if (mVarD) delete[] mVarD;
  mVarD = NULL;
  if (mVarMTV) delete[] mVarMTV;
  mVarMTV = NULL;
  mNames.clear();
  if (mInfo) delete mInfo;
  mInfo = NULL;
  mMin.clear();
  mMax.clear();
  mMinMaxNEnt.clear();
  mIColsP.clear();
  mLColsP.clear();
  mFColsP.clear();
  mDColsP.clear();
  mSColsP.clear();

  mIColIdx.clear();
  mLColIdx.clear();
  mFColIdx.clear();
  mDColIdx.clear();
  mSColIdx.clear();

  mICols.clear();
  mLCols.clear();
  mFCols.clear();
  mDCols.clear();
  mSCols.clear();
}


sa_size_t DataTable::AddColumn(FieldType ft, string const & cnom)
{
  if (NEntry() > 0) 
    throw ParmError("DataTable::AddColumn() Table contains already data ");
  CheckColName(cnom);
  sa_size_t ser; 
  sa_size_t idx = NVar(); 
  switch (ft) {
  case IntegerField :
    ser = mICols.size();
    mICols.push_back(SegDataBlock<int_4>(mSegSz));
    mIColIdx.push_back(idx);
    mIColsP.push_back(NULL);
    for(sa_size_t kk=0; kk<mICols.size(); kk++)
      mIColsP[kk] = &(mICols[kk]);
    break;
  case LongField :
    ser = mLCols.size();
    mLCols.push_back(SegDataBlock<int_8>(mSegSz));
    mLColIdx.push_back(idx);
    mLColsP.push_back(NULL);
    for(sa_size_t kk=0; kk<mLCols.size(); kk++)
      mLColsP[kk] = &(mLCols[kk]);
    break;
  case FloatField :
    ser = mFCols.size();
    mFCols.push_back(SegDataBlock<r_4>(mSegSz));
    mFColIdx.push_back(idx);
    mFColsP.push_back(NULL);
    for(sa_size_t kk=0; kk<mFCols.size(); kk++)
      mFColsP[kk] = &(mFCols[kk]);
    break;
  case DoubleField :
    ser = mDCols.size();
    mDCols.push_back(SegDataBlock<r_8>(mSegSz));
    mDColIdx.push_back(idx);
    mDColsP.push_back(NULL);
    for(sa_size_t kk=0; kk<mDCols.size(); kk++)
      mDColsP[kk] = &(mDCols[kk]);
    break;
  case StringField :
    ser = mDCols.size();
    mSCols.push_back(SegDataBlock<string>(mSegSz));
    mSColIdx.push_back(idx);
    mSColsP.push_back(NULL);
    for(sa_size_t kk=0; kk<mSCols.size(); kk++)
      mSColsP[kk] = &(mSCols[kk]);
    break;
  default:
    throw ParmError("DataTable::AddColumn() unknown field type ");
    break;
  }
  colst col;
  col.nom = cnom;
  col.type = ft;
  col.ser = ser;
  mNames.push_back(col);

  // Tableaux de calcul min-max
  mMin.push_back(0.);
  mMax.push_back(0.);
  mMinMaxNEnt.push_back(0);

  return NVar();
}

/*!
   \class SOPHYA::ObjFileIO<DataTable>
   \ingroup HiStats
   Persistence (serialisation) handler for class DataTable
*/
/* --Methode-- */ 
DECL_TEMP_SPEC  /* equivalent a template <> , pour SGI-CC en particulier */
void   ObjFileIO<DataTable>::WriteSelf(POutPersist& s)  const
//	Serialisation en ecriture du DataTable sur stream PPF
{
  if (dobj == NULL)   
    throw NullPtrError("ObjFileIO<DataTable>::WriteSelf() NULL dobj pointer ");
  //-------  On ecrit 3 uint_4 .... 
  //  [0]: Numero de version ;
  //  [1] : bit1 non nul -> has info  
  //  [2] : reserve
  uint_4 itab[3];
  itab[0] = 1;  // Numero de version a 1
  itab[1] = itab[2] = 0;
  if (dobj->mInfo)  itab[1] = 1;
  s.Put(itab, 3);

  //-------- Ecriture de segment size, nb de colonnes, nb de lignes ...  
  // [0] : SegmentSize()        [1] : NVar()
  // [2] : NEntry()             [3] : NbSegments()
  uint_8 ltab[5];
  ltab[0] = dobj->SegmentSize();
  ltab[1] = dobj->NVar();
  ltab[2] = dobj->NEntry();
  ltab[3] = dobj->NbSegments();
  ltab[4] = 0;
  s.Put(ltab, 5);

  //------ Ecriture du nom et type des colonnes
  for(sa_size_t k=0; k<dobj->NVar(); k++) { 
    uint_2 typ = dobj->mNames[k].type;
    s.Put(typ);
    s.Put(dobj->mNames[k].nom);
  }
  // ------- Ecriture des tableaux min,max et n_minmax
  for(uint_8 k=0; k<ltab[1]; k++) { 
    s.Put(dobj->mMin[k]);
    s.Put(dobj->mMax[k]);
    s.Put(dobj->mMinMaxNEnt[k]);
  }  
  //------- Ecriture du DVList Info() associe, si existant
  if (dobj->mInfo)  s << (*(dobj->mInfo));

  //------- Ecriture des SegDataBlock<T> 
  for (size_t kk=0; kk<dobj->mNames.size(); kk++) {
    sa_size_t sk = dobj->mNames[kk].ser;
    switch (dobj->mNames[kk].type) {
    case BaseDataTable::IntegerField :
      s << dobj->mICols[sk]; 
    break;
    case BaseDataTable::LongField :
      s << dobj->mLCols[sk]; 
      break;
    case BaseDataTable::FloatField :
      s << dobj->mFCols[sk]; 
      break;
    case BaseDataTable::DoubleField :
      s << dobj->mDCols[sk]; 
      break;
    case BaseDataTable::StringField :
      s << dobj->mSCols[sk]; 
      break;
    default:
      throw ForbiddenError("ObjFileIO<DataTable>::WriteSelf() : unknown column type ");
    break;
    }
  }    
  return;
}

/* --Methode-- */
DECL_TEMP_SPEC  /* equivalent a template <> , pour SGI-CC en particulier */
void  ObjFileIO<DataTable>::ReadSelf(PInPersist& s)
//	Serialisation en lecture du DataTable sur stream PPF
{
  // ------- On lit les 3 premiers uint_4
  //  [0]: Numero de version ;
  //  [1] : bit1 non nul -> has info  
  //  [2] : reserve
  uint_4 itab[3] = {0,0,0};
  s.Get(itab, 3);
  bool hadinfo = false;
  if ((itab[1]&1) == 1)  hadinfo = true;

  // -------- Lecture de segment size, nb de colonnes, nb de lignes ...  
  // [0] : SegmentSize()        [1] : NVar()
  // [2] : NEntry()             [3] : NbSegments()
  uint_8 ltab[5] = {0,0,0,0,0};
  s.Get(ltab, 5);
  if (dobj == NULL) dobj = new DataTable(ltab[0]);
  else { dobj->Clear();  dobj->mSegSz = ltab[0]; }
  // -------- Lecture nom/type colonnes et allocation des colonnes
  uint_2 typ; 
  string cnom;
  for(uint_8 k=0; k<ltab[1]; k++) { 
    s.Get(typ);
    s.Get(cnom);
    BaseDataTable::FieldType ft = (BaseDataTable::FieldType)typ;
    dobj->AddColumn(ft, cnom);
  }
  // ------- Lecture des tableaux min,max et n_minmax
  for(uint_8 k=0; k<ltab[1]; k++) { 
    s.Get(dobj->mMin[k]);
    s.Get(dobj->mMax[k]);
    s.Get(dobj->mMinMaxNEnt[k]);
  }  
  // ------- Lecture du DVList Info() associe, si necessaire
  if (hadinfo) {    // Lecture eventuelle du DVList Info
    if (dobj->mInfo == NULL)  dobj->mInfo = new DVList;
    s >> (*(dobj->mInfo));
  }
  // ------- Mise a jour des champs Nb d'entrees, nb segments ...
  dobj->mNEnt = ltab[2];
  dobj->mNSeg = ltab[3];
  // ------- Lecture des SegDataBlock<T> 
  for (size_t kk=0; kk<dobj->mNames.size(); kk++) {
    sa_size_t sk = dobj->mNames[kk].ser;
    switch (dobj->mNames[kk].type) {
    case BaseDataTable::IntegerField :
      s >> dobj->mICols[sk]; 
    break;
    case BaseDataTable::LongField :
      s >> dobj->mLCols[sk]; 
      break;
    case BaseDataTable::FloatField :
      s >> dobj->mFCols[sk]; 
      break;
    case BaseDataTable::DoubleField :
      s >> dobj->mDCols[sk]; 
      break;
    case BaseDataTable::StringField :
      s >> dobj->mSCols[sk]; 
      break;
    default:
      throw ForbiddenError("ObjFileIO<DataTable>::ReadSelf() : unknown column type ");
    break;
    }
  }    
return;
}

#ifdef __CXX_PRAGMA_TEMPLATES__
#pragma define_template ObjFileIO<DataTable>
#endif

#if defined(ANSI_TEMPLATES) || defined(GNU_TEMPLATES)
template class ObjFileIO<DataTable>;
#endif
