// This may look like C code, but it is really -*- C++ -*-
// Class BaseDataTable 
//   R. Ansari -  Avril 2005
//   (C)  LAL-IN2P3/CNRS    CEA-DAPNIA  

#ifndef BASEDTABLE_H_SEEN
#define BASEDTABLE_H_SEEN

#include "machdefs.h"

#include <iostream>
#include <string>
#include <vector>

#include "ntupintf.h"
#include "dvlist.h"
#include "segdatablock.h"

namespace SOPHYA {
//! Interface definition for classes handling data in a table.
class BaseDataTable : public AnyDataObj , public NTupleInterface {
public:
  // ===> DO NOT CHANGE EXISTING enum type VALUES !!! 
  enum FieldType {IntegerField=1, LongField=2, 
		  FloatField=3, DoubleField=4,
		  ComplexField=5, DoubleComplexField=6,
		  StringField=7, DateField=8};
  // <=====================================================> 

  static string     ColTypeToString(FieldType ft, bool fgl=false);

  // constructeur , destructeur 
  BaseDataTable(sa_size_t segsz=512);
  virtual           ~BaseDataTable();
  
  //! Adds a column holding integer, named \b cnom
  inline sa_size_t  AddIntegerColumn(const char * cnom) 
  { return AddColumn(IntegerField, cnom); }
  //! Adds a column holding integer, named \b cnom
  inline sa_size_t  AddIntegerColumn(string const & cnom) 
  { return AddColumn(IntegerField, cnom); }
  //! Adds a column holding long integer, named \b cnom
  inline sa_size_t  AddLongColumn(const char * cnom) 
  { return AddColumn(LongField, cnom); }
  //! Adds a column holding long integer, named \b cnom
  inline sa_size_t  AddLongColumn(string const & cnom) 
  { return AddColumn(LongField, cnom); }
  //! Adds a column holding floating values (r_4), named \b cnom
  inline sa_size_t  AddFloatColumn(const char * cnom) 
  { return AddColumn(FloatField, cnom); }
  //! Adds a column holding floating values (r_4), named \b cnom
  inline sa_size_t  AddFloatColumn(string const & cnom) 
  { return AddColumn(FloatField, cnom); }
  //! Adds a column holding double values (r_8), named \b cnom
  inline sa_size_t  AddDoubleColumn(const char * cnom) 
  { return AddColumn(DoubleField, cnom); }
  //! Adds a column holding double values (r_8), named \b cnom
  inline sa_size_t  AddDoubleColumn(string const & cnom) 
  { return AddColumn(DoubleField, cnom); }
  //! Adds a column holding character strings, named \b cnom
  inline sa_size_t  AddStringColumn(const char * cnom) 
  { return AddColumn(StringField, cnom); }
  //! Adds a column holding character strings, named \b cnom
  inline sa_size_t  AddStringColumn(string const & cnom) 
  { return AddColumn(StringField, cnom); }
  
  inline sa_size_t  AddColumn(FieldType ft, const char * cnom)
  { string nom = cnom;  return AddColumn(ft, nom); }
  
  //! Pure virtual method for adding a new column to the Data Table
  virtual sa_size_t AddColumn(FieldType ft, string const & cnom) = 0;
  //! Duplicate the data table structure from the source Table
  virtual sa_size_t CopyStructure(BaseDataTable const & a);
  //! Checks if the two table have the same column structure
  virtual bool      CompareStructure(BaseDataTable const & a);

  // Verifie la validite du nom de colonne:envoie une exception
  // si nom en double ou non valide
  virtual bool      CheckColName(string const & cnom);

  // Acces to various counts and parameters
  //! Return the number of lines (rows) in the table)
  inline sa_size_t  NEntry() const { return mNEnt ; }
  //! Return the number of columns in the tables (number of cells in a row)  
  inline sa_size_t  NVar() const   { return mNames.size() ; } 
  //! Return the number of columns in the tables (number of cells in a row)  
  inline sa_size_t  NVars() const   { return mNames.size() ; } 
  //! Return the segment size (SegDBInterface objects corresponding to columns)  
  inline sa_size_t  SegmentSize() const   { return mSegSz ; } 
  //! Return the number of segments (SegDBInterface objects corresponding to columns)  
  inline sa_size_t  NbSegments() const   { return mNSeg ; } 

  // Filling data structures (adding lines)
  virtual sa_size_t AddLine(const r_8* data);
  virtual sa_size_t AddLine(const MuTyV * data);

  //! Alias for AddLine()
  inline sa_size_t  Fill(const r_8* data)
  { return AddLine(data); }
  //! Alias for AddLine()
  inline sa_size_t  Fill(const MuTyV * data);
     
  virtual sa_size_t Extend();

  //! Return the information stored in line \b n of the table 
  virtual MuTyV *   GetLine(sa_size_t n) const ;

  //! Return the index for column name \b nom  
  sa_size_t         IndexNom(char const* nom) const ; 
  //! Return the index for column name \b nom  
  inline sa_size_t  IndexNom(string const & nom) const 
  { return IndexNom(nom.c_str()); }
  //! Return the column name for column index \b k
  string            NomIndex(sa_size_t k) const ; 

  //! Return the column type for column \b k (no check on index range)
  inline FieldType  GetColumType(sa_size_t k) const
  { return mNames[k].type; }
  //! Return the column name  for column \b k (no check on index range)
  inline const string & GetColumName(sa_size_t k)  const
  { return mNames[k].nom; }
  
  
  //! Copy or merges  the data from \b a into the data table (cp=true -> copy)
  virtual void      CopyMerge(BaseDataTable const& a, bool cp=false) ; 

  //! Clear/reset the table content and structure.
  virtual void Clear() = 0;

  //! Prints the table content - NOT YET IMPLEMENTED !
  void              Print(int num, int nmax=1) const ;
  
//! Prints table definition and number of entries 
  void              Show(ostream& os) const ; 
  //! Prints table definition and number of entries on the standard output stream \b cout
  inline void       Show() const { Show(cout) ; } 
  
  DVList&           Info() const ; 
  
  // Remplissage depuis fichier ASCII 
  int		    FillFromASCIIFile(string const& fn);
  
  //-------------------------------------------------------------
  //-----------  Declaration de l interface NTuple --------------

  virtual sa_size_t NbLines() const ;
  virtual sa_size_t NbColumns() const ;
  virtual r_8 *     GetLineD(sa_size_t n) const ;
  virtual r_8	    GetCell(sa_size_t n, sa_size_t k) const ;
  virtual r_8	    GetCell(sa_size_t n, string const & nom) const ;
  virtual string    GetCelltoString(sa_size_t n, sa_size_t k) const ;
  virtual void      GetMinMax(sa_size_t k, double& min, double& max)   const ; 
  virtual void      GetMinMax(string const & nom, double& min, double& max)   const ; 
  virtual sa_size_t ColumnIndex(string const & nom)  const ;
  virtual string    ColumnName(sa_size_t k) const;
  virtual string    VarList_C(const char* nomx=NULL) const ;
  virtual string    LineHeaderToString() const;
  virtual string    LineToString(sa_size_t n) const;  

  //  Pour la gestion de persistance PPF
  friend class ObjFileIO<BaseDataTable> ;
  
protected:
  uint_8   mNEnt ;   // nb total d'entrees 
  uint_8   mSegSz ;  // taille bloc/segment 
  uint_8   mNSeg;    // Nb total de segments
  mutable r_8 *  mVarD;    // Pour retourner une ligne de la table
  mutable MuTyV *  mVarMTV;    // Pour retourner une ligne de la table en MuTyV

  //! \cond   Pour pas inclure ds la documentation doxygen
  typedef struct { 
    string nom; 
    FieldType type;
    sa_size_t ser;
  } colst;
  //! \endcond  
  std::vector<colst> mNames;

  mutable DVList* mInfo;        // Infos (variables) attachees au NTuple

  // Pour calculer et garder le min/max   
  mutable std::vector<r_8> mMin;
  mutable std::vector<r_8> mMax;
  mutable std::vector<uint_8> mMinMaxNEnt;
  
  // Les pointeurs des SegDataBlock ...  
  std::vector< SegDBInterface<int_4> * > mIColsP; 
  std::vector< sa_size_t > mIColIdx;
  std::vector< SegDBInterface<int_8> * > mLColsP; 
  std::vector< sa_size_t > mLColIdx;
  std::vector< SegDBInterface<r_4> * > mFColsP;
  std::vector< sa_size_t > mFColIdx; 
  std::vector< SegDBInterface<r_8> * > mDColsP; 
  std::vector< sa_size_t > mDColIdx;
  std::vector< SegDBInterface<string> * > mSColsP; 
  std::vector< sa_size_t > mSColIdx;
  
}; 
/*! Prints table information (Column name and type, min/max) on stream \b s (bd.Show(s)) */
inline ostream& operator << (ostream& s, BaseDataTable const & bd)
  {  bd.Show(s);  return(s);  }
  
} // namespace SOPHYA

#endif
