#ifndef FITSHANDLER_H
#define FITSHANDLER_H

#include "machdefs.h"
#include <string>
#include "fitsinoutfile.h"

namespace SOPHYA {

/*! 
  \ingroup FitsIOServer
  \brief Interface definition for classes handling object storage retrieval in FITS Format 
*/
class FitsHandlerInterface {

public:

  virtual ~FitsHandlerInterface() {}
  //! Return the real data object
  virtual AnyDataObj* DataObj() = 0;       // Retourne l'objet reel 
  //! Return true if I/O for object \b o can be handled
  virtual bool        CheckHandling(AnyDataObj & o) = 0;
  //! Read/write operation will use the object o
  virtual void        SetDataObj(AnyDataObj & o) = 0;

  //! Clone (duplicate) the handler class
  virtual FitsHandlerInterface* Clone() = 0;

  //! Perform the actual write operation to the output fits file
  virtual void        Write(FitsInOutFile& os) = 0;
  //! Perform the actual read operation from input fits file
  virtual void        Read(FitsInOutFile& is) = 0;  
};

/*! 
  \ingroup FitsIOServer
  \brief Generic implementation of FitsHandlerInterface 
*/
template <class T>
class FitsHandler : public FitsHandlerInterface {

public :
	    FitsHandler() { dobj=NULL; ownobj=true; }
	    FitsHandler(T & obj) { dobj = &obj; ownobj=false; }
  virtual   ~FitsHandler() { if (ownobj && dobj) delete dobj; }

  virtual   AnyDataObj* DataObj() { return(dobj); }
  virtual   bool        CheckHandling(AnyDataObj & o) 
  {  
    T * po = dynamic_cast< T * >(& o); 
    if (po == NULL) return false;
    else return true;
  }
  virtual   void        SetDataObj(AnyDataObj & o) 
  {  
    T * po = dynamic_cast< T * >(& o); 
    if (po == NULL)  {
      string msg = "FitsHandler<T>::SetDataObj() Wrong object type: " ;
      msg += typeid(o).name(); 
      throw TypeMismatchExc(msg);    
    }
    if (ownobj && dobj) delete dobj;  dobj = po; ownobj = false; 
  }
    
  virtual FitsHandlerInterface* Clone() 
    { return new FitsHandler<T>() ; }

  inline operator T&() { return(*dobj); }

  virtual void        Read(FitsInOutFile& is);
  virtual void        Write(FitsInOutFile& os);

protected :
  T * dobj;
  bool ownobj;       // True si dobj obtenu par new
};



} // Fin du namespace

#endif

