// Gestion de block de donnees avec partage de references
//                         C.Magneville          04/99
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA
#include "machdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <complex>
#include "pexceptions.h"
#include "ndatablock.h"
#include "objfio.h"

using namespace PlanckDPC;

// define DEBUG_NDATABLOCK

#ifdef DEBUG_NDATABLOCK
  static size_t NallocData = 0;
  static size_t NallocSRef = 0;
#endif

// LOGIQUE DE BASE:
//   - le createur par copie et la surcharge de l operateur = "partage" les donnees
//   - gestion du partage de reference

////////////////////////////////////////////////////////////////
//************ Createur, Destructeur, gestion des donnees

template <class T>
NDataBlock<T>::NDataBlock(size_t n)
// Createur d'une structure de "n" donnees
: mSz(0), mSRef(NULL), mIsTemp(false)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::NDataBlock("<<this<<",n="<<n<<")"<<endl;
#endif

Alloc(n);
}

template <class T>
NDataBlock<T>::NDataBlock(size_t n, T* data, Bridge* br)
// Createur d'une structure de "n" donnees, avec donnees preallouees
//  (Voir explications dans Alloc())
: mSz(0), mSRef(NULL), mIsTemp(false)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::NDataBlock("<<this
    <<",data="<<data<<",br="<<br<<")"<<endl;
#endif

Alloc(n,data,br);
}

template <class T>
NDataBlock<T>::NDataBlock()
// Createur par default
: mSz(0), mSRef(NULL), mIsTemp(false)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::NDataBlock("<<this<<") default"<<endl;
#endif
}

template <class T>
NDataBlock<T>::NDataBlock(const NDataBlock<T>& a)
// Createur par copie
// ATTENTION: partage les donnees avec "a"
// Ecriture: NDataBlock a = b;
//           NDataBlock a(b)
: mSz(0), mSRef(NULL), mIsTemp(false)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::NDataBlock("<<this<<",&a="<<&a<<") a.(mSz="
    <<a.mSz<<" mSRef="<<a.mSRef<<" IsTemp="<<a.IsTemp()<<")"<<endl;
#endif

Share(a);
}

template <class T>
NDataBlock<T>::NDataBlock(const NDataBlock<T>& a,bool share)
// Createur avec choix de partager ou non
// Si "a" temporaire alors partage meme si share=false
: mSz(0), mSRef(NULL), mIsTemp(false)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::NDataBlock("<<this<<",&a="<<&a<<",sh=<<"<<share<<")"
    <<" a.(mSz="<<a.mSz<<" mSRef="<<a.mSRef<<" IsTemp="<<a.IsTemp()<<")"<<endl;
#endif

if(share) Share(a); else Clone(a);
}

template <class T>
NDataBlock<T>::~NDataBlock()
// Destructeur
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::~NDataBlock("<<this<<")"<<endl;
#endif

Delete();
}

////////////////////////////////////////////////////////////////

template <class T>
void NDataBlock<T>::SetTemp(bool temp) const 
// Set temporary
{
mIsTemp=temp;

#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::SetTemp("<<this<<","<<temp
    <<"), mSz="<<mSz<<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp<<endl;
#endif
}

template <class T>
void NDataBlock<T>::Alloc(size_t n,T* data,Bridge* br)
// Allocation d'un NOUVEL espace de stoquage de "n" donnees
// Si data==NULL : allocation de l'espace memoire (vide)
//    data!=NULL : partage des donnees avec l'adresse data
// Si br==NULL   : les donnees nous appartiennent
//    br!=NULL   : les donnees ne nous appartiennent pas (ex: Blitz)
//
// Exemple: on veut connecter a un tableau de T*
// 1- On veut que NDataBlock NE DESALLOUE PAS le tableau "data"
//    a- Premiere solution
//       float *x = new float[5]; ... remplissage de x[] ...;
//       NDataBlock A(5,x,new Bridge);
//       delete [] x;   // Il faut deleter explicitement
//       (et Bridge est delete par le destructeur de la classe)
//    b- Autre solution:
//       NDataBlock A(5); A.FillFrom(5,x);
//       delete [] x;   // Il faut deleter explicitement
// 2- On veut que NDataBlock desalloue le tableau
//       float *x = new float[5]; ... remplissage de x[] ...;
//       NDataBlock A(5,x);
//       (Ne Pas Faire "delete [] x;")
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::Alloc("<<this<<","
    <<n<<","<<data<<","<<br<<") mSz="<<mSz
    <<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp<<endl;
#endif

if(br && !data)
  throw(NullPtrError("NDataBlock::Alloc br!=NULL && data==NULL\n"));
if(n==0) throw(SzMismatchError("NDataBlock::Alloc n==0\n"));
if(mSRef) Delete();
mSz = n;
mSRef = new NDREF;
mSRef->nref = 1;
if(data) mSRef->data = data;
else {mSRef->data = new T[n]; memset(mSRef->data,0,n*sizeof(T));}
mSRef->bridge = br;

#ifdef DEBUG_NDATABLOCK
// Meme dans le cas data!=0 et br==0 (connexion d'un tableau
// avec destruction geree par ~NDataBlock (cas 2-) on compte
// comme si on avait fait une allocation du tableau (ce qui a ete
// fait au niveau du dessus!).
if(!br) NallocData++; NallocSRef++;
cout<<"...DEBUG_NDataBlock::Alloc mSz="<<mSz<<" mSRef="<<mSRef
    <<" mSRef->nref="<<mSRef->nref<<" mSRef->data="<< mSRef->data
    <<" mSRef->bridge="<<mSRef->bridge<<" IsTemp="<<mIsTemp
    <<" Total("<<NallocData<<","<<NallocSRef<<")"<<endl;
#endif
}

template <class T>
void NDataBlock<T>::Clone(const NDataBlock<T>& a)
// Clone (copie de donnee) a partir de "a"
// sauf si "a" est une classe temporaire (dans ce cas share!)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::Clone("<<this<<","<<&a<<") a.(mSz="
    <<a.mSz<<" mSRef="<<a.mSRef<<" IsTemp="<<a.IsTemp()
    <<"), mSz="<<mSz<<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp<<endl;
#endif

if(a.mSz==0) throw(SzMismatchError("NDataBlock::Clone a.mSz==0\n"));
else if(a.IsTemp()) Share(a);
else {Alloc(a.mSz); memcpy(Data(),a.Data(),mSz*sizeof(T));}
}

template <class T>
void NDataBlock<T>::Share(const NDataBlock<T>& a)
// Partage des donnees avec "a"
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::Share("<<this<<","<<&a<<")";
if(&a!=NULL) cout<<" a.(mSz="<<a.mSz<<" mSRef="<<a.mSRef
                 <<" IsTemp="<<a.IsTemp()<<")";
cout<<", mSz="<<mSz<<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp<<endl;
#endif

if(&a==NULL) throw(NullPtrError("NDataBlock::Share  &a==NULL\n"));
// on ne peut partager si "a" pas alloue
if(!a.mSRef) throw(NullPtrError("NDataBlock::Share not allocated a\n"));
if(mSRef) Delete();
mSz = a.mSz; mSRef = a.mSRef; mSRef->nref++;

#ifdef DEBUG_NDATABLOCK
cout<<"...DEBUG_NDataBlock::Share mSz="<<mSz<<" mSRef="<<mSRef
    <<" mSRef->nref="<<mSRef->nref<<" mSRef->data="<< mSRef->data
    <<" mSRef->bridge="<<mSRef->bridge<<" IsTemp="<<mIsTemp<<endl;
#endif
}

template <class T>
void NDataBlock<T>::Delete(void)
// Pour detruire les pointeurs en tenant compte des references
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::Delete("<<this<<") mSz="<<mSz
    <<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp;
if(mSRef)
  cout<<" mSRef->nref="<<mSRef->nref<<" mSRef->data="
      <<mSRef->data<<" mSRef->bridge="<<mSRef->bridge;
cout<<endl;
#endif

if(mSRef==NULL) return; // cas du createur par defaut
mSRef->nref--;
if(mSRef->nref != 0) {

#ifdef DEBUG_NDATABLOCK
cout<<"...DEBUG_NDataBlock::Delete() pas de desallocation il reste nref="
    <<mSRef->nref<<" Total("<<NallocData<<","<<NallocSRef<<")"<<endl;
#endif

  mSz = 0; mSRef=NULL;
  return;
}

#ifdef DEBUG_NDATABLOCK
if(!mSRef->bridge) NallocData--; NallocSRef--;
cout<<"...DEBUG_NDataBlock::Delete() desallocation complete il reste nref="
    <<mSRef->nref<<" Total("<<NallocData<<","<<NallocSRef<<")"<<endl;
#endif

// Si il y a un Bridge les donnees ne n'appartiennent pas, on detruit le Bridge
// sinon, les donnees ont ete allouees par nos soins, on libere l'espace
if(mSRef->bridge) delete mSRef->bridge; else delete [] mSRef->data;
mSRef->bridge=NULL; mSRef->data=NULL;
delete mSRef; mSRef=NULL; mSz = 0;
}

////////////////////////////////////////////////////////////////

template <class T>
void NDataBlock<T>::Reset(T v)
{
if(mSRef==NULL || mSRef->data==NULL || mSz==0) return;
T *p=Begin(), *pe=End(); while(p<pe) *p++ = v;
}

template <class T>
void NDataBlock<T>::ReSize(size_t n,bool force_alloc)
// Re-dimension, avec re-allocation de la place si n != mSz
// Si n==mSz, la place n'est re-allouee que si force_alloc=true
{
if(!force_alloc && n == mSz)  return;
Alloc(n);
}

template <class T>
void NDataBlock<T>::FillFrom(size_t n,T* data)
// Remplissage par un tableau de donnees
// - Si classe vide : creation de l'espace memoire
// - Si classe connectee : on ecrit selon la longueur
{
if(data==NULL) throw(NullPtrError("NDataBlock::FillFrom  data==NULL\n"));
if(n==0) throw(ParmError("NDataBlock::FillFrom  n<=0\n"));
if(mSRef==NULL) Alloc(n); // cas du createur par default
if(mSz<n) n = mSz;
memcpy(Data(),data,n*sizeof(T));
}

////////////////////////////////////////////////////////////////
//**** Impression

template <class T>
void NDataBlock<T>::Print(ostream& os,size_t i1,size_t n) const
// Impression de n elements a partir de i1
{
size_t nr = 0;
T* p = NULL; Bridge* br = NULL;
if(mSRef) {nr = mSRef->nref; p = mSRef->data; br = mSRef->bridge;}
os<<"NDataBlock::Print("<<this<<",Sz="<<mSz<<",IsTemp="<<mIsTemp<<")\n"
  <<"            mSRef="<<mSRef<<"(nref="<<nr<<",data="<<p
  <<",bridge="<<br<<")"<<endl;
if(i1>=mSz || n<=0 || !p) return;
size_t i2 = i1+n; if(i2>mSz) i2=mSz;
size_t im = 1; bool enl=false;
while(i1<i2) {
  enl = false;
  os<<" "<<(*this)(i1);  i1++;
  if(im==8) {os<<"\n"; im=1; enl=true;} else im++;
}
if(!enl) os<<endl;
}

////////////////////////////////////////////////////////////////
//**** Surcharge de = : NDataBlock=NDataBlock; NDataBlock=<T> b; 

template <class T>
NDataBlock<T>& NDataBlock<T>::operator = (const NDataBlock<T>& a)
// surcharge avec partage des donnees
// Ecriture:  NDataBlock a; a = b;
//            NDataBlock a(10); a = b; (a est re-affecte)
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::operator=("<<this<<","<<&a<<") a.(mSz="
    <<a.mSz<<" mSRef="<<a.mSRef<<" IsTemp="<<a.IsTemp()
    <<"), mSz="<<mSz<<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp<<endl;
#endif

if(this == &a) return *this;
if(a.mSz!=mSz)
  throw(SzMismatchError("NDataBlock::operator=A size mismatch/null\n"));
Share(a);
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator = (T v)
// surcharge avec copie des donnees (pas de partage)
// "this" est sur-ecrit, attention au partage de reference!
// NDataBlock a; a = v; ou bien NDataBlock a(10); a = v;
{
#ifdef DEBUG_NDATABLOCK
cout<<"DEBUG_NDataBlock::operator=("<<this<<","<<v<<")"
    <<" mSz="<<mSz<<" mSRef="<<mSRef<<" IsTemp="<<mIsTemp<<endl;
#endif

if(mSz==0) throw(SzMismatchError("NDataBlock::operator=v null size\n"));
T *p=Begin(), *pe=End();
while (p<pe) *p++ = v;
return *this;
}

////////////////////////////////////////////////////////////////

template <class T>
T NDataBlock<T>::Sum(size_t i1,size_t n) const
// Somme des elements de i1 a i1+n-1
{
if(i1>=mSz) return 0;
if(n>mSz) n = mSz; if(n==0) n = mSz-i1;
T const *p=Begin()+i1, *pe=p+n;
T val = 0;
while (p<pe) val += *p++;
return val;
}

template <class T>
T NDataBlock<T>::Product(size_t i1,size_t n) const
// Produit des elements de i1 a i1+n-1
{
if(i1>=mSz) return 0;
if(n>mSz) n = mSz; if(n==0) n = mSz-i1;
T const *p=Begin()+i1, *pe=p+n;
T val = 0;
while (p<pe) val *= *p++;
return val;
}

////////////////////////////////////////////////////////////////
//**** Surcharge de +=,-=,*=,/= (INPLACE): NDataBlock += <T> b;

template <class T>
NDataBlock<T>& NDataBlock<T>::operator += (T b)
{
if(mSz==0) throw(SzMismatchError("NDataBlock::operator+=v null size\n"));
T *p=Begin(), *pe=End();
while (p<pe) *p++ += b;
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator -= (T b)
{
if(mSz==0) throw(SzMismatchError("NDataBlock::operator-=v null size\n"));
T *p=Begin(), *pe=End();
while (p<pe) *p++ -= b;
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator *= (T b)
{
if(mSz==0) throw(SzMismatchError("NDataBlock::operator*=v null size\n"));
T *p=Begin(), *pe=End();
while (p<pe) *p++ *= b;
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator /= (T b)
{
if(b==(T) 0) throw(ParmError("NDataBlock::operator/=v divide by zero\n"));
if(mSz==0) throw(SzMismatchError("NDataBlock::operator/=v null size\n"));
T *p=Begin(), *pe=End();
while (p<pe) *p++ /= b;
return *this;
}


////////////////////////////////////////////////////////////////
//**** Surcharge de +=,-=,*=,/= (INPLACE): NDataBlock += NDataBlock;

template <class T>
NDataBlock<T>& NDataBlock<T>::operator += (const NDataBlock<T>& a)
{
if(mSz==0 || mSz!=a.mSz)
  throw(SzMismatchError("NDataBlock::operator+=A size mismatch/null"));
T *p=Begin(), *pe=End();
T const * pa=a.Begin();
while (p<pe) *p++ += *pa++; // ca marche meme si *this=a
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator -= (const NDataBlock<T>& a)
{
if(mSz==0 || mSz!=a.mSz)
  throw(SzMismatchError("NDataBlock::operator-=A size mismatch/null"));
T *p=Begin(), *pe=End();
T const *pa=a.Begin();
while (p<pe) *p++ -= *pa++; // ca marche meme si *this=a
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator *= (const NDataBlock<T>& a)
{
if(mSz==0 || mSz!=a.mSz)
  throw(SzMismatchError("NDataBlock::operator*=A size mismatch/null"));
T *p=Begin(), *pe=End();
T const *pa=a.Begin();
while (p<pe) *p++ *= *pa++; // ca marche meme si *this=a
return *this;
}

template <class T>
NDataBlock<T>& NDataBlock<T>::operator /= (const NDataBlock<T>& a)
{
if(mSz==0 || mSz!=a.mSz)
  throw(SzMismatchError("NDataBlock::operator/=A size mismatch/null"));
T *p=Begin(), *pe=End();
T const *pa=a.Begin();
while (p<pe) *p++ /= *pa++;
return *this;
}

////////////////////////////////////////////////////////////////
//**** Surcharge de +,-,*,/ : NDataBlock = NDataBlock+<T>b;
//                            NDataBlock = <T>b+NDataBlock;
// ATTENTION: re-affectation imposee

template <class T>
NDataBlock<T> NDataBlock<T>::Add(T b) const
// Pour A+b
{
NDataBlock<T> result(*this,false); result.SetTemp(true);
result += b;
return result;
}

template <class T>
NDataBlock<T> NDataBlock<T>::Sub(T b) const
// Pour A-b
{
NDataBlock<T> result(*this,false); result.SetTemp(true);
return result -= b;
}

template <class T>
NDataBlock<T> NDataBlock<T>::SubInv(T b) const
// Pour b-A
{
NDataBlock<T> result(*this,false); result.SetTemp(true);
T *p=result.Begin(), *pe=result.End();
T const *pa=this->Begin();
while(p<pe) {*p++ = b - *pa++;}
return result;
}

template <class T>
NDataBlock<T> NDataBlock<T>::Mul(T b) const
// Pour A*b
{
NDataBlock<T> result(*this,false); result.SetTemp(true);
return result *= b;
}

template <class T>
NDataBlock<T> NDataBlock<T>::Div(T b) const
// Pour A/b
{
NDataBlock<T> result(*this,false); result.SetTemp(true);
return result /= b;
}

template <class T>
NDataBlock<T> NDataBlock<T>::DivInv(T b) const
// Pour b/A
{
NDataBlock<T> result(*this,false); result.SetTemp(true);
T *p=result.Begin(), *pe=result.End();
T const *pa = this->Begin();
while(p<pe) {*p++ = b / *pa++;}
return result;
}


////////////////////////////////////////////////////////////////
//**** Surcharge de +,-,*,/ : NDataBlock = NDataBlock+NDataBlock;

template <class T>
NDataBlock<T> NDataBlock<T>::Add(const NDataBlock<T>& b) const
// Pour A+B
{
if(mSz!=b.mSz)
  throw(SzMismatchError("NDataBlock operator C=A+B size mismatch/null\n"));
NDataBlock<T> result; result.SetTemp(true);
if(b.IsTemp()) {
  result.Share(b);
  result += *this;
} else {
  result.Clone(*this);
  result += b;
}
return result;
}

template <class T>
NDataBlock<T> NDataBlock<T>::Mul(const NDataBlock<T>& b) const
// Pour A*B
{
if(mSz!=b.mSz)
  throw(SzMismatchError("NDataBlock operator C=A*B size mismatch/null\n"));
NDataBlock<T> result; result.SetTemp(true);
if(b.IsTemp()) {
  result.Share(b);
  result *= *this;
} else {
  result.Clone(*this);
  result *= b;
}
return result;
}

template <class T>
NDataBlock<T> NDataBlock<T>::Sub(const NDataBlock<T>& b) const
// Pour A-B
{
if(mSz!=b.mSz)
  throw(SzMismatchError("NDataBlock operator C=A-B size mismatch/null\n"));
NDataBlock<T> result; result.SetTemp(true);
if(b.IsTemp()) {
  result.Share(b);
  T *p=result.Begin(), *pe=result.End();
  T const *pa=Begin();
  while(p<pe) {*p = *pa++  - *p; p++;}
} else {
  result.Clone(*this);
  result -= b;
}
return result;
}

template <class T>
NDataBlock<T> NDataBlock<T>::Div(const NDataBlock<T>& b) const
// Pour A/B
{
if(mSz!=b.mSz)
  throw(SzMismatchError("NDataBlock operator C=A/B size mismatch/null\n"));
NDataBlock<T> result; result.SetTemp(true);
if(b.IsTemp()) {
  result.Share(b);
  T *p=result.Begin(), *pe=result.End();
  T const *pa=Begin();
  while(p<pe) {*p = *pa++  / *p; p++;}
} else {
  result.Clone(*this);
  result /= b;
}
return result;
}

////////////////////////////////////////////////////////////////
// -------------------------------------------------------------------------
//   Les objets delegues pour la gestion de persistance 
// -------------------------------------------------------------------------

/*
template <class T>
void ObjFileIO< NDataBlock<T> >::ReadSelf(PInPersist& is)
template <class T>
void ObjFileIO< NDataBlock<T> >::WriteSelf(POutPersist& os)
*/

// Pour pouvoir ecrire des tableaux de complex, en attendant 
// PIn/POutPersist::Get/Put(complex<>)
#include <piocmplx.h>

template <class T>
FIO_NDataBlock<T>::FIO_NDataBlock()
{
dobj=new NDataBlock<T>;
ownobj=true;
}

template <class T>
FIO_NDataBlock<T>::FIO_NDataBlock(string const & filename) 
{
dobj=new NDataBlock<T>;
ownobj=true; 
Read(filename);
}

template <class T>
FIO_NDataBlock<T>::FIO_NDataBlock(const NDataBlock<T> & obj) 
{ 
dobj = new NDataBlock<T>(obj);
ownobj=true; 
}

template <class T>
FIO_NDataBlock<T>::FIO_NDataBlock(NDataBlock<T> * obj) 
{ 
dobj = obj;
ownobj=false; 
}

template <class T>
FIO_NDataBlock<T>::~FIO_NDataBlock()
{
if (ownobj && dobj) delete dobj;
}

template <class T>
AnyDataObj* FIO_NDataBlock<T>::DataObj()
{
return(dobj);
}


template <class T>
void FIO_NDataBlock<T>::ReadSelf(PInPersist& is)
{
// On lit les 3 premiers uint_8
uint_8 itab[3];
is.Get(itab, 3);
if (dobj == NULL) dobj = new NDataBlock<T>(itab[1]);
else dobj->ReSize(itab[1], false);
// On lit le tableau de nombres
PIOSReadArray(is, dobj->Data(), dobj->Size()); 
}


template <class T>
void FIO_NDataBlock<T>::WriteSelf(POutPersist& os) const
{
if (dobj == NULL)   return;  // Attention - $CHECK$ Reza 26/04/99
//  On ecrit 3 uint_4 
//  0 : Numero de version,  1 : Taille,  2  reserve a l
uint_8 itab[3];
itab[0] = 1;
itab[1] = dobj->Size();
itab[2] = 0;
os.Put(itab, 3);
//  On ecrit le tableau de nombres 
PIOSWriteArray(os, dobj->Data(), dobj->Size()); 
}

///////////////////////////////////////////////////////////////
#ifdef __CXX_PRAGMA_TEMPLATES__
#pragma define_template NDataBlock<uint_1>
#pragma define_template NDataBlock<uint_2>
#pragma define_template NDataBlock<int_2>
#pragma define_template NDataBlock<int_4>
#pragma define_template NDataBlock<int_8>
#pragma define_template NDataBlock<uint_4>
#pragma define_template NDataBlock<uint_8>
#pragma define_template NDataBlock<r_4>
#pragma define_template NDataBlock<r_8>
#pragma define_template NDataBlock< complex<float> >
#pragma define_template NDataBlock< complex<double> >
// Instances des delegues FileIO (PPersist)
#pragma define_template FIO_NDataBlock<uint_1>
#pragma define_template FIO_NDataBlock<uint_2>
#pragma define_template FIO_NDataBlock<int_2>
#pragma define_template FIO_NDataBlock<int_4>
#pragma define_template FIO_NDataBlock<int_8>
#pragma define_template FIO_NDataBlock<uint_4>
#pragma define_template FIO_NDataBlock<uint_8>
#pragma define_template FIO_NDataBlock<r_8>
#pragma define_template FIO_NDataBlock<r_4>
#pragma define_template FIO_NDataBlock< complex<float> >
#pragma define_template FIO_NDataBlock< complex<double> >
#endif


#if defined(ANSI_TEMPLATES) || defined(GNU_TEMPLATES)
template class NDataBlock<uint_1>;
template class NDataBlock<uint_2>;
template class NDataBlock<int_2>;
template class NDataBlock<int_4>;
template class NDataBlock<int_8>;
template class NDataBlock<uint_4>;
template class NDataBlock<uint_8>;
template class NDataBlock<r_4>;
template class NDataBlock<r_8>;
template class NDataBlock< complex<float> >;
template class NDataBlock< complex<double> >;
// Instances des delegues FileIO (PPersist)
template class FIO_NDataBlock<uint_1>;
template class FIO_NDataBlock<uint_2>;
template class FIO_NDataBlock<int_2>;
template class FIO_NDataBlock<int_4>;
template class FIO_NDataBlock<int_8>;
template class FIO_NDataBlock<uint_4>;
template class FIO_NDataBlock<uint_8>;
template class FIO_NDataBlock<r_8>;
template class FIO_NDataBlock<r_4>;
template class FIO_NDataBlock< complex<float> >;
template class FIO_NDataBlock< complex<double> >;
#endif
