// This may look like C code, but it is really -*- C++ -*-

#ifndef VECTOR_SEEN
#define VECTOR_SEEN
#include "matrix.h"

class GeneralFit;
namespace PlanckDPC {template <class T> class TVector;}

// <summary> Vecteur colonne pour operations matricielles </summary>
// Vecteur colonne, en fait une matrice avec une seule colonne, pour faire
// des operations matricielles.
class Vector : public Matrix {
  friend class TVector<r_8>;
public:
  // Constructeur, n = nombre d'elements.
  EXPLICIT Vector(int n=1);
  // Constructeur, a partir des valeurs. Pas d'allocation.
  Vector(int n, double* values);
  // Constructeur par copie
  Vector(const Vector& v);
  // Constructeur par "copie" a partir d'une matrice, qui doit avoir une colonne
  // <thrown> sizeMismatchErr </thrown>
  Vector(const Matrix& a);
  // Constructeur par copie a partir d'un TVector<r_8>
  Vector(const TVector<r_8>& v);

  // Construction automatique pour PPF
  // <group>
  enum {classId = ClassId_Vector};
  int_4                 ClassId() const         { return classId; }
  static PPersist*      Create() {return new Vector(1);}
  // </group>
  
  // Change la taille du vecteur. Reallocation physique seulement si
  // pas assez de place, ou forcee si force=true
  void Realloc(int n, bool force=false);

  Vector& operator = (const Vector& v);
  Vector& operator = (double x);

  // Acces aux elements
  // <group>
  double& operator()(int n);
  double const& operator()(int n) const;
  // </group>

  // produit scalaire
  // <group>
  friend double operator* (const Vector& v1, const Vector& v2);
  friend Vector operator* (const Vector& v, double b)
  {return Vector(((Matrix const&)v) * b);}
  // </group>
  // produit matrice*vecteur
  friend Vector operator* (const Matrix& a, const Vector& b)
  {return Vector(a * ((Matrix const&) (b)));}

  // Nombre de lignes du vecteur
  int NElts() const {return nr;}

  // Resolution du systeme A*C = B
  friend double LinSolve(const Matrix& a, const Vector& b, Vector& c);
  
  // Resolution du systeme A*C = B, avec C retourne dans B
  friend double LinSolveInPlace(Matrix& a, Vector& b);

  // Residus et fonction fittees.
  Vector* FitResidus(GeneralFit& gfit);
  Vector* FitFunction(GeneralFit& gfit);

};


inline Vector& Vector::operator = (const Vector& v)
{ Matrix::operator =(v); return *this; }


inline double const& Vector::operator()(int n) const
{
#ifdef RGCHECK
  if (n<0 || n>=ndata) THROW(rangeCheckErr);
#endif
  return data[n];
}


inline double& Vector::operator()(int n)
{ 
#ifdef RGCHECK
  if (n<0 || n>=ndata) THROW(rangeCheckErr);
#endif
  return data[n];
}

inline void Vector::Realloc(int n, bool force)
{ Matrix::Realloc(n,1,force);}


#endif /* VECTOR_SEEN */



