#ifndef ARCHEOPSFILE_H
#define ARCHEOPSFILE_H

#include <stdio.h>
#include <string>
#include <stack>

#define _archeops		//  Acquisition Archeops (avec transputer) 
#define programme

extern "C" {
#include "archeops.h"
}

enum {
  block_param_mask = 1 << block_param,
  block_journal_mask = 1 << block_journal,
  block_reglage_mask = 1 << block_reglage,
  block_dilution_mask = 1 << block_dilution,
  block_gps_mask = 1 << block_gps,
  block_une_periode_mask = 1 << block_une_periode,
  block_synchro_sol_mask = 1 << block_synchro_sol,
  block_pointage_sol_mask = 1 << block_pointage_sol,
  block_bolo_nc_mask = 1 << block_bolo,
  block_gyro_nc_mask = 1 << block_gyro,
  block_sst_nc_mask = 1 << block_sst,
  block_11_mask = 1 << block_11,
  block_bolo_comprime_mask = 1 << block_bolo_comprime,
  block_gyro_comprime_mask = 1 << block_gyro_comprime,
  block_sst_comprime_mask = 1 << block_sst_comprime,
  block_catalog_flash_mask = 1 << block_catalog_flash,
  block_cmd_flash_mask = 1 << block_cmd_flash,
  block_data_brute_mask = 1 << block_data_brute,
  block_18_mask = 1 << block_18,
  block_19_mask = 1 << block_19,

  block_bolo_mask = block_bolo_nc_mask | block_bolo_comprime_mask,
  block_gyro_mask = block_gyro_nc_mask | block_gyro_comprime_mask,
  block_sst_mask = block_sst_nc_mask | block_sst_comprime_mask
};

class BlockSet;

class ArchExc : public exception {
public:
  ArchExc(string s) : msg(s) {}
  string Msg() {return msg;}
protected:
  string msg;
};

class ArcheopsFile {
public:
  ArcheopsFile(string const& fname);  // starting before first block
  ArcheopsFile(ArcheopsFile const&);
  virtual ~ArcheopsFile();
  
  void setUTCOffset(int UTCOffset=2); // StartMJD : decodage a partir du nom du bloc...
  double getStartMJD();
  
  bool nextBlock();   // go to next block, true if block exists
  bool nextBlock(long mask); // go to next block matching mask
  int  blockKind();   // kind of the current bloc, enum in archeops.h
  int  blockRawKind();   // kind of the current raw bloc, enum in archeops.h
  int  blockNum();   // number (timestamp) of the current bloc
  string blockKdName();
  string blockRawKdName();
  
  double perEchant(); // periode d'echantillonage pour le dernier bloc reglage
  double perBlock();  // duree (en secondes) correspondant a un bloc bolo
                      // utilise infos dernier bloc param et dernier bloc reglage.
  int nEchBlock(); // Nb d'echantillons dans un bloc bolo
  
  
  void pushMark();    // push current file position, and "last" blocks
  void popMark();     // pops last file position and "last" blocks
  
  void grabLastBlocs(ArcheopsFile const& oldFile);
  
  block_type_modele*  currentBlock(); // internal buffer
  
  block_type_param*         lastParam();  // saved blocks, including current
  block_type_journal*       lastJournal();
  block_type_reglage*       lastReglage();
  block_type_dilution*      lastDilution();
  block_type_gps*           lastGPS();
  block_type_une_periode*   lastUnePeriode();
  block_type_synchro_sol*   lastSynchroSol();
  block_type_pointage_sol*  lastPointageSol();
  block_type_bolo*          lastBolo();   // can be uncompressed bolo_comp
  block_type_bolo*          llastBolo();   
  block_type_gyro*          lastGyro();
  block_type_sst*           lastSST();
  block_type_bolo_comprime* lastBoloComp();   
  block_type_gyro_comprime* lastGyroComp();
  block_type_sst_comprime*  lastSSTComp();
  
  // Decodage des derniers blocs..
  
  // GPS
  // On travaille en MJD = JD - 2450000.0, = 9 octobre 1995 midi UTC
  // 1er juillet 1999 minuit UTC = JD 2451360.5,   MJD 1360.5 
  int getGPSBlockNum();
  double getGPSUTC();  // en secondes depuis minuit UTC
  double getGPSMJD();  // modified julian day du dernier bloc GPS, JD - 2450000
  double getGPSLat();  // degres, +  = NORD
  double getGPSLong(); // degres, +  = EST
  
  // Bolo
  long getRawBolo(int ibolo, int imesure);   // donnee brute, avec seulement soustraction offset
  double getMuVBolo(int ibolo, int imesure); // microvolts, filtre avec filtre carre
  
  // SST, gyros, etc
  
  double getAzimut(int imesure);
  double getPendDirect(int imesure);
  double getPendOrth(int imesure);
  
  double getAlpha(int imesure);
  double getDelta(int imesure);
  
  
protected:
  BlockSet*                 blockSet;
  stack<BlockSet*>          blockStack;
  stack<size_t>             fposStack;
  
  void saveCurBlock();
  void postProcessBlock();
  
  void computeMJD(string const& fname);
  
  block_type_modele         curBlock;
  int                       curKind;
  int                       curRawKind;
  size_t                    curPos;
  size_t                    fLen;
  FILE* f;
  string fn;
  
  double rawMJD;   // decodage a partir du nom de fichier, sans application d'offset heure locale
  double startMJD; // MJD du debut du fichier.
  double utcOffset;
private:
};

#endif
