#include "toiiter.h"
#include <dirent.h>

// Format bloc GPS
// $GPGGA,hhmmss.ss,ddmm.mmmm,n,dddmm.mmmm,e,q,ss,y.y,a.a,z,

TOIIter::TOIIter() {
 // Toutes les valeurs initialisees par defaut. Le TOISvr les positionnera,
 // puis lancera l'initialisation
 
 file = NULL;
 directory = "";
 fileName = "";
 
 tStart = -9.e99;
 tEnd   = -9.e99;
 
 tBlock0 = -1;

}

TOIIter::TOIIter(TOIIter const& x) {
  directory = x.directory;
  fileName = x.fileName;
  files = x.files;
  curFile = x.curFile;
  imes = x.imes;
  tStart = x.tStart;
  tEnd = x.tEnd;
  tBlock0 = x.tBlock0;
  trigMask = x.trigMask;
  infos = x.infos;   

  file = new ArcheopsFile(*x.file);
}

TOIIter::~TOIIter() {
  delete file;
}

void TOIIter::Init() {
  if (directory == "") {
    file = new ArcheopsFile(fileName);
  } else {
    DIR* dir = opendir(directory.c_str());
    struct dirent* ent;
    while ((ent = readdir(dir)) != NULL) {
      files.insert(ent->d_name);
    }
    closedir(dir);
    curFile = files.begin();
    file = new ArcheopsFile((*curFile).c_str());
  }

  if (!file->lastParam()) file->nextBlock(block_param_mask);
  if (!file->lastReglage()) file->nextBlock(block_reglage_mask);
  
  // On cherche un bloc GPS pour avoir la correspondance timestamp/UTC.
  // Pour le moment, on se fonde sur le premier bloc GPS. On pourra faire
  // mieux en prenant le min de tous les delta_T.
  
    tBlock0 = file->getStartMJD();
#if 0
  file->pushMark();
  if (file->lastGPS() || file->nextBlock(block_gps_mask)) {
    // le temps du bloc courant, en secondes
    double dt = file->blockNum() * file->perBlock();
    tBlock0 = file->getGPSMJD() - dt/86400.;
  } else { // pas de bloc GPS...
    tBlock0 = file->getStartMJD();
  }
  file->popMark();
#endif  
  
  trigMask = 0;
  for (vector<info>::iterator i = infos.begin(); i != infos.end(); i++) {
    if ((*i).triggering) {
      switch ((*i).kind) {
        case boloTens:
        case boloRaw:
          trigMask |= block_bolo_mask;
          break;
        case longitude:
        case latitude:
          trigMask |= block_gps_mask;
          break;
        case azimut:
          trigMask |= block_sst_mask;
          break;
        case alpha:
        case delta:
          trigMask |= block_sst_mask;
          break;
      }
    }
  }
  
  if (trigMask & (block_bolo_mask | block_sst_mask)) {
    imes = 9999;
  } else {
    imes = 0;
  }
}

bool TOIIter::NextFile() {
  if (directory == "") {
    return false;
  } else {
    if (curFile == files.end()) return false;
    curFile++;
    if (curFile == files.end()) return false;
    ArcheopsFile* newfile = new ArcheopsFile((*curFile).c_str());
    newfile->grabLastBlocs(*file);
    delete file;
    file = newfile;
    return true;
  }
}

bool TOIIter::Next() {


  // trigger sur info indexee dans bloc bolo ou bloc sst ?
  if (trigMask & (block_bolo_mask | block_sst_mask)) {
    imes++;
    if (imes < file->nEchBlock()) return true;
    imes = 0;
  }
  
  // soit pas d'info indexee, soit fin bloc courant...
  
  if (file->nextBlock(trigMask)) {
    return true;
  }
  if (!NextFile()) return false;
  return file->nextBlock(trigMask);
}

 double TOIIter::getTime() {                // MJD 
   // le temps du bloc courant, en secondes
   double dt = file->blockNum() * file->perBlock();
   return tBlock0 + dt/86400. + imes*file->perEchant()/86400.;
 }
 
 double TOIIter::getValue(int column) {
   if (column < 0 || column >= infos.size()) return -1;
   TOIKind kind = infos[column].kind;
   int index = infos[column].index;
   switch (kind) {
     case boloTens:
       return file->getMuVBolo(index, imes);
     case boloRaw:
       return file->getRawBolo(index, imes);
     case longitude:
       return file->getGPSLong(); // $CHECK$ TBD gerer interpolation (dans file)
     case latitude:
       return file->getGPSLat();  // $CHECK$ TBD gerer interpolation (dans file)
      case azimut:
       return file->getAzimut(imes);
      case alpha:
       return file->getAlpha(imes);
      case delta:
       return file->getDelta(imes);
   }
   return -1;
 }
 
 bool   TOIIter::newValue(int column) {
   if (column < 0 || column >= infos.size()) return false;
   TOIKind kind = infos[column].kind;
   switch (kind) {
     case boloTens:
       return file->blockKind() == block_bolo;
     case boloRaw:
       return file->blockKind() == block_bolo;
     case longitude:
       return file->blockKind() == block_gps;
     case latitude:
       return file->blockKind() == block_gps;
      case azimut:
       return file->blockKind() == block_sst || file->blockKind() == block_bolo;
      case alpha:
       return file->blockKind() == block_sst || file->blockKind() == block_bolo;
      case delta:
       return file->blockKind() == block_sst || file->blockKind() == block_bolo;
   }
   return false;
 }
 
 bool   TOIIter::extendValue(int column) {
   return !newValue(column);
 }
 
 bool   TOIIter::interpValue(int /*column*/) {
   return false; // $CHECK$ TBD gerer interpolation
 }
 
 TOIKind TOIIter::getKind(int column) {
   if (column < 0 || column >= infos.size()) return (TOIKind)-1;
   return infos[column].kind;
 }
 
 int TOIIter::getIndex(int column) {
   if (column < 0 || column >= infos.size()) return (TOIKind)-1;
   return infos[column].index;
 }
   
 int TOIIter::getSampleIndex() {
   return imes;
 }