// Classe traceur d histogramme                 96-99
// CEA-DAPNIA      LAL-IN2P3/CNRS

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <math.h>
#include <typeinfo>

#include "sopnamsp.h"
#include "pihisto.h"
#include "hisprof.h"

//++
// Class	PIHisto
// Lib		PIext
// include	pihisto.h
//
//	Classe traceur d'objet histogramme (classe *Histo*)
//--
//++
// Links	Parents
// PIDrawer
//--
//++
// Titre	Constructeur, mthodes
//--
//++
// PIHisto(Histo* histo, bool ad=false)
//	Constructeur. Si "ad == true", l'objet "histo" est dtruit par 
//	le destructeur de l'objet "PIHisto"
//	Note : "histo" doit tre cr par new
//
// void  SetStats(bool fg=true) 
//	Active/ dsactive l'indication des statistiques d'histogramme
//--


PIHisto::PIHisto(Histo* histo, bool ad)
: PIDrawer(), mHisto(histo)
{
  mAdDO = ad;     // Flag pour suppression automatique de mHisto
  SetStats();
  SetError();
  SetFilled();
  SetStatPosOffset();
  SetName("HistoDrw");
}

PIHisto::~PIHisto()
{
  if(mAdDO) delete mHisto;
}

void
PIHisto::UpdateLimits()
{
  if (!mHisto)  return;
  float hmin = mHisto->VMin();
  float hmax = mHisto->VMax()+0.2*(mHisto->VMax()-mHisto->VMin());
  // si HBProf min,max calcules en tenant compte des erreurs
  if( typeid(*mHisto) == typeid(HProf) ) {
    float v1,v2;
    for (int i=1; i<mHisto->NBins(); i++) {
      v1 = (*mHisto)(i) - mHisto->Error(i);
      v2 = (*mHisto)(i) + mHisto->Error(i);
      if(v1<hmin) hmin = v1;
      if(v2>hmax) hmax = v2;
    }
    v1 = 0.1*(hmax-hmin);
    hmin -= v1;   hmax += v1;
  }
  if(hmax<=hmin) hmax = hmin+1.;
  SetLimits(mHisto->XMin(), mHisto->XMax(), hmin, hmax);
}

void
PIHisto::Draw(PIGraphicUC* g, double /*xmin*/, double /*ymin*/, double /*xmax*/, double /*ymax*/)
{
  if (axesFlags != kAxesNone) DrawAxes(g);

  if (!mHisto)  return;
  if (mHisto->NBins()<=0) return;
 
  if( typeid(*mHisto)==typeid(HProf) ) mHisto->UpdateHisto();

  bool oktrace=false;
  // Tracer d'une polyline si demandee
  bool drawpline=false;
  if(GetGraphicAtt().GetLineAtt() != PI_NotDefLineAtt) drawpline=true;
  // Tracer des marqueurs si demande ou si HProf
  bool drawmarker=false;
  if( (GetGraphicAtt().GetMarker() != PI_NotDefMarker) ||  
      (typeid(*mHisto) == typeid(HProf)) ) drawmarker = true;
  // Tracer des erreurs ?
  bool drawerr=false;
  if(error==0) {  // Gestion automatique des erreurs
    // Tracer les erreurs si HProf
    if( typeid(*mHisto)==typeid(HProf) ) drawerr=true;
    // Trace les erreurs si marqueurs demandes
    if(drawmarker) drawerr=true;
  }
  else if(error>0) drawerr=true;
  else if(error<0) drawerr=false;
  // Fill de l'histo ?
  bool drawfill=false;
  if(filled) drawfill=true; else drawfill=false;
  // Et aussi si on ne demande ni ligne ni marqueur ?
  if( !drawmarker && !drawpline && !drawerr ) drawfill=true;

  // Remplissage des bins avec la couleur courante (trace en premier)
  if(drawfill) {
    oktrace = true;
    for(int i=0; i<mHisto->NBins(); i++) {
      double left   = mHisto->BinLowEdge(i);
      double width  = mHisto->BinWidth();
      double bottom = 0;
      double height = (*mHisto)(i);
      g->DrawFBox(left,bottom,width,height);
    }
  }

  // Trace des marqeurs
  if(drawmarker) {
    double x1,y1; oktrace = true;
    for(int i=0; i<mHisto->NBins(); i++) {
      x1 = mHisto->BinCenter(i);
      y1 = (*mHisto)(i);
      g->DrawMarker(x1,y1);
    }
  }

  // Trace des erreurs
  if(drawerr) {
    if(GetGraphicAtt().GetLineAtt()==PI_NotDefLineAtt) g->SelLine(PI_ThinLine);
    double x1,x2,y1,y2;  oktrace = true;
    double bw = mHisto->BinWidth();
    for(int i=0; i<mHisto->NBins(); i++) {
      if(mHisto->Error(i)>0.) {
        // barres d'erreur verticales
        x1 = x2 = mHisto->BinCenter(i);
        y1 = (*mHisto)(i) - mHisto->Error(i);
        y2 = (*mHisto)(i) + mHisto->Error(i);
        g->DrawLine(x1,y1, x1, y2);
        // limites de barres d'erreurs (horizontales)
        x1 -= bw/3.;  x2 += bw/3.;
        g->DrawLine(x1,y1, x2, y1);
        g->DrawLine(x1,y2, x2, y2);
      }
    }
  }

  // Trace de la ligne continue si demandee
  if(drawpline) {
    PIGrCoord* x1 = new PIGrCoord[2*mHisto->NBins()+2];
    PIGrCoord* y1 = new PIGrCoord[2*mHisto->NBins()+2];
    double dx = mHisto->BinWidth();
    int npt=0;
    x1[npt] = mHisto->BinLowEdge(0);
    y1[npt] = 0;
    npt++;
    for(int i=0; i<mHisto->NBins(); i++) {
      x1[npt] = mHisto->BinLowEdge(i);
      y1[npt] = (*mHisto)(i);
      npt++;
      x1[npt] = (double)x1[npt-1] + dx;
      y1[npt] = y1[npt-1];
      npt++;
    }
    x1[npt] = x1[npt-1];
    y1[npt] = 0;
    npt++;
    g->DrawPolygon(x1,y1,npt,false);
    delete [] x1; delete [] y1;
    oktrace = true;
  }

  // Trace/Ecriture des statistiques
  // A faire a la fin - DrawStats change l'attribut de ligne 
  if(stats) DrawStats(g);
}

int
PIHisto::DecodeOptionString(vector<string> & opt, bool rmdecopt)
{
  int optsz1 = opt.size();
  if(optsz1<1) return(0); 
  int ndec1 = PIDrawer::DecodeOptionString(opt, rmdecopt);
  if(optsz1-ndec1<1) return(ndec1);  // si tout a ete decode

  vector<string> udopt;  // On gardera ici les options non decodees
  unsigned int k = 0;
  int ndec = opt.size();
  for( k=0; k<opt.size(); k++ ) {
    string opts = opt[k];
    if(opts=="sta" || opts=="stat" || opts=="stats") SetStats(true);
    else if(  opts=="nsta"   || opts=="nstat"
           || opts=="nostat" || opts=="nostats") SetStats(false);
    else if(opts=="err")      SetError(1);
    else if(opts=="noerr" || opts=="nerr")    SetError(-1);
    else if(opts=="autoerr")  SetError(0);
    else if(opts=="fill")     SetFilled(true);
    else if(opts=="nofill" || opts=="nfill")   SetFilled(false);
    else if(opts.substr(0,11) == "statposoff=") {
      float xo=0., yo=0.;
      sscanf(opts.substr(11).c_str(),"%g,%g",&xo, &yo);
      SetStatPosOffset(xo, yo);
    }
    else {
      ndec--;
      // S'il faut supprimer les options decodees
      if (rmdecopt)  udopt.push_back(opts);
    }
  }
  // S'il faut supprimer les options decodees, on remplace l'argument opt
  // par le vecteur des options non decodees.
  if (rmdecopt)  opt = udopt;
  return(ndec+ndec1);
}

int
PIHisto::OptionToString(vector<string> & opt) const
{
 PIDrawer::OptionToString(opt);

 if(stats) opt.push_back("stat"); else opt.push_back("nstat");
 if(error==-1) opt.push_back("noerr");
   else if(error==0) opt.push_back("autoerr");
     else if(error==1) opt.push_back("err");
 if(filled) opt.push_back("fill"); else opt.push_back("nofill");

 char str[256]; sprintf(str,"statposoff=%g,%g",spoX,spoY);
 opt.push_back(str);

 return 1;
}

void
PIHisto::GetOptionsHelpInfo(string& info)
{
info += " ---- PIHisto options help info : \n" ;
info += "  sta,stat,stats:            activate   statistic display\n";
info += "  nsta,nstat,nostat,nostats: deactivate statistic display\n";
info += "  err / noerr,nerr : draw, do not draw error bars\n";
info += "  autoerr : draw error bars if Marker drawing requested OR Profile histo\n";
info += "  fill / nofill,nfill : fill, do not fill bars with selected color\n";
info += "  statposoff=OffsetX,OffsetY : Position offset for Stats drawing \n";
info += "       as a fraction of total size \n";
// On recupere ensuite la chaine info de la classe de base
PIDrawer::GetOptionsHelpInfo(info);
return;
}

void
PIHisto::DrawStats(PIGraphicUC* g)
{
  if (!mHisto) return;
  //  if (mLAtt == PI_NotDefLineAtt)  g->SelLine(PI_ThinLine);
  g->SelLine(PI_ThinLine);
  g->SelFontSz((YMax() - YMin())/30);

  // La hauteur de la cellule
  PIGrCoord a, d;
  double cH = (double)g->GetFontHeight(a,d);
  double cellHeight = 3.6 * cH;

  // Les labels et leurs longueurs -> largeur de la cellule
  char *label, label1[64], label2[64], label3[64];
  sprintf(label1, "N= %-g",mHisto->NData());
  sprintf(label2, "m= %-g",mHisto->Mean());
  sprintf(label3, "s= %-g",mHisto->Sigma());
  label = label1;
  if(strlen(label)<strlen(label2)) label = label2;
  if(strlen(label)<strlen(label3)) label = label3;
  double cellWidth =  1.1 * (double)g->CalcStringWidth(label);

  double ofpx = spoX*(XMax()-XMin());
  double ofpy = spoY*(YMax()-YMin());
  
  double xu, yu, cw;
  // Les limites du cadre
  xu = g->DeltaUCX(XMax(), -cellWidth);
  yu = g->DeltaUCY(YMax(), -cellHeight);
  double recw = XMax()-xu;
  double rech = YMax()-yu;
  xu += ofpx;  yu += ofpy;
  g->DrawBox(xu, yu, recw, rech);

  // L'ecriture des labels (attention aux inversions possibles des axes!)
  cw = (g->isAxeXDirRtoL()) ? -0.05*cellWidth : -0.95*cellWidth;
  xu = g->DeltaUCX(XMax(),cw);

  cw = (g->isAxeYDirUpDown()) ? -0.15*cH : -1.15*cH;
  yu = g->DeltaUCY(YMax(),cw);
  xu += ofpx;  yu += ofpy;
  g->DrawString(xu, yu,label1);
  cw +=  -1.15*cH;
  yu = g->DeltaUCY(YMax(),cw);  yu += ofpy;
  g->DrawString(xu, yu,label2);
  cw +=  -1.15*cH;
  yu = g->DeltaUCY(YMax(),cw);  yu += ofpy;
  g->DrawString(xu, yu,label3);
                       
}



/* --Methode-- */
double PIHisto::GetDistanceToPoint(double x, double y)
{
 if (!mHisto) return 1.e+9;

 double dist = -1.e+18;
 for(int i=0; i<mHisto->NBins(); i++) {
   double xp=mHisto->BinCenter(i);
   double yp=(*mHisto)(i);
   xp = (xp-x)/(XMax()-XMin())/0.5;
   yp = (yp-y)/(YMax()-YMin())/0.5;
   xp = xp*xp+yp*yp;
   if(dist<0. || xp<dist) dist = xp;
 }
 dist=sqrt(fabs(dist));
 //cout<<dist<<"PIHisto: xlim="<<XMin()<<","<<XMax()<<" ylim="<<YMin()<<","<<YMax()
 //    <<" NBins="<<mHisto->NBins()<<endl;
 //cout<<"....d="<<dist<<" x="<<x<<" y="<<y<<endl;

 return dist;
}
