/* Interface Fits BINARY/ASCII Table Column Reader    cmv 26/09/2001 */
#ifndef FABTCOLREAD_H_SEEN
#define FABTCOLREAD_H_SEEN

#include "machdefs.h"
#include <iostream>
#include <string.h>
#include <string>

#include "anydataobj.h"
#include "tvector.h"
#include "FitsIO/fitsio.h"

namespace SOPHYA {

///////////////////////////////////////////////////////////////////
//! Class for opening a FITS file for reading
class FitsOpenFile : public AnyDataObj {
public:
  FitsOpenFile(string fname);
  FitsOpenFile(const char *cfname);
  FitsOpenFile();
  FitsOpenFile(FitsOpenFile& fof);
  virtual ~FitsOpenFile();

  inline string FileName() const {return FitsFN;}
  //! Get the number of the HDU read
  inline int HDU(void) const {return IHdu;}
  //! Get the number of the HDU type
  inline int HDUType(void) const {return HduType;}
  //! Get the number of HDU in file
  inline int NHDU() const {return NHdu;}
  //! Get the CFISTIO fits file pointer
  inline fitsfile* GetFitsPtr() const {return FitsPtr;}
  //! Set the positionning status of the file
  inline void SetPosStatus(bool sta=true) {HasBeenPos = sta;}
  //! Get the positionning status of the file
  inline bool GetPosStatus(void) const {return HasBeenPos;}

  int MoveToHDU(int ihdu);
  int MoveToFirst(int hdutype,int ihdudeb=1);
  int MoveToLast(int hdutype,int ihdudeb=1);
  void Print(void);

  static double   ReadKey(fitsfile *fitsptr,char *keyname);
  static long     ReadKeyL(fitsfile *fitsptr,char *keyname);
  static LONGLONG ReadKeyLL(fitsfile *fitsptr,char *keyname);
  static string   ReadKeyS(fitsfile *fitsptr,char *keyname);
  static void   printerror(int sta);

protected:
  void Init(const char *cfname);
  void Delete(void);

  string FitsFN;
  int NHdu, IHdu, HduType;
  fitsfile *FitsPtr;
  bool HasBeenPos;
};

///////////////////////////////////////////////////////////////////
//! Class for reading a column in a FITS ASCII or BINARY table
class FitsABTColRd : public AnyDataObj {
public:
  FitsABTColRd(FitsOpenFile* fof,string collabel,int ihdu=0
              ,long blen=100,long bsens=1,int lp=0);
  FitsABTColRd(FitsOpenFile* fof,int colnum,int ihdu=0
              ,long blen=100,long bsens=1,int lp=0);
  FitsABTColRd(FitsABTColRd& fbt);
  FitsABTColRd();
  virtual ~FitsABTColRd();

  void ChangeBuffer(long blen=100,long bsens=1);

  double    ReadKey(char *keyname);
  long      ReadKeyL(char *keyname);
  LONGLONG  ReadKeyLL(char *keyname);
  string    ReadKeyS(char *keyname);

  double Read(LONGLONG n,bool usebuffer=true);

  LONGLONG Read(LONGLONG n1,LONGLONG n2,TVector<uint_2>& data);
  LONGLONG Read(LONGLONG n1,LONGLONG n2,TVector<int_4>& data);
  LONGLONG Read(LONGLONG n1,LONGLONG n2,TVector<int_8>& data);
  LONGLONG Read(LONGLONG n1,LONGLONG n2,TVector<float>& data);
  LONGLONG Read(LONGLONG n1,LONGLONG n2,TVector<double>& data);

  //! return the value of the first row
  double ReadFirstRow(bool usebuffer=false)
                     {return Read(0,usebuffer);}
  //! return the value of the last row
  double ReadLastRow(bool usebuffer=false)
                    {return Read(NBline-1,usebuffer);}

  LONGLONG FirstRow(double val1,double val2,LONGLONG rowstart=-1);
  LONGLONG LastRow(double val1,double val2,LONGLONG rowstart=-1);

  //! Set debug level
  inline void SetDebug(int lp=0) {DbgLevel = (unsigned short) lp;}
  //! Set null value to be return when reading null data (0=return the data)
  inline void SetNulVal(double nulval=0.) {NulVal = nulval;}
  //! Get the FITS file name
  inline string FileName(void) const
         {if(FitsOF) return FitsOF->FileName(); else return (string)"";}
  //! Get the pointer to FitsOpenFile
  inline FitsOpenFile* GetFitsOpenFile(void) const {return FitsOF;}
  //! Get the FITS file pointer (cfistio pointer)
  inline fitsfile* GetFitsPtr(void) const {return FitsOF->GetFitsPtr();}
  //! Get the number of HDU in the FITS file
  inline int NHDU(void) const
         {if(FitsOF) return FitsOF->NHDU(); else return 0;}
  //! Get the number of the HDU read
  inline int HDU(void) const
         {if(FitsOF) return FitsOF->HDU(); else return 0;}
  //! Get the HDU type
  inline int HDUType(void) const
         {if(FitsOF) return FitsOF->HDUType(); else return 0;}
  //! Get the number of rows in the FITS HDU to be read
  inline LONGLONG GetNbLine(void) const {return NBline;}
  //! Get the number of columns in the FITS HDU to be read
  inline int GetNbCol(void) const {return NBcol;}
  //! Get the columns number that is read
  inline int GetColNum(void) const {return ColNum;}
  //! Get the columns label that is read
  inline string GetColLabel(void) const {return ColLabel;}
  //! Get the columns type code that is read
  inline int GetColTypeCode(void) const {return ColTypeCode;}
  //! Get the columns fits tunit that is read
  inline string GetColTUnit(void) const {return ColTUnit;}
  //! Get the columns fits tform that is read
  inline string GetColTForm(void) const {return ColTForm;}
  //! Get the read requested buffer length
  inline long GetBLen(void) const {return BuffLen;}
  //! Get the read buffer direction
  inline long GetBSens(void) const {return BuffSens;}
  //! Print to os
  virtual void Print(ostream& os,int lp=1) const;
  //! Print to stdout
  inline  void Print(int lp=1) const {Print(cout,lp);}
  //! Get the read effective buffer length
  inline long GetNBuffer(void) const {return NBuffer;}
  //! Get the read bufferpointer
  inline double* GetBuffer(void) {return Buffer;}

protected:
  void Init(FitsOpenFile* fof,const char *collabel,int colnum
           ,int ihdu,long blen,long bsens,int lp);
  void Delete(void);

  string ColLabel,ColTUnit,ColTForm;
  int ColNum,ColTypeCode,NBcol;
  LONGLONG NBline;

  double NulVal;
  unsigned short DbgLevel;
  long BuffLen, BuffSens;

  LONGLONG NFitsRead;
  FitsOpenFile* FitsOF;
  LONGLONG LineDeb, LineFin;
  double *Buffer;
  long NBuffer;
};

///////////////////////////////////////////////////////////////////
//! Class for reading a column in a FITS ASCII or BINARY table with fits file opening
class FitsABTColRead : public FitsABTColRd {
public:
  FitsABTColRead(string fname,string collabel,int ihdu=0
                ,long blen=100,long bsens=1,int lp=0);
  FitsABTColRead(string fname,int colnum,int ihdu=0
                ,long blen=100,long bsens=1,int lp=0);
  FitsABTColRead(const char *cfname,const char *collabel,int ihdu=0
                ,long blen=100,long bsens=1,int lp=0);
  FitsABTColRead(const char *cfname,int colnum,int ihdu=0
                ,long blen=100,long bsens=1,int lp=0);
  FitsABTColRead(FitsABTColRead& fbt);
  FitsABTColRead();
  virtual ~FitsABTColRead();
};


///////////////////////////////////////////////////////////////////
//! Class for reading a 2D image from a FITS file
class FitsImg2DRd : public AnyDataObj {
public:
  FitsImg2DRd(FitsOpenFile* fof,int ihdu=0,int lp=0);
  FitsImg2DRd(FitsImg2DRd& fbt);
  FitsImg2DRd();
  virtual ~FitsImg2DRd();

  double   ReadKey(char *keyname);
  long     ReadKeyL(char *keyname);
  LONGLONG ReadKeyLL(char *keyname);
  string   ReadKeyS(char *keyname);

  LONGLONG Read(TMatrix<uint_2>& data);
  LONGLONG Read(TMatrix<int_4>& data);
  LONGLONG Read(TMatrix<int_8>& data);
  LONGLONG Read(TMatrix<float>& data);
  LONGLONG Read(TMatrix<double>& data);

  //! Set debug level
  inline void    SetDebug(int lp=0) {DbgLevel = (unsigned short) lp;}
  //! Set null value to be return when reading null data (0=return the data)
  inline void    SetNulVal(double nulval=0.) {NulVal = nulval;}
  //! Get the pointer to FitsOpenFile
  inline FitsOpenFile* GetFitsOpenFile(void) const {return FitsOF;}
  //! Get the FITS file pointer (cfistio pointer)
  inline fitsfile* GetFitsPtr(void) const {return FitsOF->GetFitsPtr();}
  //! Get the number of HDU in the FITS file
  inline int NHDU(void) const
         {if(FitsOF) return FitsOF->NHDU(); else return 0;}
  //! Get the number of the HDU read
  inline int HDU(void) const
         {if(FitsOF) return FitsOF->HDU(); else return 0;}
  //! Get the HDU type
  inline int HDUType(void) const
         {if(FitsOF) return FitsOF->HDUType(); else return 0;}
  //! Get NAXIS1
  inline LONGLONG Naxis1(void) const {return Naxis[0];}
  //! Get NAXIS2
  inline LONGLONG Naxis2(void) const {return Naxis[1];}

protected:
  void Init(FitsOpenFile* fof,int ihdu,int lp);

  LONGLONG Naxis[2];
  double NulVal;
  unsigned short DbgLevel;

  FitsOpenFile* FitsOF;
};

///////////////////////////////////////////////////////////////////
//! Class for reading a 2D image from a FITS file
class FitsImg2DRead : public FitsImg2DRd {
public:
  FitsImg2DRead(string fname,int ihdu=0,int lp=0);
  FitsImg2DRead(const char *cfname,int ihdu=0,int lp=0);
  FitsImg2DRead(FitsImg2DRead& fbt);
  FitsImg2DRead();
  virtual ~FitsImg2DRead();
};


///////////////////////////////////////////////////////////////////
//! Class for reading a 3D image from a FITS file
class FitsImg3DRd : public AnyDataObj {
public:
  FitsImg3DRd(FitsOpenFile* fof,int ihdu=0,int lp=0);
  FitsImg3DRd(FitsImg3DRd& fbt);
  FitsImg3DRd();
  virtual ~FitsImg3DRd();

  double   ReadKey(char *keyname);
  long     ReadKeyL(char *keyname);
  LONGLONG ReadKeyLL(char *keyname);
  string   ReadKeyS(char *keyname);

  LONGLONG Read(TArray<uint_2>& data);
  LONGLONG Read(TArray<int_4>& data);
  LONGLONG Read(TArray<int_8>& data);
  LONGLONG Read(TArray<float>& data);
  LONGLONG Read(TArray<double>& data);

  //! Set debug level
  inline void    SetDebug(int lp=0) {DbgLevel = (unsigned short) lp;}
  //! Set null value to be return when reading null data (0=return the data)
  inline void    SetNulVal(double nulval=0.) {NulVal = nulval;}
  //! Get the pointer to FitsOpenFile
  inline FitsOpenFile* GetFitsOpenFile(void) const {return FitsOF;}
  //! Get the FITS file pointer (cfistio pointer)
  inline fitsfile* GetFitsPtr(void) const {return FitsOF->GetFitsPtr();}
  //! Get the number of HDU in the FITS file
  inline int NHDU(void) const
         {if(FitsOF) return FitsOF->NHDU(); else return 0;}
  //! Get the number of the HDU read
  inline int HDU(void) const
         {if(FitsOF) return FitsOF->HDU(); else return 0;}
  //! Get the HDU type
  inline int HDUType(void) const
         {if(FitsOF) return FitsOF->HDUType(); else return 0;}
  //! Get NAXIS1
  inline LONGLONG Naxis1(void) const {return Naxis[0];}
  //! Get NAXIS2
  inline LONGLONG Naxis2(void) const {return Naxis[1];}
  //! Get NAXIS3
  inline LONGLONG Naxis3(void) const {return Naxis[2];}

protected:
  void Init(FitsOpenFile* fof,int ihdu,int lp);

  LONGLONG Naxis[3];
  double NulVal;
  unsigned short DbgLevel;

  FitsOpenFile* FitsOF;
};

///////////////////////////////////////////////////////////////////
//! Class for reading a 3D image from a FITS file
class FitsImg3DRead : public FitsImg3DRd {
public:
  FitsImg3DRead(string fname,int ihdu=0,int lp=0);
  FitsImg3DRead(const char *cfname,int ihdu=0,int lp=0);
  FitsImg3DRead(FitsImg3DRead& fbt);
  FitsImg3DRead();
  virtual ~FitsImg3DRead();
};

} // namespace SOPHYA
#endif    /* FABTCOLREAD_H_SEEN */
