#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include "timing.h"
#include "ntuple.h"
#include "matharr.h"
#include "srandgen.h"
#include "perandom.h"
#include "fabtwriter.h"

#include "arrctcast.h"

#include "constcosmo.h"
#include "schechter.h"
#include "geneutils.h"
#include "integfunc.h"
#include "genefluct3d.h"

void usage(void);
void usage(void)
{
 cout<<"cmvobserv3d [-a] [-0]"<<endl
     <<" -a : init auto de l aleatoire"<<endl
     <<" -0 : use methode ComputeFourier0"<<endl
     <<" -x nx,dx : taille en x (npix,Mpc)"<<endl
     <<" -y ny,dy : taille en y (npix,Mpc)"<<endl
     <<" -z nz,dz : taille en z (npix,Mpc)"<<endl
     <<" -Z zref : redshift median"<<endl
     <<" -s snoise : sigma du bruit en Msol"<<endl
     <<endl;
}

int main(int narg,char *arg[])
{
 InitTim();

 //-----------------------------------------------------------------
 // *** Survey definition
 long nx=360, ny=-1, nz=64; double dx=1., dy=-1., dz=-1.;
 //long nx=1000, ny=-1, nz=128; double dx=3., dy=-1., dz=6.;
 //long nx=1200, ny=-1, nz=128; double dx=1., dy=-1., dz=3;

 // *** Cosmography definition   (WMAP)
 double ob0 = 0.0444356;
 double h100=0.71, om0=0.267804, or0=7.9e-05, ol0=0.73,w0=-1.;
 double zref = 0.5;

 // *** Spectrum and variance definition
 double ns = 1., as = 1.;
 double R=8./h100, Rg=R/sqrt(5.);
 double sigmaR = 1.;

 double kmin=1e-5,kmax=1000.;
 int npt = 10000;
 double lkmin=log10(kmin), lkmax=log10(kmax);
 double eps=1.e-3;

 // *** Schechter mass function definition
 double h75 = 0.71 / 0.75;
 double nstar = 0.006*pow(h75,3.); 
 double mstar = pow(10.,9.8/(h75*h75));  // MSol
 double alpha = -1.37;

 double schmin=1e8, schmax=1e12;
 int schnpt = 1000;
 double lschmin=log10(schmin), lschmax=log10(schmax), dlsch=(lschmax-lschmin)/schnpt;

 // *** Niveau de bruit
 double snoise= 0.;   // en equivalent MSol

 // *** type de generation
 bool computefourier0=false;
 unsigned short nthread=4;

 // --- Decodage arguments

 char c;
 while((c = getopt(narg,arg,"ha0x:y:z:s:Z:")) != -1) {
  switch (c) {
  case 'a' :
    Auto_Ini_Ranf(5);
    break;
  case '0' :
    computefourier0 = true;
    break;
  case 'x' :
    sscanf(optarg,"%ld,%lf",&nx,&dx);
    break;
  case 'y' :
    sscanf(optarg,"%ld,%lf",&ny,&dy);
    break;
  case 'z' :
    sscanf(optarg,"%ld,%lf",&nz,&dz);
    break;
  case 's' :
    sscanf(optarg,"%lf",&snoise);
    break;
  case 'Z' :
    sscanf(optarg,"%lf",&zref);
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 }

 string tagobs = "cmvobserv3d.ppf";
 POutPersist posobs(tagobs);

 cout<<"zref="<<zref<<endl;
 cout<<"nx="<<nx<<" dx="<<dx<<" ny="<<ny<<" dy="<<dy<<" nz="<<nz<<" dz="<<dz<<endl;
 cout<<"kmin="<<kmin<<" ("<<lkmin<<"), kmax="<<kmax<<" ("<<lkmax<<") Mpc^-1"
     <<", npt="<<npt<<endl;
 cout<<"R="<<R<<" Rg="<<Rg<<" Mpc, sigmaR="<<sigmaR<<endl;
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;
 cout<<"schmin="<<schmin<<" ("<<lschmin
     <<"), schmax="<<schmax<<" ("<<lschmax<<") Msol"
     <<", schnpt="<<schnpt<<endl;
 cout<<"snoise="<<snoise<<" equivalent Msol"<<endl;

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Spectrum and mass function"<<endl;

 InitialSpectrum pkini(ns,as);

 TransfertEisenstein tf(h100,om0-ob0,ob0,T_CMB_Par,false);
 //tf.SetNoOscEnv(2);

 GrowthFactor d1(om0,ol0);

 PkSpectrum0 pk0(pkini,tf);

 PkSpectrumZ pkz(pk0,d1,zref);
 
 Schechter sch(nstar,mstar,alpha);

 //-----------------------------------------------------------------
 pkz.SetZ(0.);
 cout<<endl<<"\n--- Compute variance for top-hat R="<<R
     <<" at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_th(pkz,0);
 double kfind_th = varpk_th.FindMaximum(R,kmin,kmax,eps);
 double pkmax_th = varpk_th(kfind_th);
 cout<<"kfind_th = "<<kfind_th<<" ("<<log10(kfind_th)<<"), integrand="<<pkmax_th<<endl;
 double k1=kmin, k2=kmax;
 int rc = varpk_th.FindLimits(R,pkmax_th/1.e4,k1,k2,eps);
 cout<<"limit_th: rc="<<rc<<" : "<<k1<<" ("<<log10(k1)<<") , "
     <<k2<<" ("<<log10(k2)<<")"<<endl;

 double ldlk = (log10(k2)-log10(k1))/npt;
 varpk_th.SetInteg(0.01,ldlk,-1.,4);
 double sr2 = varpk_th.Variance(R,k1,k2);
 cout<<"varpk_th="<<sr2<<"  ->  sigma="<<sqrt(sr2)<<endl;

 double normpkz = sigmaR*sigmaR/sr2;
 pkz.SetScale(normpkz);
 cout<<"Spectrum normalisation = "<<pkz.GetScale()<<endl;

 pkz.SetZ(zref);

 Histo hpkz(lkmin,lkmax,npt); hpkz.ReCenterBin();
 FuncToHisto(pkz,hpkz,true);
 {
 tagobs = "hpkz"; posobs.PutObject(hpkz,tagobs);
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Compute variance for Pk at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_int(pkz,2);

 double kfind_int = varpk_int.FindMaximum(R,kmin,kmax,eps);
 double pkmax_int = varpk_int(kfind_int);
 cout<<"kfind_int = "<<kfind_int<<" ("<<log10(kfind_int)<<"), integrand="<<pkmax_int<<endl;
 double k1int=kmin, k2int=kmax;
 int rcint = varpk_int.FindLimits(R,pkmax_int/1.e4,k1int,k2int,eps);
 cout<<"limit_int: rc="<<rcint<<" : "<<k1int<<" ("<<log10(k1int)<<") , "
     <<k2int<<" ("<<log10(k2int)<<")"<<endl;

 double ldlkint = (log10(k2int)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int = varpk_int.Variance(R,k1int,k2int);
 cout<<"varpk_int="<<sr2int<<"  ->  sigma="<<sqrt(sr2int)<<endl;

 PrtTim("End of definition");

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Compute galaxies density number"<<endl;

 sch.SetOutValue(0);
 cout<<"sch(mstar) = "<<sch(mstar)<<" /Mpc^3/Msol"<<endl;
 double ngal_by_mpc3 = IntegrateFuncLog(sch,lschmin,lschmax,0.01,dlsch,10.*dlsch,4);
 cout<<"Galaxy density number = "<<ngal_by_mpc3<<" /Mpc^3 between limits"<<endl;

 sch.SetOutValue(1);
 cout<<"mstar*sch(mstar) = "<<sch(mstar)<<" Msol/Mpc^3/Msol"<<endl;
 double mass_by_mpc3 = IntegrateFuncLog(sch,lschmin,lschmax,0.01,dlsch,10.*dlsch,4);
 cout<<"Galaxy mass density= "<<mass_by_mpc3<<" Msol/Mpc^3 between limits"<<endl;

 //-----------------------------------------------------------------
 // FFTW3 (p26): faster if sizes 2^a 3^b 5^c 7^d 11^e 13^f  with e+f=0 ou 1
 cout<<endl<<"\n--- Compute Spectrum Realization"<<endl;

 pkz.SetZ(zref);
 TArray< complex<r_8> > pkgen;
 GeneFluct3D fluct3d(pkgen,pkz);
 fluct3d.SetNThread(nthread);
 fluct3d.SetSize(nx,ny,nz,dx,dy,dz);
 fluct3d.Print();
 double knyqmax = fluct3d.GetKmax();
 double dkmin = fluct3d.GetKinc()[0];

 cout<<"\n--- Computing spectra variance up to Kmax at z="<<pkz.GetZ()<<endl;
 // En fait on travaille sur un cube inscrit dans la sphere de rayon kmax:
 // sphere: Vs = 4Pi/3 k^3 , cube inscrit (cote k*sqrt(2)): Vc = (k*sqrt(2))^3
 // Vc/Vs = 0.675   ->  keff = kmax * (0.675)^(1/3) = kmax * 0.877
 knyqmax *= 0.877;
 ldlkint = (log10(knyqmax)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int_kmax = varpk_int.Variance(R,k1int,knyqmax);
 cout<<"varpk_int(<"<<knyqmax<<")="<<sr2int_kmax<<"  ->  sigma="<<sqrt(sr2int_kmax)<<endl;

 cout<<"\n--- Computing a realization in Fourier space"<<endl;
 if(computefourier0) fluct3d.ComputeFourier0();
   else              fluct3d.ComputeFourier();

 cout<<"\n--- Checking realization spectra"<<endl;
 long nhprof = long(fluct3d.GetKmax()/dkmin+0.5);
 HProf hpkgen(0.,fluct3d.GetKmax(),nhprof);
 hpkgen.ReCenterBin();
 fluct3d.ComputeSpectrum(hpkgen);
 {
 tagobs = "hpkgen"; posobs.PutObject(hpkgen,tagobs);
 }

 if(1) {
   cout<<"\n--- Computing convolution by pixel shape"<<endl;
   fluct3d.FilterByPixel();

   cout<<"\n--- Checking realization spectra after pixel shape convol."<<endl;
   HProf hpkgenf(hpkgen); hpkgenf.Zero();
   fluct3d.ComputeSpectrum(hpkgenf);
   {
   tagobs = "hpkgenf"; posobs.PutObject(hpkgenf,tagobs);
   }
 }

 if(0) {
   cout<<"\n--- Writing cmvobserv3dk.ppf"<<endl;
   string tag = "cmvobserv3dk.ppf";
   POutPersist pos(tag);
   tag = "pkgen"; pos.PutObject(pkgen,tag);
 }

 cout<<"\n--- Computing a realization in real space"<<endl;
 fluct3d.ComputeReal();
 r_8 undouble=0.;
 TArray<r_8> rgen = ArrayCast(pkgen,undouble);
 double rmin,rmax; rgen.MinMax(rmin,rmax);
 cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;

 cout<<"\n--- Check mean and variance in real space"<<endl;
 int_8 nlowone = fluct3d.NumberOfBad(-1.,1e+200);
 double rm,rs2; int_8 nm;
 nm = fluct3d.MeanSigma2(rm,rs2);
 cout<<"rgen:("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<",  n(<-1)="<<nlowone<<endl;

 if(1) {
   cout<<"\n--- Check variance sigmaR in real space"<<endl;
   double varr;
   int_8 nvarr = fluct3d.VarianceFrReal(R,varr);
   cout<<"R="<<R<<" : sigmaR^2="<<varr<<" -> "<<sqrt(varr)<<",   n="<<nvarr<<endl;
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Converting fluctuations into mass"<<endl;
 fluct3d.TurnFluct2Mass();
 nm = fluct3d.MeanSigma2(rm,rs2);
 cout<<"1+rgen: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;

 cout<<"\n--- Converting mass into galaxy number"<<endl;
 rm = fluct3d.TurnMass2MeanNumber(ngal_by_mpc3);
 cout<<rm<<" galaxies put into survey"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,0.);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;

 cout<<"\n--- Set negative pixels to BAD"<<endl;
 nm = fluct3d.SetToVal(0.,1e+200,-999.);
 cout<<nm<<" negative in survey set to BAD"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,-998.);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;

 cout<<"\n--- Apply poisson on galaxie number"<<endl;
 nm = fluct3d.ApplyPoisson();
 cout<<nm<<" galaxies into survey after poisson"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,-998.);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;

 cout<<"\n--- Convert Galaxies number to HI mass"<<endl;
 long nhmdndm = long( (lschmax-lschmin+1.)*100. + 0.5);
 Histo hmdndm(lschmin,lschmax,nhmdndm);
 sch.SetOutValue(1);
 FuncToHisto(sch,hmdndm,true);
 FunRan tirhmdndm(hmdndm,true);
 {
 tagobs = "hmdndm"; posobs.PutObject(hmdndm,tagobs);
 Histo hdum1(tirhmdndm);
 tagobs = "tirhmdndm"; posobs.PutObject(hdum1,tagobs);
 }
 double mhi = fluct3d.TurnNGal2Mass(tirhmdndm,true);
 cout<<mhi<<" MSol in survey / "<<mass_by_mpc3*fluct3d.GetVol()<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,0.);
 cout<<"HI mass: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 cout<<"Equivalent: "<<rm*nm/fluct3d.NPix()<<" Msol / pixels"<<endl;

 cout<<"\n--- Set BAD pixels to Zero"<<endl;
 nm = fluct3d.SetToVal(-998.,1e+200,0.);
 cout<<nm<<" BAD in survey set to zero"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;

 if(1) {
   cout<<"\n---Writing Cube to Fits file"<<endl;
   FitsImg3DWriter fwrt("!cmvobserv3dr.fits",FLOAT_IMG,5);
   fwrt.WriteKey("ZREF",zref," redshift");
   vector<long> n = fluct3d.GetNpix();
   fwrt.WriteKey("NX",n[0]," axe transverse 1");
   fwrt.WriteKey("NY",n[1]," axe transverse 2");
   fwrt.WriteKey("NZ",n[2]," axe longitudinal (redshift)");
   vector<r_8> d;
   d = fluct3d.GetDinc();
   fwrt.WriteKey("DX",d[0]," Mpc");
   fwrt.WriteKey("DY",d[1]," Mpc");
   fwrt.WriteKey("DZ",d[2]," Mpc");
   d = fluct3d.GetKinc();
   fwrt.WriteKey("DKX",d[0]," Mpc^-1");
   fwrt.WriteKey("DKY",d[1]," Mpc^-1");
   fwrt.WriteKey("DKZ",d[2]," Mpc^-1");
   fwrt.WriteKey("SNOISE",snoise," Msol");
   fwrt.Write(rgen);
 }

 cout<<"\n--- Add noise to HI Flux snoise="<<snoise<<endl;
 fluct3d.AddNoise2Real(snoise);
 nm = fluct3d.MeanSigma2(rm,rs2);
 cout<<"HI mass with noise: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;

 if(0) {
   cout<<"\n--- Writing cmvobserv3dr.ppf"<<endl;
   string tag = "cmvobserv3dr.ppf";
   POutPersist pos(tag);
   tag = "rgen"; pos.PutObject(rgen,tag);
 }

 //-----------------------------------------------------------------
 // -- NE PAS FAIRE CA SI ON VEUT CONTINUER LA SIMULATION -> d_rho/rho ecrase
 if(1) {
   cout<<endl<<"\n--- ReComputing spectrum from real space"<<endl;
   fluct3d.ReComputeFourier();
   HProf hpkrec(0.,fluct3d.GetKmax(),nhprof);
   hpkrec.ReCenterBin();
   fluct3d.ComputeSpectrum(hpkrec);
   tagobs = "hpkrec"; posobs.PutObject(hpkrec,tagobs);
 }

 return 0;
}

/*
openppf cmvobserv3dk.ppf
openppf cmvobserv3dr.ppf
openppf cmvobserv3d.ppf

objaoper pkgen sliceyz 0

n/plot hpkz.val%x ! ! "nsta connectpoints"
n/plot hpkgen.val%log10(x) x>0 ! "nsta same red connectpoints"
n/plot hpkgenf.val%log10(x) x>0 ! "nsta same orange connectpoints"
n/plot hpkrec.val%log10(x) x>0 ! "nsta same blue connectpoints"

set k pow(10.,x)
n/plot hpkz.val*$k*$k/(2*M_PI*M_PI)%x ! "connectpoints"

defscript rgensl
 objaoper rgen sliceyz $1
 disp sliceyz_$1 
endscript 

rgensl 0

disp hmdndm
disp tirhmdndm
addline 0 1 20 1 "red"

 */
