#ifndef MULTICYL_H
#define MULTICYL_H

#include "machdefs.h"
#include "sopnamsp.h"
#include "mbeamcyl.h"

/* 
   Projet BAORadio / HSHS
   Programme de simulation pour reconstruction de lobe radio.
   Reconstruction de lobe a partir de plusieurs cylindres

   R. Ansari - LAL      Jan 2007

   Note : On travaille ds un systeme sans dimension
   unite de temps = 1 = temps d'echantillonnage (f_echant = 1)
   vitesse de propagation (c) = 1
*/


//-----------------------------------------------------------------------
//--- Classe de reconstruction de plan source correspondant a un cylindre
class MultiCylinders {
public :
  // nr = nb de recepteurs
  // ns = nb d'echantillons en temps de chaque paquet
  //   nr = 256 -> resol_ang ~ pi/256 = 0.01 rad ~ 40'
  //   longueur @ f=2 ~ 64 (256*lambda/2 = 256*0.25)
 MultiCylinders(int nr=256, int ns=1024);
 MultiCylinders(char* filename);
 ~MultiCylinders();

 // Niveau de print de debug
 inline void SetPrintLevel(int prl=0) { PrtLev_=prl; }

 // Ajout d'un cylindre, en position posX, posY
 inline int AddCylinder(double posx, double posy)
   {  mCyl_.push_back( new MultiBeamCyl(NR_, NS_, posx, posy) ); return mCyl_.size(); }

 MultiBeamCyl & GetCylinder(int i);

 // Specification de la frequence de base f0 et espacement des recepteurs
 inline void SetBaseFreqDa(double f0=2., double a=0.25) 
   {  freq0_ = f0;   Da_ = a;  }

 // frequences reduites (entre 0 ... 0.5) , angle en radian, amp ~ 1
 void SetSources(BRSourceGen* brs, bool ad=true);
 inline void SetSources(BRSourceGen& brs)
   { SetSources(&brs, false); }

 // Definition du sigma du bruit gaussien sur les echantillons
 inline void SetNoiseSigma(double sig=0.) {  signoise_ = sig; }
 // Definition du sigma du jitter d'horloge (typ 0.01)
 inline void SetTimeJitter(double tjit=0.) { timejitter_ = tjit; } 
 // Definition du sigma des offsets d'horloge entre recepteurs (typ 0.02)
 inline void SetTimeOffsetSigma(double tsig=0.) { toffsig_ = tsig; }

 // Definition du gain et sigma de fluctuations de gain 
 // nzerogain: nb de recepteurs (choisis au hasard) avec gain mis a zero
 inline void SetGains(double g=1., double sigg=0., int nzerogain=0)
   {  gain_=g;  siggain_=sigg;  ngainzero_ = nzerogain; }


 // Fait la configuration des cylindres, et reconstruit le plan source
 // pour chaque cylindre
 void ReconstructCylinderPlaneS(bool fgzerocentre=true);

 // Reconstruction de la boite 3D des sources 
 // - halfny : nb de bin en angY = 2*halfny+1 
 // - stepangy : pas en angle Y (radian) 
 // NOTE: Cette methode appelle ReconstructCylinderPlaneS(true);
 // La resolution doit etre <= resol en angX --> ~ 1 deg ~ pi/
 // @f = 2 , lambda = 0.5 ===> posY <~ lambda/(2 sin resol) 
 //   ===> posY < ~ 20   
// void ReconstructSourceBox(int halfny=10, double stepangy=M_PI/300);
 void ReconstructSourceBox(int halfny, double stepangy, int ny, double stepangx);


 // Pour recuperer une tranche reconstruite angX, angY 
 // avec la moyenne des modules entre kfmin <= k (selon z) <= kfmax 
 TMatrix< r_4 > getRecXYSlice(int kfmin, int kfmax);
 TMatrix< r_4 > getRecYXSlice(int kfmin, int kfmax);

 // Acces a la boite 3D de sources reconstruite
 inline TArray< complex<r_4> > & getRecSrcBox() { return cmplx_srcbox_; }

 // Configure chaque cylindre a partir des parametres de la classe 
 // est appele automatiquement par ReconstructCylinderPlaneS
 void ConfigureCylinders();

private:
 //-------------- Variables membres ------------ 
 int NR_, NS_;  // nb recepteurs, nb d'echantillons ds chaque paquet

 int PrtLev_;   // Niveau de print de debug

 double Da_; // distance entre recepteurs
 double freq0_;  // frequence de base (freqvrai=f+freq0)
 
 double timejitter_; // le sigma du jitter en temps des coups d'horloge
 double toffsig_;    // sigma des offsets en temps
 
 BRSourceGen* src_;  // Les sources
 bool adfg_; // if true, delete src 

 double signoise_; // sigma du bruit additif (bruit ampli ... )
 double gain_, siggain_;
 int ngainzero_;

 vector<MultiBeamCyl *> mCyl_;  // Les differents cylindres 

 TArray< complex<r_4> > cmplx_srcbox_; // boite3D des sources (angX,Y,freq) reconstruit
 };


#endif
