#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include "timing.h"
#include "ntuple.h"
#include "matharr.h"

#include "constcosmo.h"
#include "cosmocalc.h"
#include "schechter.h"
#include "geneutils.h"
#include "integfunc.h"
#include "genefluct3d.h"

void usage(void);
void usage(void)
{
 cout<<"cmvobserv3d [...options...]"<<endl
     <<" -a : init auto de l aleatoire"<<endl
     <<" -0 : use methode ComputeFourier0"<<endl
     <<" -G : compute Pk(z=0) and apply growth factor in real space"<<endl
     <<" -x nx,dx : taille en x (npix,Mpc)"<<endl
     <<" -y ny,dy : taille en y (npix,Mpc)"<<endl
     <<" -z nz,dz : taille en z (npix,Mpc)"<<endl
     <<" -Z zref : redshift median"<<endl
     <<" -s snoise : sigma du bruit en Msol"<<endl
     <<" -2 : compute 2D spectrum"<<endl
     <<" -M schmin,schmax,nsch : min,max mass and nb points for schechter HI"<<endl
     <<" -W : write cube in FITS format"<<endl
     <<" -P : write cube in PPF format"<<endl
     <<" -V : compute variance from real space"<<endl
     <<endl;
}

int main(int narg,char *arg[])
{
 InitTim();

 //-----------------------------------------------------------------
 // *** Survey definition
 long nx=360, ny=-1, nz=64; double dx=1., dy=-1., dz=-1.;
 //long nx=1000, ny=-1, nz=128; double dx=3., dy=-1., dz=6.;
 //long nx=1200, ny=-1, nz=128; double dx=1., dy=-1., dz=3;

 // *** Cosmography definition   (WMAP)
 unsigned short flat = 0;
 double ob0 = 0.0444356;
 double h100=0.71, om0=0.267804, or0=7.9e-05, ol0=0.73,w0=-1.;
 double zref = 0.5;
 double perc=0.01,dzinc=-1.,dzmax=5.; unsigned short glorder=4;

 // *** Spectrum and variance definition
 double ns = 1., as = 1.;
 double R=8./h100, Rg=R/sqrt(5.);
 double sigmaR = 1.;

 double kmin=1e-5,kmax=1000.;
 int npt = 10000;
 double lkmin=log10(kmin), lkmax=log10(kmax);
 double eps=1.e-3;

 // *** Schechter mass function definition
 double h75 = h100 / 0.75;
 double nstar = 0.006*pow(h75,3.);
 double mstar = pow(10.,9.8/(h75*h75));  // MSol
 double alpha = -1.37;

 double schmin=1e8, schmax=1e12;
 int schnpt = 1000;

 // *** Niveau de bruit
 double snoise= 0.;   // en equivalent MSol

 // *** type de generation
 bool computefourier0=false;
 bool use_growth_factor = false;
 unsigned short nthread=4;

 // *** What to do
 bool comp2dspec = false;
 bool wfits = false;
 bool wppf = false;
 bool compvarreal = false;

 // --- Decodage arguments

 char c;
 while((c = getopt(narg,arg,"ha0PWV2Gx:y:z:s:Z:M:")) != -1) {
  switch (c) {
  case 'a' :
    Auto_Ini_Ranf(5);
    break;
  case '0' :
    computefourier0 = true;
    break;
  case 'G' :
    use_growth_factor = true;
    break;
  case 'x' :
    sscanf(optarg,"%ld,%lf",&nx,&dx);
    break;
  case 'y' :
    sscanf(optarg,"%ld,%lf",&ny,&dy);
    break;
  case 'z' :
    sscanf(optarg,"%ld,%lf",&nz,&dz);
    break;
  case 's' :
    sscanf(optarg,"%lf",&snoise);
    break;
  case 'Z' :
    sscanf(optarg,"%lf",&zref);
    break;
  case '2' :
    comp2dspec = true;
    break;
  case 'M' :
    sscanf(optarg,"%lf,%lf,%d",&schmin,&schmax,&schnpt);
    break;
  case 'V' :
    compvarreal = true;
    break;
  case 'W' :
    wfits = true;
    break;
  case 'P' :
    wppf = true;
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 }

 double lschmin=log10(schmin), lschmax=log10(schmax), dlsch=(lschmax-lschmin)/schnpt;

 string tagobs = "cmvobserv3d.ppf";
 POutPersist posobs(tagobs);

 cout<<"zref="<<zref<<endl;
 cout<<"nx="<<nx<<" dx="<<dx<<" ny="<<ny<<" dy="<<dy<<" nz="<<nz<<" dz="<<dz<<endl;
 cout<<"kmin="<<kmin<<" ("<<lkmin<<"), kmax="<<kmax<<" ("<<lkmax<<") Mpc^-1"
     <<", npt="<<npt<<endl;
 cout<<"R="<<R<<" Rg="<<Rg<<" Mpc, sigmaR="<<sigmaR<<endl;
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpa="<<alpha<<endl;
 cout<<"schmin="<<schmin<<" ("<<lschmin
     <<"), schmax="<<schmax<<" ("<<lschmax<<") Msol"
     <<", schnpt="<<schnpt<<endl;
 cout<<"snoise="<<snoise<<" equivalent Msol"<<endl;

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Cosmology"<<endl;

 CosmoCalc univ(flat,true,zref+1.);
 univ.SetInteg(perc,dzinc,dzmax,glorder);
 univ.SetDynParam(h100,om0,or0,ol0,w0);
 univ.Print();
 double loscomref = univ.Dloscom(zref);
 cout<<"\nzref = "<<zref<<" -> dloscom = "<<loscomref<<" Mpc"<<endl;
 univ.Print(zref);

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Spectrum"<<endl;

 InitialSpectrum pkini(ns,as);

 TransfertEisenstein tf(h100,om0-ob0,ob0,T_CMB_Par,false);
 //tf.SetNoOscEnv(2);

 GrowthFactor growth(om0,ol0);
 // GrowthFactor growth(1.,0.); // D(z) = 1/(1+z)

 PkSpectrum0 pk0(pkini,tf);

 PkSpectrumZ pkz(pk0,growth,zref);
 
 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create mass function"<<endl;

 Schechter sch(nstar,mstar,alpha);
 sch.Print();

 //-----------------------------------------------------------------
 pkz.SetZ(0.);
 cout<<endl<<"\n--- Compute variance for top-hat R="<<R
     <<" at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_th(pkz,0);
 double kfind_th = varpk_th.FindMaximum(R,kmin,kmax,eps);
 double pkmax_th = varpk_th(kfind_th);
 cout<<"kfind_th = "<<kfind_th<<" ("<<log10(kfind_th)<<"), integrand="<<pkmax_th<<endl;
 double k1=kmin, k2=kmax;
 int rc = varpk_th.FindLimits(R,pkmax_th/1.e4,k1,k2,eps);
 cout<<"limit_th: rc="<<rc<<" : "<<k1<<" ("<<log10(k1)<<") , "
     <<k2<<" ("<<log10(k2)<<")"<<endl;

 double ldlk = (log10(k2)-log10(k1))/npt;
 varpk_th.SetInteg(0.01,ldlk,-1.,4);
 double sr2 = varpk_th.Variance(R,k1,k2);
 cout<<"varpk_th="<<sr2<<"  ->  sigma="<<sqrt(sr2)<<endl;

 double normpkz = sigmaR*sigmaR/sr2;
 pkz.SetScale(normpkz);
 cout<<"Spectrum normalisation = "<<pkz.GetScale()<<endl;

 pkz.SetZ(zref);

 Histo hpkz(lkmin,lkmax,npt); hpkz.ReCenterBin();
 FuncToHisto(pkz,hpkz,true);
 {
 tagobs = "hpkz"; posobs.PutObject(hpkz,tagobs);
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Compute variance for Pk at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_int(pkz,2);

 double kfind_int = varpk_int.FindMaximum(R,kmin,kmax,eps);
 double pkmax_int = varpk_int(kfind_int);
 cout<<"kfind_int = "<<kfind_int<<" ("<<log10(kfind_int)<<"), integrand="<<pkmax_int<<endl;
 double k1int=kmin, k2int=kmax;
 int rcint = varpk_int.FindLimits(R,pkmax_int/1.e4,k1int,k2int,eps);
 cout<<"limit_int: rc="<<rcint<<" : "<<k1int<<" ("<<log10(k1int)<<") , "
     <<k2int<<" ("<<log10(k2int)<<")"<<endl;

 double ldlkint = (log10(k2int)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int = varpk_int.Variance(R,k1int,k2int);
 cout<<"varpk_int="<<sr2int<<"  ->  sigma="<<sqrt(sr2int)<<endl;

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Compute galaxy density number"<<endl;

 sch.SetOutValue(0);
 cout<<"sch(mstar) = "<<sch(mstar)<<" /Mpc^3/Msol"<<endl;
 double ngal_by_mpc3 = IntegrateFuncLog(sch,lschmin,lschmax,0.01,dlsch,10.*dlsch,4);
 cout<<"Galaxy density number = "<<ngal_by_mpc3<<" /Mpc^3 between limits"<<endl;

 sch.SetOutValue(1);
 cout<<"mstar*sch(mstar) = "<<sch(mstar)<<" Msol/Mpc^3/Msol"<<endl;
 double mass_by_mpc3 = IntegrateFuncLog(sch,lschmin,lschmax,0.01,dlsch,10.*dlsch,4);
 cout<<"Galaxy mass density= "<<mass_by_mpc3<<" Msol/Mpc^3 between limits"<<endl;
 cout<<"Omega_HI at z=0 is "<<mass_by_mpc3/(univ.Rhoc(0.)*GCm3toMsolMpc3_Cst)<<endl
     <<"         at z="<<zref<<" is "<<mass_by_mpc3/(univ.Rhoc(zref)*GCm3toMsolMpc3_Cst)<<endl;

 PrtTim(">>>> End of definition");

 //-----------------------------------------------------------------
 // FFTW3 (p26): faster if sizes 2^a 3^b 5^c 7^d 11^e 13^f  with e+f=0 ou 1
 cout<<endl<<"\n--- Initialisation de GeneFluct3D"<<endl;

 TArray< complex<r_8> > pkgen;
 GeneFluct3D fluct3d(pkgen);
 fluct3d.SetPrtLevel(2);
 fluct3d.SetNThread(nthread);
 fluct3d.SetSize(nx,ny,nz,dx,dy,dz);
 fluct3d.SetObservator(zref,nz/2.);
 fluct3d.SetCosmology(univ);
 fluct3d.SetGrowthFactor(growth);
 fluct3d.LosComRedshift(0.001);
 TArray<r_8>& rgen = fluct3d.GetRealArray();
 cout<<endl; fluct3d.Print();

 double dkmin = fluct3d.GetKincMin();
 double knyqmax = fluct3d.GetKmax();
 long nherr = long(knyqmax/dkmin+0.5);
 cout<<"For HistoErr: d="<<dkmin<<" max="<<knyqmax<<" n="<<nherr<<endl;

 double dktmin = fluct3d.GetKTincMin();
 double ktnyqmax = fluct3d.GetKTmax();
 long nherrt = long(ktnyqmax/dktmin+0.5);
 double dkzmin = fluct3d.GetKinc()[2];
 double kznyqmax = fluct3d.GetKnyq()[2];
 long nherrz = long(kznyqmax/dkzmin+0.5);
 cout<<"For Histo2DErr: d="<<dktmin<<","<<dkzmin
     <<" max="<<ktnyqmax<<","<<kznyqmax<<" n="<<nherrt<<","<<nherrz<<endl;

 //-----------------------------------------------------------------
 cout<<"\n--- Computing spectra variance up to Kmax at z="<<pkz.GetZ()<<endl;
 // En fait on travaille sur un cube inscrit dans la sphere de rayon kmax:
 // sphere: Vs = 4Pi/3 k^3 , cube inscrit (cote k*sqrt(2)): Vc = (k*sqrt(2))^3
 // Vc/Vs = 0.675   ->  keff = kmax * (0.675)^(1/3) = kmax * 0.877
 double knyqmax_mod = 0.877*knyqmax;
 ldlkint = (log10(knyqmax_mod)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int_kmax = varpk_int.Variance(R,k1int,knyqmax_mod);
 cout<<"varpk_int(<"<<knyqmax_mod<<")="<<sr2int_kmax<<"  ->  sigma="<<sqrt(sr2int_kmax)<<endl;

 PrtTim(">>>> End Initialisation de GeneFluct3D");

 //-----------------------------------------------------------------
 cout<<"\n--- Computing a realization in Fourier space"<<endl;
 if(use_growth_factor) pkz.SetZ(0.); else pkz.SetZ(zref);
 cout<<"Power spectrum set at redshift: "<<pkz.GetZ()<<endl;
 if(computefourier0) fluct3d.ComputeFourier0(pkz);
   else              fluct3d.ComputeFourier(pkz);
 PrtTim(">>>> End Computing a realization in Fourier space");

 if(1) {
   cout<<"\n--- Checking realization spectra"<<endl;
   HistoErr hpkgen(0.,knyqmax,nherr);
   hpkgen.ReCenterBin(); hpkgen.Zero();
   hpkgen.Show();
   fluct3d.ComputeSpectrum(hpkgen);
   {
   tagobs = "hpkgen"; posobs.PutObject(hpkgen,tagobs);
   }
   PrtTim(">>>> End Checking realization spectra");
 }

 if(comp2dspec) {
   cout<<"\n--- Checking realization 2D spectra"<<endl;
   Histo2DErr hpkgen2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkgen2.ReCenterBin(); hpkgen2.Zero();
   hpkgen2.Show();
   fluct3d.ComputeSpectrum2D(hpkgen2);
   {
   tagobs = "hpkgen2"; posobs.PutObject(hpkgen2,tagobs);
   }
   PrtTim(">>>> End Checking realization 2D spectra");
 }

 if(1) {
   cout<<"\n--- Computing convolution by pixel shape"<<endl;
   fluct3d.FilterByPixel();
   PrtTim(">>>> End Computing convolution by pixel shape");
 }

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_k0.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_k0.ppf",false);
   PrtTim(">>>> End WritePPF");
 }

 if(1) {
   cout<<"\n--- Checking realization spectra after pixel shape convol."<<endl;
   HistoErr hpkgenf(0.,knyqmax,nherr);
   hpkgenf.ReCenterBin(); hpkgenf.Zero();
   hpkgenf.Show();
   fluct3d.ComputeSpectrum(hpkgenf);
   {
   tagobs = "hpkgenf"; posobs.PutObject(hpkgenf,tagobs);
   }
   PrtTim(">>>> End Checking realization spectra");
 }

 if(comp2dspec) {
   cout<<"\n--- Checking realization 2D spectra after pixel shape convol."<<endl;
   Histo2DErr hpkgenf2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkgenf2.ReCenterBin(); hpkgenf2.Zero();
   hpkgenf2.Show();
   fluct3d.ComputeSpectrum2D(hpkgenf2);
   {
   tagobs = "hpkgenf2"; posobs.PutObject(hpkgenf2,tagobs);
   }
   PrtTim(">>>> End Checking realization 2D spectra");
 }

 //-----------------------------------------------------------------
 cout<<"\n--- Computing a realization in real space"<<endl;
 fluct3d.ComputeReal();
 double rmin,rmax; rgen.MinMax(rmin,rmax);
 cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
 PrtTim(">>>> End Computing a realization in real space");

 if(use_growth_factor) {
   cout<<"\n--- Apply Growth factor"<<endl;
   cout<<"...D(z=0)="<<growth(0.)<<"  D(z="<<zref<<")="<<growth(zref)<<endl;
   fluct3d.ApplyGrowthFactor(-1);
   rmin,rmax; rgen.MinMax(rmin,rmax);
   cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
   PrtTim(">>>> End Applying growth factor");
 }

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_r0.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_r0.ppf",true);
   PrtTim(">>>> End WritePPF");
 }

 int_8 nm;
 double rm,rs2;
 if(1) {
   cout<<"\n--- Check mean and variance in real space"<<endl;
   int_8 nlowone = fluct3d.NumberOfBad(-1.,1e+200);
   nm = fluct3d.MeanSigma2(rm,rs2);
   cout<<"rgen:("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
       <<rs2<<" -> "<<sqrt(rs2)<<",  n(<-1)="<<nlowone<<endl;
   PrtTim(">>>> End Check mean and variance in real space");
 }

 if(compvarreal) {
   cout<<"\n--- Check variance sigmaR in real space"<<endl;
   double varr;
   int_8 nvarr = fluct3d.VarianceFrReal(R,varr);
   cout<<"R="<<R<<" : sigmaR^2="<<varr<<" -> "<<sqrt(varr)<<",   n="<<nvarr<<endl;
   PrtTim(">>>> End Check variance sigmaR in real space");
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Converting fluctuations into mass"<<endl;
 fluct3d.TurnFluct2Mass();
 nm = fluct3d.MeanSigma2(rm,rs2);
 cout<<"1+rgen: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 PrtTim(">>>> End Converting fluctuations into mass");

 cout<<"\n--- Converting mass into galaxy number"<<endl;
 rm = fluct3d.TurnMass2MeanNumber(ngal_by_mpc3);
 cout<<rm<<" galaxies put into survey"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,0.);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 PrtTim(">>>> End Converting mass into galaxy number");

 cout<<"\n--- Set negative pixels to BAD"<<endl;
 nm = fluct3d.SetToVal(0.,1e+200,-999.);
 cout<<nm<<" negative in survey set to BAD"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,-998.);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 PrtTim(">>>> End Set negative pixels to BAD etc...");

 cout<<"\n--- Apply poisson on galaxy number"<<endl;
 nm = fluct3d.ApplyPoisson();
 cout<<nm<<" galaxies into survey after poisson"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,-998.);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 PrtTim(">>>> End Apply poisson on galaxy number");

 cout<<"\n--- Convert Galaxy number to HI mass"<<endl;
 long nhmdndm = long( (lschmax-lschmin+1.)*100. + 0.5);
 Histo hmdndm(lschmin,lschmax,nhmdndm);
 sch.SetOutValue(1);
 FuncToHisto(sch,hmdndm,true);
 FunRan tirhmdndm(hmdndm,true);
 {
 tagobs = "hmdndm"; posobs.PutObject(hmdndm,tagobs);
 Histo hdum1(tirhmdndm);
 tagobs = "tirhmdndm"; posobs.PutObject(hdum1,tagobs);
 }
 double mhi = fluct3d.TurnNGal2Mass(tirhmdndm,true);
 cout<<mhi<<" MSol in survey / "<<mass_by_mpc3*fluct3d.GetVol()<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2,0.);
 cout<<"HI mass: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 cout<<"Equivalent: "<<rm*nm/fluct3d.NPix()<<" Msol / pixels"<<endl;
 PrtTim(">>>> End Convert Galaxy number to HI mass");

 cout<<"\n--- Set BAD pixels to Zero"<<endl;
 nm = fluct3d.SetToVal(-998.,1e+200,0.);
 cout<<nm<<" BAD in survey set to zero"<<endl;
 nm = fluct3d.MeanSigma2(rm,rs2);
 cout<<"galaxy: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
     <<rs2<<" -> "<<sqrt(rs2)<<endl;
 PrtTim(">>>> End Set BAD pixels to Zero etc...");

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_r.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_r.ppf",true);
   PrtTim(">>>> End WritePPF");
 }

 if(snoise>0.) {
   cout<<"\n--- Add noise to HI Flux snoise="<<snoise<<endl;
   fluct3d.AddNoise2Real(snoise);
   nm = fluct3d.MeanSigma2(rm,rs2);
   cout<<"HI mass with noise: ("<<nm<<") Mean = "<<rm<<", Sigma^2 = "
       <<rs2<<" -> "<<sqrt(rs2)<<endl;
   PrtTim(">>>> End Add noise");
 }

 //-----------------------------------------------------------------
 // -- NE PAS FAIRE CA SI ON VEUT CONTINUER LA SIMULATION -> d_rho/rho ecrase
 
 if(1) {
   cout<<endl<<"\n--- ReComputing spectrum from real space"<<endl;
   fluct3d.ReComputeFourier();
   PrtTim(">>>> End ReComputing spectrum");
 }

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_k.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_k.ppf",false);
   PrtTim(">>>> End WritePPF");
 }

 if(1) {
   cout<<endl<<"\n--- Computing final spectrum"<<endl;
   HistoErr hpkrec(0.,knyqmax,nherr);
   hpkrec.ReCenterBin();
   hpkrec.Show();
   fluct3d.ComputeSpectrum(hpkrec);
   tagobs = "hpkrec"; posobs.PutObject(hpkrec,tagobs);
   PrtTim(">>>> End Computing final spectrum");
 }

 if(comp2dspec) {
   cout<<"\n--- Computing final 2D spectrum"<<endl;
   Histo2DErr hpkrec2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkrec2.ReCenterBin(); hpkrec2.Zero();
   hpkrec2.Show();
   fluct3d.ComputeSpectrum2D(hpkrec2);
   {
   tagobs = "hpkrec2"; posobs.PutObject(hpkrec2,tagobs);
   }
   PrtTim(">>>> End Computing final 2D spectrum");
 }

 PrtTim(">>>> End Of Job");
 return 0;
}

/*
######################################################
readfits cmvobserv3d_k0.fits
readfits cmvobserv3d_k.fits
readfits cmvobserv3d_r0.fits
readfits cmvobserv3d_r.fits

openppf cmvobserv3d_k0.ppf
openppf cmvobserv3d_k.ppf
openppf cmvobserv3d_r0.ppf
openppf cmvobserv3d_r.ppf

# pour le plot 2D d'une slice en Z du 3D: to2d nom_obj3D num_slice
defscript to2d
 objaoper $1 sliceyz $2
 mv sliceyz_${2} ${1}_$2
 disp ${1}_$2
 echo display slice $2 of $1
endscript 

to2d $cobj 0

######################################################
openppf cmvobserv3d.ppf

zone
set k pow(10.,x)
n/plot hpkz.val*$k*$k/(2*M_PI*M_PI)%x ! "connectpoints"

zone
n/plot hpkz.val%x ! ! "nsta connectpoints"
n/plot hpkgen.val%log10(x) x>0 ! "nsta same red connectpoints"
n/plot hpkgenf.val%log10(x) x>0 ! "nsta same orange connectpoints"
n/plot hpkrec.val%log10(x) x>0 ! "nsta same blue connectpoints"

disp hpkgen "hbincont err"
disp hpkgenf "hbincont err"
disp hpkrec "hbincont err"

zone 2 2
imag hpkgen2
imag hpkgenf2
imag hpkrec2

zone 2 1
disp hmdndm "nsta"
disp tirhmdndm "nsta"
addline 0 1 20 1 "red"

 */
