#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include "constcosmo.h"
#include "cosmocalc.h"
#include "geneutils.h"
#include "schechter.h"
#include "planckspectra.h"

/* --- Check Peterson at al. astro-ph/0606104 v1
cmvdefsurv -z 0.0025 -x 1 -U 0.75,0.3,0.7,-1,1 -V 300 -O 400000,6000 -N 75 -M 6.156e9 -F 3 -2 1.5
 --- */

inline double rad2deg(double trad) {return trad/M_PI*180.;}
inline double rad2min(double trad) {return trad/M_PI*180.*60.;}
inline double rad2sec(double trad) {return trad/M_PI*180.*3600.;}
inline double deg2rad(double tdeg) {return tdeg*M_PI/180.;}
inline double min2rad(double tmin) {return tmin*M_PI/(180.*60.);}
inline double sec2rad(double tsec) {return tsec*M_PI/(180.*3600.);}

void usage(void);
void usage(void) {
  cout<<"cmvdefsurv [-r] -x adtx,atxlarg [-y adty,atylarg] -z dred,redlarg redshift"<<endl
      <<" -x adtx,atxlarg : resolution en Theta_x (arcmin), largeur (degre)"<<endl
      <<" -y adty,atylarg : idem selon y, si <=0 meme que x"<<endl
      <<" -z dred,redlarg : resolution en redshift, largeur en redshift"<<endl
      <<" -P : on donne -x -y -z en Mpc au lieu d\'angles et de redshift"<<endl
      <<" -L lobewidth : taille du lobe d\'observation (FWHM) en arcmin (def= 1\')"<<endl
      <<" -O surf,tobs : surface effective (m^2) et temps d\'observation (s)"<<endl
      <<" -N Tsys : temperature du system (K)"<<endl
      <<" -S Tsynch,indnu : temperature (K) synch a 408 Mhz, index d\'evolution"<<endl
      <<"                   (indnu==0 no evolution with freq.)"<<endl
      <<" -M  : masse de HI de reference (MSol), si <=0 mean schechter in pixel"<<endl
      <<" -F  : HI flux factor to be applied for our redshift"<<endl
      <<" -V Vrot : largeur en vitesse (km/s) pour l\'elargissement doppler (def=300km/s)"<<endl
      <<" -U h100,om0,ol0,w0,or0,flat : cosmology"<<endl
      <<" -2 : two polarisations measured"<<endl
      <<" -A <log10(S_agn)> : moyenne du flux AGN en Jy dans le pixel"<<endl
      <<" redshift : redshift moyen du survey"<<endl
      <<endl;
}

int main(int narg,char *arg[])
{
 // --- Valeurs fixes
 // WMAP
 unsigned short flat = 0;
 double h100=0.71, om0=0.267804, or0=7.9e-05*0., ol0=0.73,w0=-1.;
 // Schechter
 double h75 = h100 / 0.75;
 double nstar = 0.006*pow(h75,3.);  //
 double mstar = pow(10.,9.8/(h75*h75));  // MSol
 double alpha = -1.37;
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;

 // --- Arguments
 bool inmpc = false;
 double adtx=1., atxlarg=90., adty=-1., atylarg=-1.;
 double dx=1.,txlarg=1000., dy=-1.,tylarg=1000., dz=1.,tzlarg=100.;
 int nx,ny,nz;
 double redshift = 1., dred=0.01, redlarg=0.3;
 double tobs = 6000., surfeff = 400000.;
 double lobewidth = 1.;  // taille du lobe d'observation en arcmin
 double Tsys=75.;
 // a 408 MHz (Haslam) + evol index a -2.6
 double Tsynch408=60., nuhaslam=0.408, indnu = -2.6;
 double mhiref = -1.; // reference Mass en HI (def integ schechter)
 double hifactor = 1.;
 double vrot = 300.; // largeur en vitesse (km/s) pour elargissement doppler
 double facpolar = 0.5; // si on ne mesure les 2 polars -> 1.0
 double lflux_agn = -3.;

 // --- Decodage arguments
 char c;
  while((c = getopt(narg,arg,"hP2x:y:z:N:S:O:M:F:V:U:L:A:")) != -1) {
  switch (c) {
  case 'P' :
    inmpc = true;
    break;
  case 'x' :
    sscanf(optarg,"%lf,%lf",&adtx,&atxlarg);
    break;
  case 'y' :
    sscanf(optarg,"%lf,%lf",&adty,&atylarg);
    break;
  case 'z' :
    sscanf(optarg,"%lf,%lf",&dred,&redlarg);
    break;
  case 'O' :
    sscanf(optarg,"%lf,%lf",&surfeff,&tobs);
    break;
  case 'L' :
    sscanf(optarg,"%lf",&lobewidth);
    break;
  case 'N' :
    sscanf(optarg,"%lf",&Tsys);
    break;
  case 'S' :
    sscanf(optarg,"%lf,%lf",&Tsynch408,&indnu);
    break;
  case 'M' :
    sscanf(optarg,"%lf",&mhiref);
    break;
  case 'F' :
    sscanf(optarg,"%lf",&hifactor);
    break;
  case 'V' :
    sscanf(optarg,"%lf",&vrot);
    break;
  case 'U' :
    sscanf(optarg,"%lf,%lf,%lf,%lf,%u",&h100,&om0,&ol0,&w0,&flat);
    break;
  case '2' :
    facpolar = 1.0;
    break;
  case 'A' :
    sscanf(optarg,"%lf",&lflux_agn);
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 } 
 if(optind>=narg) {usage(); return-1;}
 sscanf(arg[optind],"%lf",&redshift);
 if(redshift<=0.) {cout<<"Redshift "<<redshift<<" should be >0"<<endl; return -2;}

 // --- Initialisation de la Cosmologie
 cout<<"\nh100="<<h100<<" Om0="<<om0<<" Or0="<<or0<<" Or0="
     <<or0<<" Ol0="<<ol0<<" w0="<<w0<<" flat="<<flat<<endl;
 cout<<"\n--- Cosmology for z = "<<redshift<<endl;
 CosmoCalc univ(flat,true,2.*redshift);
 double perc=0.01,dzinc=redshift/100.,dzmax=dzinc*10.; unsigned short glorder=4;
 univ.SetInteg(perc,dzinc,dzmax,glorder);
 univ.SetDynParam(h100,om0,or0,ol0,w0);
 univ.Print(0.);
 univ.Print(redshift);

 double dang    = univ.Dang(redshift);
 double dtrcom  = univ.Dtrcom(redshift);
 double dlum    = univ.Dlum(redshift);
 double dloscom = univ.Dloscom(redshift);
 double dlosdz  = univ.Dhubble()/univ.E(redshift);
 cout<<"dang="<<dang<<" dlum="<<dlum<<" dtrcom="<<dtrcom
     <<" dloscom="<<dloscom<<" dlosdz="<<dlosdz<<" Mpc"<<endl;

 cout<<"\n1\" -> "<<dang*sec2rad(1.)<<" Mpc = "<<dtrcom*sec2rad(1.)<<" Mpc com"<<endl;
 cout<<"1\' -> "<<dang*min2rad(1.)<<" Mpc = "<<dtrcom*min2rad(1.)<<" Mpc com"<<endl;
 cout<<"1d -> "<<dang*deg2rad(1.)<<" Mpc = "<<dtrcom*deg2rad(1.)<<" Mpc com"<<endl;

 cout<<"dz=1 -> "<<dlosdz<<" Mpc com"<<endl;

 cout<<"1 Mpc los com -> dz = "<<1./dlosdz<<endl;
 cout<<"1 Mpc transv com -> "<<rad2sec(1./dtrcom)<<"\" = "
     <<rad2min(1./dtrcom)<<" \' = "<<rad2deg(1./dtrcom)<<" d"<<endl;

 // --- Mise en forme dans les unites appropriees
 if(adty<=0.) adty=adtx;
 if(atylarg<=0.) atylarg=atxlarg;
 if(inmpc) {
   dx = adtx; txlarg = atxlarg; nx = int(txlarg/dx+0.5);
   dy = adty; txlarg = atxlarg; ny = int(tylarg/dy+0.5);
   dz = dred; tzlarg = redlarg; nz = int(tzlarg/dz+0.5);
   adtx = dx/dtrcom; atxlarg = adtx*nx;
   adty = dy/dtrcom; atylarg = adty*ny;
   dred = dz/dlosdz; redlarg = dred*nz;
 } else {
   adtx = min2rad(adtx); atxlarg = deg2rad(atxlarg); nx = int(atxlarg/adtx+0.5);
   adty = min2rad(adty); atylarg = deg2rad(atylarg); ny = int(atylarg/adty+0.5);
   nz = int(redlarg/dred+0.5);
   dx = adtx*dtrcom; txlarg = dx*nx;
   dy = adty*dtrcom; tylarg = dy*ny;
   dz = dred*dlosdz; tzlarg = dz*nz;
 }
 double Npix = (double)nx*(double)ny*(double)nz;

 double redlim[2] = {redshift-redlarg/2.,redshift+redlarg/2.};
 if(redlim[0]<=0.)
   {cout<<"Lower redshift limit "<<redlim[0]<<" should be >0"<<endl; return -3;}
 double dtrlim[2] = {univ.Dtrcom(redlim[0]),univ.Dtrcom(redlim[1])};
 double loslim[2] = {univ.Dloscom(redlim[0]), univ.Dloscom(redlim[1])};
 double dlumlim[2] = {univ.Dlum(redlim[0]),univ.Dlum(redlim[1])};

 cout<<"\n---- Type de donnees: inmpc = "<<inmpc<<endl;
 cout<<"---- Line of Sight: Redshift = "<<redshift<<endl
     <<"dred = "<<dred<<"  redlarg = "<<redlarg<<endl
     <<"  dz = "<<dz<<" Mpc  redlarg = "<<tzlarg<<" Mpc, nz = "<<nz<<" pix"<<endl;
 cout<<"---- Transverse X:"<<endl
     <<"adtx = "<<rad2min(adtx)<<"\',  atxlarg = "<<rad2deg(atxlarg)<<" d"<<endl
     <<"  dx = "<<dx<<" Mpc,  txlarg = "<<txlarg<<" Mpc, nx = "<<nx<<" pix"<<endl;
 cout<<"---- Transverse Y:"<<endl
     <<"adty = "<<rad2min(adty)<<"\',  atylarg = "<<rad2deg(atylarg)<<" d"<<endl
     <<"  dy = "<<dy<<" Mpc,  tylarg = "<<tylarg<<" Mpc, ny = "<<ny<<" pix"<<endl;
 cout<<"---- Npix total = "<<Npix<<" -> "<<Npix*sizeof(double)/1.e6<<" Mo"<<endl;

 // --- Cosmolographie Transverse
 cout<<"\n--- Transverse"<<endl;
 cout<<"dang comoving = "<<dtrcom<<" Mpc (com) var_in_z ["
                         <<dtrlim[0]<<","<<dtrlim[1]<<"]"<<endl;

 cout<<"... dx = "<<dx<<" Mpc (com), with angle "<<adtx*dtrcom<<endl
     <<"       with angle var_in_z ["<<adtx*dtrlim[0]<<","<<adtx*dtrlim[1]<<"]"<<endl;
 cout<<"... largx = "<<txlarg<<" Mpc (com), with angle "<<atxlarg*dtrcom<<endl
     <<"          with angle var_in_z ["<<atxlarg*dtrlim[0]<<","<<atxlarg*dtrlim[1]<<"]"<<endl;

 cout<<"... dy = "<<dy<<" Mpc (com), with angle "<<adty*dtrcom<<endl
     <<"       with angle var_in_z ["<<adty*dtrlim[0]<<","<<adty*dtrlim[1]<<"]"<<endl;
 cout<<"... largy = "<<tylarg<<" Mpc (com), with angle "<<atylarg*dtrcom<<endl
     <<"          with angle var_in_z ["<<atylarg*dtrlim[0]<<","<<atylarg*dtrlim[1]<<"]"<<endl;

 // --- Cosmolographie Line of sight
 cout<<"\n--- Line of Sight"<<endl;
 cout<<"los comoving distance = "<<dloscom<<" Mpc (com) in ["
     <<loslim[0]<<","<<loslim[1]<<"]"<<endl
     <<"                                    diff = "
     <<loslim[1]-loslim[0]<<" Mpc"<<endl;

 cout<<"...dz = "<<dz<<" Mpc (com), with redshift approx "<<dred*dlosdz<<endl;
 cout<<"...tzlarg = "<<tzlarg<<" Mpc (com), with redshift approx "<<redlarg*dlosdz<<endl;

 // --- Solid Angle & Volume
 cout<<"\n--- Solid angle"<<endl;
 double angsol = AngSol(adtx/2.,adty/2.,M_PI/2.);
 cout<<"Elementary solid angle = "<<angsol<<" sr = "<<angsol/(4.*M_PI)<<" *4Pi sr"<<endl;
 double angsoltot = AngSol(atxlarg/2.,atylarg/2.,M_PI/2.);
 cout<<"Total solid angle = "<<angsoltot<<" sr = "<<angsoltot/(4.*M_PI)<<" *4Pi sr"<<endl;

 cout<<"\n--- Volume"<<endl;
 double dvol = dx*dy*dz;
 cout<<"Pixel volume comoving = "<<dvol<<" Mpc^3"<<endl;
 double vol = univ.Vol4Pi(redlim[0],redlim[1])/(4.*M_PI)*angsoltot;
 cout<<"Volume comoving = "<<vol<<" Mpc^3 = "<<vol/1.e9<<" Gpc^3"<<endl
     <<"Pixel volume comoving = vol/Npix = "<<vol/Npix<<" Mpc^3"<<endl;

 // --- Fourier space: k = omega = 2*Pi*Nu
 cout<<"\n--- Fourier space"<<endl;
 cout<<"Array size: nx = "<<nx<<",  ny = "<<ny<<",  nz = "<<nz<<endl;
 double dk_x = 2.*M_PI/(nx*dx), knyq_x = M_PI/dx;
 double dk_y = 2.*M_PI/(nx*dy), knyq_y = M_PI/dy;
 double dk_z = 2.*M_PI/(nz*dz), knyq_z = M_PI/dz;
 cout<<"Resolution: dk_x = "<<dk_x<<" Mpc^-1  (2Pi/dk_x="<<2.*M_PI/dk_x<<" Mpc)"<<endl
     <<"            dk_y = "<<dk_y<<" Mpc^-1  (2Pi/dk_y="<<2.*M_PI/dk_y<<" Mpc)"<<endl;
 cout<<"Nyquist:    kx = "<<knyq_x<<" Mpc^-1  (2Pi/knyq_x="<<2.*M_PI/knyq_x<<" Mpc)"<<endl
     <<"            ky = "<<knyq_y<<" Mpc^-1  (2Pi/knyq_y="<<2.*M_PI/knyq_y<<" Mpc)"<<endl;
 cout<<"Resolution: dk_z = "<<dk_z<<" Mpc^-1  (2Pi/dk_z="<<2.*M_PI/dk_z<<" Mpc)"<<endl;
 cout<<"Nyquist:    kz = "<<knyq_z<<" Mpc^-1  (2Pi/knyq_z="<<2.*M_PI/knyq_z<<" Mpc)"<<endl;

 // --- Masse de HI
 cout<<"\n--- Mass HI"<<endl;
 Schechter sch(nstar,mstar,alpha);
 sch.SetOutValue(1);
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;
 cout<<"mstar*sch(mstar) = "<<sch(mstar)<<" Msol/Mpc^3/Msol"<<endl;
 int npt = 10000;
 double lnx1=log10(1e-6), lnx2=log10(1e+14), dlnx=(lnx2-lnx1)/npt;
 double masshimpc3 = IntegrateFuncLog(sch,lnx1,lnx2,0.001,dlnx,10.*dlnx,6);
 cout<<"Mass density: "<<masshimpc3<<" Msol/Mpc^3"<<endl;

 double masshipix = masshimpc3*dvol;
 double masshitot = masshimpc3*vol;
 cout<<"Pixel mass = "<<masshipix<<" Msol"<<endl
     <<"Total mass in survey = "<<masshitot<<" Msol"<<endl;
 if(mhiref<=0.)  mhiref = masshipix;

 // --- Survey values
 double unplusz = 1.+redshift;
 double nuhiz = Fr_HyperFin_Par / unplusz;  // GHz
 // dnu = NuHi/(1.+z0-dz/2) - NuHi/(1.+z0+dz/2)
 //     = NuHi*dz/(1.+z0)^2 * 1/[1-(dz/(2*(1+z0)))^2]
 double dnuhiz = Fr_HyperFin_Par *dred/(unplusz*unplusz)
               / (1.- (dred/.2/unplusz)*(dred/.2/unplusz));
 cout<<"\n--- Observation:"<<endl
     <<"      surf_eff="<<surfeff<<" m^2, tobs="<<tobs<<" s"<<endl
     <<"      nu="<<nuhiz<<" GHz, dnu="<<dnuhiz*1.e3<<" Mhz"<<endl;
 cout<<"dang lumi = "<<dlum<<" in ["<<dlumlim[0]<<","<<dlumlim[1]<<"] Mpc"<<endl;

 double slobe = lobewidth/2.35482; // sigma du lobe en arcmin
 double lobecyl = sqrt(8.)*slobe; // diametre du lobe cylindrique equiv en arcmin
 double lobearea = M_PI*lobecyl*lobecyl/4.;  // en arcmin^2 (hypothese lobe gaussien)
 double nlobes = rad2min(adtx)*rad2min(adty)/lobearea;
 if(lobewidth<=0.) nlobes = 1.;
 cout<<"\nBeam FWHM = "<<lobewidth<<"\' -> sigma = "<<slobe<<"\' -> "
     <<" Dcyl = "<<lobecyl<<"\' -> area = "<<lobearea<<" arcmin^2"<<endl;
 cout<<"Number of beams in one transversal pixel = "<<nlobes<<endl;

 // --- Power emitted by HI
 cout<<"\n--- Power from HI for M = "<<mhiref<<" Msol at "<<nuhiz<<" GHz"<<endl;
 cout<<"flux factor = "<<hifactor<<" at redshift = "<<redshift<<endl;

 double fhi = hifactor*Msol2FluxHI(mhiref,dlum);
 cout<<"FluxHI("<<dlum<<" Mpc) all polar:"<<endl
     <<"  Flux= "<<fhi<<" W/m^2 = "<<fhi/Jansky2Watt_cst<<" Jy.Hz"<<endl
     <<"      in ["<<hifactor*Msol2FluxHI(mhiref,dlumlim[0])
     <<","<<hifactor*Msol2FluxHI(mhiref,dlumlim[1])<<"] W/m^2"<<endl;
 double sfhi = fhi / (dnuhiz*1e9) / Jansky2Watt_cst;
 cout<<"If spread over "<<dnuhiz<<" GHz, flux density = "<<sfhi<<" Jy"<<endl;

 // --- Signal analysis
 cout<<"\n--- Signal analysis"<<endl;
 cout<<"Facteur polar = "<<facpolar<<endl;

 PlanckSpectra planck(T_CMB_Par);
 planck.SetApprox(1);  // Rayleigh spectra
 planck.SetVar(0); // frequency
 planck.SetUnitOut(0); // output en W/....
 planck.SetTypSpectra(0); // radiance W/m^2/Sr/Hz

 // Signal
 double psig = facpolar * fhi * surfeff;
 double tsig = psig / k_Boltzman_Cst / (dnuhiz*1e9);
 double ssig = psig / surfeff / (dnuhiz*1e9) / Jansky2Watt_cst;
 cout<<"Signal("<<mhiref<<" Msol): P="<<psig<<" W"<<endl;
 cout<<"     flux density = "<<ssig<<" Jy  (for Dnu="<<dnuhiz<<" GHz)"<<endl;
 cout<<"     Antenna temperature: tsig="<<tsig<<" K"<<endl;

 // Elargissement doppler de la raie a 21cm: dNu = vrot/C * Nu(21cm) / (1+z)
 double doplarge = vrot / SpeedOfLight_Cst * nuhiz;
 cout<<"     Doppler width="<<doplarge*1.e3<<" MHz  for rotation width of "<<vrot<<" km/s"<<endl;
 if(doplarge>dnuhiz) {
   cout<<"Warning: doppler width "<<doplarge<<" GHz > "<<dnuhiz<<" GHz redshift bin width"<<endl;
 }

 // Synchrotron
 double tsynch = Tsynch408;
        if(fabs(indnu)>1.e-50) tsynch *= pow(nuhiz/nuhaslam,indnu);
 planck.SetTemperature(tsynch);
 double psynch = facpolar * planck(nuhiz*1.e+9) * surfeff * angsol * (dnuhiz*1e9);
 double ssynch = psynch / surfeff / (dnuhiz*1e9) / Jansky2Watt_cst;
 cout<<"Synchrotron: T="<<Tsynch408<<" K ("<<nuhaslam<<" GHz), "
     <<tsynch<<" K ("<<nuhiz<<" GHz)"<<endl
     <<"             P="<<psynch<<" W"<<endl;
 cout<<"             flux density = "<<ssynch<<" Jy"<<endl;

 // CMB
 double tcmb = T_CMB_Par;
 planck.SetTemperature(tcmb);
 double pcmb = facpolar * planck(nuhiz*1.e+9) * surfeff * angsol * (dnuhiz*1e9);
 double scmb = pcmb / surfeff / (dnuhiz*1.e+9) / Jansky2Watt_cst;
 cout<<"CMB: T="<<tcmb<<" K -> P="<<pcmb<<" W"<<endl;
 cout<<"                       flux density = "<<scmb<<" Jy"<<endl;

 // AGN
 double flux_agn = pow(10.,lflux_agn);
 double mass_agn = FluxHI2Msol(flux_agn*Jansky2Watt_cst,dlum);
 cout<<"AGN: log10(S_agn)="<<lflux_agn<<" -> S_agn="
     <<flux_agn<<" Jy -> "<<mass_agn<<" equiv. Msol/Hz"<<endl;
 double flux_agn_pix = flux_agn*(dnuhiz*1e9);
 double mass_agn_pix = FluxHI2Msol(flux_agn_pix*Jansky2Watt_cst,dlum);
 double lmass_agn_pix = log10(mass_agn_pix);
 cout<<"...pixel: f="<<flux_agn_pix<<" 10^-26 W/m^2"
     <<" -> "<<mass_agn_pix<<" Msol -> log10 = "<<lmass_agn_pix<<endl;

 // --- Noise analysis
 cout<<"\n--- Noise analysis"<<endl;
 double psys = k_Boltzman_Cst * Tsys * (dnuhiz*1.e+9);
 cout<<"Noise: T="<<Tsys<<" K, P="<<psys<<" W  (for Dnu="<<dnuhiz<<" GHz)"<<endl;

 double slim = 2. * k_Boltzman_Cst * Tsys / surfeff
               / sqrt(2.*(dnuhiz*1.e+9)*tobs) /Jansky2Watt_cst;
 cout<<"Observation flux density limit: "<<slim<<" Jy (in 1 lobe)"<<endl;

 double slim_nl = slim * sqrt(nlobes);
 cout<<"Observation flux density limit: "<<slim_nl<<" Jy (in "<<nlobes<<" lobes)"<<endl;

 double SsN = ssig/slim;
 cout<<"\nSignal to noise ratio = "<<SsN<<" (1 lobe)"<<endl;
 double SsN_nl = ssig/slim_nl;
 cout<<"\nSignal to noise ratio = "<<SsN_nl<<" ("<<nlobes<<" lobes)"<<endl;

 double smass = mhiref/ssig*slim;
 cout<<"\nSigma noise equivalent = "<<smass<<" Msol"<<endl;

 return 0;
}
