#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include "tarray.h"
#include "pexceptions.h"
#include "perandom.h"
#include "srandgen.h"

#include "fabtcolread.h"
#include "fabtwriter.h"
#include "fioarr.h"

#include "arrctcast.h"

#include "constcosmo.h"
#include "geneutils.h"
#include "schechter.h"

#include "genefluct3d.h"

//#define FFTW_THREAD

#define MODULE2(_x_) ((double)((_x_).real()*(_x_).real() + (_x_).imag()*(_x_).imag()))

//-------------------------------------------------------
GeneFluct3D::GeneFluct3D(TArray< complex<r_8 > >& T)
  : T_(T) , Nx_(0) , Ny_(0) , Nz_(0) , array_allocated_(false) , lp_(0)
  , redshref_(-999.) , kredshref_(0.) , cosmo_(NULL) , growth_(NULL)
  , loscom_ref_(-999.), loscom_min_(-999.), loscom_max_(-999.)
  , loscom2zred_min_(0.) , loscom2zred_max_(0.)
{
 xobs_[0] = xobs_[1] = xobs_[2] = 0.;
 zred_.resize(0);
 loscom_.resize(0);
 loscom2zred_.resize(0);
 SetNThread();
}

GeneFluct3D::~GeneFluct3D(void)
{
 fftw_destroy_plan(pf_);
 fftw_destroy_plan(pb_);
#ifdef FFTW_THREAD
 if(nthread_>0) fftw_cleanup_threads();
#endif
}

//-------------------------------------------------------
void GeneFluct3D::SetSize(long nx,long ny,long nz,double dx,double dy,double dz)
{
  setsize(nx,ny,nz,dx,dy,dz);
  setalloc();
  setpointers(false);
  init_fftw();
}

void GeneFluct3D::SetObservator(double redshref,double kredshref)
// L'observateur est au redshift z=0
//               est situe sur la "perpendiculaire" a la face x,y
//                         issue du centre de cette face
// Il faut positionner le cube sur l'axe des z cad des redshifts:
//     redshref  = redshift de reference
//                 Si redshref<0 alors redshref=0
//     kredshref = indice (en double) correspondant a ce redshift
//                 Si kredshref<0 alors kredshref=0
// Exemple: redshref=1.5 kredshref=250.75
//    -> Le pixel i=nx/2 j=ny/2 k=250.75 est au redshift 1.5
{
 if(redshref<0.) redshref = 0.;
 if(kredshref<0.) kredshref = 0.;
 redshref_  = redshref;
 kredshref_ = kredshref;
 if(lp_>0)
   cout<<"--- GeneFluct3D::SetObservator zref="<<redshref_<<" kref="<<kredshref_<<endl;
}

void GeneFluct3D::SetCosmology(CosmoCalc& cosmo)
{
 cosmo_ = &cosmo;
 if(lp_>1) cosmo_->Print();
}

void GeneFluct3D::SetGrowthFactor(GrowthFactor& growth)
{
 growth_ = &growth;
}

void GeneFluct3D::setsize(long nx,long ny,long nz,double dx,double dy,double dz)
{
 if(lp_>1) cout<<"--- GeneFluct3D::setsize: N="<<nx<<","<<ny<<","<<nz
                <<" D="<<dx<<","<<dy<<","<<dz<<endl;
 if(nx<=0 || dx<=0.) {
   char *bla = "GeneFluct3D::setsize_Error: bad value(s";
   cout<<bla<<endl; throw ParmError(bla);
 }

 // Les tailles des tableaux
 Nx_ = nx;
 Ny_ = ny;  if(Ny_ <= 0) Ny_ = Nx_;
 Nz_ = nz;  if(Nz_ <= 0) Nz_ = Nx_;
 N_.resize(0); N_.push_back(Nx_); N_.push_back(Ny_); N_.push_back(Nz_);
 NRtot_ = Nx_*Ny_*Nz_; // nombre de pixels dans le survey
 NCz_ =  Nz_/2 +1;
 NTz_ = 2*NCz_;

 // le pas dans l'espace (Mpc)
 Dx_ = dx;
 Dy_ = dy; if(Dy_ <= 0.) Dy_ = Dx_;
 Dz_ = dz; if(Dz_ <= 0.) Dz_ = Dx_;
 D_.resize(0); D_.push_back(Dx_); D_.push_back(Dy_); D_.push_back(Dz_);
 dVol_ = Dx_*Dy_*Dz_;
 Vol_ = (Nx_*Dx_)*(Ny_*Dy_)*(Nz_*Dz_);

 // Le pas dans l'espace de Fourier (Mpc^-1)
 Dkx_ = 2.*M_PI/(Nx_*Dx_);
 Dky_ = 2.*M_PI/(Ny_*Dy_);
 Dkz_ = 2.*M_PI/(Nz_*Dz_);
 Dk_.resize(0); Dk_.push_back(Dkx_); Dk_.push_back(Dky_); Dk_.push_back(Dkz_);
 Dk3_ = Dkx_*Dky_*Dkz_;
 
 // La frequence de Nyquist en k (Mpc^-1)
 Knyqx_ = M_PI/Dx_;
 Knyqy_ = M_PI/Dy_;
 Knyqz_ = M_PI/Dz_;
 Knyq_.resize(0); Knyq_.push_back(Knyqx_); Knyq_.push_back(Knyqy_); Knyq_.push_back(Knyqz_);
}

void GeneFluct3D::setalloc(void)
{
 if(lp_>1) cout<<"--- GeneFluct3D::setalloc ---"<<endl;
 // Dimensionnement du tableau complex<r_8>
 // ATTENTION: TArray adresse en memoire a l'envers du C
 //            Tarray(n1,n2,n3) == Carray[n3][n2][n1]
 sa_size_t SzK_[3] = {NCz_,Ny_,Nx_}; // a l'envers
 try {
   T_.ReSize(3,SzK_);
   array_allocated_ = true;
   if(lp_>1) cout<<"  allocating: "<<T_.Size()*sizeof(complex<r_8>)/1.e6<<" Mo"<<endl;
 } catch (...) {
   cout<<"GeneFluct3D::setalloc_Error: Problem allocating T_"<<endl;
 }
 T_.SetMemoryMapping(BaseArray::CMemoryMapping);
}

void GeneFluct3D::setpointers(bool from_real)
{
 if(lp_>1) cout<<"--- GeneFluct3D::setpointers ---"<<endl;
 if(from_real) T_ = ArrCastR2C(R_);
   else        R_ = ArrCastC2R(T_);
 // On remplit les pointeurs
 fdata_ = (fftw_complex *) (&T_(0,0,0));
 data_ = (double *) (&R_(0,0,0));
}

void GeneFluct3D::check_array_alloc(void)
// Pour tester si le tableau T_ est alloue
{
 if(array_allocated_) return;
 char bla[90];
 sprintf(bla,"GeneFluct3D::check_array_alloc_Error: array is not allocated");
 cout<<bla<<endl; throw ParmError(bla);
}

void GeneFluct3D::init_fftw(void)
{
 // --- Initialisation de fftw3 (attention data est sur-ecrit a l'init)
 if(lp_>1) cout<<"--- GeneFluct3D::init_fftw ---"<<endl;
#ifdef FFTW_THREAD
 if(nthread_>0) {
   cout<<"...Computing with "<<nthread_<<" threads"<<endl;
   fftw_init_threads();
   fftw_plan_with_nthreads(nthread_);
 }
#endif
 if(lp_>1) cout<<"...forward plan"<<endl;
 pf_ = fftw_plan_dft_r2c_3d(Nx_,Ny_,Nz_,data_,fdata_,FFTW_ESTIMATE);
 if(lp_>1) cout<<"...backward plan"<<endl;
 pb_ = fftw_plan_dft_c2r_3d(Nx_,Ny_,Nz_,fdata_,data_,FFTW_ESTIMATE);
}

//-------------------------------------------------------
long GeneFluct3D::LosComRedshift(double zinc,long npoints)
// Given a position of the cube relative to the observer
// and a cosmology
// (SetObservator() and SetCosmology() should have been called !)
// This routine filled:
//   the vector "zred_" of scanned redshift (by zinc increments)
//   the vector "loscom_" of corresponding los comoving distance
// -- Input:
// zinc : redshift increment for computation
// npoints : number of points required for inverting loscom -> zred
// 
{
 if(lp_>0) cout<<"--- LosComRedshift: zinc="<<zinc<<" , npoints="<<npoints<<endl;

 if(cosmo_ == NULL || redshref_<0.) {
   char *bla = "GeneFluct3D::LosComRedshift_Error: set Observator and Cosmology first";
   cout<<bla<<endl; throw ParmError(bla);
 }

 // On calcule les coordonnees de l'observateur dans le repere du cube
 // cad dans le repere ou l'origine est au centre du pixel i=j=l=0.
 // L'observateur est sur un axe centre sur le milieu de la face Oxy
 double loscom_ref_ = cosmo_->Dloscom(redshref_);
 xobs_[0] = Nx_/2.*Dx_;
 xobs_[1] = Ny_/2.*Dy_;
 xobs_[2] = kredshref_*Dz_ - loscom_ref_;

 // L'observateur est-il dans le cube?
 bool obs_in_cube = false;
 if(xobs_[2]>=0. && xobs_[2]<=Nz_*Dz_) obs_in_cube = true;

 // Find MINIMUM los com distance to the observer:
 // c'est le centre de la face a k=0
 // (ou zero si l'observateur est dans le cube)
 loscom_min_ = 0.;
 if(!obs_in_cube) loscom_min_ = -xobs_[2];

 // Find MAXIMUM los com distance to the observer:
 // ou que soit positionne l'observateur, la distance
 // maximal est sur un des coins du cube
 loscom_max_ = 0.;
 for(long i=0;i<=1;i++) {
   double dx2 = xobs_[0] - i*Nx_*Dx_; dx2 *= dx2;
   for(long j=0;j<=1;j++) {
     double dy2 = xobs_[1] - j*Ny_*Dy_; dy2 *= dy2;
     for(long k=0;k<=1;k++) {
       double dz2 = xobs_[2] - k*Nz_*Dz_; dz2 *= dz2;
       dz2 = sqrt(dx2+dy2+dz2);
       if(dz2>loscom_max_) loscom_max_ = dz2;
     }
   }
 }
 if(lp_>0) {
   cout<<"...zref="<<redshref_<<" kzref="<<kredshref_<<" losref="<<loscom_ref_<<" Mpc\n"
       <<"   xobs="<<xobs_[0]<<" , "<<xobs_[1]<<" , "<<xobs_[2]<<" Mpc "
       <<" in_cube="<<obs_in_cube
       <<" loscom_min="<<loscom_min_<<" loscom_max="<<loscom_max_<<" Mpc "<<endl;
 }

 // Fill the corresponding vectors for loscom and zred
 if(zinc<=0.) zinc = 0.01;
 for(double z=0.; ; z+=zinc) {
   double dlc = cosmo_->Dloscom(z);
   if(dlc<loscom_min_) {zred_.resize(0); loscom_.resize(0);}
   zred_.push_back(z);
   loscom_.push_back(dlc);
   z += zinc;
   if(dlc>loscom_max_) break; // on sort apres avoir stoque un dlc>dlcmax
 }

 if(lp_>0) {
   long n = zred_.size();
   cout<<"...zred/loscom tables[zinc="<<zinc<<"]: n="<<n;
   if(n>0) cout<<" z="<<zred_[0]<<" -> d="<<loscom_[0];
   if(n>1) cout<<" , z="<<zred_[n-1]<<" -> d="<<loscom_[n-1];
   cout<<endl;
 }

 // Compute the parameters and tables needed for inversion loscom->zred
 if(npoints<3) npoints = zred_.size();
 InverseFunc invfun(zred_,loscom_);
 invfun.ComputeParab(npoints,loscom2zred_);
 loscom2zred_min_ = invfun.YMin();
 loscom2zred_max_ = invfun.YMax();

 if(lp_>0) {
   long n = loscom2zred_.size();
   cout<<"...loscom -> zred[npoints="<<npoints<<"]: n="<<n
       <<" los_min="<<loscom2zred_min_
       <<" los_max="<<loscom2zred_max_
       <<" -> zred=[";
   if(n>0) cout<<loscom2zred_[0];
   cout<<",";
   if(n>1) cout<<loscom2zred_[n-1];
   cout<<"]"<<endl;
   if(lp_>1 && n>0)
     for(int i=0;i<n;i++)
       if(i==0 || abs(i-n/2)<2 || i==n-1)
         cout<<"    "<<i<<"  "<<loscom2zred_[i]<<endl;
 }

 return zred_.size();
}

//-------------------------------------------------------
void GeneFluct3D::WriteFits(string cfname,int bitpix)
{
 cout<<"--- GeneFluct3D::WriteFits: Writing Cube to "<<cfname<<endl;
 try {
   FitsImg3DWriter fwrt(cfname.c_str(),bitpix,5);
   fwrt.WriteKey("NX",Nx_," axe transverse 1");
   fwrt.WriteKey("NY",Ny_," axe transverse 2");
   fwrt.WriteKey("NZ",Nz_," axe longitudinal (redshift)");
   fwrt.WriteKey("DX",Dx_," Mpc");
   fwrt.WriteKey("DY",Dy_," Mpc");
   fwrt.WriteKey("DZ",Dz_," Mpc");
   fwrt.WriteKey("DKX",Dkx_," Mpc^-1");
   fwrt.WriteKey("DKY",Dky_," Mpc^-1");
   fwrt.WriteKey("DKZ",Dkz_," Mpc^-1");
   fwrt.WriteKey("ZREF",redshref_," reference redshift");
   fwrt.WriteKey("KZREF",kredshref_," reference redshift on z axe");
   fwrt.Write(R_);
 } catch (PThrowable & exc) {
   cout<<"Exception : "<<(string)typeid(exc).name()
       <<" - Msg= "<<exc.Msg()<<endl;
   return;
 } catch (...) {
   cout<<" some other exception was caught !"<<endl;
   return;
 }
}

void GeneFluct3D::ReadFits(string cfname)
{
 cout<<"--- GeneFluct3D::ReadFits: Reading Cube from "<<cfname<<endl;
 try {
   FitsImg3DRead fimg(cfname.c_str(),0,5);
   fimg.Read(R_);
   long nx = fimg.ReadKeyL("NX");
   long ny = fimg.ReadKeyL("NY");
   long nz = fimg.ReadKeyL("NZ");
   double dx = fimg.ReadKey("DX");
   double dy = fimg.ReadKey("DY");
   double dz = fimg.ReadKey("DZ");
   double zref = fimg.ReadKey("ZREF");
   double kzref = fimg.ReadKey("KZREF");
   setsize(nx,ny,nz,dx,dy,dz);
   setpointers(true);
   init_fftw();
   SetObservator(zref,kzref);
 } catch (PThrowable & exc) {
   cout<<"Exception : "<<(string)typeid(exc).name()
       <<" - Msg= "<<exc.Msg()<<endl;
   return;
 } catch (...) {
   cout<<" some other exception was caught !"<<endl;
   return;
 }
}

void GeneFluct3D::WritePPF(string cfname,bool write_real)
// On ecrit soit le TArray<r_8> ou le TArray<complex <r_8> >
{
 cout<<"--- GeneFluct3D::WritePPF: Writing Cube (real="<<write_real<<") to "<<cfname<<endl;
 try {
   R_.Info()["NX"] = (int_8)Nx_;
   R_.Info()["NY"] = (int_8)Ny_;
   R_.Info()["NZ"] = (int_8)Nz_;
   R_.Info()["DX"] = (r_8)Dx_;
   R_.Info()["DY"] = (r_8)Dy_;
   R_.Info()["DZ"] = (r_8)Dz_;
   R_.Info()["ZREF"] = (r_8)redshref_;
   R_.Info()["KZREF"] = (r_8)kredshref_;
   POutPersist pos(cfname.c_str());
   if(write_real) pos << PPFNameTag("rgen")  << R_;
     else         pos << PPFNameTag("pkgen") << T_;
 } catch (PThrowable & exc) {
   cout<<"Exception : "<<(string)typeid(exc).name()
       <<" - Msg= "<<exc.Msg()<<endl;
   return;
 } catch (...) {
   cout<<" some other exception was caught !"<<endl;
   return;
 }
}

void GeneFluct3D::ReadPPF(string cfname)
{
 cout<<"--- GeneFluct3D::ReadPPF: Reading Cube from "<<cfname<<endl;
 try {
   bool from_real = true;
   PInPersist pis(cfname.c_str());
   string name_tag_k = "pkgen";
   bool found_tag_k = pis.GotoNameTag("pkgen");
   if(found_tag_k) {
     cout<<"           ...reading spectrun into TArray<complex <r_8> >"<<endl;
     pis >> PPFNameTag("pkgen")  >> T_;
     from_real = false;
   } else {
     cout<<"           ...reading space into TArray<r_8>"<<endl;
     pis >> PPFNameTag("rgen")  >> R_;
   }
   setpointers(from_real);  // a mettre ici pour relire les DVInfo
   int_8 nx = R_.Info()["NX"];
   int_8 ny = R_.Info()["NY"];
   int_8 nz = R_.Info()["NZ"];
   r_8 dx = R_.Info()["DX"];
   r_8 dy = R_.Info()["DY"];
   r_8 dz = R_.Info()["DZ"];
   r_8 zref = R_.Info()["ZREF"];
   r_8 kzref = R_.Info()["KZREF"];
   setsize(nx,ny,nz,dx,dy,dz);
   init_fftw();
   SetObservator(zref,kzref);
 } catch (PThrowable & exc) {
   cout<<"Exception : "<<(string)typeid(exc).name()
       <<" - Msg= "<<exc.Msg()<<endl;
   return;
 } catch (...) {
   cout<<" some other exception was caught !"<<endl;
   return;
 }
}

//-------------------------------------------------------
void GeneFluct3D::Print(void)
{
 cout<<"GeneFluct3D(T_alloc="<<array_allocated_<<"):"<<endl;
 cout<<"Space Size : nx="<<Nx_<<" ny="<<Ny_<<" nz="<<Nz_<<" ("<<NTz_<<")  size="
     <<NRtot_<<endl;
 cout<<"      Resol: dx="<<Dx_<<" dy="<<Dy_<<" dz="<<Dz_<<" Mpc"
     <<", dVol="<<dVol_<<", Vol="<<Vol_<<" Mpc^3"<<endl;
 cout<<"Fourier Size : nx="<<Nx_<<" ny="<<Ny_<<" nz="<<NCz_<<endl;
 cout<<"        Resol: dkx="<<Dkx_<<" dky="<<Dky_<<" dkz="<<Dkz_<<" Mpc^-1"
     <<", Dk3="<<Dk3_<<" Mpc^-3"<<endl;
 cout<<"          (2Pi/k: "<<2.*M_PI/Dkx_<<" "<<2.*M_PI/Dky_<<" "<<2.*M_PI/Dkz_<<" Mpc)"<<endl;
 cout<<"      Nyquist: kx="<<Knyqx_<<" ky="<<Knyqy_<<" kz="<<Knyqz_<<" Mpc^-1"
     <<", Kmax="<<GetKmax()<<" Mpc^-1"<<endl;
 cout<<"          (2Pi/k: "<<2.*M_PI/Knyqx_<<" "<<2.*M_PI/Knyqy_<<" "<<2.*M_PI/Knyqz_<<" Mpc)"<<endl;
 cout<<"Redshift "<<redshref_<<" for z axe at k="<<kredshref_<<endl;
}

//-------------------------------------------------------
void GeneFluct3D::ComputeFourier0(GenericFunc& pk_at_z)
// cf ComputeFourier() mais avec autre methode de realisation du spectre
//    (attention on fait une fft pour realiser le spectre)
{

 // --- realisation d'un tableau de tirage gaussiens
 if(lp_>0) cout<<"--- ComputeFourier0: before gaussian filling ---"<<endl;
 // On tient compte du pb de normalisation de FFTW3
 double sntot = sqrt((double)NRtot_);
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   data_[ip] = NorRand()/sntot;
 }

 // --- realisation d'un tableau de tirage gaussiens
 if(lp_>0) cout<<"...before fft real ---"<<endl;
 fftw_execute(pf_);

 // --- On remplit avec une realisation
 if(lp_>0) cout<<"...before Fourier realization filling"<<endl;
 T_(0,0,0) = complex<r_8>(0.);  // on coupe le continue et on l'initialise
 long lmod = Nx_/10; if(lmod<1) lmod=1;
 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_;  kx *= kx;
   if(lp_>0 && i%lmod==0) cout<<"i="<<i<<" ii="<<ii<<endl;
   for(long j=0;j<Ny_;j++) {
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_;  ky *= ky;
     for(long l=0;l<NCz_;l++) {
       double kz = l*Dkz_;  kz *= kz;
       if(i==0 && j==0 && l==0) continue; // Suppression du continu
       double k = sqrt(kx+ky+kz);
       // cf normalisation: Peacock, Cosmology, formule 16.38 p504
       double pk = pk_at_z(k)/Vol_;
       // ici pas de "/2" a cause de la remarque ci-dessus
       T_(l,j,i) *= sqrt(pk);
     }
   }
 }

 if(lp_>0) cout<<"...computing power"<<endl;
 double p = compute_power_carte();
 if(lp_>0) cout<<"Puissance dans la realisation: "<<p<<endl;

}

//-------------------------------------------------------
void GeneFluct3D::ComputeFourier(GenericFunc& pk_at_z)
// Calcule une realisation du spectre "pk_at_z"
// Attention: dans TArray le premier indice varie le + vite
// Explication normalisation: see Coles & Lucchin, Cosmology, p264-265
// FFTW3: on note N=Nx*Ny*Nz
// f  --(FFT)-->  F = TF(f)  --(FFT^-1)-->  fb = TF^-1(F) = TF^-1(TF(f))
// sum(f(x_i)^2) = S
//                sum(F(nu_i)^2) = S*N
//                                          sum(fb(x_i)^2) = S*N^2
{
 // --- RaZ du tableau
 T_ = complex<r_8>(0.);

 // --- On remplit avec une realisation
 if(lp_>0) cout<<"--- ComputeFourier ---"<<endl;
 long lmod = Nx_/10; if(lmod<1) lmod=1;
 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_;  kx *= kx;
   if(lp_>0 && i%lmod==0) cout<<"i="<<i<<" ii="<<ii<<endl;
   for(long j=0;j<Ny_;j++) {
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_;  ky *= ky;
     for(long l=0;l<NCz_;l++) {
       double kz = l*Dkz_;  kz *= kz;
       if(i==0 && j==0 && l==0) continue; // Suppression du continu
       double k = sqrt(kx+ky+kz);
       // cf normalisation: Peacock, Cosmology, formule 16.38 p504
       double pk = pk_at_z(k)/Vol_;
       // Explication de la division par 2: voir perandom.cc
       // ou egalement Coles & Lucchin, Cosmology formula 13.7.2 p279
       T_(l,j,i) = ComplexGaussRan(sqrt(pk/2.));
     }
   }
 }

 manage_coefficients();   // gros effet pour les spectres que l'on utilise !

 if(lp_>0) cout<<"...computing power"<<endl;
 double p = compute_power_carte();
 if(lp_>0) cout<<"Puissance dans la realisation: "<<p<<endl;

}

long GeneFluct3D::manage_coefficients(void)
// Take into account the real and complexe conjugate coefficients
// because we want a realization of a real data in real space
{
 if(lp_>1) cout<<"...managing coefficients"<<endl;
 check_array_alloc();

 // 1./ Le Continu et Nyquist sont reels
 long nreal = 0;
 for(long kk=0;kk<2;kk++) {
   long k=0;  // continu
   if(kk==1) {if(Nz_%2!=0) continue; else k = Nz_/2;}  // Nyquist
   for(long jj=0;jj<2;jj++) {
     long j=0;
     if(jj==1) {if( Ny_%2!=0) continue; else j = Ny_/2;}
     for(long ii=0;ii<2;ii++) {
       long i=0;
       if(ii==1) {if( Nx_%2!=0) continue; else i = Nx_/2;}
       int_8 ip = IndexC(i,j,k);
       //cout<<"i="<<i<<" j="<<j<<" k="<<k<<" = ("<<fdata_[ip][0]<<","<<fdata_[ip][1]<<")"<<endl;
       fdata_[ip][1] = 0.; fdata_[ip][0] *= M_SQRT2;
       nreal++;
     }
   }
 }
 if(lp_>1) cout<<"Number of forced real number ="<<nreal<<endl;

 // 2./ Les elements complexe conjugues (tous dans le plan k=0,Nyquist)

 // a./ les lignes et colonnes du continu et de nyquist
 long nconj1 = 0;
 for(long kk=0;kk<2;kk++) {
   long k=0;  // continu
   if(kk==1) {if(Nz_%2!=0) continue; else k = Nz_/2;}  // Nyquist
   for(long jj=0;jj<2;jj++) { // selon j
     long j=0;
     if(jj==1) {if( Ny_%2!=0) continue; else j = Ny_/2;}
     for(long i=1;i<(Nx_+1)/2;i++) {
       int_8 ip = IndexC(i,j,k);
       int_8 ip1 = IndexC(Nx_-i,j,k);
       fdata_[ip1][0] = fdata_[ip][0]; fdata_[ip1][1] = -fdata_[ip][1];
       nconj1++;
     }
   }
   for(long ii=0;ii<2;ii++) {
     long i=0;
     if(ii==1) {if( Nx_%2!=0) continue; else i = Nx_/2;}
     for(long j=1;j<(Ny_+1)/2;j++) {
       int_8 ip = IndexC(i,j,k);
       int_8 ip1 = IndexC(i,Ny_-j,k);
       fdata_[ip1][0] = fdata_[ip][0]; fdata_[ip1][1] = -fdata_[ip][1];
       nconj1++;
     }
   }
 }
 if(lp_>1) cout<<"Number of forced conjugate on cont+nyq ="<<nconj1<<endl;

 // b./ les lignes et colonnes hors continu et de nyquist
 long nconj2 = 0;
 for(long kk=0;kk<2;kk++) {
   long k=0;  // continu
   if(kk==1) {if(Nz_%2!=0) continue; else k = Nz_/2;}  // Nyquist
   for(long j=1;j<(Ny_+1)/2;j++) {
     if(Ny_%2==0 && j==Ny_/2) continue; // on ne retraite pas nyquist en j
     for(long i=1;i<Nx_;i++) {
       if(Nx_%2==0 && i==Nx_/2) continue; // on ne retraite pas nyquist en i
       int_8 ip = IndexC(i,j,k);
       int_8 ip1 = IndexC(Nx_-i,Ny_-j,k);
       fdata_[ip1][0] = fdata_[ip][0]; fdata_[ip1][1] = -fdata_[ip][1];
       nconj2++;
     }
   }
 }
 if(lp_>1) cout<<"Number of forced conjugate hors cont+nyq ="<<nconj2<<endl;

 if(lp_>1) cout<<"Check: ddl= "<<NRtot_<<" =?= "<<2*(Nx_*Ny_*NCz_-nconj1-nconj2)-8<<endl;

 return nreal+nconj1+nconj2;
}

double GeneFluct3D::compute_power_carte(void)
// Calcul la puissance de la realisation du spectre Pk
{
 check_array_alloc();

 double s2 = 0.;
 for(long l=0;l<NCz_;l++)
   for(long j=0;j<Ny_;j++)
     for(long i=0;i<Nx_;i++) s2 += MODULE2(T_(l,j,i));

 double s20 = 0.;
 for(long j=0;j<Ny_;j++)
   for(long i=0;i<Nx_;i++) s20 += MODULE2(T_(0,j,i));

 double s2n = 0.;
 if(Nz_%2==0)
   for(long j=0;j<Ny_;j++)
     for(long i=0;i<Nx_;i++) s2n += MODULE2(T_(NCz_-1,j,i));

 return 2.*s2 -s20 -s2n;
}

//-------------------------------------------------------------------
void GeneFluct3D::FilterByPixel(void)
// Filtrage par la fonction fenetre du pixel (parallelepipede)
// TF = 1/(dx*dy*dz)*Int[{-dx/2,dx/2},{-dy/2,dy/2},{-dz/2,dz/2}]
//                   e^(ik_x*x) e^(ik_y*y) e^(ik_z*z) dxdydz
//    = 2/(k_x*dx) * sin(k_x*dx/2)  * (idem y) * (idem z)
// Gestion divergence en 0: sin(y)/y = 1 - y^2/6*(1-y^2/20)
//                          avec y = k_x*dx/2
{
 if(lp_>0) cout<<"--- FilterByPixel ---"<<endl;
 check_array_alloc();

 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_ *Dx_/2;
   double pk_x = pixelfilter(kx); 
   for(long j=0;j<Ny_;j++) {
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_ *Dy_/2;
     double pk_y =  pixelfilter(ky); 
     for(long l=0;l<NCz_;l++) {
       double kz = l*Dkz_ *Dz_/2;
       double pk_z =  pixelfilter(kz);
       T_(l,j,i) *= pk_x*pk_y*pk_z;
     }
   }
 }

}

//-------------------------------------------------------------------
void GeneFluct3D::ApplyGrowthFactor(void)
// Apply Growth to real space
// Using the correspondance between redshift and los comoving distance
// describe in vector "zred_" "loscom_"
{
 if(lp_>0) cout<<"--- ApplyGrowthFactor ---"<<endl;
 check_array_alloc();

 if(growth_ == NULL) {
   char *bla = "GeneFluct3D::ApplyGrowthFactor_Error: set GrowthFactor first";
   cout<<bla<<endl; throw ParmError(bla);
 }

 InterpFunc interpinv(loscom2zred_min_,loscom2zred_max_,loscom2zred_);
 unsigned short ok;

 //CHECK: Histo hgr(0.9*zred_[0],1.1*zred_[n-1],1000);
 for(long i=0;i<Nx_;i++) {
   double dx2 = xobs_[0] - i*Dx_; dx2 *= dx2;
   for(long j=0;j<Ny_;j++) {
     double dy2 = xobs_[1] - j*Dy_; dy2 *= dy2;
     for(long l=0;l<Nz_;l++) {
       double dz2 = xobs_[2] - l*Dz_; dz2 *= dz2;
       dz2 = sqrt(dx2+dy2+dz2);
       double z = interpinv(dz2);
       //CHECK: hgr.Add(z);
       double dzgr = (*growth_)(z);   // interpolation par morceau
       //double dzgr = growth_->Linear(z,ok);  // interpolation lineaire
       //double dzgr = growth_->Parab(z,ok);  // interpolation parabolique
       int_8 ip = IndexR(i,j,l);
       data_[ip] *= dzgr;
     }
   }
 }

 //CHECK: {POutPersist pos("applygrowth.ppf"); string tag="hgr"; pos.PutObject(hgr,tag);}

}

//-------------------------------------------------------------------
void GeneFluct3D::ComputeReal(void)
// Calcule une realisation dans l'espace reel
{
 if(lp_>0) cout<<"--- ComputeReal ---"<<endl;
 check_array_alloc();

 // On fait la FFT
 fftw_execute(pb_);
}

//-------------------------------------------------------------------
void GeneFluct3D::ReComputeFourier(void)
{
 if(lp_>0) cout<<"--- ReComputeFourier ---"<<endl;
 check_array_alloc();

 // On fait la FFT
 fftw_execute(pf_);
 // On corrige du pb de la normalisation de FFTW3
 double v = (double)NRtot_;
 for(long i=0;i<Nx_;i++)
   for(long j=0;j<Ny_;j++)
     for(long l=0;l<NCz_;l++) T_(l,j,i) /= complex<r_8>(v);

}

//-------------------------------------------------------------------
int GeneFluct3D::ComputeSpectrum(HistoErr& herr)
// Compute spectrum from "T" and fill HistoErr "herr"
// T : dans le format standard de GeneFuct3D: T(nz,ny,nx)
// cad T(kz,ky,kx) avec  0<kz<kz_nyq  -ky_nyq<ky<ky_nyq  -kx_nyq<kx<kx_nyq
{
 if(lp_>0) cout<<"--- ComputeSpectrum ---"<<endl;
 check_array_alloc();

 if(herr.NBins()<0) return -1;
 herr.Zero();

 // Attention a l'ordre
 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_;  kx *= kx;
   for(long j=0;j<Ny_;j++) {
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_;  ky *= ky;
     for(long l=0;l<NCz_;l++) {
       double kz = l*Dkz_;  kz *= kz;
       double k = sqrt(kx+ky+kz);
       double pk = MODULE2(T_(l,j,i));
       herr.Add(k,pk);
     }
   }
 }
 herr.ToVariance();

 // renormalize to directly compare to original spectrum
 double norm = Vol_;
 herr *= norm;

 return 0;
}

int GeneFluct3D::ComputeSpectrum2D(Histo2DErr& herr)
{
 if(lp_>0) cout<<"--- ComputeSpectrum2D ---"<<endl;
 check_array_alloc();

 if(herr.NBinX()<0 || herr.NBinY()<0) return -1;
 herr.Zero();

 // Attention a l'ordre
 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_;  kx *= kx;
   for(long j=0;j<Ny_;j++) {
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_;  ky *= ky;
     double kt = sqrt(kx+ky);
     for(long l=0;l<NCz_;l++) {
       double kz = l*Dkz_;
       double pk = MODULE2(T_(l,j,i));
       herr.Add(kt,kz,pk);
     }
   }
 }
 herr.ToVariance();

 // renormalize to directly compare to original spectrum
 double norm = Vol_;
 herr *= norm;

 return 0;
}

//-------------------------------------------------------
int_8 GeneFluct3D::VarianceFrReal(double R,double& var)
// Recompute MASS variance in spherical top-hat (rayon=R)
{
 if(lp_>0) cout<<"--- VarianceFrReal ---"<<endl;
 check_array_alloc();

 long dnx = long(R/Dx_+0.5); if(dnx<=0) dnx = 1;
 long dny = long(R/Dy_+0.5); if(dny<=0) dny = 1;
 long dnz = long(R/Dz_+0.5); if(dnz<=0) dnz = 1;
 if(lp_>0) cout<<"dnx="<<dnx<<" dny="<<dny<<" dnz="<<dnz<<endl;

 double sum=0., sum2=0., r2 = R*R; int_8 nsum=0;

 for(long i=dnx;i<Nx_-dnx;i+=dnx) {
   for(long j=dny;j<Ny_-dny;j+=dny) {
     for(long l=dnz;l<Nz_-dnz;l+=dnz) {
       double s=0.; int_8 n=0;
       for(long ii=i-dnx;ii<=i+dnx;ii++) {
         double x = (ii-i)*Dx_; x *= x;
         for(long jj=j-dny;jj<=j+dny;jj++) {
           double y = (jj-j)*Dy_; y *= y;
           for(long ll=l-dnz;ll<=l+dnz;ll++) {
             double z = (ll-l)*Dz_; z *= z;
             if(x+y+z>r2) continue;
             int_8 ip = IndexR(ii,jj,ll);
             s += 1.+data_[ip];
             n++;
	   }
	 }
       }
       if(n>0) {sum += s; sum2 += s*s; nsum++;}
       //cout<<i<<","<<j<<","<<l<<" n="<<n<<" s="<<s<<" sum="<<sum<<" sum2="<<sum2<<endl;
     }
   }
 }

 if(nsum<=1) {var=0.; return nsum;}

 sum /= nsum;
 sum2 = sum2/nsum - sum*sum;
 if(lp_>0) cout<<"VarianceFrReal: nsum="<<nsum<<" <M>="<<sum<<" <(M-<M>)^2>="<<sum2<<endl;

 var = sum2/(sum*sum);  // <dM>^2/<M>^2
 if(lp_>0) cout<<"sigmaR^2="<<var<<" -> "<<sqrt(var)<<endl;

 return nsum;
}

//-------------------------------------------------------
int_8 GeneFluct3D::NumberOfBad(double vmin,double vmax)
// number of pixels outside of ]vmin,vmax[ extremites exclues
//     ->  vmin and vmax are considered as bad
{
 check_array_alloc();

 int_8 nbad = 0;
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   double v = data_[ip];
   if(v<=vmin || v>=vmax) nbad++;
 }

 return nbad;
}

int_8 GeneFluct3D::MeanSigma2(double& rm,double& rs2,double vmin,double vmax
                             ,bool useout,double vout)
// Calcul de mean,sigma2 dans le cube reel avec valeurs ]vmin,vmax[ extremites exclues
// useout = false: ne pas utiliser les pixels hors limites pour calculer mean,sigma2
//          true : utiliser les pixels hors limites pour calculer mean,sigma2
//                 en remplacant leurs valeurs par "vout"
{
 bool tstval = (vmax>vmin)? true: false;
 if(lp_>0) {
   cout<<"--- MeanSigma2: ";
   if(tstval) cout<<"range=]"<<vmin<<","<<vmax<<"[";
   if(useout) cout<<", useout="<<useout<<" vout="<<vout;
   cout<<endl;
 }
 check_array_alloc();

 int_8 n = 0;
 rm = rs2 = 0.;

 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   double v = data_[ip];
   if(tstval) {
     if(v<=vmin || v>=vmax) {if(useout) v=vout; else continue;}
   }
   rm += v;
   rs2 += v*v;
   n++;
 }

 if(n>1) {
   rm /= (double)n;
   rs2 = rs2/(double)n - rm*rm;
 }

 if(lp_>0) cout<<"  n="<<n<<" m="<<rm<<" s2="<<rs2<<" s="<<sqrt(fabs(rs2))<<endl;

 return n;
}

int_8 GeneFluct3D::SetToVal(double vmin, double vmax,double val0)
// set to "val0" if out of range ]vmin,vmax[ extremites exclues
// cad set to "val0" if in [vmin,vmax] -> vmin and vmax are set to val0
{
 if(lp_>0) cout<<"--- SetToVal set to="<<val0
               <<" when in range=["<<vmin<<","<<vmax<<"]"<<endl;
 check_array_alloc();

 int_8 nbad = 0;
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   double v = data_[ip];
   if(v<=vmin || v>=vmax) {data_[ip] = val0; nbad++;}
 }

 return nbad;
}

//-------------------------------------------------------
void GeneFluct3D::TurnFluct2Mass(void)
// d_rho/rho -> Mass  (add one!)
{
 if(lp_>0) cout<<"--- TurnFluct2Mass ---"<<endl;
 check_array_alloc();


 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   data_[ip] += 1.;
 }
}

double GeneFluct3D::TurnMass2MeanNumber(double n_by_mpc3)
// do NOT treate negative or nul values
{
 if(lp_>0) cout<<"--- TurnMass2MeanNumber ---"<<endl;

 double m,s2;
 int_8 ngood = MeanSigma2(m,s2,0.,1e+200);
 if(lp_>0) cout<<"...ngood="<<ngood
               <<" m="<<m<<" s2="<<s2<<" -> "<<sqrt(s2)<<endl;

 // On doit mettre n*Vol galaxies dans notre survey
 // On en met uniquement dans les pixels de masse >0.
 // On NE met PAS a zero les pixels <0
 // On renormalise sur les pixels>0 pour qu'on ait n*Vol galaxies
 //   comme on ne prend que les pixels >0, on doit normaliser
 //   a la moyenne de <1+d_rho/rho> sur ces pixels
 //   (rappel sur tout les pixels <1+d_rho/rho>=1)
 double dn = n_by_mpc3*Vol_/m /(double)ngood;  // nb de gal a mettre ds 1 px
 if(lp_>0) cout<<"...galaxy density move from "
               <<n_by_mpc3*Vol_/double(NRtot_)<<" to "<<dn<<" / pixel"<<endl;
 double sum = 0.;
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   data_[ip] *= dn; // par coherence on multiplie aussi les <=0
   if(data_[ip]>0.) sum += data_[ip];  // mais on ne les compte pas
 }
 if(lp_>0) cout<<sum<<"...galaxies put into survey / "<<n_by_mpc3*Vol_<<endl;

 return sum;
}

double GeneFluct3D::ApplyPoisson(void)
// do NOT treate negative or nul mass  -> let it as it is
{
 if(lp_>0) cout<<"--- ApplyPoisson ---"<<endl;
 check_array_alloc();

 double sum = 0.;
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   double v = data_[ip];
   if(v>0.) {
     unsigned long dn = PoissRandLimit(v,10.);
     data_[ip] = (double)dn;
     sum += (double)dn;
   }
 }
 if(lp_>0) cout<<sum<<" galaxies put into survey"<<endl;

 return sum;
}

double GeneFluct3D::TurnNGal2Mass(FunRan& massdist,bool axeslog)
// do NOT treate negative or nul mass  -> let it as it is
// INPUT:
//   massdist : distribution de masse (m*dn/dm)
//   axeslog = false : retourne la masse
//           = true  : retourne le log10(mass)
// RETURN la masse totale
{
 if(lp_>0) cout<<"--- TurnNGal2Mass ---"<<endl;
 check_array_alloc();

 double sum = 0.;
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   double v = data_[ip];
   if(v>0.) {
     long ngal = long(v+0.1);
     data_[ip] = 0.;
     for(long i=0;i<ngal;i++) {
       double m = massdist.RandomInterp();  // massdist.Random();
       if(axeslog) m = pow(10.,m);
       data_[ip] += m;
     }
     sum += data_[ip];
   }
 }
 if(lp_>0) cout<<sum<<" MSol HI mass put into survey"<<endl;

 return sum;
}

void GeneFluct3D::AddAGN(double lfjy,double lsigma,double powlaw)
// Add AGN flux into simulation:
// --- Procedure:
// 1. lancer "cmvdefsurv" avec les parametres du survey
//        (au redshift de reference du survey)
//    et recuperer l'angle solide "angsol sr" du pixel elementaire
//    au centre du cube.
// 2. lancer "cmvtstagn" pour cet angle solide -> cmvtstagn.ppf
// 3. regarder l'histo "hlfang" et en deduire un equivalent gaussienne
//    cad une moyenne <log10(S)> et un sigma "sig"
//    Attention: la distribution n'est pas gaussienne les "mean,sigma"
//               de l'histo ne sont pas vraiment ce que l'on veut
// --- Limitations actuelle du code:
// . les AGN sont supposes en loi de puissance IDENTIQUE pour tout theta,phi
// . le flux des AGN est mis dans une colonne Oz (indice k) et pas sur la ligne de visee
// . la distribution est approximee a une gaussienne
// ... C'est une approximation pour un observateur loin du centre du cube
//     et pour un cube peu epais / distance observateur 
// --- Parametres de la routine:
// llfy : c'est le <log10(S)> du flux depose dans un pixel par les AGN
// lsigma : c'est le sigma de la distribution
// powlaw : c'est la pente de ls distribution cad que le flux "lmsol"
//          et considere comme le flux a 1.4GHz et qu'on suppose une loi
//             F(nu) = (1.4GHz/nu)^powlaw * F(1.4GHz)
// - Comme on est en echelle log10():
//   on tire log10(Msol) + X
//   ou X est une realisation sur une gaussienne de variance "sig^2"
//   La masse realisee est donc: Msol*10^X
// - Pas de probleme de pixel negatif car on a une multiplication!
{
  if(lp_>0) cout<<"--- AddAGN: <log10(S Jy)> = "<<lfjy<<" , sigma = "<<lsigma<<endl;
  check_array_alloc();

 if(cosmo_ == NULL || redshref_<0.| loscom2zred_.size()<1) {
   char *bla = "GeneFluct3D::AddAGN_Error: set Observator and Cosmology first";
   cout<<bla<<endl; throw ParmError(bla);
 }

 // La distance angulaire/luminosite/Dnu au centre
 double dangref = cosmo_->Dang(redshref_);
 double dlumref = cosmo_->Dlum(redshref_);
 double dredref  = Dz_/(cosmo_->Dhubble()/cosmo_->E(redshref_));
 double dnuref = Fr_HyperFin_Par *dredref/pow(1.+redshref_,2.); // GHz
 double fagnref = pow(10.,lfjy)*(dnuref*1.e9); // Jy.Hz
 double magnref = FluxHI2Msol(fagnref*Jansky2Watt_cst,dlumref); // Msol
 if(lp_>0) {
   cout<<"Au centre: z="<<redshref_<<", dredref="<<dredref<<", dnuref="<<dnuref<<" GHz"<<endl
       <<" dang="<<dangref<<" Mpc, dlum="<<dlumref<<" Mpc,"
       <<" fagnref="<<fagnref<<" Jy.Hz (a 1.4GHz), magnref="<<magnref<<" Msol"<<endl;
 }

 if(powlaw!=0.) {
   // F(nu) = (nu GHz/1.4 Ghz)^p * F(1.4GHz) et nu = 1.4 GHz / (1+z)
   magnref *= pow(1/(1.+redshref_),powlaw);
   if(lp_>0) cout<<" powlaw="<<powlaw<<"  -> change magnref to "<<magnref<<" Msol"<<endl;
 }

 // Les infos en fonction de l'indice "l" selon Oz
 vector<double> correction;
 InterpFunc interpinv(loscom2zred_min_,loscom2zred_max_,loscom2zred_);
 for(long l=0;l<Nz_;l++) {
   double z = fabs(xobs_[2] - l*Dz_);
   double zred = interpinv(z);
   double dang = cosmo_->Dang(zred);  // pour variation angle solide
   double dlum = cosmo_->Dlum(zred);  // pour variation conversion mass HI
   double dred = Dz_/(cosmo_->Dhubble()/cosmo_->E(zred));
   double dnu  = Fr_HyperFin_Par *dred/pow(1.+zred,2.); // pour variation dNu
   double corr = dnu/dnuref*pow(dangref/dang*dlum/dlumref,2.);
   if(powlaw!=0.) corr *= pow((1.+redshref_)/(1.+zred),powlaw);
   correction.push_back(corr);
   if(lp_>0 && (l==0 || abs(l-Nz_/2)<2 || l==Nz_-1)) {
     cout<<"l="<<l<<" z="<<z<<" red="<<zred
         <<" da="<<dang<<" dlu="<<dlum<<" dred="<<dred
         <<" dnu="<<dnu<<" -> corr="<<corr<<endl;
   }
 }

 double sum=0., sum2=0., nsum=0.;
 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) {
   double a = lsigma*NorRand();
   a = magnref*pow(10.,a);
   // On met le meme tirage le long de Oz (indice k)
   for(long l=0;l<Nz_;l++) {
     int_8 ip = IndexR(i,j,l);
     data_[ip] += a*correction[l];
   }
   sum += a; sum2 += a*a; nsum += 1.;
 }

 if(lp_>0 && nsum>1.) {
   sum /= nsum;
   sum2 = sum2/nsum - sum*sum;
   cout<<"...Mean mass="<<sum<<" Msol , s^2="<<sum2<<" s="<<sqrt(fabs(sum2))<<endl;
 }
 
}

void GeneFluct3D::AddNoise2Real(double snoise)
// add noise to every pixels (meme les <=0 !)
{
 if(lp_>0) cout<<"--- AddNoise2Real: snoise = "<<snoise<<endl;
 check_array_alloc();

 for(long i=0;i<Nx_;i++) for(long j=0;j<Ny_;j++) for(long l=0;l<Nz_;l++) {
   int_8 ip = IndexR(i,j,l);
   data_[ip] += snoise*NorRand();
 }
}



//-------------------------------------------------------------------
//-------------------------------------------------------------------
//--------------------- BRICOLO A NE PAS GARDER ---------------------
//-------------------------------------------------------------------
//-------------------------------------------------------------------
int GeneFluct3D::ComputeSpectrum_bricolo(HistoErr& herr)
// Compute spectrum from "T" and fill HistoErr "herr"
// T : dans le format standard de GeneFuct3D: T(nz,ny,nx)
// cad T(kz,ky,kx) avec  0<kz<kz_nyq  -ky_nyq<ky<ky_nyq  -kx_nyq<kx<kx_nyq
{
 if(lp_>0) cout<<"--- ComputeSpectrum_bricolo ---"<<endl;
 check_array_alloc();

 if(herr.NBins()<0) return -1;
 herr.Zero();

 // Attention a l'ordre
 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_;  kx *= kx;
   for(long j=0;j<Ny_;j++) {
     if(i==0 && j==0) continue;
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_;  ky *= ky;
     for(long l=1;l<NCz_;l++) {
       double kz = l*Dkz_;  kz *= kz;
       double k = sqrt(kx+ky+kz);
       double pk = MODULE2(T_(l,j,i));
       herr.Add(k,pk);
     }
   }
 }
 herr.ToVariance();

 // renormalize to directly compare to original spectrum
 double norm = Vol_;
 herr *= norm;

 return 0;
}

int GeneFluct3D::ComputeSpectrum2D_bricolo(Histo2DErr& herr)
{
 if(lp_>0) cout<<"--- ComputeSpectrum2D_bricolo ---"<<endl;
 check_array_alloc();

 if(herr.NBinX()<0 || herr.NBinY()<0) return -1;
 herr.Zero();

 // Attention a l'ordre
 for(long i=0;i<Nx_;i++) {
   long ii = (i>Nx_/2) ? Nx_-i : i;
   double kx = ii*Dkx_;  kx *= kx;
   for(long j=0;j<Ny_;j++) {
     if(i==0 && j==0) continue;
     long jj = (j>Ny_/2) ? Ny_-j : j;
     double ky = jj*Dky_;  ky *= ky;
     double kt = sqrt(kx+ky);
     for(long l=1;l<NCz_;l++) {
       double kz = l*Dkz_;
       double pk = MODULE2(T_(l,j,i));
       herr.Add(kt,kz,pk);
     }
   }
 }
 herr.ToVariance();

 // renormalize to directly compare to original spectrum
 double norm = Vol_;
 herr *= norm;

 return 0;
}
