#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <typeinfo>
#include <iostream.h>
#include <string>
#include <list>
#include <map>
#if defined(__KCC__)
using std::string ;
#include <list.h>
#include <map.h>
#endif

#include "strutil.h"
#include "datatypes.h"

#include "nobjmgr.h"
#include "servnobjm.h"
#include "nomgadapter.h"
#include "pistdimgapp.h"

#include "matrix.h"
#include "cvector.h"
#include "ntuple.h"

// EVOL-PLANCK
#ifdef SANS_EVOLPLANCK
#include "fitsimage.h"  
#endif

#include "fct1dfit.h"
#include "fct2dfit.h"

#include "pisurfdr.h"
#include "pipodrw.h"
#include "pihisto.h"
#include "hisprof.h"
#include "pihisto2d.h"
#include "pintuple.h"
#include "pintup3d.h"
#include "pigfd1.h"
#include "pigfd2.h"


//++
// Class	NamedObjMgr
// Lib		PI
// include	nobjmgr.h
//
//	Cette classe fournit les services ncssaire la gestion des objets
//	(l'ensemble des objets PPersist de PEIDA++) au sein du programme
//	d'analyse interactive *piapp* . Elle constitue en outre l'interface
//	entre les fonctions utilisateur et l'application graphique.
//--
//++
// Links	Voir aussi
// PIStdImgApp
// Services2NObjMgr
// PIACmd
//--


// ..................................................................
// ......  Gestion des objets nommes, variables globales ............
struct nobj_diritem {
  int id;
  int nobj;
};

typedef map<string, nobj_diritem, less<string> > NObjDirList;

struct nobj_item {
  AnyDataObj* obj;
  NObjMgrAdapter* obja;
  int num;
  int dirid;
  list<int> wrsid;
  bool operator==(nobj_item const& b) const
    { return (this->obj ==  b.obj); }
};

typedef map<string, nobj_item, less<string> > NObjList;

static NObjDirList* myDirs = NULL;
static NObjList* myObjs = NULL;
static int fgOInit = 0;
static int myNObj = 0;
static int myDirId = 0;
static string* currDir;

static PIStdImgApp* myImgApp=NULL;
static Services2NObjMgr* servnobjm=NULL;

static string* TmpDir;  // Repertoire pour les compilations / link dynamique

//++
// Titre	Constructeurs
//--
//++
// NamedObjMgr()
//	Constructeur. Les diffrents instantiation de la classe "NamedObjMgr" 
//	dans une mme application crent des objets qui travaillent sur la mme
//	liste d'objets. Les objets de cette classe ne possedent en effet pas
//	de variables membres. 
//--

/* --Methode-- */
NamedObjMgr::NamedObjMgr()
{
if (fgOInit == 0) { 
  myDirs = new NObjDirList;
  myObjs = new NObjList;  
  currDir = new string("home");
  char* varenv;
  TmpDir = new string("");
  if ( (varenv=getenv("PEIDA_TMP")) != NULL )  (*TmpDir) = varenv; 
  else if ( (varenv=getenv("TMPDIR")) != NULL )  (*TmpDir) = varenv; 
  int l = (*TmpDir).length();
  if ( (l>0) && ((*TmpDir)[l-1] != '/') )  (*TmpDir) += '/';
  servnobjm = new Services2NObjMgr(NULL, (*TmpDir));
  }
fgOInit++;
}

  
/* --Methode-- */
NamedObjMgr::~NamedObjMgr()
{
fgOInit--;
if (fgOInit == 0)  delete myObjs;
}

//++
// Titre	Mthodes
//--
//++
// void SetImgApp(PIStdImgApp* app)
//	Spcifie l'objet "PIStdImgApp" associ.
// PIStdImgApp* GetImgApp()
//	Accs  l'objet "PIStdImgApp" associ.
//--

/* --Methode-- */
void NamedObjMgr::SetImgApp(PIStdImgApp* app)
{
myImgApp = app;
servnobjm->SetImgApp(app);
}

/* --Methode-- */
PIStdImgApp* NamedObjMgr::GetImgApp()
{
return(myImgApp);
}

/* --Methode-- */
Services2NObjMgr* NamedObjMgr::GetServiceObj()
{
return(servnobjm);
}

/* --Methode-- */
void NamedObjMgr::CreateDir(string const& dirname)
{
}

/* --Methode-- */
void NamedObjMgr::DeleteDir(string const& dirname)
{
}

/* --Methode-- */
void NamedObjMgr::SetCurrentDir(string const& dirname)
{
}

/* --Methode-- */
void NamedObjMgr::CleanTmpDir()
{
}

/* --Methode-- */
void NamedObjMgr::CleanOldDir()
{
}

//++
// Titre	Gestion de la liste des objets
//--
//++
// void AddObj(AnyDataObj* obj, string& nom)
//	Ajoute l'objet "obj"  la liste, identifi par "nom".
//	Si un objet de mme nom existe, l'ancien objet est renomm en concatnant 
//	un numro  son nom.
// void DelObj(string const& nom, bool fgd=true)
//	Supprime l'objet "nom" de la liste. L'objet est dtruit si "fgd==true"	("delete obj")
// void DelObjects(string const& patt, bool fgd=true)
//	Supprime l'ensemble des objets dont le nom correspond au patron "patt".
//	Le patron peut contenir les caractres "*" et "?" . Les objets sont dtruits si "fgd==true"
// AnyDataObj*  GetObj(string const& nom) 
//	Retourne l'objet identifi par "nom" dans la liste. Retourne "NULL" si "nom" n'est 
//	pas dans la liste.
// void  RenameObj(string const& nom, string& nomnew)
//	Change le nom d'un objet dans la liste.
//--


/* --Methode-- */
void NamedObjMgr::AddObj(AnyDataObj* obj, string & nom, bool)
{

if (obj == NULL)  return;
// on supprime les blancs de debut et de fin 
size_t p = nom.find_first_not_of(" ");
if(p>nom.length()) {
  nom = "";
} else {
  nom = nom.substr(p);
  p = nom.find_first_of(" ");
  if(p>nom.length()) p=nom.length();
  nom = nom.substr(0, p);
}

myNObj++;
if (nom.length() < 1) nom = typeid(*obj).name();
NObjList::iterator it = myObjs->find(nom);
if (it != myObjs->end()) { // l'objet existe deja
  char strg[16];
  sprintf(strg, "%d", myNObj);
  string nomnew = nom + strg;
  RenameObj(nom, nomnew);
  }

nobj_item no;
no.obj = obj;
no.obja = servnobjm->GetAdapter(obj);  // L'adaptateur
no.num =  myNObj;
(*myObjs)[nom] = no;
if (myImgApp)  { 
  string str = nom + "    (T= " + typeid(*obj).name() + ")" ;
  (myImgApp->ObjMgrW())->AddObj(str.c_str(), myNObj+1000);
}
 
cout << "NamedObjMgr::AddObj()  Object " << nom << " ( "  
     << typeid(*obj).name() << " ) added (Total= " << myObjs->size() << ")" << endl;  
return;
}

/* --Methode-- */
void NamedObjMgr::RenameObj(string & nom, string& nomnew)
{
AnyDataObj* obj = GetObj(nom);
if (obj == NULL) return;
DelObj(nom, false);
AddObj(obj, nomnew);
return;
}

/* --Methode-- */
void NamedObjMgr::DelObj(string & nom, bool fgd)
{
NObjList::iterator it = myObjs->find(nom);
if (it == myObjs->end()) return; 
list<int>::iterator ii;
if (myImgApp) {
//DBG  cerr << " *DBG* NamedObjMgr::DelObj Sz= " << (*it).second.wrsid.size() << endl; 
  for(ii=(*it).second.wrsid.begin(); ii != (*it).second.wrsid.end(); ii++) 
      myImgApp->DelWRsId((*ii));
  (myImgApp->ObjMgrW())->DelObj((*it).second.num+1000);
}
delete (*it).second.obja;  // destruction de l'adaptateur 
if (fgd) delete (*it).second.obj;
myObjs->erase(it);
if (fgd) cout << "NamedObjMgr::DelObj()  Object " << nom << " deleted (Total= " << myObjs->size() << ")" << endl;
else  cout << "NamedObjMgr::DelObj()  Object " << nom << " removed (Total= " << myObjs->size() << ")" << endl;
return;  
}

/* --Methode-- */
void NamedObjMgr::DelObj_Id(int oid)
{
}

/* --Methode-- */
void NamedObjMgr::DelObjects(string & patt, bool fgd)
{
NObjList::iterator it;  
list<string> odel;
string cn;
for(it = myObjs->begin(); it != myObjs->end(); it++) {
  cn = (*it).first;
  if (csh_parse(cn.c_str(), patt.c_str()) != 0)  odel.push_back(cn);
  }
list<string>::iterator ii;
for(ii=odel.begin(); ii != odel.end(); ii++)  DelObj(*ii, fgd);
}

/* --Methode-- */
AnyDataObj* NamedObjMgr::GetObj(string & nom)
{
NObjList::iterator it = myObjs->find(nom);
if (it == myObjs->end()) return(NULL); 
return((*it).second.obj);
}

/* --Methode-- */
NObjMgrAdapter* NamedObjMgr::GetObjAdapter(string & nom)
{
NObjList::iterator it = myObjs->find(nom);
if (it == myObjs->end()) return(NULL); 
return((*it).second.obja);
}

/* --Methode-- */
void NamedObjMgr::ListObjs(string & patt)
{
int k;
AnyDataObj* obj=NULL;
string ctyp;
char strg[256];

cout << "NamedObjMgr::ListObjs() NObjs= " << myObjs->size() << "\n" ;
NObjList::iterator it;  k = 0;
for(it = myObjs->begin(); it != myObjs->end(); it++) {
  obj = (*it).second.obj;

  ctyp = typeid(*obj).name();
  sprintf(strg, "%2d/ %16s : %s", k, typeid(*obj).name(), ((*it).first).c_str()); 
  ctyp = strg;
  cout << ctyp << "\n" ;
  k++;
}
cout << endl;
return;
}

/* --Methode-- */
void NamedObjMgr::GetObjList(string & patt, vector<string> &)
{
}

//++
// Titre	Entres-Sorties (I/O) sur les objets
//--
//++
// void  ReadObj(PInPersist& s, int num=-1)
//	Lit l'objet  partir avec le tag numro "num" dans le flot "PInPersist s"
//	et l'ajoute  la liste. Si "num" est ngatif, tous les objets prsents
//	sur le flot "s" sont crs et ajouts  la liste.
// void  ReadObj(string const & nomppf, string nobj="")
//	Lit le premier objet  partir du fichier PPF "nomppf". L'objet est ajout
//	 la liste avec le nom "nobj". Si "nobj" est une chane vide, un nom est 
//	compos  partir du nom de fichier. 
//--

/* --Methode-- */
void NamedObjMgr::ReadObj(string const & flnm, string & nobj)
{
PPersist* obj=NULL;
bool ok = true;

TRY{
  PInPersist pis(flnm);
  obj = PPersistMgr::ReadObject(pis);
  if (obj == NULL) ok = false;
}  CATCH(merr) 
  { printf("NamedObjMgr::ReadObj()/Error Exception= %ld (%s) \n", 
    (long)merr, PeidaExc(merr));  ok = false; } ENDTRY;

if (!ok)  return;
if (nobj.length()<1)  nobj = servnobjm->FileName2Name(flnm);
AddObj(obj->DataObj(), nobj);
return;
}

/* --Methode-- */
void NamedObjMgr::ReadObj(PInPersist& s, int num)
{
int i, cid, key, ln;
int n0, n1;
bool ok = true;
PPersist* obj=NULL;
string nom;

if ( (s.NbTags() < 1) || (num >= s.NbTags()) ) {
  if (num >= 0) { 
    printf("NamedObjMgr::ReadObj(PInPersist, %d) Error! NbTags = %d \n", num, s.NbTags());
    return;
  }
  TRY {
    obj = PPersistMgr::ReadObject(s);
    if (obj == NULL)  ok = false;
  }  CATCH(merr) { 
    printf("NamedObjMgr::ReadObj()/Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr)); 
    ok = false; 
  } ENDTRY;
  if (!ok)  return;
  nom = "";
  AddObj(obj->DataObj(), nom);
}
 
if (num < 0) { n0 = 0; n1 = s.NbTags(); } 
else { n0 = num; n1 = num+1; }
for(i=n0; i<n1; i++) {
  key = s.TagKey(i, cid, ln); 
  if (ln <= 0)  nom = "";
  else nom = s.TagName(i);
  s.GotoTag(i);
  TRY {
    obj = PPersistMgr::ReadObject(s);
    if (obj == NULL)  ok = false;
  }  CATCH(merr) { 
    printf("NamedObjMgr::ReadObj()/Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr)); 
    ok = false; 
  } ENDTRY;
  if (ok)  AddObj(obj->DataObj(), nom);  
}

return;
}
/* --Methode-- */
void NamedObjMgr::ReadAll(string const & flnm)
{
bool ok = true;
PPersist* obj=NULL;

PInPersist* ppin;
TRY{
  ppin = new PInPersist(flnm);
  if (ppin->NbTags() < 1)  obj = PPersistMgr::ReadObject((*ppin));
  else obj = NULL;
}  CATCH(merr) 
  { printf("NamedObjMgr::ReadAll()/Error Exception= %ld (%s) \n", 
    (long)merr, PeidaExc(merr));  ok = false; } ENDTRY;

if (!ok)  return;
if (obj) {
  string nom = servnobjm->FileName2Name(flnm);
  AddObj(obj->DataObj(), nom);
  }
else ReadObj((*ppin), -1);
delete ppin;
return;
}

/* --Methode-- */
void NamedObjMgr::ReadFits(string const & flnm, string & nobj)
{
bool ok = true;
RzImage* obj;

TRY{
//  obj = RzReadFits((char*)flnm.c_str(), ImgOffX, ImgOffY, ImgSizX, ImgSizY, ImgBitSgn);
#ifdef SANS_EVOLPLANCK 
  obj = RzReadFits((char*)flnm.c_str());
#else
  printf("NamedObjMgr::ReadFITS( NON-Disponible EVOL-PLANCK) \n"); 
  obj = NULL;
#endif
  if (obj == NULL)   ok = false;
}  CATCH(merr) { 
  printf("NamedObjMgr::ReadFITS(_Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr)); 
  ok = false; 
} ENDTRY;
if (ok) { 
  if (nobj.length()<1)  nobj = servnobjm->FileName2Name(flnm);
  AddObj((AnyDataObj*)obj, nobj);
}
return;
}


static int key_for_write = 5000;
/* --Methode-- */
void NamedObjMgr::SaveObj(string & nom, POutPersist& s)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL)  return;
printf("NamedObjMgr::SaveObj(%s, )  (Type=%s) \n",  
       nom.c_str(), typeid(*(obja->GetDataObj())).name());
obja->SavePPF(s, nom);
return; 
}

/* --Methode-- */
void NamedObjMgr::SaveAll(string const& flnm)
{
bool ok = true;

POutPersist* pout;
TRY{
  pout = new POutPersist(flnm);
}  CATCH(merr) 
  { printf("NamedObjMgr::SaveAll()/Error Exception= %ld (%s) \n", 
    (long)merr, PeidaExc(merr));  ok = false; } ENDTRY;
if (!ok)  return;
NObjList::iterator it;
string no; 
for(it = myObjs->begin(); it != myObjs->end(); it++) {
  no = (*it).first;
  SaveObj(no, (*pout));
  }
delete pout;
return;
}

/* --Methode-- */
void NamedObjMgr::SaveFits(string& nom, string const & flnm)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL)  return;
obja->SaveFits(flnm);
return;
}



/* --Methode-- */
void NamedObjMgr::PrintObj(string& nom)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL)  return;

string ctyp = typeid(*obja->GetDataObj()).name();
cout << "NamedObjMgr::PrintObj(" << nom << ") Type: " << ctyp << endl;
obja->Print(cout);

return;
}

/* --Methode-- */
void NamedObjMgr::DisplayObj(string& nom, string dopt) 
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::DisplayObj() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return;
 
PIDrawer * dr = NULL;
P2DArrayAdapter* arr = NULL;
dr = obja->GetDrawer(dopt);
if (!dr) arr = obja->Get2DArray(dopt);

if (!dr && !arr) { 
  string ctyp = typeid(*(obja->GetDataObj())).name();
  cout << "NamedObjMgr::DisplayObj() Error , Pas de display pour " << ctyp << endl; 
  return;
  }

int wrsid = 0;
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);

if (dr) { 
  PIDrawer3D * dr3 = dynamic_cast<PIDrawer3D *>(dr);
  if(dr3)  wrsid = myImgApp->Disp3DDrawer(dr3, nom, opt);
  else wrsid = myImgApp->DispScDrawer( dr, nom, opt); 
  }
else if (arr) wrsid = myImgApp->DispImage(arr, nom, opt);

if(wrsid != 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }
if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}

/* --Methode-- */
void NamedObjMgr::DisplayImage(string& nom, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::DisplayImage() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return;
 
P2DArrayAdapter* arr = obja->Get2DArray(dopt);

if (!arr) { 
  string ctyp = typeid(*(obja->GetDataObj())).name();
  cout << "NamedObjMgr::DisplayImage() Error , Non supporte pour " << ctyp << endl; 
  return;
  }

int wrsid = 0;
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
wrsid = myImgApp->DispImage(arr, nom, opt);

if(wrsid != 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }
if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}
/* --Methode-- */
void NamedObjMgr::DisplaySurf3D(string& nom, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::DisplayImage() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return;
 
P2DArrayAdapter* arr = obja->Get2DArray(dopt);

if (!arr) { 
  string ctyp = typeid(*(obja->GetDataObj())).name();
  cout << "NamedObjMgr::DisplaySurf3D() Error , Non supporte pour " << ctyp << endl; 
  return;
  }

if ((arr->XSize() > 250) || (arr->YSize() > 250)) {
  cout << "NamedObjMgr::DisplaySurf3D() Error , 2D-Array(" <<  arr->XSize() 
       << "x" << arr->YSize() << ") trop grand (max=250x250)" << endl; 
  delete arr;
  return;
  }

int wrsid = 0;
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
PISurfaceDrawer* sdr = new PISurfaceDrawer(arr, true, true, true);
wrsid = myImgApp->Disp3DDrawer(sdr, nom, opt);
if(wrsid >= 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }

if (fgsr) myImgApp->RestoreGraphicAtt();  
return; 
}

/* --Methode-- */
void NamedObjMgr::DisplayNT(string& nom, string& nmx, string& nmy, string& nmz,
                            string& erx, string& ery, string& erz, string dopt)
{
AnyDataObj* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::DisplayNT() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return; 

NTupleInterface * nt = dynamic_cast<NTupleInterface *>(obj);
if (nt == NULL) {
// if (typeid(*obj) != typeid(NTupleInterface)) { 
  string ctyp = typeid(*obj).name();
  cout << "NamedObjMgr::DisplayNT() Error , Objet n'est pas un NTuple " << ctyp << endl; 
  return;
  }

int wrsid = 0;
bool fgsr = true;
dopt = "defline," + dopt;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);

if (nmz.length()>0)  { // Display 3D
  PINTuple3D* pin = new PINTuple3D(nt, false);
  pin->SelectXYZ(nmx.c_str(), nmy.c_str(), nmz.c_str());
  pin->SelectErrBar(erx.c_str(), ery.c_str(), erz.c_str());
  string titre = nmz + "%" + nmy + "%" + nmz;
  wrsid = myImgApp->Disp3DDrawer(pin, nom, opt, titre);  
}
else {
  PINTuple* pin = new PINTuple(nt, false);
  pin->SelectXY(nmx.c_str(), nmy.c_str());
  pin->SelectErrBar(erx.c_str(), ery.c_str());
  string titre = nmy + "%" + nmz;
  wrsid = myImgApp->DispScDrawer( (PIDrawer*)pin, nom, opt, titre);
  }

if(wrsid >= 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }

if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}

/* --Methode-- cmv 13/10/98 */
void NamedObjMgr::DisplayGFD(string& nom, string& numvarx, string& numvary, string& err, string dopt)
//	Pour le display 2D ou 3D d'un ``GeneralFitData''.
//|  nom = nom de l'objet GeneralFitData a representer.
//|  numvarx = numero (nombre entier) de la 1ere variable d'abscisse.
//|  numvary = numero (nombre entier) de la 2sd variable d'abscisse (3D).
//|            Pour le display 2D, numvary="" string vide.
//|  err = qu'elles erreurs faut il representer ?
//|      - 2D : x y xy   (display y=f(x))
//|      - 3D : x y z xy xz yz xzy   (display z=f(x,y))
//|      Ceci n'est suivi que si on a PI_NotDefLineAtt, sinon toutes
//|      les barres d'erreurs sont representees.
//|  opt = options generales pour le display.
{
AnyDataObj* obj=GetObj(nom);
if(obj == NULL)
  {cout << "NamedObjMgr::DisplayGFD() Error , Pas d'objet de nom " << nom << endl; 
   return;}
if(!myImgApp) return; 
if(typeid(*obj) != typeid(GeneralFitData))
  {string ctyp = typeid(*obj).name();
   cout<<"NamedObjMgr::DisplayGFD() Error , Objet n'est pas un GeneralFitData "<<ctyp<<endl; 
   return;}

// Decodage des options classiques
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
// Decodage des erreurs a representer
bool errx=false, erry=false, errz=false;
if(err.length()>0) {
  for(int i=0;i<err.length();i++)
    if     (err[i]=='x' || err[i]=='X') errx = true;
    else if(err[i]=='y' || err[i]=='Y') erry = true;
    else if(err[i]=='z' || err[i]=='Z') errz = true;
}
// Decodage des numeros de variables en abscisse
int numvx=-1, numvy=-1;
if(numvarx.length()>0) numvx = atoi(numvarx.c_str());
if(numvary.length()>0) numvy = atoi(numvary.c_str());

int wrsid = 0;
if(numvy>=0) {              // Display 3D
  PIGenFitDat3D* pigfd = new PIGenFitDat3D(((GeneralFitData*)obj),false);
  pigfd->SelectXY(numvx,numvy);
  pigfd->SelectErrBar(errx,erry,errz);
  wrsid = myImgApp->Disp3DDrawer(pigfd,nom,opt);  
} else {                   // Display 2D
  PIGenFitDat* pigfd = new PIGenFitDat(((GeneralFitData*)obj),false);
  pigfd->SelectX(numvx);
  pigfd->SelectErrBar(errx,erry);
  wrsid = myImgApp->DispScDrawer((PIDrawer*)pigfd,nom,opt);
}

if(wrsid >= 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
}
if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}

/* --Methode-- 
void NamedObjMgr::DisplayImage(string& nom, string dopt)
{
  cout << "NamedObjMgr::DisplayImage() a faire ! " << endl;
}
*/




/* --Methode-- */
void NamedObjMgr::SetGraphicAttributes(string gratt)
{
bool fg = false;
servnobjm->DecodeDispOption(gratt, fg);
}
/* --Methode-- */
void NamedObjMgr::SetGraphicWinZone(int nzx, int nzy, bool fcr)
{
if (!myImgApp)  return; 
if (fcr)  myImgApp->CreateGraphWin(nzx, nzy);
else myImgApp->SetZone(nzx, nzy);
}

/* --Methode-- */
void NamedObjMgr::DisplayPoints2D(string& nom,  string& expx, string& expy,
                                  string& experrx, string& experry,
                                  string& expcut, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::DisplayPoints2D() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

// Creation NTuple 
char* ntn[4] = {"expx","expy","expex","expey",};
NTuple* nt = NULL;
bool haserr = false;

if ( (experrx.length() > 0 ) && (experry.length() > 0 ) ) {  haserr = true;   nt = new NTuple(4, ntn); }
else {  haserr = false;  experrx = experry = "0."; nt = new NTuple(2, ntn); }

servnobjm->Nobj_ComputeExpressions(obja, expx, expy, experrx, experry, expcut, nt, NULL, NULL);

if (nt->NEntry() < 1) {
  cout << "NamedObjMgr::DisplayPoints2D() Warning  Zero points satisfy cut !" << endl; 
  delete nt;
  return;
  }

// nt->Show();
// nt->Print(0,10);
PINTuple* pin = new PINTuple(nt, true);
pin->SelectXY(ntn[0], ntn[1]);
if ( haserr ) pin->SelectErrBar(ntn[2], ntn[3]);

bool fgsr = true;
dopt = "defline," + dopt;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
string titre =  nom + ":" + expy + "%" + expx;
myImgApp->DispScDrawer( (PIDrawer*)pin, titre, opt);
if (fgsr) myImgApp->RestoreGraphicAtt();
return;
}

/* --Methode-- */
void NamedObjMgr::DisplayPoints3D(string& nom,  string& expx, string& expy, string& expz,
                                  string& expcut, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::DisplayPoints3D() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

char* ntn[3] = {"expx","expy","expz"};
NTuple* nt = new NTuple(3,ntn);  // Creation NTuple

string expwt = "1.";
servnobjm->Nobj_ComputeExpressions(obja, expx, expy, expz, expwt, expcut, nt, NULL, NULL);

if (nt->NEntry() < 1) {
  cout << "NamedObjMgr::DisplayPoints3D() Warning  Zero points satisfy cut !" << endl;
  delete nt;
  return;
  }
nt->Show();
nt->Print(0,10);
PINTuple3D* pin = new PINTuple3D(nt, true);
pin->SelectXYZ(ntn[0], ntn[1], ntn[2]);
bool fgsr = true;
dopt = "defline," + dopt;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);

// Pour plot a partir de DispScDrawer
// string nomdisp = "_NT3D_";
// myImgApp->DispScDrawer( (PIDrawer*)pin, nomdisp, opt);
// Pour plot a partir de Disp3DDrawer
string titre =  nom + ":" + expy + "%" + expx;
myImgApp->Disp3DDrawer(pin, titre, opt);

if (fgsr) myImgApp->RestoreGraphicAtt();
return;
}

/* --Methode-- */
void NamedObjMgr::ProjectH1(string& nom, string& expx, string& expwt, string& expcut, string& nomh1, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::ProjectH1() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

Histo* h1 = NULL;
NTuple* nt = NULL;
AnyDataObj* oh = NULL; 
if (nomh1.length() > 0) oh=GetObj(nomh1);
else nomh1 = "H1Proj";
if ( (oh != NULL) && (typeid(*oh) == typeid(Histo)) )  h1 = (Histo*)oh;  // Pas de remise a zero ! h1->Zero();
else {
  char* ntn[2]= {"hxval", "hwt"};
  nt = new NTuple(2,ntn);  // Creation NTuple 
  }
string expz = "0.";   
servnobjm->Nobj_ComputeExpressions(obja, expx, expwt, expz, expwt, expcut, nt, h1, NULL);

if ((!h1) && (!nt)) return;
if (!h1) {
  if (nt->NEntry() < 1) {
    cout << "NamedObjMgr::ProjectH1() Warning  Zero points satisfy cut !" << endl; 
    delete nt;
    return;
    }
  double xmin, xmax;
  nt->GetMinMax(0, xmin, xmax);
  h1 = new Histo(xmin, xmax, 100);
  int k;
  float* xn;
  for(k=0; k<nt->NEntry(); k++)    {
    xn = nt->GetVec(k);
    h1->Add(xn[0], xn[1]);    
    }
  delete nt;
  AddObj(h1, nomh1);
  }

DisplayObj(nomh1, dopt);
return;  
}

/* --Methode-- */
void NamedObjMgr::ProjectH2(string& nom, string& expx, string& expy, string& expwt, string& expcut, 
                            string& nomh2, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::ProjectH2() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

Histo2D* h2 = NULL;
NTuple* nt = NULL;
AnyDataObj* oh = NULL;
if (nomh2.length() > 0)  oh=GetObj(nomh2);
else nomh2 = "H2Proj";
if ( (oh != NULL) && (typeid(*oh) == typeid(Histo2D)) )  h2 = (Histo2D*)oh;  // Pas de remise a zero ! h2->Zero();
else {
  char* ntn[3]= {"hxval", "hyval", "hwt"};
  nt = new NTuple(3,ntn);  // Creation NTuple 
  }
string expz = "0.";   
servnobjm->Nobj_ComputeExpressions(obja, expx, expy, expwt, expwt, expcut, nt, NULL, h2);

if ((!h2) && (!nt)) return;
if (!h2) {
  if (nt->NEntry() < 1) {
    cout << "NamedObjMgr::ProjectH2() Warning  Zero points satisfy cut !" << endl; 
    delete nt;
    return;
    }
  double xmin, xmax, ymin, ymax;
  nt->GetMinMax(0, xmin, xmax);
  nt->GetMinMax(0, ymin, ymax);
  h2 = new Histo2D(xmin, xmax, 50, ymin, ymax, 50);
  int k;
  float* xn;
  for(k=0; k<nt->NEntry(); k++)    {
    xn = nt->GetVec(k);
    h2->Add(xn[0], xn[1], xn[2]);    
    }
  delete nt;
  AddObj(h2, nomh2);
  }

DisplayObj(nomh2, dopt);
return;  

}

/* --Methode-- cmv 13/10/98 */
void NamedObjMgr::ProjectHProf(string& nom, string& expx, string& expy, string& expwt, string& expcut, 
                              string& nomprof, string dopt)
//	Pour remplir un ``GeneralFitData'' a partir de divers objets:
//| nom = nom de l'objet a projeter dans un HProf.
//| expx = expression X de definition du bin.
//| expy = expression Y a additionner dans le bin.
//| expwt = expression W du poids a additionner.
//| expcut = expression du test de selection.
//| nomprof = nom du HProf engendre (optionnel). Si l'objet n'existe pas
//|           les limites Xmin,Xmax sont calculees automatiquement.
//|           sinon ce sont celles de l'objet preexistant.
//| opt = options generales pour le display.
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::ProjectHProf() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

HProf* hprof = NULL;
NTuple* nt = NULL;
AnyDataObj* oh = NULL;
if (nomprof.length() > 0)  oh=GetObj(nomprof);
else nomprof = "ProfProj";
if( (oh!=NULL) && (typeid(*oh) == typeid(HProf)) )  hprof = (HProf*)oh;
else {
  char* ntn[3]= {"hxval", "hyval", "hwt"};
  nt = new NTuple(3,ntn);  // Creation NTuple 
}
string expz = "0.";
servnobjm->Nobj_ComputeExpressions(obja, expx, expy, expwt, expwt, expcut, nt, NULL, NULL, hprof);

if((!hprof) && (!nt)) return;
if(!hprof) {
  if (nt->NEntry() < 1) {
    cout << "NamedObjMgr::ProjectHProf() Warning  Zero points satisfy cut !" << endl;
    delete nt;
    return;
  }
  double xmin, xmax;
  nt->GetMinMax(0, xmin, xmax);
  hprof = new HProf(xmin, xmax, 100);
  int k;
  float* xn;
  for(k=0; k<nt->NEntry(); k++)    {
    xn = nt->GetVec(k);
    hprof->Add(xn[0], xn[1], xn[2]);
    }
  delete nt;
  AddObj(hprof, nomprof);
  }
hprof->UpdateHisto();

DisplayObj(nomprof, dopt);
return;
}

/* --Methode-- */
void NamedObjMgr::FillVect(string& nom, string& expx, string& expcut, string& nomvec, string dopt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::FillVect() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

NTuple* nt = NULL;
if (nomvec.length() < 1) nomvec = "VecFill";

char* ntn[2]= {"vecval", "vecwt"};
nt = new NTuple(1,ntn);  // Creation NTuple
 
string expwt = "1.";
string expz = "0.";   
servnobjm->Nobj_ComputeExpressions(obja, expx, expz, expz, expwt, expcut, nt, NULL, NULL);

if (!nt) return;
if (nt->NEntry() < 1) {
  cout << "NamedObjMgr::FillVect() Warning  Zero points satisfy cut !" << endl; 
  delete nt;
  return;
  }
  
Vector* vec = new Vector(nt->NEntry());
int k;
float* xn;
for(k=0; k<nt->NEntry(); k++)    {
  xn = nt->GetVec(k);
  (*vec)(k) = xn[0];
  }
delete nt;
AddObj(vec, nomvec); 
DisplayObj(nomvec, dopt);
return;  
}

/* --Methode-- */
void NamedObjMgr::FillNT(string& nom, string& expx, string& expy, string& expz, string& expt, 
                         string& expcut, string& nomnt)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::FillNT() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

bool fgnnt = false;
NTuple* nt = NULL;
AnyDataObj* oh = NULL;
if (nomnt.length() > 0)  oh=GetObj(nomnt);
else nomnt = "NTFill";
if ( (oh != NULL) && (typeid(*oh) == typeid(HProf)) )  { 
  nt = (NTuple*)oh;
  if (nt->NVar() > 10) {
    cout << "NamedObjMgr::FillNT() Warning , Max 10 var ds NTuple -> new NTuple" << endl;
    nt = NULL;
    }
  }  
if (nt == NULL) {
  char* ntn[4]= {"x", "y","z","t"};
  nt = new NTuple(4,ntn);  // Creation NTuple 
  fgnnt = true;
  }

servnobjm->Nobj_ComputeExpressions(obja, expx, expy, expz, expt, expcut, nt, NULL, NULL);

if (fgnnt) AddObj(nt, nomnt);
return;  

}

/* --Methode-- cmv 13/10/98 */
void NamedObjMgr::FillGFD(string& nom, string& expx, string& expy, string& expz,
                          string& experr, string& expcut, string& nomgfd)
//	Pour remplir un ``GeneralFitData'' a partir de divers objets:
//| nom = nom de l'objet a transcrire selon 1D: Z=f(X) ou 2D: Z=f(X,Y) .
//|       Vector,Matrix,Histo,HProf,Histo2D,Image<T>,StarList,NTuple,GeneralFitData
//| expx = expression X du GeneralFitData (1er abscisse)
//| expy = expression Y du GeneralFitData (2sd abscisse si non "", Z=f(X,Y))
//| expz = expression Z du GeneralFitData (valeur de l'ordonnee)
//| experr = expression de l'erreur sur l'ordonnee Z
//| expcut = expression du test de selection
//| nomgfd = nom du GeneralFitData engendre (optionnel)
{
NObjMgrAdapter* obja=NULL;
obja = GetObjAdapter(nom);
if (obja == NULL) {
  cout << "NamedObjMgr::FillGFD() Error , Pas d'objet de nom "<<nom<<endl;
  return;
  }
if(!myImgApp)  return;

// 2D ou 3D?
int nvar = 2;
if(expy.length()<=0) {nvar = 1; expy = "0.";}

// $CHECK$ - cmv calculait le nombre d'entree ndata
// en fonction de chaque objet - Je l'ai vire Reza 11/05/99

// Creation NTuple Buffer
char* ntn[4]= {"x","y","f","e"};
NTuple*nt = new NTuple(4,ntn);

// Remplissage NTuple buffer
servnobjm->Nobj_ComputeExpressions(obja, expx, expy, expz, experr, expcut, nt, NULL, NULL);
if(nt->NEntry() < 1)
  {cout<<"NamedObjMgr::FillGFD() Warning  Zero points satisfy cut !"<<endl; 
   delete nt; return;}

//Remplissage de la structure GeneraFitData
if (nt->NEntry() <= 0) {
  cout<<"NamedObjMgr::FillGFD() Warning - NData= " << nt->NEntry() << endl;
  delete nt;
  return;
  }

GeneralFitData* gfd = new GeneralFitData(nvar,nt->NEntry(),0);
int k;
float* xn;
for(k=0; k<nt->NEntry(); k++) {
  xn = nt->GetVec(k);
  gfd->AddData(xn,xn[2],xn[3]);
}

// Menage et table d'objets
delete nt;
AddObj(gfd, nomgfd);
return;
}


///////////////////// Fit 1D et 2D //////////////////////////
/* --Function static propre aux routines de fit 1D et 2D-- cmv 13/10/98 */
struct DFOptions {
  int okres, okfun;
  int polcx,polcy; double xc,yc;
  double err_e, err_E;
  double stc2;
  int lp,lpg;
  int i1,i2,j1,j2;
};
typedef struct DFOptions DFOPTIONS;
static void DecodeFitsOptions(string par,string step,string min,string max,string opt
                      ,Vector& Par,Vector& Step,Vector& Min,Vector& Max,DFOPTIONS& O);
void DecodeFitsOptions(string par,string step,string min,string max,string opt
                      ,Vector& Par,Vector& Step,Vector& Min,Vector& Max,DFOPTIONS& O)
//| Pour decoder les "string" et remplir les vecteurs du fit (cf commentaires dans Fit1D)
{
// set des vecteurs et decodage des string correspondantes
int NParMax = 100;
Par.Realloc(NParMax); Step.Realloc(NParMax);
Min.Realloc(NParMax); Max.Realloc(NParMax);
{
  Vector* v; string* s;
  {for(int i=0;i<NParMax;i++) {Par(i)=0.; Step(i)=1.; Min(i)=1.; Max(i)=-1.;}}
  for(int j=0;j<4;j++) {
    if(j==0)      {v=&Par; s=&par;}
    else if(j==1) {v=&Step; s=&step;}
    else if(j==2) {v=&Min; s=&min;}
    else if(j==3) {v=&Max; s=&max;}
    if(s->length()>0) *s += ",";
    for(int i=0;i<NParMax;i++) {
      if(s->length()<=0) break;
      sscanf(s->c_str(),"%lf",&(*v)(i));
      size_t p = s->find_first_of(',') + 1;
      if(p>=s->length()) *s = ""; else *s = s->substr(p);
    }
  }
}

// Decodage de options de opt
O.okres = O.okfun = 0;
O.polcx = O.polcy = 0;
O.xc = O.yc = 0.;
O.stc2 = 1.e-3;
O.err_e = O.err_E = -1.;
O.lp = 1; O.lpg = 0;
O.i1 = O.j1 = O.i2 = O.j2 = -1;

if(opt.length()<=0) return;
opt = "," + opt + ",";

if(strstr(opt.c_str(),",r,")) O.okres = 1;  // residus
if(strstr(opt.c_str(),",f,")) O.okfun = 1;  // fonction fittee
if(strstr(opt.c_str(),",x")) { // Demande de centrage (fit X=x-xc)
  O.polcx = 2; // Le centrage est calcule automatiquement
  size_t p = opt.find(",x");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) {
    sscanf(dum.c_str(),",x%lf",&O.xc);
    O.polcx = 1; // Le centrage est fixe par la valeur lue
  }
}
if(strstr(opt.c_str(),",y")) { // Demande de centrage (fit Y=y-yc)
  O.polcy = 2; // Le centrage est calcule automatiquement
  size_t p = opt.find(",y");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) {
    sscanf(dum.c_str(),",y%lf",&O.yc);
    O.polcy = 1; // Le centrage est fixe par la valeur lue
  }
}
if(strstr(opt.c_str(),",E")) { // Erreurs imposees a "sqrt(val)" ou "aa.b*sqrt(val)"
  size_t p = opt.find(",E");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",E%lf",&O.err_E);
  if(O.err_E<=0.) O.err_E = 1.;
  O.err_e=-1.;
}
if(strstr(opt.c_str(),",e")) { // Erreurs imposees a "1" ou "aa.b"
  size_t p = opt.find(",e");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",e%lf",&O.err_e);
  if(O.err_e<=0.) O.err_e = 1.;
  O.err_E=-1.;
}
if(strstr(opt.c_str(),",X")) { // Valeur du StopChi2
  size_t p = opt.find(",X");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",X%lf",&O.stc2);
  if(O.stc2<=0.) O.stc2 = 1.e-3;
}
if(strstr(opt.c_str(),",l")) { // niveau de print
  size_t p = opt.find(",l");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  float ab;
  if(dum.length()>2) sscanf(dum.c_str(),",l%f",&ab);
  if(ab<0) ab = 0.;
  O.lp = (int) ab; O.lpg = (int) 10.*(ab-O.lp);
}
if(strstr(opt.c_str(),",I")) { // intervalle de fit selon X
  size_t p = opt.find(",I");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",I%d/%d",&O.i1,&O.i2);
}
if(strstr(opt.c_str(),",J")) { // intervalle de fit selon Y
  size_t p = opt.find(",J");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",J%d/%d",&O.j1,&O.j2);
}
return;
}

/* --Methode-- cmv 13/10/98 */
void  NamedObjMgr:: Fit12D(string& nom, string& func,
                          string par,string step,string min,string max,
                          string opt)
//| --------------- Fit d'objets a 1 et 2 dimensions ---------------
//| nom  : nom de l'objet qui peut etre:
//|        fit-1D:  Vector,Histo1D,HProf ou GeneraFitData(1D)
//|        fit-2D:  Matrix,Histo2D,Imagexx ou GeneraFitData(2D)
//| func : pnn = fit polynome degre nn avec classe Poly (lineaire) 1D ou 2D
//|      : Pnn = fit polynome degre nn avec GeneralFit (non-lineaire) 1D ou 2D
//|      : gnn = fit gaussienne (hauteur) + polynome de degre nn 1D
//|      : g   = fit gaussienne (hauteur) 1D
//|      : enn = fit exponentielle + polynome de degre nn 1D
//|      : e   = fit exponentielle 1D
//|      : Gnn = fit gaussienne (volume) + polynome de degre nn 1D
//|      : G   = fit gaussienne (volume) 1D
//|      :     = fit gaussienne+fond (volume) 2D
//|      : Gi  = fit gaussienne+fond integree (volume) 2D
//|      : d   = fit DL de gaussienne+fond (volume) 2D
//|      : di  = fit DL de gaussienne+fond integree (volume) 2D
//|      : D   = fit DL de gaussienne+fond avec coeff variable p6 (volume) 2D
//|      : Di  = fit DL de gaussienne+fond integree avec coeff variable p6 (volume) 2D
//|      : M   = fit Moffat+fond (expos=p6) (volume) 2D
//|      : Mi  = fit Moffat+fond integree (expos=p6) (volume) 2D
//| par  : p1,...,pn valeur d'initialisation des parametres (def=0)
//| step : s1,...,sn valeur des steps de depart (def=1)
//| min  : m1,...,mn valeur des minima (def=1)
//| max  : M1,...,Mn valeur des maxima (def=-1) (max<=min : pas de limite)
//| opt  : options "Eaa.b,eaa.b,f,r,caa.b,Xaa.b"
//|       f = generation d'un Objet identique contenant la fonction fittee
//|       r = generation d'un Objet identique contenant les residus
//|       Xaa.b = aa.b valeur du DXi2 d'arret (def=1.e-3)
//|       la.b = niveau "a.b" de print: a=niveau de print Fit1/2D
//|                                     b=niveau de debug GeneralFit
//|       Ii1/i2 numeros des bins X de l'histos utilises pour le fit [i1,i2]
//|2D     Jj1/j2 numeros des bins Y de l'histos utilises pour le fit [j1,j2]
//|       - L'erreur est celle associee a l'objet si existe, 1 sinon sauf si
//|       E = erreur prise comme la racine de la valeur a fitter
//|       Eaa.b = erreur prise aa.b*sqrt(val)
//|       e = erreur prise egale a 1 pour toutes les valeurs
//|       eaa.b = erreur prise egale a aa.b
//|       xaa.b = demande de centrage: on fit x-aa.b au lieu de x)
//|       x = demande de centrage: on fit x-xc au lieu de x
//|           avec xc=abscisse du milieu de l'histogramme
//|           Actif pour: exp+poly 1D, poly 1D
//|                       gauss+poly 1D (mais xc est le centre de la gaussienne)
//|2D     yaa.b et y = idem "xaa.b et x" mais pour y
{
AnyDataObj* obj=GetObj(nom);
if (obj == NULL) {
  cout<<"NamedObjMgr::Fit12D() Error , Pas d'objet de nom "<<nom<<endl;
  return;
}
if (!myImgApp)  return;
if(func.length()<=0)
  {cout<<"NamedObjMgr::Fit12D() Donnez un nom de fonction a fitter."<<endl;
   return;}
string ctyp = typeid(*obj).name();

int ndim = 0, nbinx=0, nbiny=0, ndata = 0;
Vector* v = NULL; Histo* h = NULL;
Matrix* m = NULL; Histo2D* h2 = NULL; RzImage* im = NULL;
GeneralFitData* g = NULL;

  // 1D
if (typeid(*obj) == typeid(Vector)) {
  ndim = 1;
  v = (Vector*) obj; nbinx = v->NElts(); nbiny = 1;
  }
else if ( (typeid(*obj) == typeid(HProf)) || (typeid(*obj) == typeid(Histo)) ) {
  ndim = 1;
  h = (Histo*)  obj; nbinx = h->NBins(); nbiny = 1;
  }
else if (typeid(*obj) == typeid(Matrix)) {
  ndim = 2;
  m = (Matrix*) obj; nbinx = m->NCol(); nbiny = m->NRows();
  }
else if (typeid(*obj) == typeid(Histo2D)) {
  ndim = 2;
  h2 = (Histo2D*) obj; nbinx = h2->NBinX(); nbiny = h2->NBinY();
  }
else if (typeid(*obj) == typeid(GeneralFitData)) {
  g = (GeneralFitData*) obj; nbinx = g->NData(); nbiny = 1;
  if(     g->NVar()==1) ndim = 1;
  else if(g->NVar()==2) ndim = 2;
  else {
     cout<<"GeneralFitData ne peut avoir que 1 ou 2 variables d'abscisse: "
         <<((GeneralFitData*) obj)->NVar()<<endl; return; }
  }
else if (dynamic_cast<RzImage*>(obj)) {
  ndim = 2;
  im = (RzImage*) obj; nbinx = im->XSize(); nbiny = im->YSize();
  }
else  {
  cout<<"NamedObjMgr::Fit12D() Error , Objet n'est pas un "
      <<"Histo1D/HProf/Vector/Histo2D/Image/Matrix/GeneralFitData "<<ctyp<<endl;
  return; 
  }

ndata = nbinx*nbiny;
if(ndata<=0)
  {cout<<"L'objet a "<<nbinx<<","<<nbiny<<" bins ("<<ndata<<")"<<endl; return;}

// Decodage des options et des parametres, mise en forme
Vector Par(1); Vector Step(1); Vector Min(1); Vector Max(1); DFOPTIONS O;
DecodeFitsOptions(par,step,min,max,opt,Par,Step,Min,Max,O);
O.i1 = (O.i1<0||O.i1>=nbinx)? 0: O.i1;
O.i2 = (O.i2<0||O.i2>=nbinx||O.i2<O.i1)? nbinx-1: O.i2;
if(ndim>=2) {
  O.j1 = (O.j1<0||O.j1>=nbiny)? 0: O.j1;
  O.j2 = (O.j2<0||O.j2>=nbiny||O.j2<O.j1)? nbiny-1: O.j2;
} else O.j2 = O.j1 = 0;
if(O.polcx==2) {
  if(v||m)    O.xc = (O.i2-O.i1+1)/2.;
  else if(h)  O.xc = (h->XMin()+h->XMax())/2.;
  else if(h2) O.xc = (h2->XMin()+h2->XMax())/2.;
  else if(g)  {double mini,maxi; g->GetMinMax(2,mini,maxi); O.xc=(mini+maxi)/2.;}
  else if(im) {O.xc = im->XOrg() * im->XPxSize()*(O.i2-O.i1+1)/2.;}
}
if(O.polcy==2 && ndim>=2) {
  if(m)  O.yc = (O.j2-O.j1+1)/2.;
  if(h2) O.yc = (h2->YMin()+h2->YMax())/2.;
  if(g)  {double mini,maxi; g->GetMinMax(12,mini,maxi); O.yc=(mini+maxi)/2.;}
  if(im) {O.yc = im->YOrg() * im->YPxSize()*(O.j2-O.j1+1)/2.;}
}
if(O.lp>0)
  cout<<"Fit["<<nbinx<<","<<nbiny<<"] ("<<ndata<<") dim="<<ndim<<":"
      <<" Int=["<<O.i1<<","<<O.i2<<"],["<<O.j1<<","<<O.j2<<"]"<<endl
      <<" Cent="<<O.polcx<<","<<O.polcy<<","<<O.xc<<"+x"<<","<<O.yc<<"+y"
      <<" TypE="<<O.err_e<<","<<O.err_E
      <<" StpX2="<<O.stc2
      <<" lp,lpg="<<O.lp<<","<<O.lpg<<endl;

///////////////////////////////////
// Remplissage de GeneralFitData //
///////////////////////////////////
GeneralFitData mydata(ndim,ndata,0);
{for(int i=O.i1;i<=O.i2;i++) for(int j=O.j1;j<=O.j2;j++) {
  double x,y,f,e;

  if(v)
    {x= (double) i; f=(*v)(i); e=1.;}
  else if(h)
    {x=h->BinCenter(i); f=(*h)(i); e=(h->HasErrors())?h->Error(i):1.;}
  else if(m)
    {x=(double) i; y=(double) j; f=(*m)(j,i); e=1.;}
  else if(h2)
    {float xf,yf; h2->BinCenter(i,j,xf,yf); x=(double)xf; y=(double)yf;
     f=(*h2)(i,j); e=(h2->HasErrors())?h2->Error(i,j):1.;}
  else if(im)
    {x=im->XOrg()+(i+0.5)*im->XPxSize(); y=im->YOrg()+(j+0.5)*im->YPxSize();
     f=im->DValue(i,j); e=1.;}
  else if(g&&ndim==1) {x= g->X(i); f=g->Val(i); e=g->EVal(i);}
  else if(g&&ndim==2) {x= g->X(i); y= g->Y(i); f=g->Val(i); e=g->EVal(i);}
  else x=y=f=e=0.;

  // Gestion des erreurs a utiliser
  if(O.err_e>0.) e=O.err_e;
  else if(O.err_E>0.) {e=(y<-1.||y>1.)?O.err_E*sqrt(fabs(y)):O.err_E;}

  // Remplissage de generalfit
  if(func[0]=='p') {x -= O.xc; if(ndim>=2) y -= O.yc;}
  if(ndim==1)      mydata.AddData1(x,f,e);
  else if(ndim==2) mydata.AddData2(x,y,f,e);
}}
if(mydata.NData()<=0)
  {cout<<"Pas de donnees dans GeneralFitData: "<<mydata.NData()<<endl;
   return;}
if(O.lpg>1) {
  mydata.PrintStatus();
  mydata.PrintData(0);
  mydata.PrintData(mydata.NData()-1);
}

////////////////////////////////////////////
// Identification de la fonction a fitter //
////////////////////////////////////////////
GeneralFunction* myfunc = NULL;
if(func[0]=='p' && ndim==1) {
  // Fit de polynome sans passer par les GeneralFit
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit (lineaire) 1D polynome de degre "<<degre<<endl;
  Poly p1(0);
  double c2rl = mydata.PolFit(0,p1,degre);
  cout<<"C2r_lineaire = "<<c2rl<<endl;
  if(O.lp>0) cout<<p1<<endl;
  return;

} else if(func[0]=='P' && ndim==1) {
  // Fit de polynome
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit polynome 1D de degre "<<degre<<endl;
  Polyn1D* myf = new Polyn1D(degre,O.xc);
  myfunc = myf;

} else if(func[0]=='e' && ndim==1) {
  // Fit d'exponentielle
  int degre =-1;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit d'exponentielle+polynome 1D de degre "<<degre<<endl;
  Exp1DPol* myf;
  if(degre>=0) myf = new Exp1DPol((unsigned int)degre,O.xc);
       else    myf = new Exp1DPol(O.xc);
  myfunc = myf;

} else if(func[0]=='g' && ndim==1) {
  // Fit de gaussienne en hauteur
  int degre =-1;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit de Gaussienne_en_hauteur+polynome 1D de degre "<<degre<<endl;
  Gauss1DPol* myf;
  if(degre>=0) myf = new Gauss1DPol((unsigned int)degre,((O.polcx)?true:false));
  else { bool bfg = (O.polcx)?true:false;   myf = new Gauss1DPol(bfg); }
  myfunc = myf;

} else if(func[0]=='G' && ndim==1) {
  // Fit de gaussienne en volume
  int degre =-1;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit de Gaussienne_en_volume+polynome 1D de degre "<<degre<<endl;
  GaussN1DPol* myf;
  if(degre>=0) myf = new GaussN1DPol((unsigned int)degre,((O.polcx)?true:false));
  else  { bool bfg = (O.polcx)?true:false;   myf = new GaussN1DPol(bfg); }
  myfunc = myf;

} else if(func[0]=='p' && ndim==2) {
  // Fit de polynome 2D sans passer par les GeneralFit
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit (lineaire) polynome 2D de degre "<<degre<<endl;
  Poly2 p2(0);
  double c2rl = mydata.PolFit(0,1,p2,degre);
  cout<<"C2r_lineaire = "<<c2rl<<endl;
  if(O.lp>0) cout<<p2<<endl;
  return;

} else if(func[0]=='P' && ndim==2) {
  // Fit de polynome 2D
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit polynome 2D de degre "<<degre<<endl;
  Polyn2D* myf = new Polyn2D(degre,O.xc,O.yc);
  myfunc = myf;

} else if(func[0]=='G' && ndim==2) {
  // Fit de gaussienne+fond en volume
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de Gaussienne+Fond 2D integ="<<integ<<endl;
  if(integ) {GauRhInt2D* myf = new GauRhInt2D; myfunc = myf;}
    else    {GauRho2D* myf = new GauRho2D; myfunc = myf;}

} else if(func[0]=='d' && ndim==2) {
  // Fit de DL gaussienne+fond en volume
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de DL de Gaussienne+Fond 2D integ="<<integ<<endl;
  if(integ) {GdlRhInt2D* myf = new GdlRhInt2D; myfunc = myf;}
    else    {GdlRho2D* myf = new GdlRho2D; myfunc = myf;}

} else if(func[0]=='D' && ndim==2) {
  // Fit de DL gaussienne+fond avec coeff variable p6 en volume
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de DL de Gaussienne+Fond avec coeff variable (p6) 2D integ="<<integ<<endl;
  if(integ) {Gdl1RhInt2D* myf = new Gdl1RhInt2D; myfunc = myf;}
    else    {Gdl1Rho2D* myf = new Gdl1Rho2D; myfunc = myf;}

} else if(func[0]=='M' && ndim==2) {
  // Fit de Moffat+fond (volume)
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de Moffat+Fond (expos=p6) 2D integ="<<integ<<endl;
  if(integ) {MofRhInt2D* myf = new MofRhInt2D; myfunc = myf;}
    else    {MofRho2D* myf = new MofRho2D; myfunc = myf;}

} else {
  cout<<"Fonction "<<func<<" inconnue pour la dim "<<ndim<<endl;
  return;
}

/////////////////////////
// Fit avec generalfit //
/////////////////////////
if(myfunc->NPar()>Par.NElts())
  {cout<<"Trop de parametres: "<<myfunc->NPar()<<">"<<Par.NElts()<<endl;
  if(myfunc) delete myfunc; return;}
GeneralFit myfit(myfunc);
myfit.SetDebug(O.lpg);
myfit.SetData(&mydata);
myfit.SetStopChi2(O.stc2);
{for(int i=0;i<myfunc->NPar();i++) {
  char str[10];
  sprintf(str,"P%d",i);
  myfit.SetParam(i,str,Par(i),Step(i),Min(i),Max(i));
}}
if(O.lp>1) myfit.PrintFit();
double c2r = (double) myfit.Fit();
if(O.lp>0) myfit.PrintFit();
if(c2r>0.) {
  c2r = myfit.GetChi2Red();
  cout<<"C2r_Reduit = "<<c2r<<"  nstep="<<myfit.GetNStep()<<endl;
  Vector ParFit(myfunc->NPar());
  for(int i=0;i<myfunc->NPar();i++) ParFit(i)=myfit.GetParm(i);
} else {
  cout<<"echec Fit, rc = "<<c2r<<"  nstep="<<myfit.GetNStep()<<endl;
}

// Mise a disposition des resultats
if(c2r>=0. && myfunc && (O.okres>0||O.okfun>0)) {
  string nomres = nom + "res";
  string nomfun = nom + "fun";
  if(v) {
    if(O.okres) {Vector* ob = v->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Vector* ob = v->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(h) {
    if(O.okres) {Histo* ob = h->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Histo* ob = h->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(m) {
    if(O.okres) {Matrix* ob = m->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Matrix* ob = m->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(h2) {
    if(O.okres) {Histo2D* ob = h2->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Histo2D* ob = h2->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(im) {
    if(O.okres) {RzImage* ob = im->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {RzImage* ob = im->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(g) {
    if(O.okres) {GeneralFitData* ob = g->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {GeneralFitData* ob = g->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  }
}

// Nettoyage
if(myfunc) delete myfunc;
return;
}

