#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include "timing.h"
#include "ntuple.h"
#include "matharr.h"

#include "constcosmo.h"
#include "cosmocalc.h"
#include "schechter.h"
#include "geneutils.h"
#include "genefluct3d.h"

void usage(void);
void usage(void)
{
 cout<<"cmvobserv3d [...options...]"<<endl
     <<" -a : auto init random seed (needed for multiple simul)"<<endl
     <<" -0 : use ComputeFourier0 method (defaut: no, use normal way)"<<endl
     <<" -G typevol: compute Pk(z=0) and apply growth factor in real space"<<endl
     <<"             typevol=1 noise evolved with distance / observateur (def)"<<endl
     <<"             typevol=2 noise evolved with distance to middle of Z planes"<<endl
     <<"      (default: no, spectrum Pk(z=z_median) for all cube)"<<endl
     <<" -F : filter spectrum by pixel shape (0=no 1=yes(default)"<<endl
     <<" -U : do not poisson fluctuate with Ngal, convert directly to HI mass"<<endl
     <<" -x nx,dx : size along x axis (npix,Mpc)"<<endl
     <<" -y ny,dy : size along y axis (npix,Mpc)"<<endl
     <<"            if ny or dy <=0 take same value as for x"<<endl
     <<" -z nz,dz : size along z axis (redshift axis, npix,Mpc)"<<endl
     <<" -Z zref : redshift for the center of the simulation cube"<<endl
     <<" -s snoise,typevol : gaussian noise sigma in equivalent Msol"<<endl
     <<"           typevol=0 no evolution (def)"<<endl
     <<"           typevol=1 noise evolved with distance / observateur"<<endl
     <<"           typevol=2 noise evolved with distance to middle of Z planes"<<endl
     <<" -2 : compute also 2D spectrum (default: no)"<<endl
     <<" -N scalecube,offsetcube: normalize cube before doing final spectrum (default: automatic)"<<endl
     <<" -M schmin,schmax,nsch : min,max mass and nb points for schechter HI"<<endl
     <<"                         If nsch<0 alors no,bre de points par decade"<<endl
     <<" -Q naleagal : use quick method for turning ngal to mass"<<endl
     <<" -R schmassdist.ppf : read mass distribution for trials from file"<<endl
     <<"               instead of computing it (ONLY if \"-Q\" option is activated)"<<endl
     <<" -A <log10(S_agn in Jy at 1.4 GHz)>,sigma,powlaw :"<<endl
     <<"    AGN mean and sigma gaussian equiv. distrib. for solid angle of centeral pixel"<<endl
     <<"    powlaw: apply S_agn evolution as (Nu/1.4)^powlaw"<<endl
     <<" -W : write cube in FITS format (complex cube is coded as real cube)"<<endl
     <<" -P : write cube in PPF format"<<endl
     <<" -S : write cube slices in PPF format"<<endl
     <<" -V : compute variance from real space (for check, default: no)"<<endl
     <<" -T nth : nombre de threads (si compil multi-thread, default: 0)"<<endl
     <<endl;
}

int main(int narg,char *arg[])
{
 InitTim();

 //-----------------------------------------------------------------
 // *** Survey definition
 long nx=360, ny=-1, nz=64; double dx=1., dy=-1., dz=-1.;
 //long nx=1000, ny=-1, nz=128; double dx=3., dy=-1., dz=6.;
 //long nx=1200, ny=-1, nz=128; double dx=1., dy=-1., dz=3;

 // *** Cosmography definition   (WMAP)
 unsigned short flat = 0;
 double ob0 = 0.0444356;
 double h100=0.71, om0=0.267804, or0=7.9e-05, ol0=0.73,w0=-1.;
 double zref = 0.5;
 double perc=0.01,dzinc=-1.,dzmax=-1.; unsigned short glorder=4;

 // *** Spectrum and variance definition
 double ns = 1., as = 1.;
 double R=8./h100, Rg=R/sqrt(5.);
 double sigmaR = 1.;

 double kmin=1e-5,kmax=1000.;
 int npt = 10000;
 double lkmin=log10(kmin), lkmax=log10(kmax);
 double eps=1.e-3;

 // *** Schechter mass function definition
 double h75 = h100 / 0.75;
 double nstar = 0.006*pow(h75,3.);
 double mstar = pow(10.,9.8);  // MSol
 double alpha = -1.37;

 double schmin=1.e7, schmax=1.e13;
 int schnpt = -100;
 bool use_schmassdist = false;
 long naleagal = 100000;
 bool recompute_schmassdist = true;
 string schmassdistfile = "";
 bool no_poisson = false;

 // *** Niveau de bruit
 double snoise= 0.;   // en equivalent MSol
 int noise_evol = 0;

 // *** AGN
 bool do_agn = false;
 double lfjy_agn=-99., lsigma_agn=0.;   // en Jy
 double powlaw_agn = 0.;

 // *** type de generation
 bool computefourier0=false;
 int use_growth_factor = 0;
 unsigned short nthread=0;
 int filter_by_pixel = 1;

 // *** What to do
 bool comp2dspec = false;
 bool wfits = false;
 bool wppf = false;
 bool wslice = false;
 bool compvarreal = false;
 double scalecube = -1., offsetcube = 0.;

 // --- Decodage arguments
 if(narg>0) {
   cout<<"\n--- Arguments: "<<endl;
   for(int i=0;i<narg;i++) cout<<arg[i]<<" ";
   cout<<endl;
 }

 char c;
 while((c = getopt(narg,arg,"ha0PWSV2UG:F:x:y:z:s:Z:M:A:T:N:Q:R:")) != -1) {
  int nth = 0;
  switch (c) {
  case 'a' :
    Auto_Ini_Ranf(5);
    break;
  case '0' :
    computefourier0 = true;
    break;
  case 'G' :
    sscanf(optarg,"%d",&use_growth_factor);
    break;
  case 'U' :
    no_poisson = true;
    break;
  case 'F' :
    sscanf(optarg,"%d",&filter_by_pixel);
    break;
  case 'x' :
    sscanf(optarg,"%ld,%lf",&nx,&dx);
    break;
  case 'y' :
    sscanf(optarg,"%ld,%lf",&ny,&dy);
    break;
  case 'z' :
    sscanf(optarg,"%ld,%lf",&nz,&dz);
    break;
  case 's' :
    sscanf(optarg,"%lf,%d",&snoise,&noise_evol);
    break;
  case 'Z' :
    sscanf(optarg,"%lf",&zref);
    break;
  case '2' :
    comp2dspec = true;
    break;
  case 'N' :
    sscanf(optarg,"%lf,%lf",&scalecube,&offsetcube);
    break;
  case 'M' :
    sscanf(optarg,"%lf,%lf,%d",&schmin,&schmax,&schnpt);
    break;
  case 'Q' :
    use_schmassdist = true;
    sscanf(optarg,"%ld",&naleagal);
    break;
  case 'R' :
    schmassdistfile = optarg;
    break;
  case 'A' :
    do_agn = true;
    sscanf(optarg,"%lf,%lf,%lf",&lfjy_agn,&lsigma_agn,&powlaw_agn);
    break;
  case 'V' :
    compvarreal = true;
    break;
  case 'W' :
    wfits = true;
    break;
  case 'P' :
    wppf = true;
    break;
  case 'S' :
    wslice = true;
    break;
  case 'T' :
    sscanf(optarg,"%d",&nth);
    nthread = (nth<1)? 0: nth;
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 }

 double lschmin=log10(schmin), lschmax=log10(schmax);
 if(schnpt<=0) {  // alors c'est un nombre de points par decade
   schnpt = long( (-schnpt)*(lschmax-lschmin+1.) + 0.5 );
   if(schnpt<=2) schnpt = 1000;
 }
 if(naleagal<=2) naleagal = 100000;

 cout<<"zref="<<zref<<endl;
 cout<<"nx="<<nx<<" dx="<<dx<<" ny="<<ny<<" dy="<<dy<<" nz="<<nz<<" dz="<<dz<<endl;
 cout<<"kmin="<<kmin<<" ("<<lkmin<<"), kmax="<<kmax<<" ("<<lkmax<<") Mpc^-1"
     <<", npt="<<npt<<endl;
 cout<<"Filter by pixel = "<<filter_by_pixel<<endl;
 cout<<"R="<<R<<" Rg="<<Rg<<" Mpc, sigmaR="<<sigmaR<<endl;
 cout<<"Use_growth_factor = "<<use_growth_factor<<endl;
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;
 cout<<"schmin="<<schmin<<" ("<<lschmin
     <<"), schmax="<<schmax<<" ("<<lschmax<<") Msol"
     <<", schnpt="<<schnpt<<endl;
 if(no_poisson) cout<<"No poisson fluctuation, direct conversion to HI mass"<<endl;
 cout<<"snoise="<<snoise<<" equivalent Msol, evolution="<<noise_evol<<endl;
 cout<<"scalecube="<<scalecube<<", offsetcube="<<offsetcube<<endl;
 if(do_agn)
   cout<<"AGN: <log10(Jy)>="<<lfjy_agn<<" , sigma="<<lsigma_agn
       <<" , powlaw="<<powlaw_agn<<endl;

 string tagobs = "cmvobserv3d.ppf";
 POutPersist posobs(tagobs);

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Cosmology"<<endl;

 CosmoCalc univ(flat,true,zref+1.);
 univ.SetInteg(perc,dzinc,dzmax,glorder);
 univ.SetDynParam(h100,om0,or0,ol0,w0);
 univ.PrtInteg();
 univ.Print();
 double loscomref = univ.Dloscom(zref);
 cout<<"\nzref = "<<zref<<" -> dloscom = "<<loscomref<<" Mpc"<<endl;
 univ.Print(zref);

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Spectrum"<<endl;

 InitialSpectrum pkini(ns,as);

 TransfertEisenstein tf(h100,om0-ob0,ob0,T_CMB_Par,false);
 //tf.SetNoOscEnv(2);

 GrowthFactor growth(om0,ol0);
 // GrowthFactor growth(1.,0.); // D(z) = 1/(1+z)
 double growth_at_z = growth(zref);
 cout<<"...Growth factor at z="<<zref<<" = "<<growth_at_z<<endl;

 PkSpectrum0 pk0(pkini,tf);

 PkSpectrumZ pkz(pk0,growth,zref);

 //-----------------------------------------------------------------
 pkz.SetZ(0.);
 cout<<endl<<"\n--- Compute variance for top-hat R="<<R
     <<" at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_th(pkz,0);
 double kfind_th = varpk_th.FindMaximum(R,kmin,kmax,eps);
 double pkmax_th = varpk_th(kfind_th);
 cout<<"kfind_th = "<<kfind_th<<" ("<<log10(kfind_th)<<"), integrand="<<pkmax_th<<endl;
 double k1=kmin, k2=kmax;
 int rc = varpk_th.FindLimits(R,pkmax_th/1.e4,k1,k2,eps);
 cout<<"limit_th: rc="<<rc<<" : "<<k1<<" ("<<log10(k1)<<") , "
     <<k2<<" ("<<log10(k2)<<")"<<endl;

 double ldlk = (log10(k2)-log10(k1))/npt;
 varpk_th.SetInteg(0.01,ldlk,-1.,4);
 double sr2 = varpk_th.Variance(R,k1,k2);
 cout<<"varpk_th="<<sr2<<"  ->  sigma="<<sqrt(sr2)<<endl;

 double normpkz = sigmaR*sigmaR/sr2;
 pkz.SetScale(normpkz);
 cout<<"Spectrum normalisation = "<<pkz.GetScale()<<endl;

 pkz.SetZ(zref);

 Histo hpkz(lkmin,lkmax,npt); hpkz.ReCenterBin();
 FuncToHisto(pkz,hpkz,true);
 {
 tagobs = "hpkz"; posobs.PutObject(hpkz,tagobs);
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Compute variance for Pk at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_int(pkz,2);

 double kfind_int = varpk_int.FindMaximum(R,kmin,kmax,eps);
 double pkmax_int = varpk_int(kfind_int);
 cout<<"kfind_int = "<<kfind_int<<" ("<<log10(kfind_int)<<"), integrand="<<pkmax_int<<endl;
 double k1int=kmin, k2int=kmax;
 int rcint = varpk_int.FindLimits(R,pkmax_int/1.e4,k1int,k2int,eps);
 cout<<"limit_int: rc="<<rcint<<" : "<<k1int<<" ("<<log10(k1int)<<") , "
     <<k2int<<" ("<<log10(k2int)<<")"<<endl;

 double ldlkint = (log10(k2int)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int = varpk_int.Variance(R,k1int,k2int);
 cout<<"varpk_int="<<sr2int<<"  ->  sigma="<<sqrt(sr2int)<<endl;
 
 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create mass function, compute number/mass density, init mass trials"<<endl;

 Schechter sch(nstar,mstar,alpha);
 sch.Print();

 sch.SetOutValue(0);
 cout<<"sch(mstar) = "<<sch(mstar)<<" /Mpc^3/Msol"<<endl;
 double ngal_by_mpc3 = sch.Integrate(schmin,schmax,schnpt);
 cout<<"Galaxy density number = "<<ngal_by_mpc3<<" /Mpc^3 between limits"<<endl;

 sch.SetOutValue(1);
 cout<<"mstar*sch(mstar) = "<<sch(mstar)<<" Msol/Mpc^3/Msol"<<endl;
 double mass_by_mpc3 = sch.Integrate(schmin,schmax,schnpt);
 cout<<"Galaxy mass density= "<<mass_by_mpc3<<" Msol/Mpc^3 between limits"<<endl;
 cout<<"Omega_HI at z=0 is "<<mass_by_mpc3/(univ.Rhoc(0.)*GCm3toMsolMpc3_Cst)<<endl
     <<"         at z="<<zref<<" is "<<mass_by_mpc3/(univ.Rhoc(zref)*GCm3toMsolMpc3_Cst)<<endl;

 SchechterMassDist schmdist(sch,schmin,schmax,schnpt);
 if(use_schmassdist && schmassdistfile.size()>0) {
   cout<<"\nWARNING: SchechterMassDist read from "<<schmassdistfile<<endl
       <<"      PLEASE CHECK CONSISTENCY WITH REQUESTED PARAMETERS"<<endl;
   schmdist.ReadPPF(schmassdistfile);
   recompute_schmassdist = false;
 }
 schmdist.Print();
 Histo hmdndm = schmdist.GetHmDnDm();
 FunRan tirhmdndm = schmdist.GetTmDnDm();
 {
 tagobs = "hmdndm"; posobs.PutObject(hmdndm,tagobs);
 Histo hdum1(tirhmdndm);
 tagobs = "tirhmdndm"; posobs.PutObject(hdum1,tagobs);
 }

 PrtTim(">>>> End of definition");

 //-----------------------------------------------------------------
 // FFTW3 (p26): faster if sizes 2^a 3^b 5^c 7^d 11^e 13^f  with e+f=0 ou 1
 cout<<endl<<"\n--- Initialisation de GeneFluct3D"<<endl;

 TArray< complex<r_8> > pkgen;
 GeneFluct3D fluct3d(pkgen);
 fluct3d.SetPrtLevel(2);
 fluct3d.SetNThread(nthread);
 fluct3d.SetSize(nx,ny,nz,dx,dy,dz);
 fluct3d.SetObservator(zref,-nz/2.);
 fluct3d.SetCosmology(univ);
 fluct3d.SetGrowthFactor(growth);
 fluct3d.LosComRedshift(0.001,-1);
 //TArray<r_8>& rgen = fluct3d.GetRealArray();
 cout<<endl; fluct3d.Print();
 cout<<"\nMean number of galaxies per pixel = "<<ngal_by_mpc3*fluct3d.GetDVol()<<endl;
 cout<<"Mean mass per pixel = "<<mass_by_mpc3*fluct3d.GetDVol()<<endl;

 double dkmin = fluct3d.GetKincMin();
 double knyqmax = fluct3d.GetKmax();
 long nherr = long(knyqmax/dkmin+0.5);
 cout<<"\nFor HistoErr: d="<<dkmin<<" max="<<knyqmax<<" n="<<nherr<<endl;

 double dktmin = fluct3d.GetKTincMin();
 double ktnyqmax = fluct3d.GetKTmax();
 long nherrt = long(ktnyqmax/dktmin+0.5);
 double dkzmin = fluct3d.GetKinc()[2];
 double kznyqmax = fluct3d.GetKnyq()[2];
 long nherrz = long(kznyqmax/dkzmin+0.5);
 cout<<"For Histo2DErr: d="<<dktmin<<","<<dkzmin
     <<" max="<<ktnyqmax<<","<<kznyqmax<<" n="<<nherrt<<","<<nherrz<<endl;

 //-----------------------------------------------------------------
 cout<<"\n--- Computing spectra variance up to Kmax at z="<<pkz.GetZ()<<endl;
 // En fait on travaille sur un cube inscrit dans la sphere de rayon kmax:
 // sphere: Vs = 4Pi/3 k^3 , cube inscrit (cote k*sqrt(2)): Vc = (k*sqrt(2))^3
 // Vc/Vs = 0.675   ->  keff = kmax * (0.675)^(1/3) = kmax * 0.877
 double knyqmax_mod = 0.877*knyqmax;
 ldlkint = (log10(knyqmax_mod)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int_kmax = varpk_int.Variance(R,k1int,knyqmax_mod);
 cout<<"varpk_int(<"<<knyqmax_mod<<")="<<sr2int_kmax<<"  ->  sigma="<<sqrt(sr2int_kmax)<<endl;

 PrtTim(">>>> End Initialisation de GeneFluct3D");

 //-----------------------------------------------------------------
 cout<<"\n--- Computing a realization in Fourier space"<<endl;
 if(use_growth_factor>0) pkz.SetZ(0.); else pkz.SetZ(zref);
 cout<<"Power spectrum set at redshift: "<<pkz.GetZ()<<endl;
 if(computefourier0) fluct3d.ComputeFourier0(pkz);
   else              fluct3d.ComputeFourier(pkz);
 PrtTim(">>>> End Computing a realization in Fourier space");

 cout<<"\n--- Checking realization spectra"<<endl;
 HistoErr hpkgen(0.,knyqmax,nherr);
 hpkgen.ReCenterBin(); hpkgen.Zero();
 hpkgen.Show();
 fluct3d.ComputeSpectrum(hpkgen);
 {
 tagobs = "hpkgen"; posobs.PutObject(hpkgen,tagobs);
 }
 PrtTim(">>>> End Checking realization spectra");

 if(comp2dspec) {
   cout<<"\n--- Checking realization 2D spectra"<<endl;
   Histo2DErr hpkgen2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkgen2.ReCenterBin(); hpkgen2.Zero();
   hpkgen2.Show();
   fluct3d.ComputeSpectrum2D(hpkgen2);
   {
   tagobs = "hpkgen2"; posobs.PutObject(hpkgen2,tagobs);
   }
   PrtTim(">>>> End Checking realization 2D spectra");
 }

 if(filter_by_pixel!=0) {
   cout<<"\n--- Computing convolution by pixel shape"<<endl;
   fluct3d.FilterByPixel();
   PrtTim(">>>> End Computing convolution by pixel shape");

   cout<<"\n--- Checking realization spectra after pixel shape convol."<<endl;
   HistoErr hpkgenfb(0.,knyqmax,nherr);
   hpkgenfb.ReCenterBin(); hpkgenfb.Zero();
   hpkgenfb.Show();
   fluct3d.ComputeSpectrum(hpkgenfb);
   {
   tagobs = "hpkgenfb"; posobs.PutObject(hpkgenfb,tagobs);
   }
   PrtTim(">>>> End Checking realization spectra");

   cout<<"\n--- Checking realization spectra after pixel shape convol. with pixel correc."<<endl;
   HistoErr hpkgenf(hpkgenfb); hpkgenf.Zero();
   fluct3d.ComputeSpectrum(hpkgenf,0.,filter_by_pixel);
   {
   tagobs = "hpkgenf"; posobs.PutObject(hpkgenf,tagobs);
   }
   PrtTim(">>>> End Checking realization spectra with pixel correc.");

   if(comp2dspec) {
     cout<<"\n--- Checking realization 2D spectra after pixel shape convol."<<endl;
     Histo2DErr hpkgenfb2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
     hpkgenfb2.ReCenterBin(); hpkgenfb2.Zero();
     hpkgenfb2.Show();
     fluct3d.ComputeSpectrum2D(hpkgenfb2);
     {
     tagobs = "hpkgenfb2"; posobs.PutObject(hpkgenfb2,tagobs);
     }
     PrtTim(">>>> End Checking realization 2D spectra");

     cout<<"\n--- Checking realization 2D spectra after pixel shape convol. with pixel correc."<<endl;
     Histo2DErr hpkgenf2(hpkgenfb2); hpkgenf2.Zero();
     fluct3d.ComputeSpectrum2D(hpkgenf2,0.,filter_by_pixel);
     {
     tagobs = "hpkgenf2"; posobs.PutObject(hpkgenf2,tagobs);
     }
     PrtTim(">>>> End Checking realization 2D spectra with pixel correc.");
   }
 }

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_k0.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_k0.ppf",false);
   PrtTim(">>>> End WritePPF");
 }

 //-----------------------------------------------------------------
 cout<<"\n--- Computing a realization in real space"<<endl;
 fluct3d.ComputeReal();
 double rmin,rmax; fluct3d.MinMax(rmin,rmax);
 cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
 PrtTim(">>>> End Computing a realization in real space");

 if(use_growth_factor>0) {
   cout<<"\n--- Apply Growth factor"<<endl;
   cout<<"...D(z=0)="<<growth(0.)<<"  D(z="<<zref<<")="<<growth(zref)<<endl;
   fluct3d.ApplyGrowthFactor(use_growth_factor);
   fluct3d.MinMax(rmin,rmax);
   cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
   PrtTim(">>>> End Applying growth factor");
 }

 int_8 nm;
 double rmref,rs2ref;
 cout<<"\n--- Computing reference variance in real space"<<endl;
 nm = fluct3d.MeanSigma2(rmref,rs2ref);
 cout<<" rs2ref= "<<rs2ref<<" , rmref="<<rmref<<" ("<<nm<<")"<<endl;
 PrtTim(">>>> End Computing reference variance in real space");

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_r0.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_r0.ppf",true);
   PrtTim(">>>> End WritePPF");
 }
 if(wslice) {
   fluct3d.WriteSlicePPF("cmvobserv3d_s_r0.ppf");
   PrtTim(">>>> End WriteSlicePPF");
 }

 cout<<"\n--- Check mean and variance in real space"<<endl;
 fluct3d.NumberOfBad(-1.,1e+200);
 double rm,rs2;
 nm = fluct3d.MeanSigma2(rm,rs2);
 PrtTim(">>>> End Check mean and variance in real space");

 if(compvarreal) {
   cout<<"\n--- Check variance sigmaR in real space"<<endl;
   double varr;
   fluct3d.VarianceFrReal(R,varr);
   cout<<"...Computed variance = "<<varr
       <<" , Theorical variance at (z=0) = "<<pow(sigmaR,1.)
       <<" , at (z="<<zref<<") = "<<pow(sigmaR*growth_at_z,1.)<<endl;
   PrtTim(">>>> End Check variance sigmaR in real space");
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Converting fluctuations into mass"<<endl;
 fluct3d.TurnFluct2Mass();
   nm = fluct3d.MeanSigma2(rm,rs2);
 PrtTim(">>>> End Converting fluctuations into mass");

 if(no_poisson) {

   cout<<"\n--- Converting !!!DIRECTLY!!! mass into HI mass: mass per pixel ="
       <<mass_by_mpc3*fluct3d.GetDVol()<<endl;
   rm = fluct3d.TurnMass2HIMass(mass_by_mpc3);
     nm = fluct3d.MeanSigma2(rm,rs2,0.,1e200);
     nm = fluct3d.MeanSigma2(rm,rs2,0.,1e200,true,0.);
   PrtTim(">>>> End Converting !!!DIRECTLY!!! mass into HI mass");

 } else {

   cout<<"\n--- Converting mass into galaxy number: gal per pixel ="
       <<ngal_by_mpc3*fluct3d.GetDVol()<<endl;
   rm = fluct3d.TurnMass2MeanNumber(ngal_by_mpc3);
     nm = fluct3d.MeanSigma2(rm,rs2,0.,1e200);
     nm = fluct3d.MeanSigma2(rm,rs2,0.,1e200,true,0.);
   PrtTim(">>>> End Converting mass into galaxy number");

   cout<<"\n--- Set negative and null pixels to BAD"<<endl;
   nm = fluct3d.SetToVal(0.,1e+200,-999.);
   PrtTim(">>>> End Set negative pixels to BAD etc...");

   cout<<"\n--- Apply poisson on galaxy number"<<endl;
   fluct3d.ApplyPoisson();
     nm = fluct3d.MeanSigma2(rm,rs2,-998.,1e200);
     nm = fluct3d.MeanSigma2(rm,rs2,0.,1e200,true,0.);
   double xgalmin,xgalmax; fluct3d.MinMax(xgalmin,xgalmax,0.1,1.e50);
   PrtTim(">>>> End Apply poisson on galaxy number");
   if(wslice) {
     fluct3d.WriteSlicePPF("cmvobserv3d_s_rn.ppf");
     PrtTim(">>>> End WriteSlicePPF");
   }

   cout<<"\n--- Convert Galaxy number to HI mass"<<endl;
   double mhi = 0.;
   if(use_schmassdist) {
     if(recompute_schmassdist) {
       int ngalmax = int(xgalmax+0.5);
       schmdist.SetNgalLim(ngalmax,1,naleagal);
       PrtTim(">>>> End creating tabulated histograms for trials");
     }
     mhi = fluct3d.TurnNGal2MassQuick(schmdist);
     schmdist.PrintStatus();
   } else {
     mhi = fluct3d.TurnNGal2Mass(tirhmdndm,true);
   }
   cout<<mhi<<" MSol in survey / "<<mass_by_mpc3*fluct3d.GetVol()<<endl;
     nm = fluct3d.MeanSigma2(rm,rs2,-998.,1e200);
     cout<<"Equivalent: "<<rm*nm/fluct3d.NPix()<<" Msol / pixels"<<endl;
     nm = fluct3d.MeanSigma2(rm,rs2,0.,1e200,true,0.);
   PrtTim(">>>> End Convert Galaxy number to HI mass");

   cout<<"\n--- Set BAD pixels to Zero"<<endl;
   nm = fluct3d.SetToVal(-998.,1e+200,0.);
   PrtTim(">>>> End Set BAD pixels to Zero etc...");

 }

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_r.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_r.ppf",true);
   PrtTim(">>>> End WritePPF");
 }
 if(wslice) {
   fluct3d.WriteSlicePPF("cmvobserv3d_s_r.ppf");
   PrtTim(">>>> End WriteSlicePPF");
 }

 //-----------------------------------------------------------------
 if(do_agn) {
   cout<<"\n--- Add AGN: <log10(S Jy)>="<<lfjy_agn<<" , sigma="<<lsigma_agn
       <<" , powlaw="<<powlaw_agn<<endl;
   fluct3d.AddAGN(lfjy_agn,lsigma_agn,powlaw_agn);
     nm = fluct3d.MeanSigma2(rm,rs2);
   PrtTim(">>>> End Add AGN");
 }

 //-----------------------------------------------------------------
 double snoisesave = 0.;
 if(snoise>0.) {
   cout<<"\n--- Add noise to HI Flux snoise="<<snoise<<", evolution="<<noise_evol<<endl;
   fluct3d.AddNoise2Real(snoise,noise_evol);
   snoisesave = snoise;
     nm = fluct3d.MeanSigma2(rm,rs2);
   PrtTim(">>>> End Add noise");
 }

 if(scalecube!=0.) {  // Si scalecube==0 pas de normalisation
   cout<<"\n--- Scale cube rs2ref="<<rs2ref<<endl;
   if(scalecube<0. && rs2ref>0.) {  // si negatif on scale automatiquement
     nm = fluct3d.MeanSigma2(rm,rs2);
     if(rs2>0.) {scalecube=sqrt(rs2ref)/sqrt(rs2); offsetcube=-rm;}
   }
   cout<<"...scale="<<scalecube<<" offset="<<offsetcube<<endl;
   if(scalecube>0.) {
     fluct3d.ScaleOffset(scalecube,offsetcube);
     snoisesave *= scalecube;
   }
   PrtTim(">>>> End Scale cube");
 }
 
 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_rf.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_rf.ppf",true);
   PrtTim(">>>> End WritePPF");
 }
 if(wslice) {
   fluct3d.WriteSlicePPF("cmvobserv3d_s_rf.ppf");
   PrtTim(">>>> End WriteSlicePPF");
 }

 //-----------------------------------------------------------------
 // -- NE PAS FAIRE CA SI ON VEUT CONTINUER LA SIMULATION -> d_rho/rho ecrase
 
 cout<<endl<<"\n--- ReComputing spectrum from real space"<<endl;
 fluct3d.ReComputeFourier();
 PrtTim(">>>> End ReComputing spectrum");

 if(wfits) {
   fluct3d.WriteFits("!cmvobserv3d_k.fits");
   PrtTim(">>>> End WriteFits");
 }
 if(wppf) {
   fluct3d.WritePPF("cmvobserv3d_k.ppf",false);
   PrtTim(">>>> End WritePPF");
 }

 cout<<endl<<"\n--- Computing final spectrum"<<endl;
 HistoErr hpkrecb(0.,knyqmax,nherr); hpkrecb.Zero();
 hpkrecb.ReCenterBin();
 hpkrecb.Show();
 fluct3d.ComputeSpectrum(hpkrecb);
 {
 tagobs = "hpkrecb"; posobs.PutObject(hpkrecb,tagobs);
 }
 PrtTim(">>>> End Computing final spectrum");

 cout<<endl<<"\n--- Computing final spectrum with pixel deconv."<<endl;
 HistoErr hpkrec(hpkrecb); hpkrec.Zero();
 fluct3d.ComputeSpectrum(hpkrec,snoisesave,filter_by_pixel);
 {
 tagobs = "hpkrec"; posobs.PutObject(hpkrec,tagobs);
 }
 PrtTim(">>>> End Computing final spectrum with pixel deconv.");

 if(comp2dspec) {
   cout<<"\n--- Computing final 2D spectrum"<<endl;
   Histo2DErr hpkrecb2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkrecb2.ReCenterBin(); hpkrecb2.Zero();
   hpkrecb2.Show();
   fluct3d.ComputeSpectrum2D(hpkrecb2);
   {
   tagobs = "hpkrecb2"; posobs.PutObject(hpkrecb2,tagobs);
   }
   PrtTim(">>>> End Computing final 2D spectrum");

   cout<<"\n--- Computing final 2D spectrum with pixel deconv."<<endl;
   Histo2DErr hpkrec2(hpkrecb2); hpkrec2.Zero();
   fluct3d.ComputeSpectrum2D(hpkrec2,snoisesave,filter_by_pixel);
   {
   tagobs = "hpkrec2"; posobs.PutObject(hpkrec2,tagobs);
   }
   PrtTim(">>>> End Computing final 2D spectrum with pixel deconv.");

 }

 PrtTim(">>>> End Of Job");
 return 0;
}

