#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include "constcosmo.h"
#include "cosmocalc.h"
#include "geneutils.h"
#include "schechter.h"
#include "planckspectra.h"

/* --- Check Peterson at al. astro-ph/0606104 v1  (pb facteur sqrt(2) sur S/N !)
cmvdefsurv -U 0.75,0.3,0.7,-1,1 -V 300 -z 0.0025,0.2,Z -x 1,90,A -O 400000,6000 -N 75 -M 6.156e9 -F 3 -2 1.5
 --- */

inline double rad2deg(double trad) {return trad/M_PI*180.;}
inline double rad2min(double trad) {return trad/M_PI*180.*60.;}
inline double rad2sec(double trad) {return trad/M_PI*180.*3600.;}
inline double deg2rad(double tdeg) {return tdeg*M_PI/180.;}
inline double min2rad(double tmin) {return tmin*M_PI/(180.*60.);}
inline double sec2rad(double tsec) {return tsec*M_PI/(180.*3600.);}

void usage(void);
void usage(void) {
  cout<<"cmvdefsurv [-r] -x adtx,atxlarg[,unit_x] -y adty,atylarg[,unit_y] -z dred,redlarg[,unit_z] redshift"<<endl
      <<"----------------"<<endl
      <<" -x adtx,atxlarg : resolution et largeur dans le plan transverse selon X"<<endl
      <<" -y adty,atylarg : idem selon Y, si <=0 meme que X"<<endl
      <<" -z dred,redlarg : resolution et largeur sur la ligne de visee"<<endl
      <<"-- Unites pour X-Y:"<<endl
      <<"   \'A\' : en angles (pour X-Y)     : resolution=ArcMin, largeur=Degre (defaut)"<<endl
      <<"   \'Z\' : en redshift (pour Z)     : resolution et largeur en redshift (defaut)"<<endl
      <<"   \'F\' : en frequence (pour Z)    : resolution et largeur MHz"<<endl
      <<"   \'M\' : en distance (pour X-Y-Z) : resolution et largeur Mpc"<<endl
      <<"----------------"<<endl
      <<" -O surf,tobs : surface effective (m^2) et temps d\'observation (s)"<<endl
      <<" -N Tsys : temperature du system (K)"<<endl
      <<" -L lobewidth,freqlob : taille du lobe d\'observation (FWHM) en arcmin (def= 1\')"<<endl
      <<"                        pour la frequence freqlob en MHz"<<endl
      <<"            Si lobewidth<=0 : l'angle solide du lobe = celui du pixel"<<endl
      <<"            Si freqlob<=0 : la frequence de reference est celle du redshift etudie"<<endl
      <<"            Si freqlob absent : la frequence de reference 1.4 GHz"<<endl
      <<" -2 : two polarisations measured"<<endl
      <<" -M  : masse de HI de reference (MSol), si <=0 mean schechter in pixel"<<endl
      <<" -F  : HI flux factor to be applied for our redshift"<<endl
      <<" -V Vrot : largeur en vitesse (km/s) pour l\'elargissement doppler (def=300km/s)"<<endl
      <<"----------------"<<endl
      <<" -S Tsynch,indnu : temperature (K) synch a 408 Mhz, index d\'evolution"<<endl
      <<"                   (indnu==0 no evolution with freq.)"<<endl
      <<"----------------"<<endl
      <<" -U h100,om0,ol0,w0,or0,flat : cosmology"<<endl
      <<"----------------"<<endl
      <<" -A <log10(S_agn)> : moyenne du flux AGN en Jy dans le pixel"<<endl
      <<" redshift : redshift moyen du survey"<<endl
      <<endl;
}

int main(int narg,char *arg[])
{
 // --- Valeurs fixes
 // WMAP
 unsigned short flat = 0;
 double h100=0.71, om0=0.267804, or0=7.9e-05*0., ol0=0.73,w0=-1.;
 // Schechter
 double h75 = h100 / 0.75;
 double nstar = 0.006*pow(h75,3.);  //
 double mstar = pow(10.,9.8);  // MSol
 double alpha = -1.37;
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;

 // --- Arguments
 double adtx=0., atxlarg=0., dx=0.,txlarg=0.;
   int nx=0; char unit_x = 'A';
 double adty=-1., atylarg=-1., dy=0.,tylarg=0.;
   int ny=0; char unit_y = 'A';
 double dred=0., redlarg=0., dz=0.,tzlarg=0.;
   int nz=0; char unit_z = 'Z';
 double redshift = 0.;
 double tobs = 6000., surfeff = 400000.;
 // taille du lobe d'observation en arcmin pour la frequence
 double lobewidth0 = -1., lobefreq0 = Fr_HyperFin_Par*1.e3;
 double Tsys=75.;
 // a 408 MHz (Haslam) + evol index a -2.6
 double Tsynch408=60., nuhaslam=0.408, indnu = -2.6;
 double mhiref = -1.; // reference Mass en HI (def integ schechter)
 double hifactor = 1.;
 double vrot = 300.; // largeur en vitesse (km/s) pour elargissement doppler
 bool ya2polar = false;
 double facpolar = 0.5; // si on mesure les 2 polars -> 1.0
 double lflux_agn = -3.;

 // --- Decodage arguments
 char c;
  while((c = getopt(narg,arg,"h2x:y:z:N:S:O:M:F:V:U:L:A:")) != -1) {
  switch (c) {
  case 'x' :
    sscanf(optarg,"%lf,%lf,%c",&adtx,&atxlarg,&unit_x);
    break;
  case 'y' :
    sscanf(optarg,"%lf,%lf,%c",&adty,&atylarg,&unit_y);
    break;
  case 'z' :
    sscanf(optarg,"%lf,%lf,%c",&dred,&redlarg,&unit_z);
    break;
  case 'O' :
    sscanf(optarg,"%lf,%lf",&surfeff,&tobs);
    break;
  case 'L' :
    sscanf(optarg,"%lf,%lf",&lobewidth0,&lobefreq0);
    break;
  case 'N' :
    sscanf(optarg,"%lf",&Tsys);
    break;
  case 'S' :
    sscanf(optarg,"%lf,%lf",&Tsynch408,&indnu);
    break;
  case 'M' :
    sscanf(optarg,"%lf",&mhiref);
    break;
  case 'F' :
    sscanf(optarg,"%lf",&hifactor);
    break;
  case 'V' :
    sscanf(optarg,"%lf",&vrot);
    break;
  case 'U' :
    sscanf(optarg,"%lf,%lf,%lf,%lf,%hu",&h100,&om0,&ol0,&w0,&flat);
    break;
  case '2' :
    ya2polar = true;
    facpolar = 1.0;
    break;
  case 'A' :
    sscanf(optarg,"%lf",&lflux_agn);
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 } 
 if(optind>=narg) {usage(); return-1;}
 sscanf(arg[optind],"%lf",&redshift);
 if(redshift<=0.) {cout<<"Redshift "<<redshift<<" should be >0"<<endl; return -2;}

 // --- Initialisation de la Cosmologie
 cout<<"\n>>>>\n>>>> Cosmologie generale\n>>>>"<<endl;
 cout<<"h100="<<h100<<" Om0="<<om0<<" Or0="<<or0<<" Or0="
     <<or0<<" Ol0="<<ol0<<" w0="<<w0<<" flat="<<flat<<endl;
 cout<<"--- Cosmology for z = "<<redshift<<endl;
 CosmoCalc univ(flat,true,2.*redshift);
 double perc=0.01,dzinc=redshift/100.,dzmax=dzinc*10.; unsigned short glorder=4;
 univ.SetInteg(perc,dzinc,dzmax,glorder);
 univ.SetDynParam(h100,om0,or0,ol0,w0);
 univ.Print(0.);
 univ.Print(redshift);

 double dang    = univ.Dang(redshift);
 double dtrcom  = univ.Dtrcom(redshift);
 double dlum    = univ.Dlum(redshift);
 double dloscom = univ.Dloscom(redshift);
 double dlosdz  = univ.Dhubble()/univ.E(redshift);
 cout<<"dang="<<dang<<" dlum="<<dlum<<" dtrcom="<<dtrcom
     <<" dloscom="<<dloscom<<" dlosdz="<<dlosdz<<" Mpc"<<endl;

 cout<<"\n1\" -> "<<dang*sec2rad(1.)<<" Mpc = "<<dtrcom*sec2rad(1.)<<" Mpc com"<<endl;
 cout<<"1\' -> "<<dang*min2rad(1.)<<" Mpc = "<<dtrcom*min2rad(1.)<<" Mpc com"<<endl;
 cout<<"1d -> "<<dang*deg2rad(1.)<<" Mpc = "<<dtrcom*deg2rad(1.)<<" Mpc com"<<endl;

 cout<<"dz=1 -> "<<dlosdz<<" Mpc com"<<endl;

 cout<<"1 Mpc los com -> dz = "<<1./dlosdz<<endl;
 cout<<"1 Mpc transv com -> "<<rad2sec(1./dtrcom)<<"\" = "
     <<rad2min(1./dtrcom)<<" \' = "<<rad2deg(1./dtrcom)<<" d"<<endl;

 // --- Mise en forme dans les unites appropriees
 cout<<"\n>>>>\n>>>> Geometrie\n>>>>"<<endl;
 if(adty<=0. || atylarg<=0.) {adty=adtx; atylarg=atxlarg; unit_y=unit_x;}
 cout<<"X values: resolution="<<adtx<<" largeur="<<atxlarg<<" unite="<<unit_x<<endl;
 if(unit_x == 'A') {
   nx = int(atxlarg*60./adtx+0.5);
   adtx = min2rad(adtx); atxlarg = deg2rad(atxlarg);
   dx = adtx*dtrcom; txlarg = dx*nx;
 } else if(unit_x == 'M') {
   nx = int(atxlarg/adtx+0.5);
   dx = adtx; txlarg = atxlarg;
   adtx = dx/dtrcom; atxlarg = adtx*nx;
 } else {
   cout<<"Unknown unit_x = "<<unit_x<<endl;
 }
 cout<<"Y values: resolution="<<adty<<" largeur="<<atylarg<<" unite="<<unit_y<<endl;
 if(unit_y == 'A') {
   ny = int(atylarg*60./adty+0.5);
   adty = min2rad(adty); atylarg = deg2rad(atylarg);
   dy = adty*dtrcom; tylarg = dy*ny;
 } else if(unit_y == 'M') {
   ny = int(atylarg/adty+0.5);
   dy = adty; tylarg = atylarg;
   adty = dy/dtrcom; atylarg = adty*ny;
 } else {
   cout<<"Unknown unit_y = "<<unit_y<<endl;
 }
 cout<<"Z values: resolution="<<dred<<" largeur="<<redlarg<<" unite="<<unit_z<<endl;
 if(unit_z == 'Z') {
   nz = int(redlarg/dred+0.5);
   dz = dred*dlosdz; tzlarg = dz*nz;
 } else if(unit_z == 'M') {
   nz = int(redlarg/dred+0.5);
   dz = dred; tzlarg = redlarg;
   dred = dz/dlosdz; redlarg = dred*nz;
 } else if(unit_z == 'F') {
   nz = int(redlarg/dred+0.5);
   dred = dred/(Fr_HyperFin_Par*1.e3)*pow(1.+redshift,2.); redlarg = dred*nz;
   dz = dred*dlosdz; tzlarg = dz*nz;
 } else {
   cout<<"Unknown unit_z = "<<unit_z<<endl;
 }

 double Npix = (double)nx*(double)ny*(double)nz;
 double redlim[2] = {redshift-redlarg/2.,redshift+redlarg/2.};
 if(redlim[0]<=0.)
   {cout<<"Lower redshift limit "<<redlim[0]<<" should be >0"<<endl; return -3;}
 double dtrlim[2]  = {univ.Dtrcom(redlim[0]),univ.Dtrcom(redlim[1])};
 double loslim[2]  = {univ.Dloscom(redlim[0]), univ.Dloscom(redlim[1])};
 double dlumlim[2] = {univ.Dlum(redlim[0]),univ.Dlum(redlim[1])};

 cout<<"---- Line of Sight: Redshift = "<<redshift<<endl
     <<"dred = "<<dred<<"  redlarg = "<<redlarg<<endl
     <<"  dz = "<<dz<<" Mpc  redlarg = "<<tzlarg<<" Mpc com, nz = "<<nz<<" pix"<<endl;
 cout<<"---- Transverse X:"<<endl
     <<"adtx = "<<rad2min(adtx)<<"\',  atxlarg = "<<rad2deg(atxlarg)<<" d"<<endl
     <<"  dx = "<<dx<<" Mpc,  txlarg = "<<txlarg<<" Mpc com, nx = "<<nx<<" pix"<<endl;
 cout<<"---- Transverse Y:"<<endl
     <<"adty = "<<rad2min(adty)<<"\',  atylarg = "<<rad2deg(atylarg)<<" d"<<endl
     <<"  dy = "<<dy<<" Mpc,  tylarg = "<<tylarg<<" Mpc com, ny = "<<ny<<" pix"<<endl;
 cout<<"---- Npix total = "<<Npix<<" -> "<<Npix*sizeof(double)/1.e6<<" Mo"<<endl;

 // --- Cosmolographie Transverse
 cout<<"\n>>>>\n>>>> Cosmologie & Geometrie transverse\n>>>>"<<endl;
 cout<<"dang comoving = "<<dtrcom<<" Mpc (com) var_in_z ["
                         <<dtrlim[0]<<","<<dtrlim[1]<<"]"<<endl;

 cout<<"... dx = "<<dx<<" Mpc (com), with angle "<<adtx*dtrcom<<endl
     <<"       with angle var_in_z ["<<adtx*dtrlim[0]<<","<<adtx*dtrlim[1]<<"]"<<endl;
 cout<<"... largx = "<<txlarg<<" Mpc (com), with angle "<<atxlarg*dtrcom<<endl
     <<"          with angle var_in_z ["<<atxlarg*dtrlim[0]<<","<<atxlarg*dtrlim[1]<<"]"<<endl;

 cout<<"... dy = "<<dy<<" Mpc (com), with angle "<<adty*dtrcom<<endl
     <<"       with angle var_in_z ["<<adty*dtrlim[0]<<","<<adty*dtrlim[1]<<"]"<<endl;
 cout<<"... largy = "<<tylarg<<" Mpc (com), with angle "<<atylarg*dtrcom<<endl
     <<"          with angle var_in_z ["<<atylarg*dtrlim[0]<<","<<atylarg*dtrlim[1]<<"]"<<endl;

 // --- Cosmolographie Line of sight
 cout<<"\n>>>>\n>>>> Cosmologie & Geometrie ligne de visee\n>>>>"<<endl;
 cout<<"los comoving distance = "<<dloscom<<" Mpc (com) in ["
     <<loslim[0]<<","<<loslim[1]<<"]"<<endl
     <<"                                    diff = "
     <<loslim[1]-loslim[0]<<" Mpc"<<endl;

 cout<<"...dz = "<<dz<<" Mpc (com), with redshift approx "<<dred*dlosdz<<endl;
 cout<<"...tzlarg = "<<tzlarg<<" Mpc (com), with redshift approx "<<redlarg*dlosdz<<endl;

 // --- Solid Angle & Volume
 cout<<"\n>>>>\n>>>> Angles solides et Volumes\n>>>>"<<endl;
 cout<<"--- Solid angle"<<endl;
 double angsol = AngSol(adtx/2.,adty/2.,M_PI/2.);
 cout<<"Elementary solid angle = "<<angsol<<" sr = "<<angsol/(4.*M_PI)<<" *4Pi sr"<<endl;
 double angsoltot = AngSol(atxlarg/2.,atylarg/2.,M_PI/2.);
 cout<<"Total solid angle = "<<angsoltot<<" sr = "<<angsoltot/(4.*M_PI)<<" *4Pi sr"<<endl;

 cout<<"\n--- Volume"<<endl;
 double dvol = dx*dy*dz;
 cout<<"Pixel volume comoving = "<<dvol<<" Mpc^3"<<endl;
 double vol = univ.Vol4Pi(redlim[0],redlim[1])/(4.*M_PI)*angsoltot;
 cout<<"Volume comoving = "<<vol<<" Mpc^3 = "<<vol/1.e9<<" Gpc^3"<<endl
     <<"Pixel volume comoving = vol/Npix = "<<vol/Npix<<" Mpc^3"<<endl;

 // --- Fourier space: k = omega = 2*Pi*Nu
 cout<<"\n>>>>\n>>>> Geometrie dans l'espace de Fourier\n>>>>"<<endl;
 cout<<"Array size: nx = "<<nx<<",  ny = "<<ny<<",  nz = "<<nz<<endl;
 double dk_x = 2.*M_PI/(nx*dx), knyq_x = M_PI/dx;
 double dk_y = 2.*M_PI/(nx*dy), knyq_y = M_PI/dy;
 double dk_z = 2.*M_PI/(nz*dz), knyq_z = M_PI/dz;
 cout<<"Resolution: dk_x = "<<dk_x<<" Mpc^-1  (2Pi/dk_x="<<2.*M_PI/dk_x<<" Mpc)"<<endl
     <<"            dk_y = "<<dk_y<<" Mpc^-1  (2Pi/dk_y="<<2.*M_PI/dk_y<<" Mpc)"<<endl;
 cout<<"Nyquist:    kx = "<<knyq_x<<" Mpc^-1  (2Pi/knyq_x="<<2.*M_PI/knyq_x<<" Mpc)"<<endl
     <<"            ky = "<<knyq_y<<" Mpc^-1  (2Pi/knyq_y="<<2.*M_PI/knyq_y<<" Mpc)"<<endl;
 cout<<"Resolution: dk_z = "<<dk_z<<" Mpc^-1  (2Pi/dk_z="<<2.*M_PI/dk_z<<" Mpc)"<<endl;
 cout<<"Nyquist:    kz = "<<knyq_z<<" Mpc^-1  (2Pi/knyq_z="<<2.*M_PI/knyq_z<<" Mpc)"<<endl;

 // --- Masse de HI
 cout<<"\n>>>>\n>>>> Mass HI\n>>>>"<<endl;
 Schechter sch(nstar,mstar,alpha);
 sch.SetOutValue(1);
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;
 cout<<"mstar*sch(mstar) = "<<sch(mstar)<<" Msol/Mpc^3/Msol"<<endl;
 int npt = 10000;
 double lnx1=log10(1.e+6), lnx2=log10(1.e+13), dlnx=(lnx2-lnx1)/npt;
 double masshimpc3 = IntegrateFuncLog(sch,lnx1,lnx2,0.001,dlnx,10.*dlnx,6);
 cout<<"Mass density: "<<masshimpc3<<" Msol/Mpc^3"<<endl;

 double masshipix = masshimpc3*dvol;
 double masshitot = masshimpc3*vol;
 cout<<"Pixel mass = "<<masshipix<<" Msol"<<endl
     <<"Total mass in survey = "<<masshitot<<" Msol"<<endl;
 if(mhiref<=0.)  mhiref = masshipix;

 sch.SetOutValue(0);
 cout<<"\nsch(mstar) = "<<sch(mstar)<<" /Mpc^3/Msol"<<endl;
 cout<<"Galaxy number density:"<<endl;
 for(double x=lnx1; x<lnx2-0.5; x+=1.) {
   double n = IntegrateFuncLog(sch,x,lnx2,0.001,dlnx,10.*dlnx,6);
   cout<<"  m>"<<x<<" Msol: "<<n<<" /Mpc^3, "<<n*dvol<<" /pixel, "
       <<n*vol<<" in survey"<<endl;
 }
 sch.SetOutValue(1);


 // --- Survey values
 cout<<"\n>>>>\n>>>> Observations\n>>>>"<<endl;
 double unplusz = 1.+redshift;
 double nuhiz = Fr_HyperFin_Par / unplusz;  // GHz
 // dnu = NuHi/(1.+z0-dz/2) - NuHi/(1.+z0+dz/2)
 //     = NuHi*dz/(1.+z0)^2 * 1/[1-(dz/(2*(1+z0)))^2]
 //    ~= NuHi*dz/(1.+z0)^2
 double dnuhiz = Fr_HyperFin_Par *dred/(unplusz*unplusz)
               / (1.- pow(dred/.2/unplusz,2.));
 cout<<"      surf_eff="<<surfeff<<" m^2, tobs="<<tobs<<" s"<<endl
     <<"      nu="<<nuhiz<<" GHz, dnu="<<dnuhiz*1.e3<<" Mhz"<<endl;
 cout<<"dang lumi = "<<dlum<<" in ["<<dlumlim[0]<<","<<dlumlim[1]<<"] Mpc"<<endl;

 double nlobes = 1.;
 if(lobewidth0>0.) {
   double lobewidth = lobewidth0;   // ArcMin
   if(lobefreq0<=0.) lobefreq0 = nuhiz*1.e3; // MHz
   // La taille angulaire du lobe change avec la frequence donc avec le redshift
   lobewidth *= lobefreq0/(nuhiz*1.e3);
   cout<<"\n--- Lobe: width="<<lobewidth0<<" pour "<<lobefreq0<<" MHz"<<endl
       <<"          changed to "<<lobewidth<<" pour "<<nuhiz*1.e3<<" MHz"<<endl;
   double slobe = lobewidth/2.35482; // sigma du lobe en arcmin
   double lobecyl = sqrt(8.)*slobe; // diametre du lobe cylindrique equiv en arcmin
   double lobearea = M_PI*lobecyl*lobecyl/4.;  // en arcmin^2 (hypothese lobe gaussien)
   nlobes = rad2min(adtx)*rad2min(adty)/lobearea;
   cout<<"Beam FWHM = "<<lobewidth<<"\' -> sigma = "<<slobe<<"\' -> "
       <<" Dcyl = "<<lobecyl<<"\' -> area = "<<lobearea<<" arcmin^2"<<endl;
   cout<<"Number of beams in one transversal pixel = "<<nlobes<<endl;
 }

 // --- Power emitted by HI
 cout<<"\n--- Power from HI for M = "<<mhiref<<" Msol at "<<nuhiz<<" GHz"<<endl;
 cout<<"flux factor = "<<hifactor<<" at redshift = "<<redshift<<endl;

 double fhi = hifactor*Msol2FluxHI(mhiref,dlum);
 cout<<"FluxHI("<<dlum<<" Mpc) all polar:"<<endl
     <<"  Flux= "<<fhi<<" W/m^2 = "<<fhi/Jansky2Watt_cst<<" Jy.Hz"<<endl
     <<"      in ["<<hifactor*Msol2FluxHI(mhiref,dlumlim[0])
     <<","<<hifactor*Msol2FluxHI(mhiref,dlumlim[1])<<"] W/m^2"<<endl;
 double sfhi = fhi / (dnuhiz*1e9) / Jansky2Watt_cst;
 cout<<"If spread over pixel depth ("<<dnuhiz<<" GHz), flux density = "<<sfhi<<" Jy"<<endl;

 // --- Signal analysis
 cout<<"\n--- Signal analysis"<<endl;
 cout<<"Facteur polar = "<<facpolar<<endl;

 PlanckSpectra planck(T_CMB_Par);
 planck.SetSpectraApprox(PlanckSpectra::RAYLEIGH);  // Rayleigh spectra
 planck.SetSpectraVar(PlanckSpectra::NU); // frequency
 planck.SetSpectraPower(PlanckSpectra::POWER); // output en W/....
 planck.SetSpectraUnit(PlanckSpectra::ANGSFLUX); // radiance W/m^2/Sr/Hz

 // Signal
 double psig_2polar = fhi * surfeff;
 double tsig_2polar = psig_2polar / k_Boltzman_Cst / (dnuhiz*1e9);
 double ssig_2polar = psig_2polar / surfeff / (dnuhiz*1e9) / Jansky2Watt_cst;
 double psig = facpolar * psig_2polar;
 double tsig = facpolar * tsig_2polar;
 double ssig = facpolar * ssig_2polar;
 cout<<"\nSignal("<<mhiref<<" Msol):"<<endl
     <<"     P="<<psig<<" W"<<endl
     <<"     flux density = "<<ssig*1.e6<<" mu_Jy  (for Dnu="<<dnuhiz<<" GHz)"<<endl
     <<"     Antenna temperature: tsig="<<tsig<<" K"<<endl;

 // Elargissement doppler de la raie a 21cm: dNu = vrot/C * Nu(21cm) / (1+z)
 double doplarge = vrot / SpeedOfLight_Cst * nuhiz;
 double dzvrot = vrot / SpeedOfLight_Cst * unplusz;
 cout<<"     Doppler width="<<doplarge*1.e3<<" MHz  for rotation width of "<<vrot<<" km/s"<<endl
     <<"                 dx= "<<dzvrot<<" a z="<<redshift<<endl;
 if(doplarge>dnuhiz)
   cout<<"Warning: doppler width "<<doplarge<<" GHz > "<<dnuhiz<<" GHz redshift bin width"<<endl;

 // Synchrotron  (T en -2.7  -> Flux en -0.7  dans l'approximation Rayleigh)
 double tsynch = Tsynch408;
        if(fabs(indnu)>1.e-50) tsynch *= pow(nuhiz/nuhaslam,indnu);
 planck.SetTemperature(tsynch);
 double psynch_2polar = planck(nuhiz*1.e+9) * surfeff * angsol * (dnuhiz*1e9);
 double ssynch_2polar = psynch_2polar / surfeff / (dnuhiz*1e9) / Jansky2Watt_cst;
 double psynch = facpolar * psynch_2polar;
 double ssynch = facpolar * ssynch_2polar;
 cout<<"\nSynchrotron: T="<<Tsynch408<<" K ("<<nuhaslam<<" GHz), "
     <<tsynch<<" K ("<<nuhiz<<" GHz)"<<endl
     <<"             P="<<psynch<<" W for pixel"<<endl
     <<"             flux density = "<<ssynch<<" Jy for pixel solid angle"<<endl;

 // CMB
 double tcmb = T_CMB_Par;
 planck.SetTemperature(tcmb);
 double pcmb_2polar = planck(nuhiz*1.e+9) * surfeff * angsol * (dnuhiz*1e9);
 double scmb_2polar = pcmb_2polar / surfeff / (dnuhiz*1.e+9) / Jansky2Watt_cst;
 double pcmb = facpolar * pcmb_2polar;
 double scmb = facpolar * scmb_2polar;
 cout<<"\nCMB: T="<<tcmb<<" K"<<endl
     <<"     P="<<pcmb<<" W for pixel"<<endl
     <<"     flux density = "<<scmb<<" Jy for pixel solid angle"<<endl;

 // AGN
 double flux_agn = pow(10.,lflux_agn);
 double mass_agn = FluxHI2Msol(flux_agn*Jansky2Watt_cst,dlum);
 cout<<"\nAGN: log10(S_agn)="<<lflux_agn<<" -> S_agn="
     <<flux_agn<<" Jy -> "<<mass_agn<<" equiv. Msol/Hz"<<endl;
 double flux_agn_pix = flux_agn*(dnuhiz*1e9);
 double mass_agn_pix = FluxHI2Msol(flux_agn_pix*Jansky2Watt_cst,dlum);
 double lmass_agn_pix = log10(mass_agn_pix);
 cout<<"...pixel: f="<<flux_agn_pix<<" 10^-26 W/m^2"
     <<" -> "<<mass_agn_pix<<" Msol -> log10 = "<<lmass_agn_pix<<endl;

 // =====================================================================
 // ---
 // --- Noise analysis
 // ---
 // --- Puissance du bruit pour un telescope de surface Ae et de BW dNu
 // Par definition la puissance du bruit est:
 //     Pb = k * Tsys * dNu  (W)
 // Pour une source (non-polarisee) de densite de flux (totale 2 polars)
 //     St (exprimee en Jy=W/m^2/Hz)
 //     Pt = St * Ae * dNu   (puissance totale emise en W pour 2 polars)
 //     P1 = 1/2 * St * Ae * dNu   (puissance emise en W pour une polar)
 // la SEFD (system equivalent flux density en Jy) est definie comme
 //   la densite de flux total (2 polars) "St" d'une source (non-polarisee)
 //   dont la puissance P1 mesuree pour une seule polarisation
 //   serait egale a la puissance du bruit. De P1 = Pb on deduit:
 //   SEFD = 2 * k * Tsys / Ae    (en Jy)
 //   la puissance du bruit est: Pb = 1/2 * SEFD * Ae * dNu  (en W)
 // la sensibilite Slim tient compte du temps d'integration et de la BW:
 //   le nombre de mesures independantes est "2*dNu*Tobs" donc
 //   Slim = SEFD / sqrt(2*dNu*Tobs) =  2*k*Tsys/[Ae*sqrt(2*dNu*Tobs) (en Jy)
 // --- Puissance du bruit pour un interferometre
 // Ae = surface d'un telescope elementaire
 // N  = nombre de telescopes dans l'interferometre (Atot = N*Ae)
 // La sensibilite Slim en Jy est:
 // Slim = 2 * k * Tsys / [ Ae * Sqrt(2*N(N-1)/2 *dnu*Tobs) ]
 //      = 2 * k * Tsys / [ Atot/N * Sqrt(2*N(N-1)/2*dnu*Tobs) ]
 //      = 2 * k * Tsys / [ Atot * Sqrt((N-1)/N *dnu*Tobs) ]
 // - Interferometre a deux antennes:
 // Slim =  2 * k * Tsys / [ Atot * Sqrt(1/2 *dnu*Tobs) ]
 // - Interferometre a N antennes (N grand):
 // Slim ->  2 * k * Tsys / [ Atot * Sqrt(dnu*Tobs) ]
 //      C'est aussi la formule pour un telescope unique de surface Atot
 // --- On ne mesure qu'une seule polarisation
 // Ces formules sont valables si on mesure 1 polarisation:
 // Slim est la densite de flux total "St" (2 polars) d'une source (non-polarisee)
 //   qui donne la meme puissance que le bruit dans un detecteur qui ne
 //   mesure qu'une seule polarisation:
 // Le rapport S/N pour une source de densite de flux St (totale 2 polars):
 //    S/N = St / Slim
 // La puissance de bruit est, par definition:
 //    Pb = 1/2 *Slim*Atot*dNu
 //         = k*Tsys*sqrt(2*dNu/Tobs)  pour N=2
 //         = k*Tsys*sqrt(dNu/Tobs)    pour N>>grand
 // La densite de flux d'une source a S/N=1 est:
 //    St = Slim
 // La puissance d'une source a S/N=1 mesuree par un detecteur
 //    qui ne mesure qu'une polar est:
 //    P1_lim = 1/2 *Slim*Atot*dNu
 // --- On mesure les 2 polarisations avec deux voies d'electronique distinctes
 // la puissance du signal mesure est multipliee par 2
 // la puissance du bruit est multipliee par sqrt(2)
 // on a donc un gain d'un facteur sqrt(2) sur le rapport S/N
 // (cela revient d'ailleur a doubler le temps de pose: Tobs -> 2*Tobs)
 // En notant arbitrairement: Slim' = Slim / sqrt(2)
 //    ou Slim est defini par les formules ci-dessus
 // Le rapport S/N pour une source de densite de flux St (totale 2 polars):
 //   (S/N)_2 = (S/N)_1 * sqrt(2) = (St / Slim) * sqrt(2) = St / Slim'
 // La densite de flux d'une source a S/N=1 est:
 //    St = Slim' = Slim / sqrt(2)
 // La puissance d'une source a S/N=1 cumulee par les 2 detecteurs est:
 //    P_lim = St*Atot*dNu = Slim'*Atot*dNu = 1/sqrt(2) *Slim*Atot*dNu
 //          = P1_lim * sqrt(2)
 // La puissance de bruit cumulee par les 2 detecteurs est, par definition:
 //    Pb = P_lim = Slim'*Atot*dNu = P1_lim * sqrt(2)
 //               = 2*k*Tsys*sqrt(dNu/Tobs)   pour N=2
 //               = k*Tsys*sqrt(2*dNu/Tobs)   pour N>>grand
 // =====================================================================

 cout<<"\n---\n--- Noise analysis \n---"<<endl;
 double psys = k_Boltzman_Cst * Tsys * (dnuhiz*1.e+9);
 cout<<"Noise: T="<<Tsys<<" K, P="<<psys<<" W  (for Dnu="<<dnuhiz<<" GHz)"<<endl;

 cout<<"...Computation assume that noise dominate the signal."<<endl;
 if(ya2polar)
   cout<<"...Assuming 2 polarisations measurements with 2 different electronics."<<endl;

 double slim,slim_nl,SsN,SsN_nl,smass,smass_nl;

 //---
 for(unsigned short it=0;it<2;it++) {

   double fac = 1.;
   if(it==0) { // Interferometre a 2 telescopes
     fac = 0.5;
     cout<<"\n...Observation limits for a 2 telescope interferometer (with complex correlator)"<<endl
         <<"   (sensitivity is given for real or complex correlator output)"<<endl;
   } else if (it==1) { // Interferometre a N>> telescopes
     fac = 1.;
     cout<<"\n...Observation limits for a N (large) telescope interferometer (with complex correlator)"<<endl
         <<"     (weak source limit sensitivity in a synthetised image)"<<endl
         <<"     Also valid for a single dish telescope."<<endl;
   } else continue;

   slim = 2. * k_Boltzman_Cst * Tsys / surfeff
               / sqrt(fac*(dnuhiz*1.e+9)*tobs) /Jansky2Watt_cst;
   if(ya2polar) slim /= sqrt(2.);
   SsN = ssig_2polar / slim;
   smass = mhiref / ssig_2polar * slim;
   cout<<"for 1 lobe:"<<endl
       <<"   Slim    = "<<slim<<" Jy"<<endl
       <<"   S/N     = "<<SsN<<endl
       <<"   Mass HI = "<<smass<<" Msol"<<endl;

   slim_nl = slim * sqrt(nlobes);
   SsN_nl = ssig_2polar / slim_nl;
   smass_nl = mhiref / ssig_2polar * slim_nl;
   cout<<"for "<<nlobes<<" lobes:"<<endl
       <<"   Flux    = "<<slim_nl<<" Jy"<<endl
       <<"   S/N     = "<<SsN_nl<<endl
       <<"   Mass HI = "<<smass_nl<<" Msol"<<endl;

 }

 return 0;
}
