#ifndef STSRAND_H_SEEN
#define STSRAND_H_SEEN

// Classe RandomGenerator 
// Generateur aleatoire compatible multi-thread
//
// R. Ansari          (C) UPS+LAL IN2P3/CNRS  2007
// C. Magneville      (C) DAPNIA/SPP  CEA     2007


#include "machdefs.h"
#include "objfio.h"   // Pour rendre la classe PPersist

#include "ndatablock.h"

namespace SOPHYA {

class RandomGenerator : public AnyDataObj {

 public:
  RandomGenerator(size_t n = 1024, bool tsafe=true);
  RandomGenerator(RandomGenerator const & rg);

  virtual ~RandomGenerator();

  /*! \brief Automatic initialization using the current time */
  static  void AutoInit(int lp=0);
  /*! \brief Initialization using the specified seed  */
  static  void Init(long seed_val, int lp=0);
  /*! \brief Initialization using the specified seed  */
  static  void Init(unsigned short seed_16v[3], int lp=0);
  /*! \brief Getting the current seed state */
  static  void GetSeed(unsigned short seed_16v[3], int lp=0);

  /*! \brief Return a random number \b r with a flat distribution  0<=r<1 */ 
  inline r_8 Flat01() {return Next();}

  /*! \brief Return a random number \b r with a flat distribution  -1<=r<1 */ 
  inline r_8 Flatpm1() {return 2.*Next()-1.;}

  /*! \brief Return a random number following a gaussian distribution (sigma=1, mean=0*/ 
  virtual r_8 Gaussian(); 

  /*! \brief Return a random number following a gaussian distribution "sigma", (mean=0)*/ 
  virtual r_8 Gaussian(double sigma) 
    { return sigma*Gaussian(); }
 /*! \brief Return a random number following a gaussian distribution with mean=mu, and sigma */ 
  virtual r_8 Gaussian(double sigma,double mu) 
    { return sigma*Gaussian()+mu; }

  /*! \brief alias for Gaussian() */ 
  inline r_8 NorRand()
    { return Gaussian(); }

  /*! \brief Return a random number following a poisson distribution with mean mu */ 
  virtual  uint_8 Poisson(double mu, double mumax);
  
  /*! \brief Return a random number following a poisson distribution with mean mu */ 
  inline uint_8 Poisson(double mu)
    { return Poisson(mu, -1.); }

   /*! \brief Return a random number following a poisson distribution with mean mu */ 
  inline uint_8  PoissRandLimit(double mu,double mumax=10.)
    { return Poisson(mu, mumax); }
 
  //  Pour la gestion de persistance PPF
  friend class ObjFileIO<RandomGenerator> ;

 protected:
  // ---- protected data members
  NDataBlock<r_8>  rseq_;  
  size_t idx_;
  bool fg_nothrsafe;  // if true --> NOT thread-safe
  // ---- protected methods 
  void GenSeq();
  inline r_8 Next() 
    {
      if (rseq_.Size() == 0)  return drand48(); 
      else { 
	if(idx_==rseq_.Size()) GenSeq(); 
	return(rseq_(idx_++));
      }
    }
  // Non thread-safe version of Init() and GetSeed()
  static  void Init_P(unsigned short seed_16v[3]);
  static  void GetSeed_P(unsigned short seed_16v[3]);

};  // Fin de la classe RandomGenerator

// Classe pour la gestion de persistance PPF :  ObjFileIO<RandomGenerator>

/*! Writes the random generator object state in the POutPersist stream \b os */
inline POutPersist& operator << (POutPersist& os, RandomGenerator & obj)
{ ObjFileIO<RandomGenerator> fio(&obj);  fio.Write(os);  return(os); }
/*! Reads the random generator object state from the PInPersist stream \b is */
inline PInPersist& operator >> (PInPersist& is, RandomGenerator & obj)
{ ObjFileIO<RandomGenerator> fio(&obj); is.SkipToNextObject(); fio.Read(is); return(is); }

} /* namespace SOPHYA */

#endif 
