// toiiter.h
// Eric Aubourg         CEA/DAPNIA/SPP   juillet 1999

#ifndef TOIITER_H
#define TOIITER_H


#include <vector>
#include <set>
#include <map>
#include <string>
#include "archeopsfile.h"

enum TOIKind {
  sampleNum,  // Numero d''echantillon (timer interne transputer)
  internalTime, // temps transputer depuis debut manip, en secondes.
  mjd,        // UTC (en jours, MJD = JD - 2450000). index=0 : premier bloc GPS. 1: + sioux.
  boloTens,   // tension en microVolts, filtree avec filtre carre indexe par bolometre
  boloRaw,    // tension brute, non filtree indexe par bolometre
  boloTemp,   // Kelvins
  sstDiode,   // signal brut SST, indice = diode, 0-45
  sstChannel, // signal brut SST, indice = canal, 0-47
  sstStarZ,   // numero de diode d''une eventuelle etoile,   i=etoile 0..n
  sstStarF,   // flux d''une eventuelle etoile,             <0 si moins de i+1 etoiles
  gyroRaw,    // valeur brute du gyro, index = 0,1,2
  gpsTime,    // temps fourni par GPS
  longitude,  // position ballon
  latitude,   // position ballon
  altitude,   // position ballon
  tsid,       // temps sideral en secondes
  azimut,     // a preciser, reconstruit SST, magneto, autres...
  alphaAxis,  // 
  deltaAxis,  //
  alphaSst,  // 
  deltaSst,  //
  alphaBolo,  // sur le ciel, indexe par bolometre
  deltaBolo   // sur le ciel, indexe par bolometre
};

class TOIInterpolator;

class TOIIter {
public:
   TOIIter(TOIIter const&);   // $CHECK$ prevoir operator = ?
   ~TOIIter();
   bool Next();
   
   bool    canGetValue(int column);  // data is available for that...
   double  getValue(int column);
   bool    newValue(int column);     // a juste change ?
   bool    extendValue(int column);  // une valeur plus ancienne, etendue ?
   bool    interpValue(int column);  // une valeur interpolee avec valeur future ?
   bool    isTrig(int column);  
   TOIKind getKind(int column);
   int     getIndex(int column);     // si plusieurs, exemple, tension bolo 2
   
   bool    canGetTOI(TOIKind kind, int index=0);
   double  getTOI(TOIKind kind, int index=0);
   
   
   int    getBlockSampleIndex();        // numero d''echantillon dans dernier bloc bolo
   int    getSampleIndex();             // numero d''echantillon 

   block_type_param*         lastParam();  
   block_type_journal*       lastJournal();
   block_type_reglage*       lastReglage();
   block_type_dilution*      lastDilution();
   block_type_gps*           lastGPS();
   block_type_une_periode*   lastUnePeriode();
   block_type_synchro_sol*   lastSynchroSol();
   block_type_pointage_sol*  lastPointageSol();
   block_type_bolo*          lastBolo();   
   block_type_gyro*          lastGyro();
   block_type_sst*           lastSST();
   block_type_bolo_comprime* lastBoloComp();   
   block_type_gyro_comprime* lastGyroComp();
   block_type_sst_comprime*  lastSSTComp();
   
   ArcheopsFile* currentFile() {return file;}

protected:
   TOIIter();
   void Init(); // After setting the options, opens the first file.
   void ScanFiles(); // Parcourt tous les fichiers, cherche le premier echantillon...
   
   bool NextSample(); // no test of trange
   bool NextFile();
   
   ArcheopsFile* file;
   string directory;   // If directory scanning
   set<string> files;  // list of files to be read...
   map<string, long> file1stSamp; // premier echantillon de chaque fichier
   set<string>::iterator curFile;
   bool  isOnBoardRecorder;
   
   int imes;
   
   double tStart, tEnd;
   
   double tBlock0;  // MJD du bloc numero zero...
   double perEch;   // periode d''echantillonage en secondes
   
   struct info {
     TOIKind kind;
     int index;
     bool triggering;
     bool interpolated;
   };
   
   int getColTOI(TOIKind kind, int index=0);
   
   long trigMask;
   
   vector<info>   infos;
   
   friend class TOISvr;
   
   // Gestion de l''interpolation, par delegation a un autre iterateur
   TOIIter* rawIter; // iterateur sans interpolation, qui avance avec retard
                     // en bufferisant les valeurs. S'il existe, on lui delegue
                     // toutes les operations agissant sur le fichier.
   TOIInterpolator* interp; // interpolateurs pour stocker les valeurs
   long lastSample;  // last interpolated sample
   long maxLookAhead;
   bool fetchAhead(); // avance d''une TOI en nourrissant les interpolateurs
   
   private:
};

#endif
