#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include <vector>

#include "constcosmo.h"
#include "cosmocalc.h"
#include "geneutils.h"
#include "schechter.h"
#include "pkspectrum.h"
#include "planckspectra.h"

/* --- Check Peterson at al. astro-ph/0606104 v1  (pb facteur sqrt(2) sur S/N !)
cmvdefsurv -U 0.75,0.3,0.7,-1,1 -V 300 -z 0.0025,0.2,Z -x 1,90,A -O 400000,6000 -N 75 -M 6.156e9 -F 3 -2 1.5
 --- */

//-----------------------------------------------------------------------------------------------------------
inline double rad2deg(double trad) {return trad/M_PI*180.;}
inline double rad2min(double trad) {return trad/M_PI*180.*60.;}
inline double rad2sec(double trad) {return trad/M_PI*180.*3600.;}
inline double deg2rad(double tdeg) {return tdeg*M_PI/180.;}
inline double min2rad(double tmin) {return tmin*M_PI/(180.*60.);}
inline double sec2rad(double tsec) {return tsec*M_PI/(180.*3600.);}

inline double sr2deg2(double asr) {return asr*pow(rad2deg(1.),2.);}
inline double sr2min2(double asr) {return asr*pow(rad2min(1.),2.);}
inline double deg22sr(double adeg) {return adeg*pow(deg2rad(1.),2.);}
inline double min22sr(double amin) {return amin*pow(min2rad(1.),2.);}

double LargeurDoppler(double v /* km/s */, double nu);
double DzFrV(double v /* km/s */, double zred);
double DNuFrDz(double dzred,double nu_at_0,double zred);
double DzFrDNu(double dnu_at_0,double nu_at_0,double zred);
double DzFrDNuApprox(double dnu_at_0,double nu_at_0,double zred);
double ZFrLos(double loscom /* Mpc com */,CosmoCalc& univ);
double AngsolEqTelescope(double nu /* GHz */,double telsurf /* m^2 */);
void usage(void);

void usage(void) {
  cout<<"cmvdefsurv [options] -x dx,txlarg[,unit_x] -y dy,tylarg[,unit_y] -z dz,tzlarg[,unit_z] redshift"<<endl
      <<"----------------"<<endl
      <<" -x dx,txlarg[,unit_x] : resolution et largeur dans le plan transverse selon X"<<endl
      <<" -y dy,tylarg[,unit_y] : idem selon Y, si <=0 meme que X"<<endl
      <<" -z dz,tzlarg[,unit_z] : resolution et largeur sur la ligne de visee"<<endl
      <<"-- Unites pour X-Y:"<<endl
      <<"   \'A\' : en angles (pour X-Y)  : resolution=ArcMin, largeur=Degre (defaut)"<<endl
      <<"   \'Z\' : en redshift (pour Z)  : resolution et largeur en redshift (defaut)"<<endl
      <<"   \'F\' : en frequence (pour Z) : resolution et largeur MHz"<<endl
      <<"   \'M\' : en longeur comobile (pour X-Y-Z) : resolution et largeur Mpc"<<endl
      <<"----------------"<<endl
      <<" -K k,dk,pk : k(Mpc^-1) dk(Mpc^-1) pk(a z=0 en Mpc^-3) pour estimer la variance cosmique"<<endl
      <<"----------------"<<endl
      <<" -O surf,tobs,eta : surface effective (m^2), temps d\'observation (s), efficacite d\'ouverture"<<endl
      <<" -N Tsys : temperature du system (K)"<<endl
      <<" -L lobearea,freqlob : angle solide du lobe d\'observation en arcmin^2 (def= celle du pixel)"<<endl
      <<"                       pour la frequence freqlob en MHz"<<endl
      <<"            Si freqlob<=0 : la frequence de reference est celle du redshift etudie (def)"<<endl
      <<" -2 : two polarisations measured"<<endl
      <<" -M  : masse de HI de reference (MSol), si <=0 mean schechter in pixel"<<endl
      <<" -F  : HI flux factor to be applied for our redshift"<<endl
      <<" -V Vrot : largeur en vitesse (km/s) pour l\'elargissement doppler (def=300km/s)"<<endl
      <<"----------------"<<endl
      <<" -S Tsynch,indnu : temperature (K) synch a 408 Mhz, index d\'evolution"<<endl
      <<"                   (indnu==0 no evolution with freq.)"<<endl
      <<"----------------"<<endl
      <<" -U h100,om0,ol0,w0,or0,flat : cosmology"<<endl
      <<"----------------"<<endl
      <<" -A <log10(S_agn)> : moyenne du flux AGN en Jy dans le pixel"<<endl
      <<" redshift : redshift moyen du survey"<<endl
      <<endl;
}

//-------------------------------------------------------------------------------------------
int main(int narg,char *arg[])
{
 // --- 
 // --- Valeurs fixes ou par defaut
 // --- 

 //-- WMAP
 unsigned short flat = 0;
 double h100=0.71, om0=0.267804, or0=7.9e-05*0., ol0=0.73,w0=-1.;

 //-- Schechter HIMASS
 double h75 = h100 / 0.75;
 double nstar = 0.006*pow(h75,3.);  //
 double mstar = pow(10.,9.8);  // MSol
 double alpha = -1.37;
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;

 double hifactor = 1.;
 double vrot = 300.; // largeur en vitesse (km/s) pour elargissement doppler

 //-- CMB , Synchrotron et AGN
 double tcmb = T_CMB_Par;
 // a 408 MHz (Haslam) + evol index a -2.6
 double Tsynch408=60., nuhaslam=0.408, indnu = -2.6;
 double lflux_agn = -3.;

 //-- Appareillage
 bool ya2polar = false;
 double facpolar = 0.5; // si on mesure les 2 polars -> 1.0
 double Tsys=75.;
 double tobs = 6000., surftot = 400000., eta_eff  = 1.;
 // taille du lobe d'observation en arcmin pour la frequence
 double lobearea0 = -1., lobefreq0 = -1.;

 //-- Variance cosmique (default = standard SDSSII)
 double kcosm = 0.05, dkcosm = -1., pkcosm = 40000.;

 // --- Pour taille du survey
 double dx=1., dy=-1., dz=0.0007, txlarg=100., tylarg=-1., tzlarg=0.1;
 int nx=0, ny=0, nz=0;
 char unit_x = 'A', unit_y = 'A', unit_z = 'Z';
 double redshift0 = 0.;
 double mhiref = -1.; // reference Mass en HI (def integ schechter)

 // --- 
 // --- Decodage arguments
 // --- 
 char c;
  while((c = getopt(narg,arg,"h2x:y:z:N:S:O:M:F:V:U:L:A:K:")) != -1) {
  switch (c) {
  case 'x' :
    sscanf(optarg,"%lf,%lf,%c",&dx,&txlarg,&unit_x);
    break;
  case 'y' :
    sscanf(optarg,"%lf,%lf,%c",&dy,&tylarg,&unit_y);
    break;
  case 'z' :
    sscanf(optarg,"%lf,%lf,%c",&dz,&tzlarg,&unit_z);
    break;
  case 'O' :
    sscanf(optarg,"%lf,%lf,%lf",&surftot,&tobs,&eta_eff);
    break;
  case 'L' :
    sscanf(optarg,"%lf,%lf",&lobearea0,&lobefreq0);
    break;
  case 'N' :
    sscanf(optarg,"%lf",&Tsys);
    break;
  case 'S' :
    sscanf(optarg,"%lf,%lf",&Tsynch408,&indnu);
    break;
  case 'M' :
    sscanf(optarg,"%lf",&mhiref);
    break;
  case 'F' :
    sscanf(optarg,"%lf",&hifactor);
    break;
  case 'V' :
    sscanf(optarg,"%lf",&vrot);
    break;
  case 'U' :
    sscanf(optarg,"%lf,%lf,%lf,%lf,%hu",&h100,&om0,&ol0,&w0,&flat);
    break;
  case '2' :
    ya2polar = true;
    facpolar = 1.0;
    break;
  case 'A' :
    sscanf(optarg,"%lf",&lflux_agn);
    break;
  case 'K' :
    sscanf(optarg,"%lf,%lf,%lf",&kcosm,&dkcosm,&pkcosm);
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 } 
 if(optind>=narg) {usage(); return -1;}
 sscanf(arg[optind],"%lf",&redshift0);
 if(redshift0<=0.) {cout<<"Redshift "<<redshift0<<" should be >0"<<endl; return -2;}

 // --- 
 // --- Initialisation de la Cosmologie
 // --- 
 CosmoCalc univ(flat,true,2.*redshift0);
 double perc=0.01,dzinc=redshift0/100.,dzmax=dzinc*10.; unsigned short glorder=4;
 univ.SetInteg(perc,dzinc,dzmax,glorder);
 univ.SetDynParam(h100,om0,or0,ol0,w0);

 GrowthFactor growth(om0,ol0);

 // --- 
 // --- Mise en forme dans les unites appropriees
 // --- 
 // ATTENTION: le cube de simulation est en Mpc avec des pixels de taille comobile fixe
 cout<<"\n>>>>\n>>>> Geometrie\n>>>>"<<endl;
 if(dy<=0. || tylarg<=0.) {dy=dx; tylarg=txlarg; unit_y=unit_x;}
 cout<<"X values: resolution="<<dx<<" largeur="<<txlarg<<" unite="<<unit_x<<endl;
 if(unit_x == 'A') {
   nx = int(txlarg*60./dx+0.5);
   dx = min2rad(dx)*univ.Dtrcom(redshift0);
   txlarg = dx*nx;
 } else if(unit_x == 'M') {
   nx = int(txlarg/dx+0.5);
   txlarg = dx*nx;
 } else {
   cout<<"Unknown unit_x = "<<unit_x<<endl; return -2;
 }
 cout<<"Y values: resolution="<<dy<<" largeur="<<tylarg<<" unite="<<unit_y<<endl;
 if(unit_y == 'A') {
   ny = int(tylarg*60./dy+0.5);
   dy = min2rad(dy)*univ.Dtrcom(redshift0);
   tylarg = dy*ny;
 } else if(unit_y == 'M') {
   ny = int(tylarg/dy+0.5);
   tylarg = dy*ny;
 } else {
   cout<<"Unknown unit_y = "<<unit_y<<endl; return -2;
 }
 cout<<"Z values: resolution="<<dz<<" largeur="<<tzlarg<<" unite="<<unit_z<<endl;
 if(unit_z == 'Z') {
   nz = int(tzlarg/dz+0.5);
   dz = dz*univ.Dhubble()/univ.E(redshift0);
   tzlarg = dz*nz;
 } else if(unit_z == 'M') {
   nz = int(tzlarg/dz+0.5);
   tzlarg = dz*nz;
 } else if(unit_z == 'F') { // unite en MHz
   nz = int(tzlarg/dz+0.5);
   dz = DzFrDNu(dz,Fr_HyperFin_Par*1.e3,redshift0);  //dzred
   dz = dz*univ.Dhubble()/univ.E(redshift0);
   tzlarg = dz*nz;
 } else {
   cout<<"Unknown unit_z = "<<unit_z<<endl; return -2;
 }

 // On estime la valeur du redshift le plus grand
 double bigred = redshift0 + dz/univ.Dhubble()*univ.E(redshift0) * nz/2.;
 cout<<"biggest redshift estimated to be "<<bigred<<endl;
 dzinc=bigred/100.; dzmax=dzinc*10.;
 univ.SetInteg(perc,dzinc,dzmax,glorder);

 // --- 
 // --- Calcul des valeurs au centre du cube et aux faces limites av/ar
 // --- 
 cout<<"\n>>>>\n>>>> Cosmologie generale\n>>>>"<<endl;
 double adtx[3], adty[3], atxlarg[3], atylarg[3];
 double loscom[3], redshift[3], unplusz[3], dred[3], growthfac[3], rhocz[3];
 double dang[3], dtrcom[3], dlum[3], dloscom[3], dlosdz[3];
 double nuhiz[3], lambdahiz[3], dnuhiz[3];
 double angsol_pix[3], angsol_tot[3];

 redshift[0] = redshift0;
 loscom[0] = univ.Dloscom(redshift0);
 loscom[1] = loscom[0]-tzlarg/2.;
 loscom[2] = loscom[0]+tzlarg/2.;
 //if(loscom[1]<=0.) {cout<<"Lower distance limit "<<loscom[1]<<" should be >0"<<endl; return -3;}
 for(int i=0;i<3;i++) {
   double l = loscom[i]; if(l<=0.) l = dz/2.;
   redshift[i]  = ZFrLos(l,univ);
   unplusz[i]   = 1. + redshift[i];
   growthfac[i] = growth(redshift[i]);
   rhocz[i]     = univ.Rhoc(redshift[i])*GCm3toMsolMpc3_Cst;;
   dang[i]      = univ.Dang(redshift[i]);
   dtrcom[i]    = univ.Dtrcom(redshift[i]);
   dlum[i]      = univ.Dlum(redshift[i]);
   dloscom[i]   = univ.Dloscom(redshift[i]);
   dlosdz[i]    = univ.Dhubble()/univ.E(redshift[i]);
   dred[i]      = dz/dlosdz[i];
   adtx[i]      = dx/dtrcom[i];
   atxlarg[i]   = adtx[i]*nx;
   adty[i]      = dy/dtrcom[i];
   atylarg[i]   = adty[i]*ny;
   nuhiz[i]     = Fr_HyperFin_Par / unplusz[i];  // GHz
   lambdahiz[i] = SpeedOfLight_Cst*1000./(nuhiz[i]*1.e9);  // m
   dnuhiz[i]    = DNuFrDz(dred[i],Fr_HyperFin_Par,redshift[i]);  // GHz
   angsol_pix[i] = AngSol(adtx[i]/2.,adty[i]/2.,M_PI/2.);
   angsol_tot[i] = AngSol(atxlarg[i]/2.,atylarg[i]/2.,M_PI/2.);
 }
 

 cout<<"--- Cosmology for z = "<<0.<<endl;
 univ.Print(0.);
 double rhoc0 = univ.Rhoc(0.)*GCm3toMsolMpc3_Cst;

 for(int i=0;i<3;i++) {
   cout<<"\n--- Cosmology for z = "<<redshift[i]<<endl;
   univ.Print(redshift[i]);
   cout<<"Nu(HI) = "<<Fr_HyperFin_Par*1000./(1.+redshift[i])<<" MHz"<<endl;
 }

 cout<<endl;
 cout<<"--- Resume"<<endl;
 cout<<"Facteur de croissance lineaire = "<<growthfac[0]
     <<" in ["<<growthfac[1]<<","<<growthfac[2]<<"]"<<endl;
 cout<<"                       1/(1+z) = "<<1./(1.+redshift[0])
     <<" in ["<<1./(1.+redshift[1])<<","<<1./(1.+redshift[2])<<"]"<<endl;
 cout<<"Facteur de croissance lineaire^2 = "<<growthfac[0]*growthfac[0]
     <<" in ["<<growthfac[1]*growthfac[1]<<","<<growthfac[2]*growthfac[2]<<"]"<<endl;

 cout<<"Rho_c (z=0) =  "<<rhoc0<<", a z="<<0<<": "<<rhoc0<<" Msol/Mpc^3"<<endl;
 cout<<"Rho_c a z = "<<rhocz[0]
     <<" Msol/Mpc^3 in ["<<rhocz[1]<<","<<rhocz[2]<<"]"<<endl;

 cout<<endl;
 cout<<"dang= "<<dang[0]<<" Mpc in ["<<dang[1]<<","<<dang[2]<<"]"<<endl;
 cout<<"dtrcom= "<<dtrcom[0]<<" Mpc com in ["<<dtrcom[1]<<","<<dtrcom[2]<<"]"<<endl;
 cout<<"dlum= "<<dlum[0]<<" Mpc in ["<<dlum[1]<<","<<dlum[2]<<"]"<<endl;
 cout<<"dloscom= "<<dloscom[0]<<" Mpc com in ["<<dloscom[1]<<","<<dloscom[2]<<"]"<<endl;
 cout<<"dz=1 -> dlosdz= "<<dlosdz[0]<<" Mpc com in ["<<dlosdz[1]<<","<<dlosdz[2]<<"]"<<endl;
 cout<<"1 Mpc los com -> dz = "<<1./dlosdz[0]<<" in ["<<1./dlosdz[1]<<","<<1./dlosdz[2]<<"]"<<endl;

 cout<<endl;
 for(int i=0;i<3;i++) {
   cout<<"...Redshift="<<redshift[i]<<endl;
   cout<<"1\" -> "<<dang[i]*sec2rad(1.)<<" Mpc = "<<dtrcom[i]*sec2rad(1.)<<" Mpc com"<<endl;
   cout<<"1\' -> "<<dang[i]*min2rad(1.)<<" Mpc = "<<dtrcom[i]*min2rad(1.)<<" Mpc com"<<endl;
   cout<<"1d -> "<<dang[i]*deg2rad(1.)<<" Mpc = "<<dtrcom[i]*deg2rad(1.)<<" Mpc com"<<endl;
   cout<<"1 Mpc transv com -> "<<rad2sec(1./dtrcom[i])<<"\" = "
       <<rad2min(1./dtrcom[i])<<" \' = "<<rad2deg(1./dtrcom[i])<<" d"<<endl;
   cout<<"dz=1 -> dlosdz= "<<dlosdz[i]<<" Mpc com"<<endl;
   cout<<"1 Mpc los com -> dz = "<<1./dlosdz[0]<<endl;
 }

 // --- 
 // --- Cosmolographie Transverse
 // --- 
 cout<<"\n>>>>\n>>>> Cosmologie & Geometrie transverse\n>>>>"<<endl;

 cout<<"dx = "<<dx<<", txlarg = "<<txlarg<<" Mpc (com), nx="<<nx<<endl;
 cout<<"adtx = "<<adtx[0]<<" rd in ["<<adtx[1]<<","<<adtx[2]<<"]"<<endl;
 cout<<"     = "<<rad2min(adtx[0])<<"\' in ["<<rad2min(adtx[1])<<","<<rad2min(adtx[2])<<"]"<<endl;
 cout<<"atxlarg = "<<atxlarg[0]<<" rd in ["<<atxlarg[1]<<","<<atxlarg[2]<<"]"<<endl;
 cout<<"          "<<rad2deg(atxlarg[0])<<" d in ["<<rad2deg(atxlarg[1])<<","<<rad2deg(atxlarg[2])<<"]"<<endl;

 if(fabs(dx-dy)>1.e-20 && fabs(txlarg-tylarg)>1.e-20) {
   cout<<"\ndy = "<<dy<<", tylarg = "<<tylarg<<" Mpc (com), ny="<<ny<<endl;
   cout<<"adty = "<<adty[0]<<" rd in ["<<adty[1]<<","<<adty[2]<<"]"<<endl;
   cout<<"     = "<<rad2min(adty[0])<<"\' in ["<<rad2min(adty[1])<<","<<rad2min(adty[2])<<"]"<<endl;
   cout<<"atylarg = "<<atylarg[0]<<" rd in ["<<atylarg[1]<<","<<atylarg[2]<<"]"<<endl;
   cout<<"          "<<rad2deg(atylarg[0])<<" d in ["<<rad2deg(atylarg[1])<<","<<rad2deg(atylarg[2])<<"]"<<endl;
 }

 // --- 
 // --- Cosmolographie Line of sight
 // --- 
 cout<<"\n>>>>\n>>>> Cosmologie & Geometrie ligne de visee\n>>>>"<<endl;
 cout<<"dz = "<<dz<<", tzlarg = "<<tzlarg<<" Mpc (com), nz="<<nz<<endl;
 cout<<"Redshift = "<<redshift[0]<<" in ["<<redshift[1]<<","<<redshift[2]<<"]"<<endl;
 cout<<"dred = "<<dred[0]<<" in ["<<dred[1]<<","<<dred[2]<<"]"<<endl;
 cout<<"nu HI = "<<nuhiz[0]<<" GHz in ["<<nuhiz[1]<<","<<nuhiz[2]<<"]"<<endl;
 cout<<"lambda HI = "<<lambdahiz[0]<<" m in ["<<lambdahiz[1]<<","<<lambdahiz[2]<<"]"<<endl;
 cout<<"dnu HI= "<<dnuhiz[0]*1e3<<" MHz in ["<<dnuhiz[1]*1e3<<","<<dnuhiz[2]*1e3<<"]"<<endl;

 // --- 
 // --- Volume
 // --- 
 cout<<"\n>>>>\n>>>> Volumes\n>>>>"<<endl;
 double Npix = (double)nx*(double)ny*(double)nz;
 double vol_pixel = dx*dy*dz;
 double vol_survey = vol_pixel*Npix;
 cout<<"Npix total = "<<Npix<<" -> "<<Npix*sizeof(double)/1.e6<<" Mo"<<endl;
 cout<<"Volume pixel = "<<vol_pixel<<" Mpc^3 com"<<endl;
 cout<<"Volume total = "<<vol_survey<<" Mpc^3 com = "<<vol_survey/1.e9<<" Gpc^3"<<endl;

 double vol = univ.Vol4Pi(redshift[1],redshift[2])/(4.*M_PI)*angsol_tot[0];
 cout<<"Calcul avec angsol et redshift: vol = "<<vol<<" Mpc^3 = "<<vol/1.e9<<" Gpc^3"<<endl
     <<"       -> pixel volume comoving = vol/Npix = "<<vol/Npix<<" Mpc^3"<<endl;

 // --- 
 // --- Angles solides
 // --- 
 cout<<"\n>>>>\n>>>> Angles solides\n>>>>"<<endl;
 cout<<"Pixel solid angle = "<<angsol_pix[0]<<" sr ("<<angsol_pix[0]/(4.*M_PI)<<" *4Pi)"
     <<" in ["<<angsol_pix[1]<<","<<angsol_pix[2]<<"]"<<endl;
 cout<<"                  = "<<sr2min2(angsol_pix[0])<<"\'^2"
     <<" in ["<<sr2min2(angsol_pix[1])<<","<<sr2min2(angsol_pix[2])<<"]"<<endl;
 cout<<"Total solid angle = "<<angsol_tot[0]<<" sr ("<<angsol_tot[0]/(4.*M_PI)<<" *4Pi)"
     <<" in ["<<angsol_tot[1]<<","<<angsol_tot[2]<<"]"<<endl;

 // --- 
 // --- Fourier space: k = omega = 2*Pi*Nu = 2*Pi*C/Lambda
 // --- 
 cout<<"\n>>>>\n>>>> Geometrie dans l'espace de Fourier\n>>>>"<<endl;
 cout<<"Array size: nx = "<<nx<<",  ny = "<<ny<<",  nz = "<<nz<<endl;
 double dk_x = 2.*M_PI/(nx*dx), knyq_x = M_PI/dx;
 double dk_y = 2.*M_PI/(nx*dy), knyq_y = M_PI/dy;
 double dk_z = 2.*M_PI/(nz*dz), knyq_z = M_PI/dz;
 cout<<"Transverse:"<<endl
     <<"  Resolution: dk_x = "<<dk_x<<" Mpc^-1  (2Pi/dk_x="<<2.*M_PI/dk_x<<" Mpc)"<<endl
     <<"              dk_y = "<<dk_y<<" Mpc^-1  (2Pi/dk_y="<<2.*M_PI/dk_y<<" Mpc)"<<endl
     <<"  Nyquist:    kx = "<<knyq_x<<" Mpc^-1  (2Pi/knyq_x="<<2.*M_PI/knyq_x<<" Mpc)"<<endl
     <<"              ky = "<<knyq_y<<" Mpc^-1  (2Pi/knyq_y="<<2.*M_PI/knyq_y<<" Mpc)"<<endl;
 cout<<"Line of sight:"<<endl
     <<"  Resolution: dk_z = "<<dk_z<<" Mpc^-1  (2Pi/dk_z="<<2.*M_PI/dk_z<<" Mpc)"<<endl
     <<"  Nyquist:    kz = "<<knyq_z<<" Mpc^-1  (2Pi/knyq_z="<<2.*M_PI/knyq_z<<" Mpc)"<<endl;

 // --- 
 // --- Variance cosmique
 // --- 
 //                      cosmique                    poisson
 // (sigma/P)^2 = 2*(2Pi)^3 / (4Pi k^2 dk Vsurvey) * [(1+n*P)/(n*P)]^2
 // nombre de mode = 1/2 * Vsurvey/(2Pi)^3 * 4Pi*k^2*dk
 if(kcosm>0. && pkcosm>0.) {
   double pk = pkcosm*pow(growthfac[0],2.);
   cout<<"\n>>>>\n>>>> variance cosmique pour k="<<kcosm<<" Mpc^-1, pk(z=0)="
       <<pkcosm<<", pk(z="<<redshift[0]<<")="<<pk<<"\n>>>>"<<endl;
   for(int i=0;i<3;i++) {  // la correction de variance du au bruit de poisson
     double v = 1.1; if(i==1) v=1.5; if(i==2) v=2.0;
     double ngal = 1./(v-1.)/pk;
     cout<<"  pour "<<ngal<<" gal/Mpc^3 on multiplie par "<<v<<" sigma/P"<<endl;
   }

   cout<<endl;
   vector<double> dk; if(dkcosm>0.) dk.push_back(dkcosm);
   dk.push_back(dk_x); dk.push_back(dk_y); dk.push_back(dk_z);
   for(int i=0;i<(int)dk.size();i++) { // la variance cosmique pure
     double vcosm = sqrt( 2.*pow(2.*M_PI,3.)/(4.*M_PI*pow(kcosm,2.)*dk[i]*vol_survey) );
     double nmode = 0.5*vol_survey/pow(2.*M_PI,3.) * 4.*M_PI*pow(kcosm,2.)*dk[i];
     cout<<"  pour dk = "<<dk[i]<<" Mpc^-1:  sigma/P = "<<vcosm
         <<" , Nmode = "<<nmode<<endl;
   }
 }

 // --- 
 // --- Masse de HI
 // --- 
 cout<<"\n>>>>\n>>>> Mass HI\n>>>>"<<endl;
 Schechter sch(nstar,mstar,alpha);
 sch.SetOutValue(1);
 cout<<"nstar= "<<nstar<<"  mstar="<<mstar<<"  alpha="<<alpha<<endl;
 cout<<"mstar*sch(mstar) = "<<sch(mstar)<<" Msol/Mpc^3/Msol"<<endl;
 int npt = 10000;
 double lnx1=log10(1.e+6), lnx2=log10(1.e+13), dlnx=(lnx2-lnx1)/npt;
 double masshimpc3 = IntegrateFuncLog(sch,lnx1,lnx2,0.001,dlnx,10.*dlnx,6);
 cout<<"Mass density: "<<masshimpc3<<" Msol/Mpc^3"<<endl;

 double masshipix = masshimpc3*vol_pixel;
 double masshitot = masshimpc3*vol_survey;
 cout<<"Pixel mass = "<<masshipix<<" Msol"<<endl
     <<"Total mass in survey = "<<masshitot<<" Msol"<<endl;
 cout<<"OmegaHI a z=0: "<<masshimpc3/rhoc0<<endl;
 for(int i=0;i<3;i++)
   cout<<"        a z="<<redshift[i]<<": "<<masshimpc3/rhocz[i]<<endl;
 if(mhiref<=0.)  mhiref = masshipix;

 sch.SetOutValue(0);
 cout<<"\nsch(mstar) = "<<sch(mstar)<<" /Mpc^3/Msol"<<endl;
 cout<<"Galaxy number density:"<<endl;
 for(double x=lnx1; x<lnx2-0.5; x+=1.) {
   double n = IntegrateFuncLog(sch,x,lnx2,0.001,dlnx,10.*dlnx,6);
   cout<<"  m>10^"<<x<<" Msol: "<<n<<" /Mpc^3, "<<n*vol_pixel<<" /pixel, "
       <<n*vol_survey<<" in survey"<<endl;
 }
 sch.SetOutValue(1);

 // --- 
 // --- Survey values
 // --- 
 cout<<"\n>>>>\n>>>> Observations\n>>>>"<<endl;
 double surfeff = surftot * eta_eff;
 cout<<"surf_tot="<<surftot<<" m^2, eta="<<eta_eff<<"  surf_eff="<<surfeff<<" m^2, tobs="<<tobs<<" s"<<endl;

 // Angles solides pour un telescope plein
 double angSEq[4], angEq[4];
 for(int i=0;i<4;i++) {
   double nu = Fr_HyperFin_Par;
   if(i<3) nu = nuhiz[i];
   angSEq[i] = AngsolEqTelescope(nu,surftot);
   angEq[i] = 2.*FrAngSol(angSEq[i]);
 }
 cout<<"\nFor a "<<surftot<<" m^2 telescope:"<<endl
     <<"    equivalent Omega = "<<angSEq[0]<<" sr in ["<<angSEq[1]<<","<<angSEq[2]<<"]"<<endl
     <<"    angular diameter = "<<angEq[0]<<" rd = "<<rad2min(angEq[0])
     <<"\' in ["<<angEq[2]<<","<<angEq[2]<<"]"<<endl;
 cout<<"At z=0: equivalent Omega = "<<angSEq[3]<<" sr"<<endl
     <<"        angular diameter = "<<angEq[3]<<" rd = "<<rad2min(angEq[3])<<"\'"<<endl;

 // Pour une observation ou le lobe est + petit ou grand que le pixel de simulation
 // La taille angulaire du lobe change avec la frequence donc avec le redshift
 // La taille du lobe "lobearea0" est donnee pour une frequence de reference "lobefreq0" en MHz
 double nlobes[3] = {1.,1.,1.};
 // Si "lobefreq0" negatif c'est la frequence du centre du cube
 if(lobefreq0<=0.) lobefreq0 = nuhiz[0]*1.e3;
 // Si "lobearea0" negatif c'est la taille du pixel ramenee a lobefreq0
 if(lobearea0<=0.) lobearea0 = sr2min2(angsol_pix[0])*pow(nuhiz[0]*1.e3/lobefreq0,2.);
 cout<<"\n--- Lobe: angsol="<<lobearea0<<"\'^2 pour "<<lobefreq0<<" MHz"<<endl;
 double lobearea[3];
 for(int i=0;i<3;i++) {
   lobearea[i] = lobearea0*pow(lobefreq0/(nuhiz[0]*1.e3),2.);
   nlobes[i] = sr2min2(angsol_pix[i])/lobearea[i];
 }
 cout<<"Lobe cylindrique area "<<lobearea[0]<<"\'^2 in ["<<lobearea[1]<<","<<lobearea[2]<<"]"<<endl;
 cout<<"Number of beams in one transversal pixel "<<nlobes[0]<<" in ["<<nlobes[1]<<","<<nlobes[2]<<"]"<<endl;

 // --- 
 // --- Signal analysis
 // --- 
 // --- Temperature d'antenne Ta et temperature de brillance Tb
 // La puissance d'une source de brillance I non polarisee est:
 // P = I * S * Omega * dNu
 // La puissance recue pour une seule polarisation est:
 // P = 1/2 * I * S * Omega * dNu
 // La puissance recue pour un telescope (plein) est
 //    en remplacant Omega = lambda^2/S
 // P = 1/2 * I * lambda^2 * dNu
 // En appliquant la loi de Rayleigh: I = 2*k*Tb/lambda^2
 //   on obtient
 // P = 1/2 * 2*k*Tb * dNu = k * Tb * dNu
 // La temperature d'antenne est definie comme
 // P = k * Ta * dNu
 // Donc pour un ciel de temperature de brillance Tb
 //   et pour une antenne qui mesure une seule polarisation
 // on a: Ta = Tb

 cout<<"\n>>>>\n>>>> Signal Analysis\n>>>>"<<endl;
 cout<<"Facteur polar = "<<facpolar<<endl;

 PlanckSpectra planck(T_CMB_Par);
 planck.SetSpectraApprox(PlanckSpectra::RAYLEIGH);  // Rayleigh spectra
 planck.SetSpectraVar(PlanckSpectra::NU); // frequency
 planck.SetSpectraPower(PlanckSpectra::POWER); // output en W/....
 planck.SetSpectraUnit(PlanckSpectra::ANGSFLUX); // radiance W/m^2/Sr/Hz

 // --- 
 // --- Signal HI
 // --- 
 // Power emitted by HI
 cout<<"--- Power from HI for M = "<<mhiref<<" Msol"<<endl;
 cout<<"Flux factor = "<<hifactor<<" at redshift = "<<redshift[0]<<endl;
 cout<<"Luminosite = "<<hifactor*Msol2LumiHI(mhiref)<<" W"<<endl;

 double fhi_ref[3], sfhi_ref[3];
 for(int i=0;i<3;i++) {
   fhi_ref[i] = hifactor*Msol2FluxHI(mhiref,dlum[i]);
   sfhi_ref[i] = fhi_ref[i] / (dnuhiz[i]*1e9) / Jansky2Watt_cst;
 }
 cout<<"FluxHI("<<dlum[0]<<" Mpc) all polar: "<<fhi_ref[0]<<" W/m^2 = "
     <<fhi_ref[0]/Jansky2Watt_cst<<" Jy.Hz"
     <<" in ["<<fhi_ref[1]/Jansky2Watt_cst<<","<<fhi_ref[2]/Jansky2Watt_cst<<"]"<<endl;
 cout<<"If spread over pixel depth ("<<dnuhiz[0]<<" GHz):"<<endl
     <<"  flux density = "<<sfhi_ref[0]*1.e6<<" mu_Jy"
     <<" in ["<<sfhi_ref[1]<<","<<sfhi_ref[2]<<"]"<<endl;

 // Signal HI
 double psig_2polar[3], tasig_2polar[3], ssig_2polar[3], isig_2polar[3], tsig_2polar[3];
 double psig[3], tasig[3], ssig[3], isig[3], tsig[3];
 double doplarge[3], dzvrot[3];
 
 for(int i=0;i<3;i++) {
   psig_2polar[i] = fhi_ref[i] * surfeff;
   tasig_2polar[i] = psig_2polar[i] / k_Boltzman_Cst / (dnuhiz[i]*1e9);
   ssig_2polar[i] = psig_2polar[i] / surfeff / (dnuhiz[i]*1e9) / Jansky2Watt_cst;
   isig_2polar[i] = ssig_2polar[i] / angsol_pix[i];
   tsig_2polar[i] = isig_2polar[i]*Jansky2Watt_cst
                    / (2.*pow(nuhiz[i]*1e9/(SpeedOfLight_Cst*1000.),2.)*k_Boltzman_Cst);
   psig[i] = facpolar * psig_2polar[i];
   tasig[i] = facpolar * tasig_2polar[i];
   ssig[i] = facpolar * ssig_2polar[i];
   isig[i] = facpolar * isig_2polar[i];
   tsig[i] = facpolar * tsig_2polar[i];
   doplarge[i] = LargeurDoppler(vrot,nuhiz[i]);
   dzvrot[i] = DzFrV(vrot,redshift[i]);
 }
 cout<<"\n--- Signal HI("<<mhiref<<" Msol) for Dnu="<<dnuhiz[0]<<" GHz :"<<endl
     <<" Observation:"<<endl
     <<"    Power="<<psig[0]<<" W in ["<<psig[1]<<","<<psig[2]<<"]"<<endl
     <<"    Flux density = "<<ssig[0]*1.e6<<" mu_Jy in ["<<ssig[1]*1.e6<<","<<ssig[2]*1.e6<<"]"<<endl
     <<"    Intensity = "<<isig[0]<<" Jy/sr in ["<<isig[1]<<","<<isig[2]<<"]"<<endl
     <<"    Antenna temperature = "<<tasig[0]<<" K in ["<<tasig[1]<<","<<tasig[2]<<"]"<<endl
     <<"    Brightness temperature = "<<tsig[0]<<" K in ["<<tsig[1]<<","<<tsig[2]<<"]"<<endl
     <<" 2 polars:"<<endl
     <<"    Power="<<psig_2polar[0]<<" W in ["<<psig_2polar[1]<<","<<psig_2polar[2]<<"]"<<endl
     <<"    Flux density = "<<ssig_2polar[0]*1.e6<<" mu_Jy in ["<<ssig_2polar[1]*1.e6<<","<<ssig_2polar[2]*1.e6<<"]"<<endl
     <<"    Intensity = "<<isig_2polar[0]<<" Jy/sr in ["<<isig_2polar[1]<<","<<isig_2polar[2]<<"]"<<endl
     <<"    Antenna temperature = "<<tasig_2polar[0]<<" K in ["<<tasig_2polar[1]<<","<<tasig_2polar[2]<<"]"<<endl
     <<"    Brightness temperature = "<<tsig_2polar[0]<<" K in ["<<tsig_2polar[1]<<","<<tsig_2polar[2]<<"]"<<endl;

 // Elargissement doppler de la raie a 21cm: dNu = vrot/C * Nu(21cm) / (1+z)
 cout<<"    Doppler for rotation width of "<<vrot<<" km/s"<<endl
     <<"            width="<<doplarge[0]*1.e3<<" MHz  in ["<<doplarge[1]*1.e3<<","<<doplarge[2]*1.e3<<"]"<<endl
     <<"            dzvrot= "<<dzvrot[0]<<" in ["<<dzvrot[1]<<","<<dzvrot[2]<<"]"<<endl;
 if(doplarge[0]>dnuhiz[0])
   cout<<"Warning: doppler width "<<doplarge[0]<<" GHz > "<<dnuhiz[0]<<" GHz redshift bin width"<<endl;

 // --- 
 // --- Synchrotron  (T en -2.7  -> Flux en -0.7  dans l'approximation Rayleigh)
 // --- 
 double tsynch[3];
 double psynch_2polar[3], tasynch_2polar[3], ssynch_2polar[3], isynch_2polar[3];
 double psynch[3], tasynch[3], ssynch[3], isynch[3];

 for(int i=0;i<3;i++) {
   tsynch[i] = Tsynch408;
   if(fabs(indnu)>1.e-50) tsynch[i] *= pow(nuhiz[i]/nuhaslam,indnu);
   planck.SetTemperature(tsynch[i]);
   psynch_2polar[i] = planck(nuhiz[i]*1.e+9) * surfeff * angsol_pix[i] * (dnuhiz[i]*1e9);
   tasynch_2polar[i] = psynch_2polar[i] / k_Boltzman_Cst / (dnuhiz[i]*1e9);
   ssynch_2polar[i] = psynch_2polar[i] / surfeff / (dnuhiz[i]*1e9) / Jansky2Watt_cst;
   isynch_2polar[i] = ssynch_2polar[i] / angsol_pix[i];
   psynch[i] = facpolar * psynch_2polar[i];
   tasynch[i] = facpolar * tasynch_2polar[i];
   ssynch[i] = facpolar * ssynch_2polar[i];
   isynch[i] = ssynch[i] / angsol_pix[i];
 }
 cout<<"\n--- Synchrotron: T="<<Tsynch408<<" K ("<<nuhaslam<<" GHz), "<<endl
     <<"    tsynch="<<tsynch[0]<<" K ("<<nuhiz[0]<<" GHz) in ["<<tsynch[1]<<","<<tsynch[2]<<"]"<<endl
     <<" Observation:"<<endl
     <<"    Power="<<psynch[0]<<" W for pixel in ["<<psynch[1]<<","<<psynch[2]<<"]"<<endl
     <<"    Flux density = "<<ssynch[0]<<" Jy for pixel solid angle in ["<<ssynch[1]<<","<<ssynch[2]<<"]"<<endl
     <<"    Intensity = "<<isynch[0]<<" Jy/sr in ["<<isynch[1]<<","<<isynch[2]<<"]"<<endl
     <<"    Antenna temperature = "<<tasynch[0]<<" K in ["<<tasynch[1]<<","<<tasynch[2]<<"]"<<endl
     <<" 2 polars:"<<endl
     <<"    Power="<<psynch_2polar[0]<<" W for pixel in ["<<psynch_2polar[1]<<","<<psynch_2polar[2]<<"]"<<endl
     <<"    Flux density = "<<ssynch_2polar[0]<<" Jy for pixel solid angle in ["<<ssynch_2polar[1]<<","<<ssynch_2polar[2]<<"]"<<endl
     <<"    Intensity = "<<isynch_2polar[0]<<" Jy/sr in ["<<isynch_2polar[1]<<","<<isynch_2polar[2]<<"]"<<endl
     <<"    Antenna temperature = "<<tasynch_2polar[0]<<" K in ["<<tasynch_2polar[1]<<","<<tasynch_2polar[2]<<"]"<<endl;

 // --- 
 // --- CMB
 // --- 
 planck.SetTemperature(tcmb);
 double pcmb_2polar[3], tacmb_2polar[3], scmb_2polar[3], icmb_2polar[3];
 double pcmb[3], tacmb[3], scmb[3], icmb[3];

 for(int i=0;i<3;i++) {
   pcmb_2polar[i] = planck(nuhiz[i]*1.e+9) * surfeff * angsol_pix[i] * (dnuhiz[i]*1e9);
   tacmb_2polar[i] = pcmb_2polar[i] / k_Boltzman_Cst / (dnuhiz[i]*1e9);
   scmb_2polar[i] = pcmb_2polar[i] / surfeff / (dnuhiz[i]*1e9) / Jansky2Watt_cst;
   icmb_2polar[i] = scmb_2polar[i] / angsol_pix[i];
   pcmb[i] = facpolar * pcmb_2polar[i];
   tacmb[i] = facpolar * tacmb_2polar[i];
   scmb[i] = facpolar * scmb_2polar[i];
   icmb[i] = scmb[i] / angsol_pix[i];
 }
 cout<<"\n--- CMB: T="<<tcmb<<" K"<<endl
     <<" Observation:"<<endl
     <<"    Power="<<pcmb[0]<<" W for pixel in ["<<pcmb[1]<<","<<pcmb[2]<<"]"<<endl
     <<"    Flux density = "<<scmb[0]<<" Jy for pixel solid angle in ["<<scmb[1]<<","<<scmb[2]<<"]"<<endl
     <<"    Intensity = "<<icmb[0]<<" Jy/sr in ["<<icmb[1]<<","<<icmb[2]<<"]"<<endl
     <<"    Antenna temperature = "<<tacmb[0]<<" K in ["<<tacmb[1]<<","<<tacmb[2]<<"]"<<endl
     <<" 2 polars:"<<endl
     <<"    Power="<<pcmb_2polar[0]<<" W for pixel in ["<<pcmb_2polar[1]<<","<<pcmb_2polar[2]<<"]"<<endl
     <<"    Flux density = "<<scmb_2polar[0]<<" Jy for pixel solid angle in ["<<scmb_2polar[1]<<","<<scmb_2polar[2]<<"]"<<endl
     <<"    Intensity = "<<icmb_2polar[0]<<" Jy/sr in ["<<icmb_2polar[1]<<","<<icmb_2polar[2]<<"]"<<endl
     <<"    Antenna temperature = "<<tacmb_2polar[0]<<" K in ["<<tacmb_2polar[1]<<","<<tacmb_2polar[2]<<"]"<<endl;

 // --- 
 // --- AGN
 // --- 
 double flux_agn = pow(10.,lflux_agn);
 double mass_agn[3], flux_agn_pix[3], mass_agn_pix[3], lmass_agn_pix[3];
 for(int i=0;i<3;i++) {
   mass_agn[i] = FluxHI2Msol(flux_agn*Jansky2Watt_cst,dlum[i]);
   flux_agn_pix[i] = flux_agn*(dnuhiz[i]*1e9);
   mass_agn_pix[i] = FluxHI2Msol(flux_agn_pix[i]*Jansky2Watt_cst,dlum[i]);
   lmass_agn_pix[i] = log10(mass_agn_pix[i]);
 }
 cout<<"\n--- AGN: log10(S_agn)="<<lflux_agn<<"   S_agn="<<flux_agn<<" Jy :"<<endl
     <<"  mass_agn = "<<mass_agn[0]<<" equiv. Msol/Hz in ["<<mass_agn[1]<<","<<mass_agn[2]<<"]"<<endl
     <<"  flux_agn_pix = "<<flux_agn_pix[0]<<" 10^-26 W/m^2 in ["<<flux_agn_pix[1]<<","<<flux_agn_pix[2]<<"]"<<endl
     <<"  mass_agn_pix = "<<mass_agn_pix[0]<<" Msol in ["<<mass_agn_pix[1]<<","<<mass_agn_pix[2]<<"]"<<endl
     <<"  log10(mass_agn_pix) = "<<lmass_agn_pix[0]<<"  in ["<<lmass_agn_pix[1]<<","<<lmass_agn_pix[2]<<"]"<<endl;

 // =====================================================================
 // ---
 // --- Noise analysis
 // ---
 // --- Puissance du bruit pour un telescope de surface Ae et de BW dNu
 // Par definition la puissance du bruit est:
 //     Pb = k * Tsys * dNu  (W)
 // Pour une source (non-polarisee) de densite de flux (totale 2 polars)
 //     St (exprimee en Jy = 10^-26 W/m^2/Hz)
 //     Pt = St * Ae * dNu   (puissance totale emise en W pour 2 polars)
 //     P1 = 1/2 * St * Ae * dNu   (puissance emise en W pour une polar)
 // la SEFD (system equivalent flux density en Jy) est definie comme
 //   la densite de flux total (2 polars) "St" d'une source (non-polarisee)
 //   dont la puissance P1 mesuree pour une seule polarisation
 //   serait egale a la puissance du bruit. De P1 = Pb on deduit:
 //   SEFD = 2 * k * Tsys / Ae    (en Jy)
 //   la puissance du bruit est: Pb = 1/2 * SEFD * Ae * dNu  (en W)
 // la sensibilite Slim tient compte du temps d'integration et de la BW:
 //   le nombre de mesures independantes est "2*dNu*Tobs" donc
 //   Slim = SEFD / sqrt(2*dNu*Tobs) =  2*k*Tsys/[Ae*sqrt(2*dNu*Tobs) (en Jy)
 // --- Puissance du bruit pour un interferometre
 // Ae = surface d'un telescope elementaire
 // N  = nombre de telescopes dans l'interferometre (Atot = N*Ae)
 // La sensibilite Slim en Jy est:
 // Slim = 2 * k * Tsys / [ Ae * Sqrt(2*N(N-1)/2 *dnu*Tobs) ]
 //      = 2 * k * Tsys / [ Atot/N * Sqrt(2*N(N-1)/2*dnu*Tobs) ]
 //      = 2 * k * Tsys / [ Atot * Sqrt((N-1)/N *dnu*Tobs) ]
 // - Interferometre a deux antennes:
 // Slim =  2 * k * Tsys / [ Atot * Sqrt(1/2 *dnu*Tobs) ]
 // - Interferometre a N antennes (N grand):
 // Slim ->  2 * k * Tsys / [ Atot * Sqrt(dnu*Tobs) ]
 //      C'est aussi la formule pour un telescope unique de surface Atot
 // --- On ne mesure qu'une seule polarisation
 // Ces formules sont valables si on mesure 1 polarisation:
 // Slim est la densite de flux total "St" (2 polars) d'une source (non-polarisee)
 //   qui donne la meme puissance que le bruit dans un detecteur qui ne
 //   mesure qu'une seule polarisation:
 // Le rapport S/N pour une source de densite de flux St (totale 2 polars):
 //    S/N = St / Slim
 // La puissance de bruit est, par definition:
 //    Pb = 1/2 *Slim*Atot*dNu
 //         = k*Tsys*sqrt(2*dNu/Tobs)  pour N=2
 //         = k*Tsys*sqrt(dNu/Tobs)    pour N>>grand
 // La densite de flux d'une source a S/N=1 est:
 //    St = Slim
 // La puissance d'une source a S/N=1 mesuree par un detecteur
 //    qui ne mesure qu'une polar est:
 //    P1_lim = 1/2 *Slim*Atot*dNu
 // --- On mesure les 2 polarisations avec deux voies d'electronique distinctes
 // la puissance du signal mesure est multipliee par 2
 // la puissance du bruit est multipliee par sqrt(2)
 // on a donc un gain d'un facteur sqrt(2) sur le rapport S/N
 // (cela revient d'ailleur a doubler le temps de pose: Tobs -> 2*Tobs)
 // En notant arbitrairement: Slim' = Slim / sqrt(2)
 //    ou Slim est defini par les formules ci-dessus
 // Le rapport S/N pour une source de densite de flux St (totale 2 polars):
 //   (S/N)_2 = (S/N)_1 * sqrt(2) = (St / Slim) * sqrt(2) = St / Slim'
 // La densite de flux d'une source a S/N=1 est:
 //    St = Slim' = Slim / sqrt(2)
 // La puissance d'une source a S/N=1 cumulee par les 2 detecteurs est:
 //    P_lim = St*Atot*dNu = Slim'*Atot*dNu = 1/sqrt(2) *Slim*Atot*dNu
 //          = P1_lim * sqrt(2)
 // La puissance de bruit cumulee par les 2 detecteurs est, par definition:
 //    Pb = P_lim = Slim'*Atot*dNu = P1_lim * sqrt(2)
 //               = 2*k*Tsys*sqrt(dNu/Tobs)   pour N=2
 //               = k*Tsys*sqrt(2*dNu/Tobs)   pour N>>grand
 // =====================================================================

 cout<<"\n>>>>\n>>>> Noise analysis\n>>>>"<<endl;
 double psys[3];
 for(int i=0;i<3;i++) psys[i] = k_Boltzman_Cst * Tsys * (dnuhiz[i]*1.e+9);
 cout<<"Noise: T="<<Tsys<<" K"<<endl
     <<"       P="<<psys[0]<<" W in ["<<psys[1]<<","<<psys[2]<<"]"<<endl;

 cout<<"...Computation assume that noise dominate the signal."<<endl;
 if(ya2polar)
   cout<<"...Assuming 2 polarisations measurements with 2 different electronics."<<endl;


 //---
 for(unsigned short it=0;it<2;it++) {

   double fac = 1.;
   if(it==0) { // Interferometre a 2 telescopes
     fac = 0.5;
     cout<<"\n...Observation limits for a 2 telescope interferometer (with complex correlator)"<<endl
         <<"   (sensitivity is given for real or complex correlator output)"<<endl;
   } else if (it==1) { // Interferometre a N>> telescopes
     fac = 1.;
     cout<<"\n...Observation limits for a N (large) telescope interferometer (with complex correlator)"<<endl
         <<"     (weak source limit sensitivity in a synthetised image)"<<endl
         <<"     Also valid for a single dish telescope"<<endl
         <<"          (add factor sqrt(2) if ON-OFF with T_ON=t_OFF)"<<endl;
   } else continue;

   double slim[3], SsN[3], smass[3], pkbruit[3];
   for(int i=0;i<3;i++) {
     slim[i] = 2. * k_Boltzman_Cst * Tsys / surfeff
               / sqrt(fac*(dnuhiz[i]*1.e+9)*tobs) /Jansky2Watt_cst;
     if(ya2polar) slim[i] /= sqrt(2.);
     SsN[i] = ssig_2polar[i] / slim[i];  // independant de  angsol_pix*surfeff
     smass[i] = mhiref / ssig_2polar[i] * slim[i];
     pkbruit[i] = pow(smass[i]/mhiref,2.)*vol_pixel;
   }
   cout<<"for 1 lobe:"<<endl
       <<"   Slim    = "<<slim[0]*1.e6<<" mu_Jy in ["<<slim[1]*1.e6<<","<<slim[2]*1.e6<<"]"<<endl
       <<"   S/N     = "<<SsN[0]<<" in ["<<SsN[1]<<","<<SsN[2]<<"]"<<endl
       <<"   Mass HI = "<<smass[0]<<" Msol in ["<<smass[1]<<","<<smass[2]<<"]"<<endl
       <<"   Pk      = "<<pkbruit[0]<<" Mpc^3 in ["<<pkbruit[1]<<","<<pkbruit[2]<<"]"<<endl;

   double slim_nl[3], SsN_nl[3], smass_nl[3], pkbruit_nl[3];
   for(int i=0;i<3;i++) {
     slim_nl[i] = slim[i] * sqrt(nlobes[i]);
     SsN_nl[i] = ssig_2polar[i] / slim_nl[i];
     smass_nl[i] = mhiref / ssig_2polar[i] * slim_nl[i];
     pkbruit_nl[i] = pow(smass_nl[i]/mhiref,2.)*vol_pixel;
   }
   cout<<"for "<<nlobes[0]<<" lobes[i] in ["<<nlobes[1]<<","<<nlobes[2]<<"] :"<<endl
       <<"   Slim    = "<<slim_nl[0]*1.e6<<" mu_Jy in ["<<slim_nl[1]*1.e6<<","<<slim_nl[2]*1.e6<<"]"<<endl
       <<"   S/N     = "<<SsN_nl[0]<<" in ["<<SsN_nl[1]<<","<<SsN_nl[2]<<"]"<<endl
       <<"   Mass HI = "<<smass_nl[0]<<" Msol in ["<<smass_nl[1]<<","<<smass_nl[2]<<"]"<<endl
       <<"   Pk      = "<<pkbruit_nl[0]<<" Mpc^3 in ["<<pkbruit_nl[1]<<","<<pkbruit_nl[2]<<"]"<<endl;

 }

 return 0;
 }


//-------------------------------------------------------------------------------------------
double LargeurDoppler(double v, double nu)
// largeur doppler pour une vitesse v en km/s et une frequence nu
{
 return v / SpeedOfLight_Cst * nu;
}

double DzFrV(double v, double zred)
// largeur en redshift pour une vitesse v en km/s au redshift zred
{
  return v / SpeedOfLight_Cst * (1. + zred);
}

double DNuFrDz(double dzred,double nu_at_0,double zred)
// Largeur DNu pour une largeur en redshift "dzred" au redshift "zred"
//    pour la frequence "nu_at_0" a z=0
// nu =  NuHi(z=0)/(1.+z0)
// dnu = NuHi(z=0)/(1.+z0-dz/2) - NuHi/(1.+z0+dz/2)
//     = NuHi(z=0)*dz/[ (1+z0)^2 - (dz/2)^2 ]
//     = NuHi(z=0)*dz/(1.+z0)^2 / [ 1 - [dz/(1+z0)/2)]^2 ]
//     = NuHi(z=0)*dz/(1.+z0)^2 / [1 - dz/(1+z0)/2] / [1 + dz/(1+z0)/2]
//    ~= NuHi(z=0)*dz/(1.+z0)^2   (approx. pour dz<<z0 a l'ordre (dz/z0)^2)
{
  double zp1 = 1.+zred;
  return nu_at_0*dzred/(zp1*zp1)/(1.-dzred/zp1/2.)/(1.+dzred/zp1/2.);
}

double DzFrDNu(double dnu_at_0,double nu_at_0,double zred)
// Largeur en redshift au redshift "zred" pour une largeur
// en frequence "dnu_at_0" a la frequence "nu_at_0" a z=0
{
  if(dnu_at_0<=0.) return 0.;
  double zp1 = 1.+zred;
  double dnusnu0 = dnu_at_0/nu_at_0;
  return 2./dnusnu0 * (sqrt(1.+(dnusnu0*zp1)*(dnusnu0*zp1)) - 1.);
}
double DzFrDNuApprox(double dnu_at_0,double nu_at_0,double zred)
// idem DzFrDNu mais on utilise l'approximation: dnu=NuHi(z=0)*dz/(1.+z0)^2
{
  double zp1 = 1.+zred;
  return dnu_at_0/nu_at_0 *(zp1*zp1);
}

double ZFrLos(double loscom,CosmoCalc& univ)
// Recherche du redshift correspondant a une distance comobile
// le long de la ligne de visee egale a "loscom" Mpc
// et pour un univers "univ"
{
  double dz = univ.ZMax()/10.; if(dz<=0.) dz = 0.1;
  double zmin=0., zmax=0.;
  while(univ.Dloscom(zmax)<loscom) zmax += dz;
  if(zmax==0.) return 0.;
  for(int i=0; i<6; i++) {
    zmin=zmax-dz; if(zmin<0.) zmin=0.;
    dz /= 10.;
    for(double z=zmin; z<zmax+dz; z+=dz) {
      double d = univ.Dloscom(z);
      if(d<loscom) continue;
      zmax = z;
      //cout<<"ZFrLos: z="<<zmax<<" d="<<d<<" / "<<loscom<<endl;
      break;
    }
  }
  return zmax;
}

double AngsolEqTelescope(double nu /* GHz */,double telsurf /* m^2 */)
/*
Calcul de l'angle solide (sr) equivalent pour un telescope
de surface totale "telsurf" (m^2) a la frequence "nu" (GHz)
- Soit D(t) la figure de diffraction du telescope telle que D(t=0)=1
       (t = angle depuis l'axe optique)
  telescope circulaire de diametre D:
      D(t) = [2J1(Pi*D*t/l)/(Pi*D*t/l)]
  telescope rectangulaire axb:
      D(t) = [sin(Pi*a*t/l)/(Pi*a*t/l)]*[sin(Pi*b*t/l)/(Pi*b*t/l)]
- On cherche l'angle solide equivalent (par ex d'un cylindre de hauteur 1)
  Int[ D(t)^2 dOmega ] =  Int[ D(t)^2 2Pi t dt ] = Lambda^2/S
- En conclusion, pour un ciel d'intensite uniforme I, on a:
  P = I * S * Omega * dNu = I * S * (Lambda^2/S) * dNu
*/
{
  double lambda = SpeedOfLight_Cst*1000./(nu*1.e9);
  return lambda*lambda / telsurf;
}
