// This may look like C code, but it is really -*- C++ -*-
// Classe d'emballage (wrapper) d'histos pour trace PI 
//       C. Magneville / R. Ansari             2007
// (C)     CEA-DAPNIA      LAL-IN2P3/CNRS

#ifndef PHISTOWRAPPER_H
#define PHISTOWRAPPER_H

#include <string> 
#include <vector> 
#include "parradapter.h"


//! Wrapper class interface for 1d-histogram like objects , used by PIHisto
// methodes par defaut dans pihisto.cc
class P1DHistoWrapper : public P1DArrayAdapter
{
public:
  P1DHistoWrapper(int asz);  // asz: taille du tableau pour P1DArrayAdapter
  virtual ~P1DHistoWrapper(); 

  virtual int_4 NBins() = 0; 
  virtual r_8 XMin() = 0; 
  virtual r_8 XMax() = 0;
  virtual r_8 BinWidth() = 0; 

  virtual r_8 BinLowEdge(int_4 i) = 0;
  virtual r_8 BinCenter(int_4 i) = 0;
  virtual r_8 BinHighEdge(int_4 i) = 0;

  // Renvoie contenu du bin i
  virtual r_8 Content(int_4 i) = 0;
  virtual r_8 Error(int_4 i) = 0;
  virtual r_8 NbEntries(int_4 i) = 0;

  inline  r_8 operator()(int_4 i) 
  { 
    r_8 rv = (mRetFg == 2) ? NbEntries(i) : ( (mRetFg == 1) ? Error(i) : Content(i) ) ;
    return ( rv*mScale+mOff ); 
  }

  // Implementation de la methode P1DArrayAdapter::Value(int i) 
  // Identique a l'operateur ()(i)
  virtual double Value(int i) { return (*this)(i); }

  // Methode de mise a jour du contenu 
  // avec implementation par defaut - ne faisant rien 
  virtual void Update() { return; }

  // ajoute des lignes de texte avec les infos statistiques (dans text)
  // renvoie le nombre de lignes ajoutees - avec implementation par defaut
  virtual int GetStatInfoAsText(vector<string> &  text );

//   Methode de decodage des options - herite de P1DArrayAdapter
//   implementation ds le fichier pihisto.cc 
  virtual int DecodeOptionString(vector<string> & opt, bool rmdecopt=true);
//   liste des options sous forme text - herite de P1DArrayAdapter (ds pihisto.cc )
  virtual int OptionToString(vector<string> & opt) const;

protected:
  r_8 mScale, mOff;   // scaling coefficient, offset coefficient used in operator ()
  short mRetFg;       // operator() : 0 -> bin content , 1 -> bin error , 2 -> bin nentries
};

//-----------------------------------------------------------------------------

//! Wrapper class interface for 2d-histogram like objects , used by PIHisto2D
// methodes par defaut dans pihisto2d.cc
class P2DHistoWrapper : public P2DArrayAdapter
{
public:
  P2DHistoWrapper(int_4 asx, int_4 asy); // asx,asy: tailles du tableau pour P2DArrayAdapter
  virtual ~P2DHistoWrapper(); 

  virtual int_4 NBinX() = 0; 
  virtual int_4 NBinY() = 0; 
  virtual r_8 XMin() = 0; 
  virtual r_8 XMax() = 0;
  virtual r_8 YMin() = 0; 
  virtual r_8 YMax() = 0;
  virtual r_8 WBinX() = 0; 
  virtual r_8 WBinY() = 0; 

  virtual void BinLowEdge(int_4 i, int_4 j, r_8& xr, r_8& yr) = 0;
  virtual void BinCenter(int_4 i, int_4 j, r_8& xr, r_8& yr) = 0;
  virtual void BinHighEdge(int_4 i, int_4 j, r_8& xr, r_8& yr) = 0;

  // Renvoie contenu du bin i
  virtual r_8 Content(int_4 i, int_4 j) = 0;
  virtual r_8 Error(int_4 i, int_4 j) = 0;
  virtual r_8 NbEntries(int_4 i, int_4 j) = 0;

  inline  r_8 operator()(int_4 i, int_4 j) 
  { 
    r_8 rv = (mRetFg == 2) ? NbEntries(i,j) : ( (mRetFg == 1) ? Error(i,j) : Content(i,j) ) ;
    return ( rv*mScale+mOff ); 
  }
  // Implementation de la methode P2DArrayAdapter::Value(int i, j) 
  // Identique a l'operateur ()(i)
  virtual double Value(int i, int j) { return (*this)(i, j); }
  // Calcule la moyenne des bins entre (ix1,jy1) <= (ix2, jy2)
  virtual double MeanVal(int ix1, int ix2, int jy1, int jy2);

  // Methode de mise a jour du contenu 
  // avec implementation par defaut - ne faisant rien 
  virtual void Update() { return; }

  // ajoute des lignes de texte avec les infos statistiques (dans text)
  // renvoie le nombre de lignes ajoutees - avec implementation par defaut
  virtual int GetStatInfoAsText(vector<string> &  text );

//   Methode de decodage des options - herite de P2DArrayAdapter
//   l'implementation ds le fichier pihisto2d.cc 
  virtual int DecodeOptionString(vector<string> & opt, bool rmdecopt=true);
//   liste des options sous forme text - herite de P2DArrayAdapter (ds pihisto2d.cc )
  virtual int OptionToString(vector<string> & opt) const;

protected:
  r_8 mScale, mOff;   // scaling coefficient, offset coefficient used in operator ()
  short mRetFg;       // operator() : 0 -> bin content , 1 -> bin error , 2 -> bin nentries
};




#endif
