#include <stdio.h>
#include <stdlib.h>
#include <typeinfo>
#include <iostream>
#include <string>

#include "sopnamsp.h"
#include "ppersist.h"
#include "anydataobj.h"
#include "sambainit.h"
#include "histinit.h"


/*!
  \defgroup PrgUtil PrgUtil module
  This module contains simple programs to perform various utility tasks:
  <UL>
  <LI> scanppf : Check and scan PPF files (scanppf.cc)
  <LI> scanfits : Check and scan FITS files (scanfits.cc)
  <LI> runcxx : Compile and run simple C++ code using SOPHYA (runcxx.cc)
  </UL>
*/

/*!
  \ingroup PrgUtil
  \file scanppf.cc
  \brief \b scanppf: Check and scan PPF files

  \verbatim

  csh> scanppf -h
 PIOPersist::Initialize() Starting Sophya Persistence management service 
SOPHYA Version  2.0 Revision 0 (V_Jul2006) -- Jul 17 2006 14:13:27 cxx 
 Usage: scanppf [flags] filename 
 flags = -s -n -a0 -a1 -a2 -a3 -lh -lho -lmod
   -s[=default} : Sequential reading of objects 
   -n : Object reading at NameTags 
   -a0...a3 : Tag List with PInPersist.AnalyseTags(0...3) 
   -lh : List PPersist handler classes 
   -lho : List PPersist handler and dataobj classes 
   -lmod : List initialized/registered modules 

  \endverbatim
*/

int main(int narg, char* arg[])
{

SambaInitiator smbinit;
HiStatsInitiator hisinit;

if ((narg < 2) || (strcmp(arg[1],"-h") == 0) ) {
  cout << " Usage: scanppf [flags] filename \n" 
       << " flags = -s -n -a0 -a1 -a2 -a3 -lh -lho -lmod \n"
       << "   -s[=default} : Sequential reading of objects \n"  
       << "   -n : Object reading at NameTags \n" 
       << "   -a0...a3 : Tag List with PInPersist.AnalyseTags(0...3) \n" 
       << "   -lh : List PPersist handler classes \n" 
       << "   -lho : List PPersist handler and dataobj classes \n" 
       << "   -lmod : List initialized/registered modules \n" << endl; 
  return(0);
  }

try {
  string flnm;
  bool seq=true;
  bool ana=false;
  int analev = 0;
  string opt = "s";
  if ((narg >= 2) && (*arg[1] == '-')) { 
    opt = arg[1];
    if (narg > 2) flnm = arg[2];
  }
  else flnm = arg[1];

  if (opt == "-lh")  {
    cout << " --- scanppf : List of registered handler classes --- " << endl;
    PIOPersist::ListPPHandlers();
  }
  else if (opt == "-lho") {
    cout << " --- scanppf : List of registered handler and DataObj classes --- " << endl;
    PIOPersist::ListPPHandlers();
    PIOPersist::ListDataObjClasses();
  }
  else if (opt == "-lmod") {
    cout << " --- scanppf : List of registered module names and version --- " << endl;
    SophyaInitiator::ListModules(cout);
  }
  else {
    if (opt == "-n")  seq = false;
    else if ((opt[0] == '-') && (opt[1] == 'a')) {  
      ana = true;  
      analev = 0;
      if (opt.length()>2)  analev = opt[2]-'0'; 
    }
    
    if (ana)   cout << " Analyse PInPersist( " << flnm << ")  Level=" << analev << endl;
    else { 
      if (!seq) cout << "PInPersist( " << flnm << ") Object Reading at NameTags " << endl;
      else  cout << "PInPersist( " << flnm << ") Sequential Object Reading " << endl;
    }
    PPersist* op = NULL;
    cout << " Opening PPF file " << flnm << endl; 
    PInPersist s(flnm);
    
    if (ana) s.AnalyseTags(analev);   // Analysing all tags in file 
    
    else {
      cout << " Version= " << s.Version() << " CreationDate= " << s.CreationDateStr() << endl;
      int nt = s.NbNameTags();
      cout << " Number of tags in file = " << nt << endl;
      if ( seq || (nt < 1) ) {
	while (1) {
	  op = s.ReadObject();
	  cout << " Object Type " << typeid(*op).name() << endl;
	  if (op) delete op;
	}
      }
      for(int i=0; i<nt; i++) {
	cout << ">>> TagNum= " << i << " TagName= " << s.GetTagName(i) << endl;
	s.GotoNameTagNum(i);
	op = s.ReadObject();
	cout << " Object Type " << typeid(*op).name() << endl;
	if (op) delete op;
      }
    }
  }
}
catch (PThrowable & pex) {
  cerr << " scanppf/Error - Exception catched " << (string)typeid(pex).name()
       << " - Msg= " << pex.Msg() << endl;
}

cout << " ----------- End of scanppf ------------- " << endl;
}
