#ifndef RANDOMGENERATORINTERFACE_H_SEEN
#define RANDOMGENERATORINTERFACE_H_SEEN

// Classes RandomGeneratorInterface 
//
// R. Ansari          (C) UPS+LAL IN2P3/CNRS  2009
// C. Magneville      (C) DAPNIA/SPP  CEA     2009


#include "machdefs.h"
#include "anydataobj.h"
#include <complex>
#include <vector>

namespace SOPHYA {

//! Definition of the interface for Random number generator classes 
class RandomGeneratorInterface : public AnyDataObj {
public:
  RandomGeneratorInterface();
  virtual ~RandomGeneratorInterface();

  void UseGaussian(uint_2 type=0);
  void UsePoisson(uint_2 type=0);
  void UseExpo(uint_2 type=0);

  void GenerateSeedVector(int nseed,vector<uint_2>& seed,int lp=0);

  // --- Le tirage sur une distribution plate
  /*! \brief Return a random number \b r with a flat distribution  0<=r<1 */ 
  inline r_8 Flat01() {return Next();}
  /*! \brief Return a random number \b r with a flat distribution  -1<=r<1 */ 
  inline r_8 Flatpm1() {return 2.*Next()-1.;}

  // --- Le tirage sur une distribution gaussienne
  /*! \brief Return a random number following a gaussian distribution (sigma=1, mean=0)*/ 
  inline r_8 Gaussian() 
         {
           if(usegaussian_==0) return GaussianBoxMuller();
           if(usegaussian_==1) return GaussianSNorm();
           if(usegaussian_==2) return GaussianPolarBoxMuller();
           if(usegaussian_==3) return GaussianRatioUnif();
           if(usegaussian_==4) return GaussianLevaRatioUnif();
           return GaussianBoxMuller();
         }
  virtual r_8 GaussianBoxMuller(); 
  virtual r_8 GaussianSNorm(); 
  virtual r_8 GaussianPolarBoxMuller(); 
  virtual r_8 GaussianRatioUnif(); 
  virtual r_8 GaussianLevaRatioUnif();
  /*! \brief Return a random number following a gaussian distribution "sigma", (mean=0)*/ 
  inline r_8 Gaussian(double sigma) {return sigma*Gaussian();}
 /*! \brief Return a random number following a gaussian distribution with mean=mu, and sigma */ 
  inline r_8 Gaussian(double sigma,double mu) {return sigma*Gaussian()+mu;}

  /*! \brief Return a random number following a gaussian tail distribution for x>sdev */ 
  virtual r_8 GaussianTail(double sdev);

  // --- Le tirage sur une distribution de poisson
  /*! \brief Return a random number following a poisson distribution with mean mu */ 
  inline uint_8 Poisson(double mu, double mumax) 
         {
           if(usepoisson_==0) return PoissonSimple(mu,mumax);
           if(usepoisson_==1) return PoissonAhrens(mu);
           return PoissonSimple(mu,mumax);
         }
  virtual uint_8 PoissonSimple(double mu, double mumax=-1);
  virtual uint_8 PoissonAhrens(double mu);

  // --- Le tirage sur une distribution exponentielle
  inline r_8 Expo() 
         {
           if(useexpo_==0) return ExpoSimple();
           if(useexpo_==1) return ExpoAhrens();
           return ExpoSimple();
         }
  virtual r_8 ExpoSimple();
  virtual r_8 ExpoAhrens();

  // --- Le tirage gaussien complexe (cf texte a la fin du .cc)
  inline complex< r_8 > ComplexGaussRan(void)
         {return complex< r_8 >(Gaussian(),Gaussian());}
  inline complex< r_8 > ComplexGaussRan(double sig)
         {return complex< r_8 >(sig*Gaussian(),sig*Gaussian());}
  /*! \brief Returns the module of a random complex number generated by ComplexGaussRan */
  inline double ModComplexGaussRan(double sig=1.)
         {double r=-log(1.-Next()); return sig*sqrt(2.*r);}

protected:
  //! Return a random number in [0,1] 
  virtual r_8 Next();
  uint_2 usegaussian_, usepoisson_, useexpo_;
  
};

} /* namespace SOPHYA */

#endif
