#include <string.h>

#include "brpaqu.h"

/* ----------------------------------------
     Projet BAORadio --- LAL - 2008-2010 
     R. Ansari , M.Taurigna 
-------------------------------------------  */

/* --Methode-- */
BAORadioException::BAORadioException(const char * m)   throw()
{
  if (m!=NULL) { 
    strncpy(msg_, m, BREX_MAXMSGLEN-1);
    msg_[BREX_MAXMSGLEN-1] = '\0';
  }
  else msg_[0] = '\0';
}
/* --Methode-- */
BAORadioException::BAORadioException(const string& m)   throw()
{
  strncpy(msg_, m.c_str(), BREX_MAXMSGLEN-1);
  msg_[BREX_MAXMSGLEN-1] = '\0';
}

/* --Methode-- */
BAORadioException::~BAORadioException()  throw()
{
}

/* --Methode-- */
const char* BAORadioException::what() const throw()
{
  return msg_;
}

/* --Methode-- */
string const BAORadioException::Msg() const 
{
  return (string(msg_));
}

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////

static inline void bswap4(void* p)
{
  UInt32 tmp = *(UInt32*)p;
  *(UInt32*)p = ((tmp >> 24) & 0x000000FF) |
                ((tmp >> 8)  & 0x0000FF00) |
                ((tmp & 0x0000FF00) << 8)  |
                ((tmp & 0x000000FF) << 24);
}

/* --Methode__ */
BRPaquet::BRPaquet(Byte* src, Byte* dst, int paqsz, BRDataFmtConv fgconv)
  // swapall = true -> on swap tout le paquet, sinon swap entete seulement
{
  dst_ = dst;
  sz_ = paqsz;
  /*
  if ((dst == NULL) || (paqsz<=(BRHDRSIZE+BRTRLSIZE)) ) 
    throw BAORadioException("BRPaquet::BRPaquet(src,dst...) NULL src or dst pointer OR paqsz<HDR+TRLSize");
  */
  if ((dst==NULL)||(src==NULL))  return;

  UInt32* src32 = (UInt32*)src;
  UInt32* dst32 = (UInt32*)dst;
 
 
  switch ( fgconv ) {
  case BR_DoNothing :  // rien a faire
    break;
  case BR_Copy : // copie directe 
    memcpy(dst_, src, sz_); 
    break;
  case BR_Swap32 : // On swappe toutes les donnees du paquet 
  //  les bytes sont dans l'ordre par paquet de 4 octets (Int32) , les deux Int32 de 
  // On copie la zone donnees en faisant un byte-swap correspondant a 8 octets (4->8 Reza/firmware SGDMA)
    for(int ka=0; ka<sz_/4; ka+=2) {
      dst32[ka] = src32[ka+1];
      dst32[ka+1] = src32[ka];
    }   
    break;
  case BR_SwapAll:
    // Byte swap complet sur 8 bytes 
    for(int ka=0; ka<sz_; ka+=8) {
      for(int kb=0; kb<4; kb++) {
        dst_[ka+kb] = src[ka+3-kb+4];    
        dst_[ka+kb+4] = src[ka+3-kb];    
      }
    }
    break;
  case BR_CopyHDR:
    for(int ka=0; ka<BRHDRSIZE/4; ka++)  dst32[ka] = src32[ka];
    for(int ka=(HeaderSize()+DataSize())/4;ka < sz_/4; ka++)  dst32[ka] = src32[ka];
    break;
  case BR_SwapHDR :
    // ByteSwap 8 (4->8 Reza/firmware SGDMA) de l'enete 
    for(int ka=0; ka<BRHDRSIZE; ka+=8) {
      for(int kb=0; kb<4; kb++) {
	dst_[ka+kb] = src[ka+3-kb+4];    
        dst_[ka+kb+4] = src[ka+3-kb];    
      }
    }
    // Byte swap (sur 8 octets) du trailer 
    for(int ka=HeaderSize()+DataSize(); ka<sz_; ka+=8) {
      for(int kb=0; kb<4; kb++) {
	dst_[ka+kb] = src[ka+3-kb+4];    
        dst_[ka+kb+4] = src[ka+3-kb];    
      }
    }
    break;

    //  --------------------
    //  --- Copie/remise en ordre des donnees FFT 
  case BR_FFTOneChan :
  case BR_FFTTwoChan :
    // copie du header uniquement 
    for(int ka=0; ka<BRHDRSIZE/4; ka++)  dst32[ka] = src32[ka];
    // on reoordonne les coeff FFT (pas de swap) 
    if (fgconv == BR_FFTOneChan) 
      ReorderFFTData((SByte*)(src+HeaderSize()), (SByte*)(dst_+HeaderSize()), DataSize());
    else  {
      ReorderFFTData((SByte*)(src+HeaderSize()), (SByte*)(dst_+HeaderSize()), DataSize()/2);
      ReorderFFTData((SByte*)(src+HeaderSize()+DataSize()/2), (SByte*)(dst_+HeaderSize()+DataSize()/2), DataSize()/2);
    }
    // copie du trailler uniquement : 
    for(int ka=(HeaderSize()+DataSize())/4;ka < sz_/4; ka++)  dst32[ka] = src32[ka];
    break;

  case BR_FFTOneChanSwapAll :
  case BR_FFTTwoChanSwapAll :
    // Byte swap (sur 8 octets) de l'entete
    // ByteSwap 8 (4->8 Reza/firmware SGDMA) de l'enete 
    for(int ka=0; ka<BRHDRSIZE; ka+=8) {
      for(int kb=0; kb<4; kb++) {
	dst_[ka+kb] = src[ka+3-kb+4];    
        dst_[ka+kb+4] = src[ka+3-kb];    
      }
    }
    // on reoordonne les coeff FFT et on swappe en mem temps  (Byte-swap sur 8 octets)
    if (fgconv == BR_FFTOneChanSwapAll) 
      ReorderFFTDataSwapAll((SByte*)(src+HeaderSize()), (SByte*)(dst_+HeaderSize()), DataSize());
    else  {
      ReorderFFTDataSwapAll((SByte*)(src+HeaderSize()), (SByte*)(dst_+HeaderSize()), DataSize()/2);
      ReorderFFTDataSwapAll((SByte*)(src+HeaderSize()+DataSize()/2), (SByte*)(dst_+HeaderSize()+DataSize()/2), DataSize()/2);
    }
    // Byte swap (sur 8 octets) du trailer 
    for(int ka=HeaderSize()+DataSize(); ka<sz_; ka+=8) {
      for(int kb=0; kb<4; kb++) {
	dst_[ka+kb] = src[ka+3-kb+4];    
        dst_[ka+kb+4] = src[ka+3-kb];    
      }
    }
    break;
 
  case BR_FFTOneChanSwap32 :
  case BR_FFTTwoChanSwap32 :
    // swap du header uniquement : Echange de deux mots de 4 octets
    for(int ka=0; ka<BRHDRSIZE/4; ka+=2) {
      dst32[ka] = src32[ka+1];
      dst32[ka+1] = src32[ka];
    }
    // on reoordonne et on swappe en mem temps (Echange de deux mots de 4 octets)
    if (fgconv == BR_FFTOneChanSwap32) 
      ReorderFFTDataSwap32((SByte*)(src+HeaderSize()), (SByte*)(dst_+HeaderSize()), DataSize());
    else  {
      ReorderFFTDataSwap32((SByte*)(src+HeaderSize()), (SByte*)(dst_+HeaderSize()), DataSize()/2);
      ReorderFFTDataSwap32((SByte*)(src+HeaderSize()+DataSize()/2), (SByte*)(dst_+HeaderSize()+DataSize()/2), DataSize()/2);
    }
    // swap du trailler uniquement : Echange de deux mots de 4 octets
    for(int ka=(HeaderSize()+DataSize())/4;ka < sz_/4; ka+=2) {
      dst32[ka] = src32[ka+1];
      dst32[ka+1] = src32[ka];
    }
    break;

  default:
    // on ne fait rien 
    break;
  }  // Fin switch 
  
}

/* --Methode__ */
BRPaquet::BRPaquet(Byte* srcdst, int paqsz)
{
  /*
  if ((srcdst==NULL)||(paqsz<=(BRHDRSIZE+BRTRLSIZE))) 
    throw BAORadioException("BRPaquet::BRPaquet(srcdst) NULL pointer OR paqsz<HDR+TRLSize");  
  */
  dst_ = srcdst;
  sz_ = paqsz;
}

/* --Methode__ */
BRPaquet::BRPaquet(int paqsz)
{
  /*
  if (paqsz<=(BRHDRSIZE+BRTRLSIZE))
    throw BAORadioException("BRPaquet::BRPaquet(paqsz) paqsz<HDR+TRLSize");  
  */
  dst_ = NULL;
  sz_ = paqsz;
}

/* --Methode__ */
void BRPaquet::CopyFrom(BRPaquet& pq, BRPaqReducAction ract, int offset)
{
  if (ract==BR_CopyRA) {
    if (PaquetSize()!=pq.PaquetSize()) {
      throw BAORadioException("BRPaquet::CopyFrom(BR_CopyRA) - NOT equal paquet size");
    }
    memcpy(Begin(), pq.Begin(), PaquetSize()); 
    return;
  }
  // Recopie de l'entete HDR et trailer TRL 
  memcpy(Header(), pq.Header(), HeaderSize()); 
  //memcpy(dst_, pq.dst_, 24); 
  // SetHDRMarker64(pq.HDRMarker64());
  // SetTimeTag(pq.TimeTag());
  memcpy(Trailer(), pq.Trailer(), TrailerSize());
  // Recopie partielle des donnees 
  size_t szc, szcmx;
  switch (ract) {
  case BR_OneChanReduc:
  case BR_OneChanReducK0:
    szc = DataSize();
    szcmx = pq.DataSize()-offset;
    if (szcmx<szc) szc=szcmx;
    memcpy(Data1(), pq.Data1()+offset, szc);
    if (ract==BR_OneChanReducK0)   *(Data1C()) = *(pq.Data1C());  // On garde le continu et le nyquist   
    SetPaqLen(szc/4);  
    break;
  case BR_TwoChanReduc:
  case BR_TwoChanReducK0:
    szc = DataSize()/2;
    szcmx = pq.DataSize()/2-offset;
    if (szcmx<szc) szc=szcmx;
    memcpy(Data1(), pq.Data1()+offset, szc);
    memcpy(Data2(), pq.Data2()+offset, szc);
    if (ract==BR_TwoChanReducK0)  {   // On garde le continu et le nyquist  
      *(Data1C()) = *(pq.Data1C()); 
      *(Data2C()) = *(pq.Data2C()); 
    }
    SetPaqLen(szc/2);  
    break;
  default:
    break;
  }
}


/* --Methode__ */
UInt16 BRPaquet::ChannelID()
{

  UInt16 ChnId=ChanId();
  UInt16 ChpId=ChipId();
 
     
  if (ChpId == 2) 
    {
      if (ChnId == 1) ChnId = Ch3;
      if (ChnId == 2) ChnId = Ch4;
      if (ChnId == 3) ChnId = Ch3_4;
    }
  return(ChnId);
}

/* --Methode__ */
UInt16 BRPaquet::ModeAcquisition()
{
  UInt16 ModAq;
//DEL  printf("Mod Acq %x \n",ModeAcq());
  ModAq = ((ModeAcq() & 0x30)>> 4);
  return(ModAq);
}

/* --Methode__ */
void BRPaquet::SetHDRMarker64(UInt64 htag)
{
  *((UInt64*)(dst_+OFFSET)) = htag;
  return;  	
}

/* --Methode__ */
void BRPaquet::SetTRLMarker64(UInt64 ttag)
{
  *((UInt64*)(dst_+(sz_-BRTRLSIZE+OFFSET+1))) = 0;	
  *((UInt64*)(dst_+(sz_-BRTRLSIZE+OFFSET))) = ttag; 
  return;         	          	   	
}

/* --Methode__ */
void BRPaquet::SetFrameCounter(UInt32 fc)
{
  UInt32* wp =  (UInt32*)(dst_+(BRFRCPTOFF+OFFSET));
  *wp = (*wp & 0x0000FFFF) | ((fc<<16) & 0xFFFF0000); 
  return;         	          	
}

/* --Methode__ */
void BRPaquet::SetTimeTag(UInt64 timtag)
{
  UInt32* wp =  (UInt32*)(dst_+(BRFRCPTOFF+OFFSET));
  UInt32 fc = *wp;
  *((UInt64*)(dst_+(BRTMTAGOFF+OFFSET))) = timtag;
  *wp = (*wp & 0x0000FFFF) | (fc & 0xFFFF0000);
  return;         	
}

/* --Methode__ */
ostream& BRPaquet::Print(ostream & os, int nelt, bool prht)
{
  os << endl << "BRPaquet::Print() PaqSz=" << PaquetSize() << " DataSize=" << DataSize() 
     << " dst_pointer=(hex)" << hex << (unsigned long)dst_ << dec << endl; 
  if (dst_ == NULL) {
    os << " ...NULL paquet " << endl;
    return os; 
  }
  os << " BR AcqMode: " << ModeAcquisition() << " Channel: " << ChannelID()  
     << " FrameCounter=" << FrameCounter() << " FrameDataLen=" << PaqLen() << endl;
  if (TrailerSize() > 0) 
    os << " ...HDRMarker(hex)=" << hex <<  HDRMarker() << " TRLMarker=" << TRLMarker() << dec << endl;
  else 
    os << " ...HDRMarker(hex)=" << hex <<  HDRMarker() << " NO TRLMarker=" << dec << endl;
  UInt32 tt1, tt2; 
  tt2 = TimeTag1(); 
  tt1 = TimeTag2();
  os << " ...TimeTag (hex)=" << hex << " TT1= " << tt1 << " TT2=" << tt2 
     << dec << " ->TimeTag()=" << TimeTag() << endl;
  // os << " ...Position Chariot (hex)= " << hex << PositionChariot() << endl;
  if (nelt > DataSize()/2) nelt = DataSize()/2;
  os << " ...Data[1.." << nelt << "]= ";

  for(int k=0; k<nelt; k++) os << (int)(*(Data1()+k)) << " , ";
  os << endl;
  os << " ...Data[" << DataSize()-nelt << ".." << DataSize()-1 << "]= ";
  for(int k=DataSize()-nelt; k<DataSize(); k++) os << (int)(*(Data1()+k)) << " , ";
  os << endl;
  if (prht) {   // Impression header / trailer
    UInt32* hdr = (UInt32*)Header();
    os << " ...Header (hex):" << hex ;
    for(int k=0; k<HeaderSize()/sizeof(UInt32); k++) 
      os << hdr[k] << " , " ;
    os << dec << endl;
    if (TrailerSize() > 0) {
      UInt32* trl = (UInt32*)Trailer();
      os << " ...Trailer (hex):" << hex ;
      for(int k=0; k<TrailerSize()/sizeof(UInt32); k++) 
        os << trl[k] << " , " ;
      os << dec << endl;
    }
  }
  return os;
}


// ---------------------------------------------------------
//  **** REMARQUE   N/2+1 complexes -> N/2 complexes *****
// N = Nb d'echantillon en temps -> N/2 paires (real, imag)
// Il y a le continu, et N/2 frequences ---> N/2+1 nombres complexes,
// mais avec la contrainte Z(0).imag = 0  Z(N/2).imag = 0 
// f(i) i=0...N-1   ===> Z(k) ( Z complexe , k=0...N/2 )
// mais avec la contrainte Z(0).imag = 0  Z(N/2).imag = 0 
// On peut donc tout mettre ds N/2 complexes en choisissant 
// de mettre ds Z(0).imag  Z(N/2).real 
// ---------------------------------------------------------- 

// Fonction magique qui donne le pointeur permettant de tenir compte du byte-swp sur 8 octets
static inline int IndexByteSwap8(int idx) 
{
  return ( (idx-(idx%8))+(7-idx%8) ) ;
}

/* --Methode__ */
void BRPaquet::ReorderFFTDataSwapAll(SByte* src, SByte* dst, int N)
{
  // Remise en ordre des donnees avec swap complet sur 8 bytes 
  // Code recopie depuis /Dev/DisplayData/HistoWindow.cc 
  // fonction TraceWind::DisplayBaoDatasFFT() et adapte aux structures BRPaquet et Cie 
  // Modif par rapport au code de Bruno : N/2 elements complexes au lieu de N/2+1 - Remarque ci-dessus
 
  int nCoef = N / 2; // to change
  int debutIndex = N / 4 + 1;
  int fifoSize =  N / 4 - 1;
  int i; 

  TwoByteComplex* dstcmplx = (TwoByteComplex*)dst;
  
  //   cout << " Display BAO Datas FFT (" << N << ")" << " : from 0 to "<< nCoef << endl;
  //   cout << " Variables : debutIndex, fifoSize " << debutIndex << ", " << fifoSize << endl;

  
  // Sortie 1
  for (i = 0; i < fifoSize ; i++)
    {
      dstcmplx[debutIndex + i].realB() = src[IndexByteSwap8(2*i)];
      dstcmplx[debutIndex + i].imagB() = src[IndexByteSwap8(2*i + 1)];
    }

  // element au milieu
  dstcmplx[N / 4].realB() = src[IndexByteSwap8(2*fifoSize)];
  dstcmplx[N / 4].imagB() = src[IndexByteSwap8(2*fifoSize + 1)];

  // Sortie 2
  for (i = 0; i < fifoSize ; i++)
    {
      dstcmplx[fifoSize - i].realB() = src[IndexByteSwap8(nCoef + 2*i)];
      dstcmplx[fifoSize - i].imagB() = src[IndexByteSwap8(nCoef + 2*i + 1)];
    }

  // k = 0 et k = N/2  
  dstcmplx[0].realB() = src[IndexByteSwap8(N - 2)];
  // Voir Remarque ci-dessus Z(N/2).real -> Z(0).image
  dstcmplx[0].imagB() = src[IndexByteSwap8(N - 1)];  // Attention, on met ici la real(fmax)

  return ; 
}

static inline int IndexByteSwap8_32(int idx) 
{
  return ( (idx-(idx%8))+((4+idx%8)%8) ) ;
}

/* --Methode__ */
void BRPaquet::ReorderFFTDataSwap32(SByte* src, SByte* dst, int N)
{
  // Remise en ordre avec echange (swap) des mots de 32 bits 
  // Code recopie depuis /Dev/DisplayData/HistoWindow.cc 
  // fonction TraceWind::DisplayBaoDatasFFT() et adapte aux structures BRPaquet et Cie 
  // Modif par rapport au code de Bruno : N/2 elements complexes au lieu de N/2+1 - Remarque ci-dessus

  int nCoef = N / 2; // to change
  int debutIndex = N / 4 + 1;
  int fifoSize =  N / 4 - 1;
  int i; 

  TwoByteComplex* dstcmplx = (TwoByteComplex*)dst;
  
  //   cout << " Display BAO Datas FFT (" << N << ")" << " : from 0 to "<< nCoef << endl;
  //   cout << " Variables : debutIndex, fifoSize " << debutIndex << ", " << fifoSize << endl;

  
  // Sortie 1
  for (i = 0; i < fifoSize ; i++)
    {
      dstcmplx[debutIndex + i].realB() = src[IndexByteSwap8_32(2*i)];
      dstcmplx[debutIndex + i].imagB() = src[IndexByteSwap8_32(2*i + 1)];
    }

  // element au milieu
  dstcmplx[N / 4].realB() = src[IndexByteSwap8_32(2*fifoSize)];
  dstcmplx[N / 4].imagB() = src[IndexByteSwap8_32(2*fifoSize + 1)];

  // Sortie 2
  for (i = 0; i < fifoSize ; i++)
    {
      dstcmplx[fifoSize - i].realB() = src[IndexByteSwap8_32(nCoef + 2*i)];
      dstcmplx[fifoSize - i].imagB() = src[IndexByteSwap8_32(nCoef + 2*i + 1)];
    }

  // k = 0 et k = N/2  
  dstcmplx[0].realB() = src[IndexByteSwap8_32(N - 2)];
  // Voir Remarque ci-dessus Z(N/2).real -> Z(0).image
  dstcmplx[0].imagB() = src[IndexByteSwap8_32(N - 1)];  // Attention, on met ici la real(fmax)

  return ; 
}

/* --Methode__ */
void BRPaquet::ReorderFFTData(SByte* src, SByte* dst, int N)
{
  // Remise en ordre des donnees FFT (sans swap) 
  // Code recopie depuis /Dev/DisplayData/HistoWindow.cc 
  // fonction TraceWind::DisplayBaoDatasFFT() et adapte aux structures BRPaquet et Cie 
  // Modif par rapport au code de Bruno : N/2 elements complexes au lieu de N/2+1 - Remarque ci-dessus
 
  int nCoef = N / 2; // to change
  int debutIndex = N / 4 + 1;
  int fifoSize =  N / 4 - 1;
  int i; 

  TwoByteComplex* dstcmplx = (TwoByteComplex*)dst;
  
  //   cout << " Display BAO Datas FFT (" << N << ")" << " : from 0 to "<< nCoef << endl;
  //   cout << " Variables : debutIndex, fifoSize " << debutIndex << ", " << fifoSize << endl;

  
  // Sortie 1
  for (i = 0; i < fifoSize ; i++)
    {
      dstcmplx[debutIndex + i].realB() = src[(2*i)];
      dstcmplx[debutIndex + i].imagB() = src[(2*i + 1)];
    }

  // element au milieu
  dstcmplx[N / 4].realB() = src[(2*fifoSize)];
  dstcmplx[N / 4].imagB() = src[(2*fifoSize + 1)];

  // Sortie 2
  for (i = 0; i < fifoSize ; i++)
    {
      dstcmplx[fifoSize - i].realB() = src[(nCoef + 2*i)];
      dstcmplx[fifoSize - i].imagB() = src[(nCoef + 2*i + 1)];
    }

  // k = 0 et k = N/2  
  dstcmplx[0].realB() = src[(N - 2)];
  // Voir Remarque ci-dessus Z(N/2).real -> Z(0).image
  dstcmplx[0].imagB() = src[(N - 1)];  // Attention, on met ici la real(fmax)

  return ; 
}

/* --Methode__ */
const char* BRPaquet::FmtConvToString(BRDataFmtConv fgconv)
{
  const char * rs="";
  switch ( fgconv ) {
  case BR_DoNothing :  
    rs = "BR_DoNothing";
    break;
  case BR_Copy : 
    rs = "BR_Copy";
    break;
  case BR_SwapAll : 
    rs = "BR_SwapAll";
    break;
  case BR_Swap32 :
    rs = "BR_Swap32";
    break;
  case BR_CopyHDR :
    rs = "BR_CopyHDR";
    break;
  case BR_SwapHDR :
    rs = "BR_SwapHDR";
    break;
  case BR_FFTOneChan :
    rs = "BR_FFTOneChan";
    break;
  case BR_FFTTwoChan :
    rs = "BR_FFTTwoChan";
    break;
  case BR_FFTOneChanSwapAll :
    rs = "BR_FFTOneChanSwapAll";
    break;
  case BR_FFTTwoChanSwapAll :
    rs = "BR_FFTTwoChanSwapAll";
    break;
  case BR_FFTOneChanSwap32 :
    rs = "BR_FFTOneChanSwap32";
    break;
  case BR_FFTTwoChanSwap32 :
    rs = "BR_FFTTwoChanSwap32";
    break;
  default:
    rs = "?????";
    break;
  }  // Fin switch 
  return rs;
}

/* --Methode__ */
const char* BRPaquet::ReducActionToString(BRPaqReducAction rac)
{
  const char * rs="";
  switch ( rac ) {
  case BR_CopyRA :  
    rs = "BR_CopyRA";
    break;
  case BR_OneChanReduc : 
    rs = "BR_OneChanReduc";
    break;
  case BR_TwoChanReduc : 
    rs = "BR_TwoChanReduc";
    break;
  case BR_OneChanReducK0 : 
    rs = "BR_OneChanReducK0";
    break;
  case BR_TwoChanReducK0 : 
    rs = "BR_TwoChanReducK0";
    break;
  default:
    rs = "?????";
    break;
  }  // Fin switch 
  return rs;
}

// --------------------------------------------------------------------------
// Classe pour effectuer des verifications d'integrite sur les paquets/frames
// --------------------------------------------------------------------------

/* --Methode__ */
BRPaqChecker::BRPaqChecker(bool cktrl, int maxprt)
{
  cktrl_ = cktrl;	
  totnframes = 0;
  nframeok = 0;
  lostframes = 0;
  frclst = 0;
  lastframenum = 0;
  cnt_saut = 0;
  maxprt_ = maxprt;
  DefineHDRTag();
  DefineTRLTag();
}

/* --Methode__ */
BRPaqChecker::~BRPaqChecker()
{
}

UInt64 BRPaqChecker::DefineHDRTag(UInt32 hdr1, UInt32 hdr2)
{
  hdrtag_ = (UInt64)hdr1 + ((UInt64)hdr2 << 32);
  return hdrtag_;
}

/* --Methode__ */
UInt64 BRPaqChecker::DefineTRLTag(UInt32 trl1, UInt32 trl2)
{
  trltag_ = (UInt64)trl1 + ((UInt64)trl2 << 32);
  return trltag_;
}

/* --Methode__ */
bool BRPaqChecker::Check(BRPaquet& paq, UInt64& numframe)
{
  totnframes++; 
  numframe=0;
  if (paq.HDRMarker64() != HDRTag()) return false;
  if (cktrl_&&(paq.TRLMarker64() != TRLTag())) return false;
  /*  DBG 
  if (paq.TRLMarker64() != TRLTag()) { 
    cnt_pb++; 
    if (cnt_pb < 5) paq.Print();
    return false;
  }
  */
  unsigned int curfc = paq.FrameCounter();
  unsigned int delfc = 0;
  if (nframeok > 0) {
    if (curfc>frclst)  delfc = (curfc-frclst);
    else delfc = (65536-frclst+curfc);
    lostframes += (unsigned long long)delfc - 1;
    if (delfc != 1) { 
      cnt_saut++; 
      if (cnt_saut < maxprt_) { 
        cout << "BRPaqChecker::Check([NumFrameOK=" << nframeok 
             << ")/Debug  FrameCounter Cur=" << curfc 
             << " Last=" << frclst << " -> delta=" << delfc << endl;  
        paq.Print();
      }
    }
  }
  nframeok++; frclst=curfc; 
  lastframenum+=(unsigned long long)delfc;  // numero de frame sans repliement a 65535 
  numframe=lastframenum;
//DBG  if (cnt_pb<5) { cnt_pb++; paq.Print(); }
  return true;
}

/* --Methode__ */
ostream& BRPaqChecker::Print(ostream& os) const
{
  //  os << "BRPaqChecker:  HDRTag=" << hex << HDRTag() << " TRLTag=" << TRLTag() << dec << "\n"
  //     << " ... Tot.Nb.Frames.Proc=" << totnframes << " NbFrame HDR/TRL OK=" << nframeok 
  os << "BRPaqChecker: Tot.Nb.Frames.Proc=" << totnframes << " Nb.HDR/TRL OK=" << nframeok;
  if (cktrl_) os << " (Check Header AND Trailer)" << endl;
  else os << " (Header Check only)" << endl;
  float meangap = (cnt_saut>0)?((float)lostframes/(float)cnt_saut):0.;
  os   << " ... LostFrames=" << lostframes 
       << " LossRate=" << (double)lostframes*100./(double)totnframes << " %" 
       << " NbGaps=" << cnt_saut << " MeanGap=" << meangap << endl;
  return os;
}

/* --Methode__ */
string BRPaqChecker::Summary(bool detail) const
{
  double meangap = (cnt_saut>0)?((double)lostframes/(double)cnt_saut):0.;
  double lossrate= (double)lostframes*100./(double)totnframes;
  char buff[256];
  if (detail) 
    sprintf(buff, " TotNPaq= %ld  HDR/TRL OK= %ld LostFrames= %ld LossRate= %lg NbGaps= %ld MeanGap= %lg",
	    (long)totnframes, (long)nframeok, (long)lostframes, lossrate, (long)cnt_saut, meangap);
  else 
    sprintf(buff, "TotNPaq=%ld  N_Ok=%ld LossRate=%lg", (long)totnframes, (long)nframeok, lossrate); 
  return buff;
}

