#include "piscdrawwdg.h"
#include "generalfit.h"
#include <algo.h>


PIScDrawWdg::PIScDrawWdg(PIContainerGen *par, char *nom,
                         int sx, int sy, int px, int py)
:PIBaseWdg(par, nom, sx, sy, px, py)
{
  SetLimits(-1, 1, -1, 1);
  limitsFixed = 0;
  xEndDrag = yEndDrag = xBegDrag = yBegDrag = 0;
  SetAxesFlags(kAxesDflt);
  clip = true;
  padsup = false;
  mTrlb = mTxw = NULL;
  mTrW = NULL;
  SetTextWin(NULL);
  ActivateButton(1);    // Pour pouvoir activer la fenetre et coordonnees 
  ActivateMove(1);      //    "         "         "        "
  ActivateButton(2);
  ActivateMove(2);

}

PIScDrawWdg::~PIScDrawWdg()
{
  DeleteDrawers();
  if (mTrW)  delete mTrW;
}


void
PIScDrawWdg::SetLimits(double xmin, double xmax, double ymin, double ymax,
                       bool axrl, bool ayud, bool tmp)
{
  if (xmax <= xmin || ymax <= ymin)
    return; // $CHECK$ exception ?
  aXdir = axrl;      // Vrai si Axe X de Droite vers Gauche
  aYdir = ayud;      // Vrai si Axe Y de Haut vers Bas
  xMin = xmin;
  xMax = xmax;
  yMin = ymin;
  yMax = ymax;
  mPPos[0] = 0.5*(xMin+xMax);
  mPPos[1] = 0.5*(yMin+yMax);
  if (!tmp) {
    xMinS = xmin;
    xMaxS = xmax;
    yMinS = ymin;
    yMaxS = ymax;
  }
  CalcScale();
  CalcTicks();
  limitsFixed = 1;
}

void
PIScDrawWdg::Resize()
{
  CalcScale();
  CalcTicks();
  Refresh();
}

void
PIScDrawWdg::CalcScale()
{
  double margeH = (xMax-xMin)*0.15;
  double margeV = (yMax-yMin)*0.15;
  xScale = XSize() / (xMax-xMin + 2*margeH);
  if (aXdir)  { 
    xScale = -xScale;
    xOrg   = (-xMax - margeH) * xScale;// + XPos();
  }
  else xOrg   = (-xMin + margeH) * xScale;// + XPos();
  yScale = YSize() / (yMax-yMin + 2*margeV);
  if (!aYdir) {
    yScale = -yScale;
    yOrg   = (-yMax - margeV) * yScale;// + YPos();
  }
  else yOrg   = (-yMin + margeV) * yScale;// + YPos();
return;
}

void
PIScDrawWdg::CalcTicks()
{
	double xExt = xMax-xMin;
	int lx = (int) ceil(log10(xExt)-0.1);
	xExt = pow(10.0,lx);
	xMajTickStep = xExt/10;
	xMinTickStep = xMajTickStep/10;
    xFirstMajTick = int(xMin / xMajTickStep) * xMajTickStep;
    if (xFirstMajTick < xMin) xFirstMajTick += xMajTickStep;
    xFirstMinTick = int(xMin / xMinTickStep) * xMinTickStep;
    if (xFirstMinTick < xMin) xFirstMinTick += xMinTickStep;
    
    
	double yExt = yMax-yMin;
	int ly = (int) ceil(log10(yExt)-0.1);
	yExt = pow(10.0,ly);
	yMajTickStep = yExt/10;
	yMinTickStep = yMajTickStep/10;
    yFirstMajTick = int(yMin / yMajTickStep) * yMajTickStep;
    if (yFirstMajTick < yMin) yFirstMajTick += yMajTickStep;
    yFirstMinTick = int(yMin / yMinTickStep) * yMinTickStep;
    if (yFirstMinTick < yMin) yFirstMinTick += yMinTickStep;
    
//    xMajTickLen = -2/yScale;
//    xMinTickLen = -1/yScale;
//    yMajTickLen = 2/xScale;
//    yMinTickLen = 1/xScale;

    yMajTickLen = (xMax-xMin)/100;
    yMinTickLen = (xMax-xMin)/250;
    xMajTickLen = (yMax-yMin)/100;
    xMinTickLen = (yMax-yMin)/250;

}

void
PIScDrawWdg::Sc2Local(double x, double y, int& xpix, int& ypix)
{
  xpix = int(xOrg + x*xScale + .5);
  ypix = int(yOrg + y*yScale + .5);
}

void
PIScDrawWdg::Local2Sc(int xpix, int ypix, double& x, double& y)
{
  x = (xpix-xOrg)/xScale;
  y = (ypix-yOrg)/yScale;
}


void
PIScDrawWdg::DSc2Local(double dx, double dy, int& dxpix, int& dypix)
{
  dxpix = int(dx*xScale + .5);
  dypix = int(dy*yScale + .5);
}

double
PIScDrawWdg::ScStringWidth(char const* s)
{
  return CalcStringWidth(s) / xScale;
}

void
PIScDrawWdg::ScSelFontSz(double size, PIFontAtt att)
{
  int npt = int(fabs(size * yScale)+.5);
  if (npt < 8) npt = 8;
  if (npt > 127) npt = 127;
  SelFontSzPt(npt, att);
}

void
PIScDrawWdg::ScSelMarker(double size, PIMarker mrk)
{
  int isz = int(fabs(size * xScale)+.5);
  SelMarker( isz, mrk);
}

void
PIScDrawWdg::ScDrawString(double x, double y, char* s, int pos)
{
  int xPix, yPix;
  Sc2Local(x, y, xPix, yPix);
  DrawString(xPix, yPix, s, pos);
}

void
PIScDrawWdg::ScDrawLine(double x1, double y1, double x2, double y2)
{
  
  if (clip) {
    if (x2 < xMin && x1 < xMin) return;
    if (x2 > xMax && x1 > xMax) return;
    if (y1 < yMin && y2 < yMin) return;
    if (y1 > yMax && y2 > yMax) return;
    if (x2 < x1) {
      swap(x1,x2);
      swap(y1,y2);
    }
    if (x2>xMax) {
      y2 = y1 + (y2-y1)/(x2-x1)*(xMax-x1);
      x2 = xMax;
    }
    if (x1<xMin) {
      y1 = y2 + (y1-y2)/(x1-x2)*(xMin-x2);
      x1 = xMin;
    }
    if (y2 < y1) {
      swap(x1,x2);
      swap(y1,y2);
    }
    if (y2>yMax) {
      x2 = x1 + (x2-x1)/(y2-y1)*(yMax-y1);
      y2 = yMax;
    }
    if (y1<yMin) {
      x1 = x2 + (x1-x2)/(y1-y2)*(yMin-y2);
      y1 = yMin;
    }
  }
  int x1Pix, y1Pix, x2Pix, y2Pix;
  Sc2Local(x1, y1, x1Pix, y1Pix);
  Sc2Local(x2, y2, x2Pix, y2Pix);
  DrawLine(x1Pix, y1Pix, x2Pix, y2Pix);
}  


void
PIScDrawWdg::ScDrawBox(double x, double y, double dx, double dy)
{
  if (clip) {
    if (dx < 0) {
      x = x+dx;
      dx = -dx;
    }
    if (dy < 0) {
      y = y+dy;
      dy = -dy;
    }
    if (x > xMax || x+dx < xMin || y > yMax || y+dy < yMin) return;
    if (x < xMin) {
      dx -= (xMin-x);
      x = xMin;
    }
    if (y < yMin) {
      dy -= (yMin-y);
      y = yMin;
    }
    if (x+dx > xMax)
      dx = (xMax-x);
    if (y+dy > yMax)
      dy = (yMax-y);
  }
  int xPix, yPix, x2Pix, y2Pix;
  Sc2Local(x, y, xPix, yPix);
  Sc2Local(x+dx, y+dy, x2Pix, y2Pix);
  DrawBox(xPix, yPix, x2Pix-xPix, y2Pix-yPix);
}  

void
PIScDrawWdg::ScDrawFBox(double x, double y, double dx, double dy)
{
  if (clip) {
    if (dx < 0) {
      x = x+dx;
      dx = -dx;
    }
    if (dy < 0) {
      y = y+dy;
      dy = -dy;
    }
    if (x > xMax || x+dx < xMin || y > yMax || y+dy < yMin) return;
    if (x < xMin) {
      dx -= (xMin-x);
      x = xMin;
    }
    if (y < yMin) {
      dy -= (yMin-y);
      y = yMin;
    }
    if (x+dx > xMax)
      dx = (xMax-x);
    if (y+dy > yMax)
      dy = (yMax-y);
  }
  int xPix, yPix, x2Pix, y2Pix;
  Sc2Local(x, y, xPix, yPix);
  Sc2Local(x+dx, y+dy, x2Pix, y2Pix);
  DrawFBox(xPix, yPix, x2Pix-xPix, y2Pix-yPix);
}  

void
PIScDrawWdg::ScDrawCircle(double x, double y, double r)
{
  int xPix, yPix, rPix;
  Sc2Local(x, y, xPix, yPix);
  rPix = r * xScale;
  DrawCircle(xPix, yPix, rPix);
}  

void
PIScDrawWdg::ScDrawFCircle(double x, double y, double r)
{
  int xPix, yPix, rPix;
  Sc2Local(x, y, xPix, yPix);
  rPix = r * xScale;
  DrawFCircle(xPix, yPix, rPix);
}

#define NMXMULTP  30   // Pour multipoint sans new

void  
PIScDrawWdg::ScDrawPolygon(double *x, double *y, int n)
{
int ix[NMXMULTP], iy[NMXMULTP];
int *pxi, *pyi;
int i;
 
if (n <= 0)  return;
if (n > NMXMULTP) { pxi = new int[n];  pyi = new int[n]; }
else { pxi = ix;  pyi = iy; }
for(i=0; i<n; i++)  
  Sc2Local(x[i], y[i], pxi[i], pyi[i]);
DrawPolygon(pxi, pyi, n);
if (n > NMXMULTP) { delete[] pxi;  delete[] pyi; }
return;
}
  
void  
PIScDrawWdg::ScDrawFPolygon(double *x, double *y, int n)
{
int xi[NMXMULTP], yi[NMXMULTP];
int *pxi, *pyi;
int i;
 
if (n <= 0)  return;
if (n > NMXMULTP) { pxi = new int[n];  pyi = new int[n]; }
else { pxi = xi;  pyi = yi; }
for(i=0; i<n; i++)  
  Sc2Local(x[i], y[i], pxi[i], pyi[i]);
DrawFPolygon(pxi, pyi, n);
if (n > NMXMULTP) { delete[] pxi;  delete[] pyi; }
return;
}

void  
PIScDrawWdg::ScDrawMarker(double x0, double y0)
{
  int xPix, yPix;
  Sc2Local(x0, y0, xPix, yPix);
  DrawMarker(xPix, yPix);
  return;
}
void  
PIScDrawWdg::ScDrawMarkers(double *x, double *y, int n)
{
int xi[NMXMULTP], yi[NMXMULTP];
int *pxi, *pyi;
int i;
 
if (n <= 0)  return;
if (n > NMXMULTP) { pxi = new int[n];  pyi = new int[n]; }
else { pxi = xi;  pyi = yi; }
for(i=0; i<n; i++)  
  Sc2Local(x[i], y[i], pxi[i], pyi[i]);
DrawMarkers(pxi, pyi, n);
if (n > NMXMULTP) { delete[] pxi;  delete[] pyi; }
return;
}

void
PIScDrawWdg::DrawHTicks(double y, double tickUp, double tickDown, double xBeg, double xStep)
{
  for (double x=xBeg; x<=xMax; x += xStep)
    ScDrawLine(x, y-tickDown, x, y+tickUp);
}

void
PIScDrawWdg::DrawVTicks(double x, double tickLeft, double tickRight, double yBeg, double yStep)
{
  for (double y=yBeg; y<=yMax; y += yStep)
    ScDrawLine(x-tickLeft, y, x+tickRight, y);
}

void
PIScDrawWdg::DrawHLabels(double y, double xBeg, double xStep, int just)
{
  double xOffset = 0;
  int kk;
  ScSelFontSz(xMajTickLen*4);
  for (double x=xBeg; x<=xMax; x += xStep) {
      char label[20]; sprintf(label, "%-6g", x);
      for(kk=0; kk<20; kk++)
        if (label[kk] == ' ')   { label[kk] = '\0' ; break; }
      double largeur = ScStringWidth(label);
      if (just == 1)
        xOffset = -largeur;
      else if (just == 0)
        xOffset = -largeur/2;
      else
        xOffset = 0;
      if (aXdir)  xOffset = -xOffset;
      ScDrawString(x+xOffset, y, label);
  }
}

void
PIScDrawWdg::DrawVLabels(double x, double yBeg, double yStep, int just)
{
  double xOffset = 0;
  ScSelFontSz(xMajTickLen*4);
  for (double y=yBeg; y<=yMax; y += yStep) {
      char label[20]; sprintf(label, "%-6g", y);
      double largeur = ScStringWidth(label);
      if (just == 1)
        xOffset = -largeur;
      else if (just == 0)
        xOffset = -largeur/2;
      else
        xOffset = 0;
      if (aXdir)  xOffset = -xOffset;
      ScDrawString(x+xOffset, y, label);
  }
}


void
PIScDrawWdg::DrawAxes(int flags)
{
  NoClip();
  if (flags == -1)
    flags = axesFlags;
    
  
  if (flags & kStdAxes) {
  
    // Les axes
    
    ScDrawLine(xMin, 0, xMax, 0);
    ScDrawLine(0, yMin, 0, yMax);
  
    // Les ticks majeurs
  
    if (flags & kMajTicks) {
        DrawHTicks(0, xMajTickLen, xMajTickLen, xFirstMajTick, xMajTickStep);
        DrawVTicks(0, yMajTickLen, yMajTickLen, yFirstMajTick, yMajTickStep);
    }
    
    // Les ticks mineurs
  
    if (flags & kMinTicks) {
        DrawHTicks(0, xMinTickLen, xMinTickLen, xFirstMinTick, xMinTickStep);
        DrawVTicks(0, yMinTickLen, yMinTickLen, yFirstMinTick, yMinTickStep);
    }
    
    // Les labels
    
    if (flags & kLabels) {
      if (!aYdir)
        DrawHLabels(-xMajTickLen*8, xFirstMajTick, xMajTickStep,0);
      else 
        DrawHLabels(xMajTickLen*8, xFirstMajTick, xMajTickStep,0);
      if (!aXdir)
        DrawVLabels(-yMajTickLen*2, yFirstMajTick, yMajTickStep,1);
      else 
        DrawVLabels(yMajTickLen*2, yFirstMajTick, yMajTickStep,1);
    }
    
  }
  
  if (flags & kBoxAxes) {
  
    // La boite
    
    ScDrawLine(xMin, yMin, xMax, yMin);
    ScDrawLine(xMax, yMin, xMax, yMax);
    ScDrawLine(xMax, yMax, xMin, yMax);
    ScDrawLine(xMin, yMax, xMin, yMin);

    // Longueur des ticks
    
    double extXMajTickLen = flags&kExtTicks ? xMajTickLen : 0;
    double intXMajTickLen = flags&kIntTicks ? xMajTickLen : 0;
    double extXMinTickLen = flags&kExtTicks ? xMinTickLen : 0;
    double intXMinTickLen = flags&kIntTicks ? xMinTickLen : 0;
    double extYMajTickLen = flags&kExtTicks ? yMajTickLen : 0;
    double intYMajTickLen = flags&kIntTicks ? yMajTickLen : 0;
    double extYMinTickLen = flags&kExtTicks ? yMinTickLen : 0;
    double intYMinTickLen = flags&kIntTicks ? yMinTickLen : 0;

    // Les ticks majeurs
  
    if (flags & kMajTicks) {
        DrawHTicks(yMin, intXMajTickLen, extXMajTickLen, xFirstMajTick, xMajTickStep);
        DrawHTicks(yMax, extXMajTickLen, intXMajTickLen, xFirstMajTick, xMajTickStep);
        DrawVTicks(xMin, extYMajTickLen, intYMajTickLen, yFirstMajTick, yMajTickStep);
        DrawVTicks(xMax, intYMajTickLen, extYMajTickLen, yFirstMajTick, yMajTickStep);
    }
    
    // Les ticks mineurs
  
    if (flags & kMinTicks) {
        DrawHTicks(yMin, intXMinTickLen, extXMinTickLen, xFirstMinTick, xMinTickStep);
        DrawHTicks(yMax, extXMinTickLen, intXMinTickLen, xFirstMinTick, xMinTickStep);
        DrawVTicks(xMin, extYMinTickLen, intYMinTickLen, yFirstMinTick, yMinTickStep);
        DrawVTicks(xMax, intYMinTickLen, extYMinTickLen, yFirstMinTick, yMinTickStep);
    }


    // Les labels
    
    if (flags & kLabels) {
      if (!aYdir) {
        DrawHLabels(yMin-xMajTickLen*8, xFirstMajTick, xMajTickStep,0);
        DrawHLabels(yMax+xMajTickLen*2, xFirstMajTick, xMajTickStep,0);
      }
      else {
        DrawHLabels(yMin-xMajTickLen*2, xFirstMajTick, xMajTickStep,0);
        DrawHLabels(yMax+xMajTickLen*8, xFirstMajTick, xMajTickStep,0);
      }
      if (!aXdir) {
        DrawVLabels(xMin-yMajTickLen*2, yFirstMajTick, yMajTickStep,1);
        DrawVLabels(xMax+yMajTickLen*2, yFirstMajTick, yMajTickStep,-1);
      }
      else {
        DrawVLabels(xMin-yMajTickLen*2, yFirstMajTick, yMajTickStep,-1);
        DrawVLabels(xMax+yMajTickLen*2, yFirstMajTick, yMajTickStep,1);
      }
    }
  }
  Clip();
}

  

void
PIScDrawWdg::AddDrawer(PIScDrawer* d)
{
  mDrawers.push_back(d);
  d->AttachTo(this);
}

void
PIScDrawWdg::RemoveDrawer(PIScDrawer* d)
{
  if (padsup)  return;
  mDrawers.remove(d);
  d->mDrawWdg = NULL;
}

void
PIScDrawWdg::DeleteDrawers()
{
  padsup = true;
  for(list<PIScDrawer*>::iterator i = mDrawers.begin(); i != mDrawers.end(); i++)
    delete *i;
  mDrawers.erase(mDrawers.begin(),mDrawers.end());
  padsup = false;
}

void
PIScDrawWdg::CallDrawer(PIScDrawer* d)
{
  d->Draw();
}

void
PIScDrawWdg::CallDrawers()
{
  for_each(mDrawers.begin(), mDrawers.end(), CallDrawer);
}

void
PIScDrawWdg::SetAxesFlags(int flags)
{
  axesFlags = flags;
  if (axesFlags & (kIntTicks | kExtTicks | kMajTicks | kMinTicks))
    axesFlags |= kTicks;
  if ((axesFlags & (kTicks | kIntTicks | kExtTicks)) == kTicks)
    axesFlags |= kIntTicks | kExtTicks;
  if ((axesFlags & (kTicks | kMajTicks | kMinTicks)) == kTicks)
    axesFlags |= kMajTicks;
}
void
PIScDrawWdg::SetTextWin(PILabel * tw, bool trw, int tx, int ty)
{
mTxw = tw;
if (trw)
  {
  if (mTrW)  mTrW->SetSize(tx, ty);
  else
    {
    mTrW = new PIWindow(this, "Point", PIWK_transient, tx, ty, 0, 0);
    mTrW->SetAutoDelChilds(true);
    mTrlb = new PILabel(mTrW, "PixVal", tx, ty, 0, 0);
    mTrlb->SetBinding(PIBK_fixed,PIBK_fixed, PIBK_fixed,PIBK_fixed);
    }
  }
else
  if (mTrW) { delete mTrW; mTrW = NULL;  mTrlb = NULL; }

return;
}

void
PIScDrawWdg::Draw()
{
  EraseWindow();
  DrawAxes();
  CallDrawers();
  DrawSelf();
}



// Gestion affichage coordonnees du point 
void
PIScDrawWdg::But1Press(int x, int y)
{
  if (UpdateText(x, y))
     {
     if (mTrW)
       { 
       int tpx, tpy;
       GetScreenPos(tpx, tpy);       
       tpy -= (mTrW->YSize()+5);
       if (tpy < 0)  tpy = 0;
       mTrW->SetPos(tpx, tpy);
       mTrW->Show();
       }
     SelPointerShape(PI_CrossPointer);
     }
  Send(Msg(), PIMsg_Active);

}

void
PIScDrawWdg::Ptr1Move(int x, int y)
{
  UpdateText(x, y);
}

void
PIScDrawWdg::But1Release(int x, int y)
{
  if (UpdateText(x, y)) 
    SelPointerShape(PI_ArrowPointer);
  if (mTrW)  mTrW->Hide();
}

bool
PIScDrawWdg::UpdateText(int xp, int yp)
{

  double dx, dy;
  Local2Sc(xp, yp, dx, dy); 
  mPPos[0] = dx;  mPPos[1] = dy; 
  if ((mTxw == NULL) && (mTrlb == NULL)) return(false);

  char buff[128];
  sprintf(buff, "X= %g  Y= %g",  dx, dy);
  if (mTxw)
    mTxw->SetLabel((string)buff);
  if (mTrlb)
    mTrlb->SetLabel((string)buff);
  return(true);

}

// Gestion du zoom
void
PIScDrawWdg::But2Press(int x, int y)
{
  xEndDrag = xBegDrag = x;
  yEndDrag = yBegDrag = y;
  cForCol = GetForeground();
  cGOmod  = GetGOMode();
  cPointer = GetPointerShape();
  cLatt = GetLineAtt();
  SelPointerShape(PI_CrossPointer);
  SelForeground(PI_Magenta);
  SelGOMode(PI_GOXOR);
  SelLine(PI_ThinLine);
  Send(Msg(), PIMsg_Active);
}

void
PIScDrawWdg::But2Release(int /*x*/, int /*y*/)
{

  DrawBox(xBegDrag, yBegDrag, xEndDrag-xBegDrag, yEndDrag-yBegDrag);

  SelForeground(cForCol);
  SelGOMode(cGOmod);
  SelPointerShape(cPointer); 
  SelLine(cLatt); 
  
  if (xBegDrag == xEndDrag || yBegDrag == yEndDrag)
    SetLimits(xMinS, xMaxS, yMinS, yMaxS, aXdir, aYdir );
  else {
    double xl,yl,xh,yh;
    Local2Sc(xBegDrag, yBegDrag, xl, yl);
    Local2Sc(xEndDrag, yEndDrag, xh, yh);
    if (xl > xh) swap(xl, xh);
    if (yl > yh) swap(yl, yh);
    SetLimits(xl,xh,yl,yh, aXdir, aYdir, true);
  }
  
  Refresh();
}

void
PIScDrawWdg::Ptr2Move(int x, int y)
{
  DrawBox(xBegDrag, yBegDrag, xEndDrag-xBegDrag, yEndDrag-yBegDrag);
  xEndDrag = x;
  yEndDrag = y;
  DrawBox(xBegDrag, yBegDrag, xEndDrag-xBegDrag, yEndDrag-yBegDrag);
}

//================================================================
// PIScDrawer
//================================================================

PIScDrawer::PIScDrawer()
: mDrawWdg(NULL)
{}

PIScDrawer::~PIScDrawer()
{
  if (mDrawWdg)
    mDrawWdg->RemoveDrawer(this);
}

void
PIScDrawer::AttachTo(PIScDrawWdg* wdg)
{
  if (mDrawWdg)
    mDrawWdg->RemoveDrawer(this);
  mDrawWdg = wdg;
}



//================================================================
// PIScFuncDrawer
//================================================================

PIScFuncDrawer::PIScFuncDrawer(PIScFuncDrawer::FUNC f)
: mFunc(f)
{}

PIScFuncDrawer::~PIScFuncDrawer()
{
}

void
PIScFuncDrawer::Draw()
{
  if (!mDrawWdg) return;
  double xMax = mDrawWdg->XMax();
  double xMin = mDrawWdg->XMin();
  double yMax = mDrawWdg->YMax();
  double yMin = mDrawWdg->YMin();
  double xStep = (xMax - xMin)/100;
  double xOld = xMin;
  double yOld = mFunc(xMin);
  for (double x = xMin+xStep; x<xMax; x+=xStep) {
    double y = mFunc(x);
    if (y>yMin && yOld>yMin && 
        y<yMax && yOld<yMax) 
           mDrawWdg->ScDrawLine(xOld, yOld, x, y);
    xOld = x;
    yOld = y;
  }
}

//================================================================
// PIScGFFDrawer
//================================================================

PIScGFFDrawer::PIScGFFDrawer(GeneralFunction* f)
: mFunc(f), mNParms(f->NPar()), mParms(new double[mNParms])
{
  ASSERT(f->NVar() == 1);
}

PIScGFFDrawer::~PIScGFFDrawer()
{
  delete[] mParms;
}

void
PIScGFFDrawer::SetParms(double const* p)
{
  for (int i=0; i<mNParms; i++)
    mParms[i] = p[i];
}


void
PIScGFFDrawer::Draw()
{
  if (!mDrawWdg) return;
  double xMax = mDrawWdg->XMax();
  double xMin = mDrawWdg->XMin();
  double yMax = mDrawWdg->YMax();
  double yMin = mDrawWdg->YMin();
  double xStep = (xMax - xMin)/100;
  double xOld = xMin;
  double yOld = 0;
//  double yOld = f->Value(&xMin, mParms);
  for (double x = xMin+xStep; x<xMax; x+=xStep) {
   double y = 0; // $CHECK$
//    double y = f->Value(&x, mParms);
    if (y>yMin && yOld>yMin && 
        y<yMax && yOld<yMax) 
           mDrawWdg->ScDrawLine(xOld, yOld, x, y);
    xOld = x;
    yOld = y;
  }
}

