/*
 *  PSDict.cc
 *
 *  Fonctions d'IO qui ecrivent effectivement le code PostScript
 *   dans les fichiers
 */
#include <stdio.h>
#include <time.h>
#include "psfile.h"


/*****************************************************************/
/****               Gestion du PostScript                     ****/
/*****************************************************************/
#define PIDICT_PAR   \
  \
  LANDSCAPE, PORTRAIT,\
  \
  PI_Black, PI_White, PI_Grey, PI_Red, PI_Blue,\
  PI_Green, PI_Yellow, PI_Magenta, PI_NotDefColor,\
  \
  PI_RomanFont, PI_BoldFont, PI_ItalicFont, PI_NotDefFontAtt,\
  PI_SmallSizeFont, PI_NormalSizeFont, PI_BigSizeFont, PI_NotDefFontSize, \
  \
  PI_NormalLine, PI_ThinLine, PI_ThickLine, PI_NotDefLineAtt

#define PIDICT       "\
/$PIDict 50 dict def\n\
$PIDict begin $PIDict /mtrx matrix put /Landscape %d def /Portrait %d def\n\
%% Couleurs et fontes courantes\n\
/C%-2d  {0.0   0.0   0.0   setrgbcolor} bind def %% PI_Black\n\
/C%-2d  {0.996 0.996 0.996 setrgbcolor} bind def %% PI_White\n\
/C%-2d  {0.824 0.824 0.824 setrgbcolor} bind def %% PI_Grey\n\
/C%-2d  {0.996 0.0   0.0   setrgbcolor} bind def %% PI_Red\n\
/C%-2d  {0.0   0.0   0.996 setrgbcolor} bind def %% PI_Blue\n\
/C%-2d  {0.0   0.996 0.0   setrgbcolor} bind def %% PI_Green\n\
/C%-2d  {0.996 0.996 0.0   setrgbcolor} bind def %% PI_Yellow\n\
/C%-2d  {0.996 0.0   0.996 setrgbcolor} bind def %% PI_Magenta\n\
/C%-2d  {}                              bind def %% PI_NotDefColor\n\
/F%-2d  {/Times-Roman   findfont} bind def       %% PI_RomanFont\n\
/F%-2d  {/Times-Bold    findfont} bind def       %% PI_BoldFont\n\
/F%-2d  {/Times-Italic  findfont} bind def       %% PI_ItalicFont\n\
/F%-2d  {}\
                        bind def       %% PI_NotDefFontAtt\n\
/FS {scalefont setfont} bind def\n\
%%/FS%-2d {8  scalefont setfont} bind def          %% PI_SmallSizeFont\n\
%%/FS%-2d {12 scalefont setfont} bind def          %% PI_NormalSizeFont\n\
%%/FS%-2d {16 scalefont setfont} bind def          %% PI_BigSizeFont\n\
/FS%-2d {}                     bind def          %% PI_NotDefFontSize\n\
/L%-2d  {1    slw} bind def                      %% PI_NormalLine\n\
/L%-2d  {0.5  slw} bind def                      %% PI_ThinLine\n\
/L%-2d  {3    slw} bind def                      %% PI_ThickLine\n\
/L%-2d  {}         bind def                      %% PI_NotDefLineAtt\n\
\
%% Conversion\n\
/cm   {28.3 mul}   bind def  %% cm --> points\n\
/Ux   {$Cvrtx mul} bind def  %% Unites utilisateur /x --> points\n\
/Uy   {$Cvrty mul} bind def  %% Unites utilisateur /y --> points\n\
\
%% Pages et Blocs\n\
/newref  {%% Dx Dy Tx Ty X0 Y0 newref\n\
translate 3 -1 roll cm div 28.3 mul /$Cvrty exch def\n\
exch cm div 28.3 mul /$Cvrtx exch def} bind def\n\
/NewPage {%% Dx Dy orientation NewPage\n\
/$PageEnteredState save def 1 -1 scale Portrait eq {w cm h cm 0 h cm neg\n\
newref} { -90 rotate h cm w cm 0 0 newref} ifelse} bind def\n\
/EndPage { %% EndPage\n\
$PageEnteredState restore} bind def\n\
/NewBloc { %% Dx Dy Tx Ty X0 Y0 NewBloc\n\
/$BlocEnteredState save def newref} bind def\n\
/EndBloc { %% EndBloc\n\
$BlocEnteredState restore} bind def\n\
\
%% Trace\n\
/n  {newpath} bind def    /c  {closepath} bind def\n\
/gs {gsave}  bind def     /gr {grestore} bind def\n\
/m  {moveto} bind def     /l  {lineto} bind def\n\
/s  {stroke} bind def     /f  {fill}   bind def\n\
/sc {scale}  bind def     /S  {gs 1 -1 sc show gr} bind def \n\
/slw {setlinewidth} bind def\n\
/ff {findfont} bind def   /sf {setfont} bind def /scf {scalefont} bind def\n\
/rl {rlineto} bind def    /tr {translate} bind def\n\
%% Tx Ty X0 Y0 box\n\
/box {tr 0 3 1 roll neg exch 0 0 3 index neg 0 0 m rl rl rl c} bind def\n\
%% R X0 Y0 circle\n\
/circle {tr dup 0 m 0 0 3 -1 roll 0 380 arc} bind def\n\
%% DrawEllipse pompee sur transfig\n\
/ellipse {/endangle exch def /startangle exch def /yrad exch def\n\
/xrad exch def /y exch def /x exch def /savematrix mtrx currentmatrix def\n\
x y tr xrad yrad sc 0 0 1 startangle endangle arc\n\
closepath savematrix setmatrix } def\n\
\
%% Gestion des images\n\
%% ColorImage, pompee sur xv (qui les avait pompees sur xgrab)\n\
/colorimage where   %% do we know about 'colorimage'?\n\
  { pop }           %% yes: pop off the 'dict' returned\n\
  {                 %% no:  define one\n\
    /colortogray {  %% define an RGB->I function\n\
      /rgbdata exch store    %% call input 'rgbdata'\n\
      rgbdata length 3 idiv\n\
      /npixls exch store\n\
      /rgbindx 0 store\n\
      0 1 npixls 1 sub {\n\
        grays exch\n\
        rgbdata rgbindx       get 20 mul    %% Red\n\
        rgbdata rgbindx 1 add get 32 mul    %% Green\n\
        rgbdata rgbindx 2 add get 12 mul    %% Blue\n\
        add add 64 idiv      %% I = .5G + .31R + .18B\n\
        put\n\
        /rgbindx rgbindx 3 add store\n\
      } for\n\
      grays 0 npixls getinterval\n\
    } bind def\n\
    /mergeprocs { %% def\n\
      dup length 3 -1 roll dup length dup 5 1 roll 3 -1 roll add\n\
      array cvx dup 3 -1 roll 0 exch putinterval dup 4 2 roll\n\
      putinterval } bind def\n\
    /colorimage {\n\
      pop pop     %% remove 'false 3' operands\n\
      {colortogray} mergeprocs image} bind def\n\
  } ifelse          %% end of 'false' case\n\
\
%% GetRGB\n\
/GetRGB { %% ImageStr --> RGBStr\n\
  /ImageData exch store\n\
  %% Pour chaque element de ImageData,\n\
  %% On va chercher les valeurs RGB correspondantes\n\
  0 1 Taille 1 sub {\n\
    /i      exch store  %% On met a jour l'indice de parcours\n\
    ImageData i get     %% Recupere la Indice e val dans la chaine ImData\n\
    ColorMap exch get   %% On recupere la valeur RGB Correspondante\n\
    aload pop           %% On met tout ca dans la pile\n\
    %% Maintenant, on va mettre tout ca dans la chaine RGBStr\n\
    RGBStr i 3 mul 2 add 3 -1 roll put\n\
    RGBStr i 3 mul 1 add 3 -1 roll put\n\
    RGBStr i 3 mul       3 -1 roll put\n\
    } for\n\
  RGBStr 0 Taille 3 mul getinterval %% On renvoie la chaine RGB\n\
} bind def\n\
end\n\
"


/*****************************************************************/
/****                       PIMarkers                         ****/
/*****************************************************************/
#define PIMARKERS_PAR  PI_DotMarker, PI_PlusMarker, PI_CrossMarker,\
  PI_CircleMarker, PI_FCircleMarker, PI_BoxMarker, PI_FBoxMarker,\
  PI_TriangleMarker, PI_FTriangleMarker, PI_StarMarker, PI_FStarMarker

#define PIMARKERS  "\
%% Definition de la fonte $PIMarkers\n\
/$PIMarkers 26 dict def\n\
$PIMarkers begin\n\
/FontType 3 def\n\
/FontMatrix [0.001 0 0 0.001 -0.375 -0.375] def\n\
/FontBBox   [0 0 1000 1000] def\n\
/Encoding 256 array def\n\
0 1 255 {Encoding exch /.notdef put} for\n\
Encoding  %d /PIDot       put %% PI_DotMarker\n\
Encoding  %d /PIPlus      put %% PI_PlusMarker\n\
Encoding  %d /PICross     put %% PI_CrossMarker\n\
Encoding  %d /PICircle    put %% PI_CircleMarker\n\
Encoding  %d /PIFCircle   put %% PI_FCircleMarker\n\
Encoding  %d /PIBox       put %% PI_BoxMarker\n\
Encoding  %d /PIFBox      put %% PI_FBoxMarker\n\
Encoding  %d /PITriangle  put %% PI_TriangleMarker\n\
Encoding  %d /PIFTriangle put %% PI_FTriangleMarker\n\
Encoding  %d /PIStar      put %% PI_StarMarker\n\
Encoding  %d /PIFStar     put %% PI_FStarMarker\n\
/CharProcs 26 dict def\n\
$PIDict begin\n\
CharProcs begin\n\
/.notdef     {} def\n\
/PIDot       {100 375 375 circle f} bind def\n\
/PIPlus      {0 375 m 750 375 l 375 0 m 375 750 l s} bind def\n\
/PICross     {0 0 m 750 750 l 750 0 m 0 750 l s} bind def\n\
/PICircle    {375 375 375 circle s} bind def\n\
/PIFCircle   {375 375 375 circle f} bind def\n\
/PIBox       {0 0 m 750 0 l 750 750 l 0 750 l c s} bind def\n\
/PIFBox      {0 0 m 750 0 l 750 750 l 0 750 l c f} bind def\n\
/PITriangle  {0 750 m 375 0 l 750 750 l c s} bind def\n\
/PIFTriangle {0 750 m 375 0 l 750 750 l c f} bind def\n\
/PIStar      {} def\n\
/PIFStar     {} def\n\
end %%$PIDict\n\
end %%$CharProcs\n\
/BuildChar {\n\
1000 0      %% Largeur \n\
0 0 750 750 %% Bounding Box\n\
setcachedevice \n\
exch begin\n\
Encoding exch get\n\
CharProcs exch get\n\
end exec} def\n\
end\n\
/PIMarkers $PIMarkers definefont pop\n\
save\n\
"



#define END_PROLOG_PAR  PAGE_WIDTH, PAGE_HEIGHT
#define END_PROLOG  "/NewFile { %%%% w h NewFile (--> h et w en cm <--)\n\
$PIDict begin /PIMarkers ff 14 scf sf /h exch def /w exch def \n\
/$PIEnteredState save def} bind def\n\
/EndFile {$PIEnteredState restore end} bind def\n\
%%%%EndProlog\n\n\
"


#define HEADER_PS  "%%!PS-Adobe-2.0\n"
#define HEADER_EPS "%%!PS-Adobe-2.0 EPSF-2.0\n%%%%BoundingBox: %d %d %d %d\n"
#define HEADER     "\
%%%%Title: %s\n\
%%%%Creator: PSFile\n\
%%%%CreationDate: %s\
%%%%Pages: (atend)\n\
%%%%EndComments\n\n\
"
void  PSFile::WriteHeader(){

    int X;
    int Y;
    int H;
    int W;
    time_t timer;
    char *date = (char*)malloc(64*sizeof(char));

    if(isEPS) {
        X = (int)(mFormat.X0*28.35);
        Y = (int)(mFormat.Y0*28.35);
	W = (int)((mFormat.X0 + mFormat.Tx)*28.35);
	H = (int)((mFormat.Y0 + mFormat.Ty)*28.35);
	fprintf(mPSFile,HEADER_EPS, X, Y, W, H);
    }
    else
	fprintf(mPSFile,HEADER_PS);

    time(&timer);
    date = ctime(&timer);
    fprintf(mPSFile, HEADER, mPSFileName, date);
    fprintf(mPSFile, PIDICT, PIDICT_PAR);
    fprintf(mPSFile, PIMARKERS, PIMARKERS_PAR);
    fprintf(mPSFile, END_PROLOG);
    if(!isEPS)
      fprintf(mPSFile, "%.2f %.2f NewFile\n", END_PROLOG_PAR);
    else
      fprintf(mPSFile, "%.2f %.2f NewFile\n", 
	      mFormat.Tx, mFormat.Ty);
}


void PSFile::WriteTrailer() {

    fprintf(mPSFile, "%%%%Trailer\n%%%%Pages: %d\n%%%%EOF\n", PageCounter);
}


void PSFile::WriteNewPage(Page *mPage) {

    fprintf(mPSFile, "%%%%Page: %d %d\n%.2f %.2f %d NewPage\n",
	    mPage->num, mPage->num, 
	    mPage->format.Dx, mPage->format.Dy,
	    mPage->format.orientation);
}

void PSFile::WriteEndPage(Page *mPage) {

    fprintf(mPSFile, "EndPage showpage\n");
    //    if(!isEPS)
    //	fprintf(mPSFile, "showpage");
    fprintf(mPSFile, "%%>EndPage %d\n\n\n", mPage->num);
}

void PSFile::WriteNewBloc(Bloc *mBloc) {

    fprintf(mPSFile, "%%>BeginBloc %d\n", mBloc->num);
    fprintf(mPSFile, "%.2f %.2f %.2f Ux %.2f Uy %.2f Ux %.2f Uy NewBloc\n",
	    mBloc->format.Dx, mBloc->format.Dy,
	    mBloc->format.Tx, mBloc->format.Ty,
	    mBloc->format.X0, mBloc->format.Y0);
}

void PSFile::WriteEndBloc(Bloc *mBloc) {

    fprintf(mPSFile, "EndBloc\n");
    fprintf(mPSFile, "%%>EndBloc %d\n", mBloc->num);
}



/*****************************************************************/
/****                Fonctions de trace                       ****/
/*****************************************************************/
void PSFile::WriteDrawString(float x, float y, char * s) {
    
    fprintf(mPSFile, "gs %.2f Ux %.2f Uy m (%s) S gr\n", x, y, s);
}

void PSFile::WriteDrawLine(float x1, float y1, float x2, float y2) {

    fprintf(mPSFile, "n %.2f Ux %.2f Uy m %.2f Ux %.2f Uy l s\n", 
	    x1, y1, x2, y2);
}

void PSFile::WriteDrawBox(float X0, float Y0, float Tx, float Ty) {

    fprintf(mPSFile, "gs n %.2f Ux %.2f Uy %.2f Ux %.2f Uy box s gr\n",
	    Tx, Ty, X0, Y0);
}

void PSFile::WriteDrawFBox(float X0, float Y0, float Tx, float Ty) {

    fprintf(mPSFile, "gs n %.2f Ux %.2f Uy %.2f Ux %.2f Uy box f gr\n",
	    Tx, Ty, X0, Y0);
}

void PSFile::WriteDrawCircle(float X0, float Y0,float R) {

    fprintf(mPSFile, "gs n %.2f Ux %.2f Uy %.2f Ux %.2f Uy 0 360 ellipse s gr\n",
	    X0, Y0, R, R);
}

void PSFile::WriteDrawFCircle(float X0, float Y0,float R) {

    WriteSetColor(mFillColor);
    fprintf(mPSFile, "gs n %.2f Ux %.2f Uy %.2f Ux %.2f Uy \
                      0 360 ellipse f gr\n",
	    X0, Y0, R, R);
}

void PSFile::WriteDrawPolygon(float * x, float * y, int n) {

    int i;

    fprintf(mPSFile, "%.2f Ux %.2f Uy m\n", x[0], y[0]);
    for(i=0; i<n; i++) 
	fprintf(mPSFile, "%.2f Ux %.2f Uy %.2f Ux %.2f Uy l\n",
		x[i+1], y[i+1], x[i], y[i]);
    fprintf(mPSFile, "c s\n");
}

void PSFile::WriteDrawFPolygon(float * x, float * y, int n) {

    int i;

    fprintf(mPSFile, "%.2f Ux %.2f Uy m\n", x[0], y[0]);
    for(i=0; i<n; i++) 
	fprintf(mPSFile, "%.2f Ux %.2f Uy %.2f Ux %.2f Uy l\n",
		x[i+1], y[i+1], x[i], y[i]);
    fprintf(mPSFile, "c f\n");
}


void PSFile::WriteDrawMarker(float x, float y) {

    fprintf(mPSFile, "%.2f Ux %.2f Uy m (\\%#o) show\n", 
	    x, y, mMarker);
}


void PSFile::WriteDrawMarkers(float * x, float * y, int n) {

    for(int i = 0; i<n; i++)
	fprintf(mPSFile, "%.2f Ux %.2f Uy m (\\%#o) show\n", 
		x[i], y[i], mMarker);
}


void PSFile::WriteImage(float x0, float y0, float Tx, float Ty,
			int Nx, int Ny, unsigned char *pixmap, 
			PIColorMap *mColorMap) {

    int i, j;

    /* Buffers PostScript */
    fprintf(mPSFile, "/Taille %d def\n", Nx);
    fprintf(mPSFile, "/ImStr Taille string def\n");
    fprintf(mPSFile, "/RGBStr Taille 3  mul string def\n");

    /* Ecriture ColorMap */
    fprintf(mPSFile, "/ColorMap %02d array def\n", mColorMap->NCol() );
    for(i = 0; i < mColorMap->NCol(); i++) {
	fprintf(mPSFile, "ColorMap %d [16#%02X 16#%02X 16#%02X] put\n",
		i,
		(int)(mColorMap->GetColor(i).red/256),
		(int)(mColorMap->GetColor(i).green/256),
		(int)(mColorMap->GetColor(i).blue/256));
    }
    /* Emplacement de l'image (et gsave...) */
    fprintf(mPSFile, "gs %.2f  %.2f  tr %.2f Ux %.2f Uy sc\n", x0, y0, Tx, Ty);
    /* Nbr pix-X, Nbr pix-Y bits/pixels */
    fprintf(mPSFile, "%d %d 8\n", Nx, Ny);
    /* Matrice de passage */
    fprintf(mPSFile, "[%d 0 0 %d 0 0]", Nx, Ny);
    /* Fonction colorimage */
    fprintf(mPSFile, "{currentfile ImStr readhexstring pop GetRGB}\
false 3 colorimage\n\n");
    /* Data */
    for(i = 0; i < Ny; i++) {
	for(j=0; j<Nx-1; j++ )
	    fprintf(mPSFile, "%02X", pixmap[i*Nx+j]);
    fprintf(mPSFile, "%02X\n", pixmap[i*Nx+j]);
    }
    /* Un petit grestore ...*/
    fprintf(mPSFile, "gr\n");
}

void PSFile::WriteSetColor(PIColors mDrawColor) {

    fprintf(mPSFile, "C%d ", mDrawColor);
}

void PSFile::WriteSetFont(PIFontAtt mFontAtt, int mFontSize) {

  //  int fntsz;
  //  PageList::iterator it;

  //  it = mPageList.end(); it--;
  //  fntsz = (int)(28.3 * mFormat.Tx * ( mFontSize / (*it).format.Tx) );
  fprintf(mPSFile, "F%d %d Ux FS ", mFontAtt, mFontSize);
  //  printf("%d %f %f\n",fntsz,(float)mFormat.Tx,(float)(*it).format.Tx);
}

void PSFile::WriteSetLineWidth(PILineAtt mLineAtt) {

    fprintf(mPSFile, "%d slw ", mLineAtt);
}
