#include "machdefs.h"
#include "sopnamsp.h"
#include <iostream>
#include <string>
#include <math.h>

#include <typeinfo>

#include "mdish.h"
#include "specpk.h"
#include "histinit.h"
// #include "fiosinit.h"
// #include "fitsioserver.h"

#include "randr48.h"      

#include "timing.h"
#include "ctimer.h"

typedef DR48RandGen RandomGenerator ;

// ---------------------------------------------------------------------
// Test main program for computing interferometer noise power spectrum
// R. Ansari  - Avril 2010 
// ---------------------------------------------------------------------

class PkNoiseCalculator
{
public:
  PkNoiseCalculator(Four3DPk& pk3, Four2DResponse& rep, 
		    double s2cut=100., int ngen=1, const char* tit="PkNoise")
    : pkn3d(pk3), frep(rep), S2CUT(s2cut), NGEN(ngen), title(tit) 
  { } 
  inline void SetS2Cut(double s2cut=100.)
  {  S2CUT=s2cut; }
  HProf Compute()
  {
    Timer tm(title.c_str());
    tm.Nop();
    HProf hnd;
    cout << "PkNoiseCalculator::Compute() " << title << "  NGEN=" << NGEN << " S2CUT=" << S2CUT << endl;
    for(int igen=0; igen<NGEN; igen++) {
      pkn3d.ComputeNoiseFourierAmp(frep);
      if (igen==0) hnd = pkn3d.ComputePk(S2CUT);
      else pkn3d.ComputePkCumul(hnd,S2CUT);
    }
    return hnd;
  }

  Four3DPk& pkn3d;
  Four2DResponse& frep;
  double S2CUT;
  int NGEN;
  string title;
};

int main(int narg, const char* arg[])
{
  if ((narg>1)&&(strcmp(arg[1],"-h")==0))  {
    cout<< " Usage: pknoise [OutPPFName NGen S2Cut Lambda] " << endl;
    cout<< "   Default: OutPPFName=pknoise.ppf, NGen=1 " << endl;
    cout<< "            S2CUT=0. , Lambda=0.357  \n" << endl;

    return 1;
  }
  cout << " ==== pknoise.cc program , test of SpectralShape and MassDist2D classes ==== " << endl;
  // make sure SOPHYA modules are initialized 
  SophyaInit();  
  //  FitsIOServerInit();
  InitTim();
  //--- decoding command line arguments 
  string outfile = "pknoise.ppf";  
  if (narg>1) outfile = arg[1];
  if (outfile==".")  outfile = "pknoise.ppf";
  int NMAX = 1;
  if (narg>2) NMAX = atoi(arg[2]);
  double SCut=0.;
  if (narg>3) SCut = atof(arg[3]);
  double LAMBDA=0.357 ;  // 21 cm at z=0.7 
  if (narg>4) LAMBDA = atof(arg[4]);

  //-- end command line arguments
  
  int rc = 1;  
  try {  // exception handling try bloc at top level
    cout << "0/ pknoise.cc: Executing, output file= " << outfile << endl;  
    POutPersist po(outfile);
    cout << " 1.a/ Instanciating object type SpectralShape " << endl;
    SpectralShape spec(2);
    cout << " 1.b/ Wrinting spectral shape vector (name= Pk) to output PPF " << endl; 
    Histo hpk = spec.GetPk(1024);
    po << PPFNameTag("Pk") << hpk;

    double D = 100.;
    double lambda = LAMBDA;
    double Dol = D/lambda;
    cout << " 2.a/ Instanciating Four2DResponse(1/2/3...) " << endl;
    Four2DResponse dishg(1,Dol,Dol);
    Four2DResponse dish(2,Dol,Dol);
    Four2DResponse dish2(2,Dol*2.,Dol*2.);
    Four2DResponse dishsq(3,Dol,Dol/5.);
    cout << " 2.b/ Writing Four2DResponse Histo2D to output ppf " << endl;
    Histo2D hdg = dishg.GetResponse();
    Histo2D hd = dish.GetResponse();
    Histo2D hd2 = dish2.GetResponse();
    Histo2D hdsq = dishsq.GetResponse();
    po << PPFNameTag("dishg") << hdg;
    po << PPFNameTag("dish") << hd;
    po << PPFNameTag("dish2") << hd2;
    po << PPFNameTag("dishsq") << hdsq;

    cout << " 2.c/ Creating MultiDish Filled Array " << endl;
    double Ddish = 5.;
    double Ddish2 = 7.5;
    double Eta=0.95;
    int cnt=0;
    vector<Dish> vdplein;
    for(int i=0; i<20; i++) 
      for(int j=0; j<20; j++) {
	cnt++; 
	vdplein.push_back(Dish(i*20+j+1, i*Ddish, j*Ddish, Eta*Ddish));
      }
    vector<Dish> vdsparse;
    vector<Dish> vdsparseD7;

    cnt=0;
    for(int i=0; i<=18; i++) {
      cnt++; vdsparse.push_back(Dish(cnt, i*Ddish,0.,Eta*Ddish));
      vdsparseD7.push_back(Dish(cnt, i*Ddish2,0.,Eta*Ddish2));
    }
    for(int i=-18; i<=18; i++) {
      if (i==0)  continue;
      cnt++; vdsparse.push_back(Dish(cnt, 0.,i*Ddish,Eta*Ddish));
      vdsparseD7.push_back(Dish(cnt, 0.,i*Ddish2,Eta*Ddish2));
    }
    for(int i=0; i<4; i++) {
	cnt++;  vdsparse.push_back(Dish(cnt, (3+2*i)*Ddish,(3+2*i)*Ddish,Eta*Ddish));
	vdsparseD7.push_back(Dish(cnt, (3+2*i)*Ddish2,(3+2*i)*Ddish2,Eta*Ddish2));
	cnt++;	vdsparse.push_back(Dish(cnt, (3+2*i)*Ddish,-(3+2*i)*Ddish,Eta*Ddish));
	vdsparseD7.push_back(Dish(cnt, (3+2*i)*Ddish2,-(3+2*i)*Ddish2,Eta*Ddish2));
	/*
	if ((i>0)||(j>0)) {
	  cnt++;  vdsparse.push_back(Dish(cnt, (5+3*i)*Ddish,(3+2*j)*Ddish,Eta*Ddish));
	  cnt++;  vdsparse.push_back(Dish(cnt, (5+3*i)*Ddish,-(3+2*j)*Ddish,Eta*Ddish));
	}
	*/
      }


    vector<Dish> vcylplein, vcylplP;
    cnt=0;
    double cylW=12.;   // Largeur des cylindres 
    double cylRL=0.5;  // Longeur des elements de reception le long du cylindre
    for(int i=0; i<8; i++) 
      for(int j=0; j<192; j++) {
	vcylplein.push_back(Dish(i+1, i*cylW, j*cylRL, 0.9*cylW, 0.9*cylRL));
	cnt++; vcylplP.push_back(Dish(cnt, i*cylW, j*cylRL, 1.*cylW, 1.*cylRL));
      }
    vector<Dish> v2cyl, v2cylP;
    cnt=0;
    for(int i=0; i<2; i++) 
      for(int j=0; j<32; j++) {
	v2cyl.push_back(Dish(i+1, i*25, j*cylRL, 0.9*9., 0.9*cylRL));
	cnt++; v2cylP.push_back(Dish(cnt, i*25, j*cylRL, 0.9*9., 1.*cylRL));
      }
    double LMAX = D;
    bool fgnoauto = true;
    int NRX=160;
    int NRY=160;

    MultiDish mdfill(lambda, LMAX, vdplein, fgnoauto);
    mdfill.SetRespHisNBins(NRX,NRY);   
    Histo2D hrfill = mdfill.GetResponse();
    PrtTim("Apres mdfill.GetResponse()");

    MultiDish mdsparse(lambda, LMAX, vdsparse, fgnoauto); 
    mdsparse.SetThetaPhiRange(M_PI/4.,16, M_PI/4., 16);
    mdsparse.SetRespHisNBins(NRX,NRY);
    Histo2D hrsp = mdsparse.GetResponse();
    PrtTim("Apres mdsparse.GetResponse()");

    MultiDish mdsparsefp(lambda, LMAX, vdsparse, fgnoauto); 
    mdsparsefp.SetRespHisNBins(NRX,NRY);
    Histo2D hrspfp = mdsparsefp.GetResponse();
    PrtTim("Apres mdsparsefp.GetResponse()");

    MultiDish mdsparseD7(lambda, LMAX, vdsparseD7, fgnoauto); 
    mdsparseD7.SetThetaPhiRange(M_PI/4.,16, M_PI/4., 16);
    mdsparseD7.SetRespHisNBins(NRX,NRY);
    Histo2D hrspd7 = mdsparseD7.GetResponse();
    PrtTim("Apres mdsparseD7.GetResponse()");
			     
    MultiDish mcylfill(lambda, LMAX, vcylplein, fgnoauto);
    mcylfill.SetRespHisNBins(NRX,NRY);
    Histo2D hfcyl = mcylfill.GetResponse();
    PrtTim("Apres mcylfill.GetResponse()");
    MultiDish mcylfillP(lambda, LMAX, vcylplP, fgnoauto);
    mcylfillP.SetRespHisNBins(NRX,NRY);
    Histo2D hfcylP = mcylfillP.GetResponse();

    MultiDish md2cyl(lambda, LMAX, v2cyl, fgnoauto);
    md2cyl.SetRespHisNBins(NRX,NRY);
    Histo2D h2cyl = md2cyl.GetResponse();
    MultiDish md2cylP(lambda, LMAX, v2cylP, fgnoauto);
    md2cylP.SetRespHisNBins(NRX,NRY);
    Histo2D h2cylP = md2cylP.GetResponse();

    po << PPFNameTag("mfill") << hrfill;
    po << PPFNameTag("mspars") << hrsp;
    po << PPFNameTag("msparsfp") << hrspfp;
    po << PPFNameTag("msparsd7") << hrspd7;
    po << PPFNameTag("mcylf") << hfcyl;
    po << PPFNameTag("m2cyl") << h2cyl;
    po << PPFNameTag("mcylfP") << hfcylP;
    po << PPFNameTag("m2cylP") << h2cylP;

    PrtTim("Done computing multi-dish response");
 

    Four2DRespTable mdf(hrfill, Dol);
    Four2DRespTable mds(hrsp,   Dol);
    Four2DRespTable mdsfp(hrspfp,   Dol);
    Four2DRespTable mdsd7(hrspd7,   Dol);

    Four2DRespTable mcylf(hfcyl, Dol);
    Four2DRespTable m2cyl(h2cyl, Dol);
    Four2DRespTable mcylfP(hfcylP, Dol);
    Four2DRespTable m2cylP(h2cylP, Dol);

    
    cout << " 3.a/ Instanciating object type Four3DPk  " << endl;
    RandomGenerator rg;
    Four3DPk m3d(rg);
    m3d.SetCellSize(2.*DeuxPI, 2.*DeuxPI, 2.*DeuxPI); 
    cout << " 3.b/ call ComputeFourierAmp()  NGEN=" << NMAX << endl;
    HProf hrpk;
    for(int igen=0; igen<NMAX; igen++) {
      m3d.ComputeFourierAmp(spec);
      if (igen==0)  hrpk = m3d.ComputePk();
      else m3d.ComputePkCumul(hrpk);
    }
    PrtTim("md.ComputeFourierAmp() done");
    po << PPFNameTag("recPk") << hrpk;

    cout << " 4/ Computing Noise P(k) using PkNoiseCalculator ..." << endl;
#define NCONFIG 10
    Four2DResponse* f2rep[NCONFIG]={&dish, &dish2, &mdf, &mds, &mdsfp, &mdsd7, &mcylf, &mcylfP, &m2cyl, &m2cylP};
    const char* tits[NCONFIG]={"Dish100m", "Dish200m","F20x20Dish5m","S63Dish5m","S63Dish5mFP","S63Dish7m",
			       "F8Cyl","F8CylP","BiCyl","BiCylP"};
    const char* tags[NCONFIG]={"noiseD", "noiseD2","noisemdf","noisemds","noisemdsfp","noisemdsd7",
			       "noisefcyl","noisefcylP","noise2cyl","noise2cylP"};
    vector<int> nbdishes;
    nbdishes.push_back(1);
    nbdishes.push_back(1);
    nbdishes.push_back(vdplein.size());
    nbdishes.push_back(vdsparse.size());
    nbdishes.push_back(vdsparse.size());
    nbdishes.push_back(vdsparseD7.size());
    nbdishes.push_back(vcylplein.size());
    nbdishes.push_back(vcylplP.size());
    nbdishes.push_back(v2cyl.size());
    nbdishes.push_back(v2cylP.size());

    for(int lc=0; lc<NCONFIG; lc++) {
      PkNoiseCalculator pkn(m3d, *(f2rep[lc]), SCut/(double)nbdishes[lc], NMAX, tits[lc]);
      HProf hpn = pkn.Compute();
      po << PPFNameTag(tags[lc]) << hpn;
    } 
    rc = 0;
  }  // End of try bloc 
  catch (PThrowable & exc) {  // catching SOPHYA exceptions
    cerr << " pknoise.cc: Catched Exception (PThrowable)" << (string)typeid(exc).name() 
         << "\n...exc.Msg= " << exc.Msg() << endl;
    rc = 99;
  }
  catch (std::exception & e) {  // catching standard C++ exceptions
    cerr << " pknoise.cc: Catched std::exception "  << " - what()= " << e.what() << endl;
    rc = 98;
  }
  catch (...) {  // catching other exceptions
    cerr << " pknoise.cc: some other exception (...) was caught ! " << endl;
    rc = 97;
  }
  PrtTim("End-pknoise");
  cout << " ==== End of pknoise.cc program  Rc= " << rc << endl;
  return rc;	
}


