/*  ------------------------ Projet BAORadio -------------------- 
  Programme de calcul du spectre de puissance (3D) a partir d'un 
  cube de  delta T/T LSS, d'un cube  delta T/T LSS synchrotron
  ou radio-sources, apres ajustement / soustraction d'une loi de 
  puissance en frequence (l'axe Z du tableau doit etre en frequence)

    R. Ansari , C. Magneville - Juin 2010 

Usage: calcpk2 InMapLSS convFacLSS InMapSync convFacSync InMapRadioSource convFacRsc OutPk 
               [PixNoiseLevel] [TargetBeamArcmin] [NSigSrcThr]
---------------------------------------------------------------  */

#include "machdefs.h"
#include "sopnamsp.h"
#include <iostream>
#include <string>
#include <math.h>

#include <typeinfo>

#include "specpk.h"
#include "histats.h"
#include "vector3d.h"

#include "qhist.h" 
#include "lobe.h" 
#include "cubedef.h" 
#include "fgndsub.h" 
#include "radutil.h" 

#include "histinit.h"
#include "fftwserver.h"
#include "randr48.h"      

#include "ctimer.h"

typedef ThSDR48RandGen RandomGenerator ;

//-------------------------------------------------------------------------
//      ------------------ MAIN PROGRAM ------------------------------
//-------------------------------------------------------------------------
/* --Fonction-- */
int main(int narg, const char* arg[])
{
  if (narg<6) {
    cout << " Usage: calcpk2 InMapLSS convFacLSS InMapSync convFacSync  OutPk \n" 
         << "        [PixNoiseLevel] [TargetBeamArcmin] [NSigSrcThr]" << endl;
    return 1;
  }
  Timer tm("calcpk2");
  int rc = 0;
  try {
    string inppflss = arg[1];
    r_4 rfaclss = atof(arg[2]);
    string inppfsync = arg[3];
    r_4 rfacsync = atof(arg[4]);
    string outname = arg[5];

    double pixsignoise = 0.;
    bool fgaddnoise=false;
    if (narg>6) {
      pixsignoise=atof(arg[6]);
      if (pixsignoise>1.e-6)  fgaddnoise=true;
    }
    
    bool fgcorrbeam=true;
    double tbeamarcmin=15.;
    if (narg>7) {
      tbeamarcmin=atof(arg[7]);
      if (tbeamarcmin<1.e-6)  fgcorrbeam=false;
    }
    bool fgclnsrc=true;
    double nsigsrc=5.;
    if (narg>8) {
      nsigsrc=atof(arg[8]);
      if (nsigsrc<1.e-6)  fgclnsrc=false;
    }

    TArray<r_4> maplss, mapsync;
    const char * tits[2]={"LSS", "Sync/RadioSrc"};
    for(int ks=0; ks<2; ks++)  {
      string& ppfname=inppflss;
      r_4 rfac=rfaclss;
      TArray<r_4>* inmap=&maplss; 
      if (ks==1) {  ppfname=inppfsync;  rfac=rfacsync;  inmap=&mapsync; }
      cout << "calcpk2[" << ks+1 << "] : reading 3D map " << tits[ks] << " from file " << ppfname 
	   << "  RenormFactor=" << rfac << endl;
      PInPersist pin(ppfname);
      pin >> (*inmap);
      (*inmap) *= rfac;
      double mean, sigma;
      MeanSigma(*inmap, mean, sigma);
      cout << " ...InMap sizes " << inmap->InfoString() << endl;
      inmap->Show(); 
      cout << " ... Mean=" << mean << " Sigma=" << sigma << endl;
    }

    bool smo; 
    if (!maplss.CompareSizes(mapsync,smo) ) {
      cout << " calcpk2/ERROR sizes " << endl;
      maplss.Show();  mapsync.Show(); 
      return 99;
    }
    
    TArray<r_4> skycube(mapsync);
    skycube += maplss;

    if (fgaddnoise) { 
      cout << " calcpk2: adding noise to skycube cube ... " << endl;
      BeamEffect::AddNoise(skycube, pixsignoise);
    }

    double mean, sigma;
    MeanSigma(skycube, mean, sigma);
    cout << " input sky cube : Mean=" << mean << " Sigma=" << sigma << endl;      
    tm.Split(" After input ");

    H21Conversions conv;
    conv.setFrequency(Freq0MHz);
    double lambda = conv.getLambda();
    Four2DResponse arep(2, InterfArrayDiametre/lambda, InterfArrayDiametre/lambda, lambda);
    double DoL = 1.22/ArcminToRadian(tbeamarcmin);
    Four2DResponse tbeam(2, DoL, DoL );
    ForegroundCleaner  cleaner(arep, tbeam, skycube); 
    if (fgcorrbeam) {
      cout << "calcpk2[3.a] : calling cleaner.BeamCorrections() for target beam (" << tbeamarcmin << " arcmin)" 
	   << " Diam/Lambda=" << DoL << endl;
      cleaner.BeamCorrections();
    }
    cout << " calcpk2[3.b] : calling cleaner.CleanNegatives() ... " << endl;
    cleaner.CleanNegatives();
    if (fgclnsrc) {
      cout << "calcpk2[3.c] : calling cleaner.CleanPointSources() with threshold NSigma=" << nsigsrc  << endl;
      cleaner.CleanPointSources(nsigsrc);
    }

    cout << "calcpk2[4] : calling cleaner.extractLSSCube(...) " << endl;
    TArray<r_4> synctemp, specidx; 
    TArray<r_4> exlss = cleaner.extractLSSCube(synctemp, specidx);

    MeanSigma(exlss, mean, sigma);
    cout << " After cleaning/extractLSS: Mean=" << mean << " Sigma=" << sigma << endl;      
    tm.Split(" After CleanForeground");

    cout << "calcpk2[5] : computing 3D Fourier coefficients ... " << endl;
    FFTWServer ffts;
    TArray< complex<r_4> > four3d;
    ffts.FFTForward(exlss, four3d);
    tm.Split(" After FFTForward ");

    cout << "calcpk2[6] : computing power spectrum ... " << endl;
    RandomGenerator rg;
    Four3DPk pkc(four3d, rg);
    double dkxmpc = DeuxPI/(double)exlss.SizeX()/XCellSizeMpc;
    double dkympc = DeuxPI/(double)exlss.SizeY()/YCellSizeMpc;
    double dkzmpc = DeuxPI/(double)exlss.SizeZ()/ZCellSizeMpc;
    pkc.SetCellSize(dkxmpc, dkympc, dkzmpc);
    
    HProf hp = pkc.ComputePk(0.,256);

    tm.Split(" Done ComputePk ");    
    
    cout << "calcpk2[7.a] : writing profile P(k)  to  " << outname << endl;
    POutPersist po(outname);
    po << hp;
    outname = "fgm_" + outname;
    cout << "calcpk2[7.b] : writing foreground maps to  " << outname << endl;
    POutPersist pom(outname);
    pom << PPFNameTag("Tsync") << synctemp;
    pom << PPFNameTag("async") << specidx;

  }  // End of try bloc 
  catch (PThrowable & exc) {  // catching SOPHYA exceptions
    cerr << " calcpk2.cc: Catched Exception (PThrowable)" << (string)typeid(exc).name() 
         << "\n...exc.Msg= " << exc.Msg() << endl;
    rc = 99;
  }
  catch (std::exception & e) {  // catching standard C++ exceptions
    cerr << " calcpk2.cc: Catched std::exception "  << " - what()= " << e.what() << endl;
    rc = 98;
  }
  catch (...) {  // catching other exceptions
    cerr << " calcpk2.cc: some other exception (...) was caught ! " << endl;
    rc = 97;
  }
  cout << " ==== End of calcpk2.cc program  Rc= " << rc << endl;
  return rc;    
}


