#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include "sophyainit.h"
#include "timing.h"
#include "dvlist.h"
#include "ntuple.h"
#include "matharr.h"
#include "randfmt.h"
//#include "randr48.h"
#include "srandgen.h"

#include "constcosmo.h"
#include "geneutils.h"
#include "genefluct3d.h"

void usage(void);
void usage(void)
{
 cout<<"cmvginit3d [...options...]"<<endl
     <<" -a : auto init random seed (needed for multiple simul)"<<endl
     <<" -G typevol: compute Pk(z=0) and apply growth factor in real space"<<endl
     <<"       typevol=1 evolved with distance / observateur (def)"<<endl
     <<"       typevol=2 evolved with distance to middle of Z planes"<<endl
     <<"       else : no evol, spectrum Pk(z=z_median) for all cube (def)"<<endl
     <<" -F : filter spectrum by pixel shape (0=no 1=yes(default)"<<endl
     <<" -x nx,dx : size along x axis (npix,Mpc)"<<endl
     <<" -y ny,dy : size along y axis (npix,Mpc)"<<endl
     <<"            if ny or dy <=0 take same value as for x"<<endl
     <<" -z nz,dz : size along z axis (redshift axis, npix,Mpc)"<<endl
     <<" -Z zref : redshift for the center of the simulation cube"<<endl
     <<" -1 : compute 1D spectrum (default: no)"<<endl
     <<" -2 : compute 2D spectrum (default: no)"<<endl
     <<" -C : go back to cube in FT space and compute P(k) for check (def: do nothing)"<<endl
     <<" -8 sigmaR,R : normalisation du spectre de puissance, R en Mpc"<<endl
     <<"               (default sigmaR=1, R=8/h100 Mpc)"<<endl
     <<" -v temp_file.ppf: generate los velocity cube"<<endl
     <<"                   temporary save cube in temp_file.ppf"<<endl
     <<" -n nent : fill control ntuple with nent entries (def: do not fill)"<<endl
     <<" -O a,b : tell what you want to write"<<endl
     <<"      a : write generated fourier cube (_k0)"<<endl
     <<"      b : write real space cube dRho/Rho at z (_r0)"<<endl
     <<"      a,b= 0 no write, 1 ppf write, 2 fits write, 3 ppf+fits write"<<endl
     <<" -S : write cube slices in PPF format"<<endl
     <<" -o out_base_name : base string for output file name (def: cmvginit3d)"<<endl
     <<" -V : compute variance from real space (for check, default: no)"<<endl
     <<" -T nth : nombre de threads (si compil multi-thread, default: 0)"<<endl
     <<endl;
}

int main(int narg,char *arg[])
{
 SophyaInit();
 InitTim();

 //-----------------------------------------------------------------
 // *** Survey definition
 long nx=360, ny=-1, nz=64; double dx=1., dy=-1., dz=-1.;
 //long nx=1000, ny=-1, nz=128; double dx=3., dy=-1., dz=6.;
 //long nx=1200, ny=-1, nz=128; double dx=1., dy=-1., dz=3;

 // *** Cosmography definition   (WMAP)
 unsigned short flat = 0;
 double ob0 = 0.0444356;
 double h100=0.71, om0=0.267804, or0=7.9e-05, ol0=0.73,w0=-1.;
 double zref = 0.5;
 double perc=0.01,dzinc=-1.,dzmax=-1.; unsigned short glorder=4;

 // *** Spectrum and variance definition
 double ns = 1., as = 1.;
 double R=8./h100, Rg=R/sqrt(5.);
 double sigmaR = 1.;

 double kmin=1e-5,kmax=1000.;
 int npt = 10000;
 double lkmin=log10(kmin), lkmax=log10(kmax);
 double eps=1.e-3;

 // *** type de generation
 int use_growth_factor = 0;
 unsigned short nthread=0;
 int filter_by_pixel = 1;

 // *** What to do
 bool comptransveloc = false;
 string temporay_file = "cmvginit3d_tmp.ppf";
 // compute: 1=1D spectra,  2=2D spectra,
 //          4=recompute spectra after real space generation
 unsigned short compdspec = 0;
 long ntnent = 0;  // fill control NTuple
 bool wslice = false;
 bool compvarreal = false;
 unsigned short whattowrt[2] = {1,1};
 string rootnameout = "cmvobserv3d";

 // --- Decodage arguments
 if(narg>0) {
   cout<<"\n--- Arguments: "<<endl;
   for(int i=0;i<narg;i++) cout<<arg[i]<<" ";
   cout<<endl;
 }
 system("date -u");

 // --- Choix du generateur aleatoire (a faire ICI imperativement avant AutoInitRand)
 FMTRandGen *RandGen = new FMTRandGen;
 RandGen->SelectGaussianAlgo(C_Gaussian_RandLibSNorm);
 RandomGeneratorInterface::SetGlobalRandGenP(RandGen);
 
 // --- Decodage des arguments
 char c;
 while((c = getopt(narg,arg,"haG:F:x:y:z:Z:128:v:n:CO:So:VT:")) != -1) {
  int nth = 0;
  switch (c) {
  case 'a' :
    AutoInitRand(5);
    break;
  case 'G' :
    use_growth_factor = atoi(optarg);
    break;
  case 'F' :
    filter_by_pixel = atoi(optarg);
    break;
  case 'x' :
    sscanf(optarg,"%ld,%lf",&nx,&dx);
    break;
  case 'y' :
    sscanf(optarg,"%ld,%lf",&ny,&dy);
    break;
  case 'z' :
    sscanf(optarg,"%ld,%lf",&nz,&dz);
    break;
  case 'Z' :
    zref = atof(optarg);
    break;
  case 'v' :
    comptransveloc = true;
    temporay_file = optarg;
    break;
  case '1' :
    compdspec |= 1;
    break;
  case '2' :
    compdspec |= 2;
    break;
  case 'C' :
    compdspec |= 4;
    break;
  case '8' :
    sscanf(optarg,"%lf,%lf",&sigmaR,&R);
    break;
  case 'V' :
    compvarreal = true;
    break;
  case 'n':
    ntnent = atol(optarg);
    if(ntnent<0) ntnent = 0;
    break;
  case 'O' :
    sscanf(optarg,"%hu,%hu",&whattowrt[0],&whattowrt[1]);
    break;
  case 'S' :
    wslice = true;
    break;
  case 'o' :
    rootnameout = optarg;
    break;
  case 'T' :
    nth = atoi(optarg);
    nthread = (nth<1)? 0: nth;
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 }

 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH
 try {
 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Initialisation and parameters"<<endl;
 cout<<"zref="<<zref<<endl;
 cout<<"nx="<<nx<<" dx="<<dx<<" ny="<<ny<<" dy="<<dy<<" nz="<<nz<<" dz="<<dz<<endl;
 cout<<"kmin="<<kmin<<" ("<<lkmin<<"), kmax="<<kmax<<" ("<<lkmax<<") Mpc^-1"<<", npt="<<npt<<endl;
 cout<<"Filter by pixel = "<<filter_by_pixel<<endl;
 if(comptransveloc) cout<<"Tansverse velocity generation requested"<<endl;
 cout<<"R="<<R<<" Rg="<<Rg<<" Mpc, sigmaR="<<sigmaR<<endl;
 cout<<"Use_growth_factor = "<<use_growth_factor<<endl;
 cout<<"wslice="<<wslice<<" what?="<<whattowrt[0]<<","<<whattowrt[1]<<endl;
 cout<<"rootnameout="<<rootnameout<<endl;
 ShowRandom();
 cout<<"   First random is: "<<drand01()<<endl;

 string tagobs = rootnameout + ".ppf";
 cout<<"Writing result in "<<tagobs<<endl;
 POutPersist posobs(tagobs);

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Cosmology"<<endl;

 CosmoCalc univ(flat,true,zref+1.);
 univ.SetInteg(perc,dzinc,dzmax,glorder);
 univ.SetDynParam(h100,om0,or0,ol0,w0);
 univ.PrtInteg();
 univ.Print();
 double loscomref = univ.Dloscom(zref);
 cout<<"\nzref = "<<zref<<" -> dloscom = "<<loscomref<<" Mpc"<<endl;
 univ.Print(zref);

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Create Spectrum"<<endl;

 InitialSpectrum pkini(ns,as);

 TransfertEisenstein tf(h100,om0-ob0,ob0,T_CMB_Par,false);
 TransfertEisenstein tfnos(h100,om0-ob0,ob0,T_CMB_Par,false);
   tfnos.SetNoOscEnv(2);

 GrowthFactor growth(om0,ol0);
 // GrowthFactor growth(1.,0.); // D(z) = 1/(1+z)
 double growth_at_z = growth(zref);
 cout<<"...Growth factor at z="<<zref<<" = "<<growth_at_z<<endl;

 PkSpectrum0 pk0(pkini,tf);
 PkSpectrum0 pk0nos(pkini,tfnos);

 PkSpectrumZ pkz(pk0,growth,zref);
 PkSpectrumZ pkznos(pk0nos,growth,zref);

 //-----------------------------------------------------------------
 pkz.SetZ(0.);
 pkznos.SetZ(0.);
 cout<<endl<<"\n--- Compute variance for top-hat R="<<R<<" at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_th(pkz,R,VarianceSpectrum::TOPHAT);
 double kfind_th = varpk_th.FindMaximum(kmin,kmax,eps);
 double pkmax_th = varpk_th(kfind_th);
 cout<<"kfind_th = "<<kfind_th<<" ("<<log10(kfind_th)<<"), integrand="<<pkmax_th<<endl;
 double k1=kmin, k2=kmax;
 int rc = varpk_th.FindLimits(pkmax_th/1.e4,k1,k2,eps);
 cout<<"limit_th: rc="<<rc<<" : "<<k1<<" ("<<log10(k1)<<") , "
     <<k2<<" ("<<log10(k2)<<")"<<endl;

 double ldlk = (log10(k2)-log10(k1))/npt;
 varpk_th.SetInteg(0.01,ldlk,-1.,4);
 double sr2 = varpk_th.Variance(k1,k2);
 cout<<"varpk_th="<<sr2<<"  ->  sigma="<<sqrt(sr2)<<endl;

 double normpkz = sigmaR*sigmaR/sr2;
 pkz.SetScale(normpkz);
 pkznos.SetScale(normpkz);
 cout<<"Spectrum normalisation = "<<pkz.GetScale()<<endl;
 {
 Histo hpkz0(lkmin,lkmax,npt); hpkz0.ReCenterBin();
 FuncToHisto(pkz,hpkz0,true);
 posobs.PutObject(hpkz0,"hpkz0");
 Histo hpkz0nos(hpkz0);
 FuncToHisto(pkznos,hpkz0nos,true);
 posobs.PutObject(hpkz0nos,"hpkz0nos");
 }

 pkz.SetZ(zref);
 pkznos.SetZ(zref);
 {
 Histo hpkz(lkmin,lkmax,npt); hpkz.ReCenterBin();
 FuncToHisto(pkz,hpkz,true);
 posobs.PutObject(hpkz,"hpkz");
 Histo hpkznos(hpkz);
 FuncToHisto(pkznos,hpkznos,true);
 posobs.PutObject(hpkznos,"hpkznos");
 }

 //-----------------------------------------------------------------
 cout<<endl<<"\n--- Compute variance for Pk at z="<<pkz.GetZ()<<endl;
 VarianceSpectrum varpk_int(pkz,R,VarianceSpectrum::NOFILTER);
 double kfind_int = varpk_int.FindMaximum(kmin,kmax,eps);
 double pkmax_int = varpk_int(kfind_int);
 cout<<"kfind_int = "<<kfind_int<<" ("<<log10(kfind_int)<<"), integrand="<<pkmax_int<<endl;
 double k1int=kmin, k2int=kmax;
 int rcint = varpk_int.FindLimits(pkmax_int/1.e4,k1int,k2int,eps);
 cout<<"limit_int: rc="<<rcint<<" : "<<k1int<<" ("<<log10(k1int)<<") , "
     <<k2int<<" ("<<log10(k2int)<<")"<<endl;

 double ldlkint = (log10(k2int)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int = varpk_int.Variance(k1int,k2int);
 cout<<"varpk_int="<<sr2int<<"  ->  sigma="<<sqrt(sr2int)<<endl;

 //-----------------------------------------------------------------

 PrtTim(">>>> End of definition");

 //-----------------------------------------------------------------
 // Le cube et sa cosmlogie
 // FFTW3 (p26): faster if sizes 2^a 3^b 5^c 7^d 11^e 13^f  with e+f=0 ou 1
 cout<<endl<<"\n--- Initialisation de GeneFluct3D"<<endl;

 GeneFluct3D fluct3d(nx,ny,nz,dx,dy,dz,nthread,2);
 fluct3d.SetCosmology(univ);
 fluct3d.SetGrowthFactor(growth);
 fluct3d.SetObservator(zref,-nz/2.);
 fluct3d.LosComRedshift(0.001,-1);
 //TArray< complex<GEN3D_TYPE> >& pkgen = fluct3d.GetComplexArray();
 //TArray<GEN3D_TYPE>& rgen = fluct3d.GetRealArray();
 cout<<endl; fluct3d.Print();

 //-----------------------------------------------------------------
 // Les bins et ranges en k pour les histos 1D et 2D
 double dkmin = fluct3d.GetKincMin();
 double knyqmax = fluct3d.GetKmax();
 long nherr = long(knyqmax/dkmin+0.5);
 cout<<"\nFor HistoErr: d="<<dkmin<<" max="<<knyqmax<<" n="<<nherr<<endl;

 double dktmin = fluct3d.GetKTincMin();
 double ktnyqmax = fluct3d.GetKTmax();
 long nherrt = long(ktnyqmax/dktmin+0.5);
 double dkzmin = fluct3d.GetKinc()[2];
 double kznyqmax = fluct3d.GetKnyq()[2];
 long nherrz = long(kznyqmax/dkzmin+0.5);
 cout<<"For Histo2DErr: d="<<dktmin<<","<<dkzmin
     <<" max="<<ktnyqmax<<","<<kznyqmax<<" n="<<nherrt<<","<<nherrz<<endl;

 //-----------------------------------------------------------------
 cout<<"\n--- Computing spectra variance up to Kmax at z="<<pkz.GetZ()<<endl;
 // En fait on travaille sur un cube inscrit dans la sphere de rayon kmax:
 // sphere: Vs = 4Pi/3 k^3 , cube inscrit (cote k*sqrt(2)): Vc = (k*sqrt(2))^3
 // Vc/Vs = 0.675   ->  keff = kmax * (0.675)^(1/3) = kmax * 0.877
 double knyqmax_mod = 0.877*knyqmax;
 ldlkint = (log10(knyqmax_mod)-log10(k1int))/npt;
 varpk_int.SetInteg(0.01,ldlkint,-1.,4);
 double sr2int_kmax = varpk_int.Variance(k1int,knyqmax_mod);
 cout<<"varpk_int(<"<<knyqmax_mod<<")="<<sr2int_kmax<<"  ->  sigma="<<sqrt(sr2int_kmax)<<endl;

 PrtTim(">>>> End Initialisation of GeneFluct3D");

 //-----------------------------------------------------------------
 cout<<"\n--- Computing a realization in Fourier space"<<endl;
 if(use_growth_factor>0) pkz.SetZ(0.); else pkz.SetZ(zref);
 cout<<"Power spectrum set at redshift: "<<pkz.GetZ()<<endl;
 fluct3d.ComputeFourier(pkz);
 fluct3d.NTupleCheck(posobs,string("ntpkgen"),ntnent);
 PrtTim(">>>> End Computing a realization in Fourier space");

 HistoErr hpkgen(0.,knyqmax,nherr);
 if(compdspec&1) {
   cout<<"\n--- Checking realization spectra"<<endl;
   hpkgen.ReCenterBin(); hpkgen.Zero(); hpkgen.Show();
   fluct3d.ComputeSpectrum(hpkgen);
   posobs.PutObject(hpkgen,"hpkgen");
   PrtTim(">>>> End Checking realization 1D spectra");
 }
 Histo2DErr hpkgen2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
 if(compdspec&2) {
   cout<<"\n--- Checking realization 2D spectra"<<endl;
   hpkgen2.ReCenterBin(); hpkgen2.Zero(); hpkgen2.Show();
   fluct3d.ComputeSpectrum2D(hpkgen2);
   posobs.PutObject(hpkgen2,"hpkgen2");
   PrtTim(">>>> End Checking realization 2D spectra");
 }

 //-----------------------------------------------------------------
 if(filter_by_pixel!=0) {
   cout<<"\n--- Computing convolution by pixel shape"<<endl;
   fluct3d.FilterByPixel();
   fluct3d.NTupleCheck(posobs,string("ntpkgenf"),ntnent);
   PrtTim(">>>> End Computing convolution by pixel shape");
 }

 if(compdspec&1) {
   HistoErr hpkgenf(hpkgen);
   if(filter_by_pixel!=0) {
     cout<<"\n--- Checking realization spectra"<<endl;
     hpkgenf.Zero(); hpkgenf.Show();
     fluct3d.ComputeSpectrum(hpkgenf);
     PrtTim(">>>> End Checking realization 1D spectra");
   }
   posobs.PutObject(hpkgenf,"hpkgenf");
 }
 if(compdspec&2) {
   Histo2DErr hpkgenf2(hpkgen2);
   if(filter_by_pixel!=0) {
     cout<<"\n--- Checking realization 2D spectra"<<endl;
     hpkgenf2.Zero(); hpkgenf2.Show();
     fluct3d.ComputeSpectrum2D(hpkgenf2);
     PrtTim(">>>> End Checking realization 2D spectra");
   }
   posobs.PutObject(hpkgenf2,"hpkgenf2");
 }

 //-----------------------------------------------------------------
 if(whattowrt[0]&1) {
   tagobs = rootnameout + "_k.ppf";
   fluct3d.WritePPF(tagobs,false);
   PrtTim(">>>> End WritePPF");
   temporay_file = tagobs;
 }
 if(whattowrt[0]&2) {
   tagobs = "!" + rootnameout + "_k.fits";
   fluct3d.WriteFits(tagobs);
   PrtTim(">>>> End WriteFits");
 }
 if(comptransveloc && !(whattowrt[0]&1)) {
   cout<<">>> Writing FT cube (for transv veloc.) in "<<temporay_file<<endl;
   fluct3d.WritePPF(temporay_file,false);
   PrtTim(">>>> End Writing FT cube");
 }

 //-----------------------------------------------------------------
 cout<<"\n--- Computing a realization in real space"<<endl;
 fluct3d.ComputeReal();
 double rmin,rmax; fluct3d.MinMax(rmin,rmax);
 cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
 fluct3d.NTupleCheck(posobs,string("ntreal"),ntnent);
 PrtTim(">>>> End Computing a realization in real space");

 if(use_growth_factor>0) {
   cout<<"\n--- Apply Growth factor"<<endl;
   cout<<"...D(z=0)="<<growth(0.)<<"  D(z="<<zref<<")="<<growth(zref)<<endl;
   fluct3d.ApplyGrowthFactor(use_growth_factor);
   fluct3d.MinMax(rmin,rmax);
   cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
   fluct3d.NTupleCheck(posobs,string("ntgrow"),ntnent);
   PrtTim(">>>> End Applying growth factor");
 }

 int_8 nm;
 double rmref,rs2ref;
 cout<<"\n--- Computing reference variance in real space"<<endl;
 nm = fluct3d.MeanSigma2(rmref,rs2ref);
 cout<<" rs2ref= "<<rs2ref<<" , rmref="<<rmref<<" ("<<nm<<")"<<endl;
 PrtTim(">>>> End Computing reference variance in real space");

 if(whattowrt[1]&1) {
   tagobs = rootnameout + "_r.ppf";
   fluct3d.WritePPF(tagobs,true);
   PrtTim(">>>> End WritePPF");
 }
 if(whattowrt[1]&2) {
     tagobs = "!" + rootnameout + "_r.fits";
     fluct3d.WriteFits(tagobs);
     PrtTim(">>>> End WriteFits");
 }
 if(wslice) {
   tagobs = rootnameout + "_s_r.ppf";
   fluct3d.WriteSlicePPF(tagobs);
   PrtTim(">>>> End WriteSlicePPF");
 }

 //-----------------------------------------------------------------
 if(compvarreal) {
   cout<<"\n--- Check variance sigmaR in real space"<<endl;
   double varr;
   fluct3d.VarianceFrReal(R,varr);
   cout<<"...Computed variance = "<<varr
       <<" , Theorical variance at (z=0) = "<<pow(sigmaR,2.)
       <<" , at (z="<<zref<<") = "<<pow(sigmaR*growth_at_z,2.)<<endl;
   PrtTim(">>>> End Check variance sigmaR in real space");
 }

 //-----------------------------------------------------------------
 if(compdspec&4) { // ATTENTION:  d_rho/rho ecrase
   cout<<endl<<"\n--- ReComputing spectrum from real space"
       <<"\n    Warning: REAL SPACE CUBE OVERWRITTEN"<<endl;
   fluct3d.ReComputeFourier();
   fluct3d.NTupleCheck(posobs,string("ntpkrec"),ntnent);
   PrtTim(">>>> End ReComputing spectrum");

   cout<<endl<<"\n--- Computing final spectrum"<<endl;
   HistoErr hpkrecb(0.,knyqmax,nherr); hpkrecb.Zero();
   hpkrecb.ReCenterBin(); hpkrecb.Show();
   fluct3d.ComputeSpectrum(hpkrecb);
   posobs.PutObject(hpkrecb,"hpkrecb");
   PrtTim(">>>> End Computing final spectrum");

   HistoErr hpkrec(hpkrecb);
   if(filter_by_pixel!=0) {
     cout<<endl<<"\n--- Computing final spectrum with pixel deconv."<<endl;
     hpkrec.Zero();
     fluct3d.ComputeSpectrum(hpkrec,0.,filter_by_pixel);
     PrtTim(">>>> End Computing final spectrum with pixel deconv.");
   }
   posobs.PutObject(hpkrec,"hpkrec");

   cout<<"\n--- Computing final 2D spectrum"<<endl;
   Histo2DErr hpkrecb2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkrecb2.ReCenterBin(); hpkrecb2.Zero(); hpkrecb2.Show();
   fluct3d.ComputeSpectrum2D(hpkrecb2);
   posobs.PutObject(hpkrecb2,"hpkrecb2");
   PrtTim(">>>> End Computing final 2D spectrum");

   Histo2DErr hpkrec2(hpkrecb2);
   if(filter_by_pixel!=0) {
     cout<<"\n--- Computing final 2D spectrum with pixel deconv."<<endl;
     hpkrec2.Zero();
     fluct3d.ComputeSpectrum2D(hpkrec2,0.,filter_by_pixel);
     PrtTim(">>>> End Computing final 2D spectrum with pixel deconv.");
   }
   posobs.PutObject(hpkrec2,"hpkrec2");
 }

 //-----------------------------------------------------------------
 //-----------------------------------------------------------------
 //-----------------------------------------------------------------

 if(comptransveloc) {
   cout<<"\n\n\n"<<"---------------------------------------------\n"
                 <<"--- Transverse velocity space computation ---\n"
                 <<"---------------------------------------------\n"<<endl;

   //-----------------------------------------------------------------
   cout<<"\n--- Reading back the Pk(vec(k)) cube from "<<temporay_file<<endl;
   fluct3d.ReadPPF(temporay_file);
   PrtTim(">>>> End Reading the Pk(vec(k)) cube");

   if(compdspec&1) {
     cout<<"\n--- Check realization spectra that has been re-read"<<endl;
     HistoErr hpkgenf_rr(hpkgen); hpkgenf_rr.Zero();
     fluct3d.ComputeSpectrum(hpkgenf_rr);
     PrtTim(">>>> End Checking re-read realization 1D spectra");
     posobs.PutObject(hpkgenf_rr,"hpkgenf_rr");
   }

   //-----------------------------------------------------------------
   cout<<"\n--- Modifying cube for Transverse velocity"<<endl;
   fluct3d.ToVelLoS();
   fluct3d.NTupleCheck(posobs,string("ntpvgenf"),ntnent);
   PrtTim(">>>> End Modifying cube for Transverse velocity");

   if(compdspec&1) {
     cout<<"\n--- Checking realization spectra"<<endl;
     HistoErr hpvgen(0.,knyqmax,nherr);
     hpvgen.ReCenterBin(); hpvgen.Zero(); hpvgen.Show();
     fluct3d.ComputeSpectrum(hpvgen);
     posobs.PutObject(hpvgen,"hpvgen");
     PrtTim(">>>> End Checking realization 1D spectra");
   }
   if(compdspec&2) {
     cout<<"\n--- Checking realization 2D spectra"<<endl;
     Histo2DErr hpvgen2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
     hpvgen2.ReCenterBin(); hpvgen2.Zero(); hpvgen2.Show();
     fluct3d.ComputeSpectrum2D(hpvgen2);
     posobs.PutObject(hpvgen2,"hpvgen2");
     PrtTim(">>>> End Checking realization 2D spectra");
   }

    if(whattowrt[0]&1) {
      tagobs = rootnameout + "_kv.ppf";
      fluct3d.WritePPF(tagobs,false);
      PrtTim(">>>> End WritePPF");
    }
    if(whattowrt[0]&2) {
        tagobs = "!" + rootnameout + "_kv.fits";
        fluct3d.WriteFits(tagobs);
        PrtTim(">>>> End WriteFits");
    }

   //-----------------------------------------------------------------
   cout<<"\n--- Computing a realization in real space for Transverse velocity"<<endl;
   fluct3d.ComputeReal();
   double rmin,rmax; fluct3d.MinMax(rmin,rmax);
   cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
   fluct3d.NTupleCheck(posobs,string("nvreal"),ntnent);
   PrtTim(">>>> End Computing a realization in real space");

    if(use_growth_factor>0) {
      cout<<"\n--- Apply Growth factor"<<endl;
      cout<<"...D(z=0)="<<growth(0.)<<"  D(z="<<zref<<")="<<growth(zref)<<endl;
      fluct3d.ApplyVelLosGrowthFactor(use_growth_factor);
      fluct3d.MinMax(rmin,rmax);
      cout<<"rgen.Min = "<<rmin<<" , Max="<<rmax<<endl;
      fluct3d.NTupleCheck(posobs,string("nvgrow"),ntnent);
      PrtTim(">>>> End Applying growth factor");
    }

    int_8 nmv;
    double vmref,vs2ref;
    cout<<"\n--- Computing reference variance in real space"<<endl;
    nmv = fluct3d.MeanSigma2(vmref,vs2ref);
    cout<<" vs2ref= "<<vs2ref<<" , vmref="<<vmref<<" ("<<nmv<<")"<<endl;
    PrtTim(">>>> End Computing reference variance in real space");

    if(whattowrt[1]&1) {
      tagobs = rootnameout + "_rv.ppf";
      fluct3d.WritePPF(tagobs,true);
      PrtTim(">>>> End WritePPF");
    }
    if(whattowrt[1]&2) {
        tagobs = "!" + rootnameout + "_rv.fits";
        fluct3d.WriteFits(tagobs);
        PrtTim(">>>> End WriteFits");
    }
    if(wslice) {
      tagobs = rootnameout + "_s_rv.ppf";
      fluct3d.WriteSlicePPF(tagobs);
      PrtTim(">>>> End WriteSlicePPF");
    }

 }

 //-----------------------------------------------------------------
 {
 DVList dvl;
 dvl("Nx") = (int_4)fluct3d.Nx(); dvl("Ny") = (int_4)fluct3d.Ny(); dvl("Nz") = (int_4)fluct3d.Nz();
 dvl("Dx") = fluct3d.Dx(); dvl("Dy") = fluct3d.Dy(); dvl("Dz") = fluct3d.Dz();
 dvl("Z") = fluct3d.Zref(); dvl("Zpk") = fluct3d.ZrefPk();
 dvl("D") = fluct3d.Dref(); dvl("Dpk") = fluct3d.DrefPk();
 dvl("s8") = sigmaR;
 dvl("Dtype") = (int_4)use_growth_factor; dvl("Pxfilt") = (int_4)filter_by_pixel; 
 posobs.PutObject(dvl,"siminfo");
 }

 delete RandGen;
 PrtTim(">>>> End Of Job");

 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH
 } catch (PException& exc) {
   cerr<<"cmvginit3d.cc catched PException"<<exc.Msg()<<endl;
   return 77;
 } catch (std::exception& sex) {
   cerr << "cmvginit3d.cc std::exception :" 
        << (string)typeid(sex).name() << "\n msg= " 
        << sex.what() << endl;
   return 78;
 } catch (...) {
   cerr << "cmvginit3d.cc catched unknown (...) exception  " << endl; 
   return 79;
 }
 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH

 return 0;
}

