#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include "sophyainit.h"
#include "timing.h"
#include "dvlist.h"
#include "histos.h"
#include "fabtcolread.h"
#include "fftwserver.h"

#include "constcosmo.h"
#include "geneutils.h"
#include "genefluct3d.h"
// set simul = 6_0p0_780
// cmvrvloscorf -o rvlos_${simul}.ppf -n 1,1 -z -1,0 -H 50 -2 -S 10000 -s 1. ginit3d_${simul}_r.fits ginit3d_${simul}_rv.fits

void usage(void);
void usage(void)
{
cout
<<"cmvrvloscor [options] rho.fits vlos.fits"<<endl
<<"-n nplany,nplanx: process one Y (X) plane every \"nplany\" (\"nplanx\") (def:1,1(all))"<<endl
<<"-z nplanz,iplanz0: process only \"nplanz\" Z plane beginnig at plane \"iplanz0\" (def:all Z planes)"<<endl
<<"-H nhfill: fill displacement histo with \"nhfill\" los (def:25)"<<endl
<<"-s scaldis: scale the displacement by factor \"scaldis\" (def: 1.)"<<endl
<<"-p: do not distribute dRho/Rho in pixel, just take the nearest (def: no, distribute rho)"<<endl
<<"-S npkmax: compute cross-power spectrum of V*conj(R) with \"npkmax\" los (def: no)"<<endl
<<"-2: compute 2D projection for dRho and dRho(corrected) for 3 Y-Z plans (def=no)"<<endl
<<"-N: do not create 3D cube and recompute 1D and 2D spectra (def: no do-it !)"<<endl
<<"-o ppfout.ppf: output ppf file name (def=\"cmvrvloscor.ppf\")"<<endl
<<endl;
}

int main(int narg,char *arg[])
{
 int nthread = 1, nplany=1, nplanx=1, nplanz=-1, iplanz0=0, nhfilllos = 25, npkmax = 0;
 double scaldis = 1.;
 bool docube=true, do2d = false, dodistrib=true;
 string fnppf = "cmvrvloscor.ppf";
 
 // --- Decodage des arguments
 char c;
 while((c = getopt(narg,arg,"hn:H:S:Np2o:s:z:")) != -1) {
  switch (c) {
  case 'n' :
    sscanf(optarg,"%d,%d",&nplany,&nplanx);
    if(nplany<=0) nplany = 1;
    if(nplanx<=0) nplanx = 1;
    break;
  case 'z' :
    sscanf(optarg,"%d,%d",&nplanz,&iplanz0);
    break;
  case 'H' :
    nhfilllos = atoi(optarg);
    if(nhfilllos<=0) nhfilllos = 0;
    break;
  case 'S' :
    npkmax = atoi(optarg);
    break;
  case 'N' :
    docube = false;
    break;
  case 'p' :
    dodistrib = false;
    break;
  case '2' :
    do2d = true;
    break;
  case 'o' :
    fnppf = optarg;
    break;
  case 's' :
    scaldis = atof(optarg);
    break;
  case 'h' :
  default :
    usage(); return -1;
  }
 }
 if(optind>=narg-1) {usage(); return -1;}

 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH
 try {
 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH

 SophyaInit();
 InitTim();

 // --- open FITS files (dRho/Rho and Vlos)
 cout<<"> read rho: "<<arg[optind]<<endl;
 FitsImg3DRead f3dr(arg[optind],0,5);
 cout<<"> read vlos: "<<arg[optind+1]<<endl;
 FitsImg3DRead f3dv(arg[optind+1],0,5);
 long Nx = f3dr.ReadKeyL("Nx");
 long Ny = f3dr.ReadKeyL("Ny");;
 long Nz = f3dr.ReadKeyL("Nz");;
 cout<<"N: x="<<Nx<<" y="<<Ny<<" z="<<Nz<<endl;
 double Dx = f3dr.ReadKey("Dx");
 double Dy = f3dr.ReadKey("Dy");
 double Dz = f3dr.ReadKey("Dz");
 cout<<"D: x="<<Dx<<" y="<<Dy<<" z="<<Dz<<endl;
 double Zref = f3dr.ReadKey("ZREF");
 double Href = f3dr.ReadKey("HREF");
 cout<<"Zref="<<Zref<<" Href="<<Href<<endl;

 POutPersist pos(fnppf.c_str());
 DVList dvllos;

 // --- Read and process data
 if(nplany>Ny || nplany<=0) nplany = Ny;
 if(nplanx>Nx || nplanx<=0) nplanx = Nx;
 cout<<"...Will read one Y plane every "<<nplany<<endl;
 cout<<"             one X plane every "<<nplanx<<endl;
 int nlospred = Nx*Ny/nplany/nplanx;
 cout<<"   predicted number of los to be read "<<nlospred<<endl;
 if(nplanz>Nz || nplanz<=0) nplanz = Nz;
 if(iplanz0<0)iplanz0 = 0;
 if(iplanz0>=Nz) {cout<<"Error: iplanz0="<<iplanz0<<" out of range"<<endl; return -2;}
 if(iplanz0+nplanz>Nz) nplanz = Nz - iplanz0;
 cout<<"   process "<<nplanz<<" Z planes in ["<<iplanz0<<","<<iplanz0+nplanz<<"["<<endl;
 TVector<r_4> V4read(Nz);
 TVector<r_4> R(nplanz), V(nplanz);
 TVector<r_8> Rdis(nplanz);

 // --- Histo des deplacements
 Histo* hmpc = NULL;
 if(nhfilllos>0) {
   cout<<"...Fill Mpc displacement histo with "<<nhfilllos<<" los"<<endl;
   double dmin = min(Dx,min(Dy,Dz)); if(fabs(scaldis)>0.) dmin *= fabs(scaldis);
   long nmax = max(Nx,max(Ny,Nz));
   hmpc = new Histo(-dmin*nmax/10.,dmin*nmax/10.,nmax);
   cout<<"   dmin="<<dmin<<" nmax="<<nmax<<endl;
   nhfilllos = int((double)nlospred/nhfilllos + 0.5);
   if(nhfilllos<=0) nhfilllos = 1;
   cout<<"   -> fill one los every "<<nhfilllos<<endl;
 }

 // --- Vector for PK cross-correlation computation
 if(npkmax<0) npkmax = Nx*Ny;
 TVector< complex<r_4> > FR, FV;
 TVector< complex<r_8> > pkvr, FRdis;
 TVector<r_8> pkr, pkrc;
 FFTWServer fftserv;
 fftserv.setNormalize(false); // pour accord avec GeneFluct3D
 if(npkmax>0) {
   cout<<"...compute R*conj(R) V*conj(V) V*conj(R) cross-power spectrum with "<<npkmax<<" los"<<endl;
   npkmax = int((double)nlospred/npkmax + 0.5);
   if(npkmax<=0) npkmax = 1;
   cout<<"   -> fill one los every "<<npkmax<<endl;
 }
 int npkav = 0;
 double dkzpk = 2.*M_PI/(nplanz*Dz);  // Mpc^-1

 // --- Create a Cube for analysis
 GeneFluct3D *fluct3d = NULL;
 TArray<GEN3D_TYPE>* rgen = NULL;
 if(docube) {
   cout<<"...Create and fill 3D cube"<<endl;
   fluct3d = new GeneFluct3D(Nx,Ny,nplanz,Dx,Dy,Dz,nthread,2);
   fluct3d->Print();
   rgen = &(fluct3d->GetRealArray());
   *rgen = 0.;
   cout<<"rgen:  size [1]="<<rgen->SizeX()
                           <<" [2]="<<rgen->SizeY()
                           <<" [3]="<<rgen->SizeZ()<<endl;
   cout<<"pkgen: size [1]="<<fluct3d->GetComplexArray().SizeX()
                           <<" [2]="<<fluct3d->GetComplexArray().SizeY()
                           <<" [3]="<<fluct3d->GetComplexArray().SizeZ()<<endl;
 }

 // --- Do the jobs !!!
 cout<<">>> filling redshift distorted cube, scale displacement by "<<scaldis<<endl;
 int nlosread = 0;
 int lpri = nlospred/25; if(lpri==0) lpri = 1;
 for(int i=0;i<Nx;i+=nplanx) {
   TMatrix<r_4> M2d, M2dv, M2dc;
   if(do2d && (i==0 || i==Nx/2 || i==Nx-1)) {
     M2d.ReSize(Ny,nplanz); M2d = 0.;
     M2dv.ReSize(Ny,nplanz); M2dv = 0.;
     M2dc.ReSize(Ny,nplanz); M2dc = 0.;
   }
 for(int j=0;j<Ny;j+=nplany) {
   if(nlosread%lpri==0) cout<<"   nlosread="<<nlosread<<" (i="<<i<<",j="<<j<<")"<<endl;
   bool fhis = (hmpc != NULL && nlosread%nhfilllos==0) ? true: false;
   //for(int l=0;l<nplanz;l++) R(l) = f3dr.Read(iplanz0+l,j,i);
   //for(int l=0;l<nplanz;l++) V(l) = f3dv.Read(iplanz0+l,j,i);
   f3dr.Read(j,i,V4read);
     for(int l=0;l<nplanz;l++) R(l) = V4read(iplanz0+l);
   f3dv.Read(j,i,V4read);
     for(int l=0;l<nplanz;l++) V(l) = V4read(iplanz0+l);
   Rdis = 0.;
   // Calcul du champ R redshift distordu
   for(int l=0;l<nplanz;l++) {
     double d = scaldis * V(l) / Href;
     if(fhis) hmpc->Add(d);
     double lpd = (double)l + d/Dz; // valeur du deplacee
     if(dodistrib) { 
       // on repartit proportionellement au recouvrement sur 2 pixels
       long l1 = long(lpd); // pixel de gauche
       long l2 = l1 + 1;  // pixel de droite
       lpd -= (double)l1;  // recouvrement du pixel du dessus
       if(l1>=0 && l1<nplanz) Rdis(l1) += R(l) * (1.-lpd);
       if(l2>=0 && l2<nplanz) Rdis(l2) += R(l) * lpd;
     } else { // take nearest pixel
       long l1 = long(lpd + 0.5);
       if(l1>=0 && l1<nplanz) Rdis(l1) += R(l);
     }
   }
   // On remplit eventuellement les matrices 2D
   if(do2d && M2d.Size()>0)
     for(int l=0;l<nplanz;l++) {M2d(j,l) = R(l); M2dv(j,l) = V(l); M2dc(j,l) = Rdis(l);}
   // On remplit le cube avec le champ R redshift distordu
   if(fluct3d) for(int l=0;l<nplanz;l++) (*rgen)(l,j,i) += Rdis(l);
   // Cross-power spectrum computation
   if(npkmax>0 && nlosread%npkmax==0) {
     fftserv.FFTForward(V,FV);
     int nf = FV.Size();
     if(pkvr.Size()<=0) {
       cout<<"...Create vector for cross-power spectrum computation sz="<<nf<<endl;
       pkr.ReSize(nf);  pkr = 0.;
       pkrc.ReSize(nf); pkrc = 0.;
       pkvr.ReSize(nf); pkvr = complex<r_8>(0.);
     }
     fftserv.FFTForward(R,FR);
     fftserv.FFTForward(Rdis,FRdis);
     for(int l=0;l<nf;l++) {
       pkr(l)  += norm(FR(l));
       pkrc(l) += norm(FRdis(l));
       pkvr(l) += FV(l)*conj(FR(l));
     }
     npkav++;
   }
   nlosread++;
 }   // fin boucle sur ny
   if(do2d && M2d.Size()>0) {
     char str[64];
     sprintf(str,"mx_%d",i);
     pos.PutObject(M2d,str);
     sprintf(str,"mxv_%d",i);
     pos.PutObject(M2dv,str);
     sprintf(str,"mxc_%d",i);
     pos.PutObject(M2dc,str);
   }
 }   // fin boucle sur nx

 cout<<"Number of processed los: "<<nlosread<<" / "<<Nx*Ny<<endl;
 if(hmpc != NULL) {
   hmpc->Show();
   if(hmpc->NEntries()>0) pos.PutObject(*hmpc,"hmpc");
 }
 if(npkav>0) {
   double nn = Dx*Dy*Dz/double(nplanz)/double(npkav); // moyenne + normalisation / hpkrec
   cout<<"Number of averaged spectra: npkav="<<npkav<<" norm="<<nn<<endl;
   pkr  *= nn;
   pkrc *= nn;
   pkvr *= nn;
   pos.PutObject(pkr,"pkr");
   pos.PutObject(pkrc,"pkrc");
   pos.PutObject(pkvr,"pkvr");
 }
 PrtTim(">>>> End filling redshift distorted cube");

 // --- Fourier transform 3D cube and compute 1D and 2D power spectra
 if(fluct3d) {
   cout<<">>> Fourier transform 3D cube and compute 1D and 2D power spectra"<<endl;
   // do the FFT for spectrum analysis
   fluct3d->ReComputeFourier();
   PrtTim(">>>> End ReComputing spectrum");

   // Compute 1D spectrum
   cout<<endl<<"\n--- Computing final 1D spectrum"<<endl;
   double dkmin = fluct3d->GetKincMin(), knyqmax = fluct3d->GetKmax();
   long nherr = long(knyqmax/dkmin+0.5);
   cout<<"\nFor HistoErr: d="<<dkmin<<" max="<<knyqmax<<" n="<<nherr<<endl;
   HistoErr hpkrec(0.,knyqmax,nherr); hpkrec.Zero();
   hpkrec.ReCenterBin(); hpkrec.Show();
   fluct3d->ComputeSpectrum(hpkrec);
   pos.PutObject(hpkrec,"hpkrec");
   PrtTim(">>>> End Computing final spectrum");

   // Compute 2D spectrum
   cout<<"\n--- Computing final 2D spectrum"<<endl;
   double dktmin = fluct3d->GetKTincMin(), ktnyqmax = fluct3d->GetKTmax();
   double dkzmin = fluct3d->GetKinc()[2], kznyqmax = fluct3d->GetKnyq()[2];
   long nherrt = long(ktnyqmax/dktmin+0.5), nherrz = long(kznyqmax/dkzmin+0.5);
   cout<<"For Histo2DErr: d="<<dktmin<<","<<dkzmin
       <<" max="<<ktnyqmax<<","<<kznyqmax<<" n="<<nherrt<<","<<nherrz<<endl;
   Histo2DErr hpkrec2(0.,ktnyqmax,nherrt,0.,kznyqmax,nherrz);
   hpkrec2.ReCenterBin(); hpkrec2.Zero(); hpkrec2.Show();
   fluct3d->ComputeSpectrum2D(hpkrec2);
   pos.PutObject(hpkrec2,"hpkrec2");
   PrtTim(">>>> End Computing final 2D spectrum");
 }

 // --- end of job, write objects in ppf
 dvllos("nlosread") = (int_4)nlosread;
 dvllos("nlospred") = (int_4)nlospred;
 dvllos("Nx") = (int_4)Nx;
 dvllos("Ny") = (int_4)Ny;
 dvllos("Nz") = (int_4)Nz;
 dvllos("Dx") = (r_8)Dx;
 dvllos("Dy") = (r_8)Dy;
 dvllos("Dz") = (r_8)Dz;
 dvllos("nplanz") = (int_4)nplanz;
 dvllos("iplanz0") = (int_4)iplanz0;
 dvllos("nplany") = (int_4)nplany;
 dvllos("nplanx") = (int_4)nplanx;
 dvllos("npkav") = (int_4)npkav;
 dvllos("dkzpk") = (r_8)dkzpk;
 pos.PutObject(dvllos,"dvllos");
 if(fluct3d != NULL) delete fluct3d;
 if(hmpc != NULL) delete hmpc;

 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH
 } catch (PException& exc) {
   cerr<<"cmvrvloscor.cc catched PException"<<exc.Msg()<<endl;
   return 77;
 } catch (std::exception& sex) {
   cerr << "cmvrvloscor.cc std::exception :" 
        << (string)typeid(sex).name() << "\n msg= " 
        << sex.what() << endl;
   return 78;
 } catch (...) {
   cerr << "cmvrvloscor.cc catched unknown (...) exception  " << endl; 
   return 79;
 }
 //----TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH-TRY-CATCH

 return 0;
}

/*
openppf cmvrvloscor.ppf

c++exec dvllos.Print();
set dkz ${dvllos.dkzpk}

disp hmpc

# cross-power spectrum
n/plot pkr.val%n*${dkz} n>0&&val>0 ! "nsta cpts logx logy"
n/plot pkrc.val%n*${dkz} n>0&&val>0 ! "nsta cpts logx logy same red"

n/plot pkvr.val%n*${dkz} n>0&&val>0 ! "nsta cpts logx logy"

# reconstructed 1D power spectrum
n/plot hpkrec.val%x x>0&&val>0 ! "nsta cpts logx logy"

# reconstructed 2D power spectrum
imag hpkrec2
addoval 0 0 0.05 0.05 "green" false
addoval 0 0 0.1 0.1 "green" false
addoval 0 0 0.25 0.25 "green" false
addoval 0 0 0.5 0.5 "green" false
x = ${hpkrec2.xmax} / 2.
addoval 0 0 $x $x "green" false
x = ${hpkrec2.ymax} / 2.
addoval 0 0 $x $x "green" false

# proj selon kT (black), selon kZ (red)
n/plot hpkrec2.val%sqrt(x*x+y*y) ! ! "nsta crossmarker3 logx"

# les matrices 2D
set n 0
disp mx_$n
newwin
disp mxc_$n

defscript tom
del m2
c++exec TMatrix<r_8> m2(hpkrec2.NBinX(),hpkrec2.NBinY()); \
for(int i=0;i<hpkrec2.NBinX();i++) \
for(int j=0;j<hpkrec2.NBinY();j++) m2(i,j) = hpkrec2(i,j); \
KeepObj(m2); cout<<"OK"<<endl;
endscript

###### Comparaison avec le spectre initial
cd /home
mkdir ok
mkdir gene
cd /home

cd /ok
openppf cmvrvloscor.ppf
tom
cd /home

cd /gene
openppf ginit3d_?_?p?_???.ppf
tom
cd /home

h/oper / /ok/m2 /gene/m2 mdiff
disp mdiff
 */
