// Module PI : Peida Interactive     PIGraphicUC 
// Trace graphiques User-Coordinates    R. Ansari  97
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include "pigraphuc.h"

//++
// Class	PIGraphicUC
// Lib		PI
// include	pigraphuc.h
//
//   	Classe fournissant les services de trac graphique 
//	en coordonnes utilisateur.
//	Toutes les mthodes de trac de "PIGraphic" sont redfinies pour 
//	accepter les arguments en systme de coordonnes utilisateur.
//--
//++
// Links	Parent
// PIGraphic
//--

// Pour echanger deux valeurs PIGrCoord
static inline void dbl_swap(double& a, double& b) { double tmp = a; a = b; b = tmp; }

//++
// Titre	Constructeurs, mthodes
//--
//++
// PIGraphicUC(PIGraphicGen* g, PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
//	Constructeur d'un objet "PIGraphUC"  partir d'un "PIGraphicGen* g". L'espace
//	de travail du nouvel objet "PIGraphic" correspond au rectangle dfini par
//	"(x0,y0)" , "dx,dy" du "PIGraphic" de dpart.
// 
// int  kind()
//	Renvoie le type ("= PI_UCGraphics")
// void SetBaseGraphic(PIGraphicGen* g, PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
//	Permet de modifier l'objet "PIGraphic" de dpart et l'espace graphique correspondant.
//
// void  SelFontSz(PIGrCoord sz, PIFontAtt att=PI_RomanFont)
//	Slection de fonte avec la taille exprime en coordonnes utilisateur
// void  SelMarkerSz(PIGrCoord msz, PIMarker mrk=PI_DotMarker)
//	Slection de marker avec la taille exprime en coordonnes utilisateur
//--

/* --Methode-- */
PIGraphicUC::PIGraphicUC(PIGraphicGen* g, PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
  : PIGraphicGen()
{
  mGrC = g;
  xW0 = x0;  yW0 = y0; 
  xWd = dx;  yWd = dy; 
  aXdir = false;  // Vrai si Axe X de Droite vers Gauche 
  aYdir = false;  // Vrai si Axe Y de Haut vers Bas
  SetUCS(0., 1., 0., 1., kAxeDirLtoR, kAxeDirDownUp);
  clip = true;  
}

/* --Methode-- */
PIGraphicUC::~PIGraphicUC()
{
}

/* --Methode-- */
int PIGraphicUC::kind() 
{
return PI_UCGraphics; 
}


/* --Methode-- */
void PIGraphicUC::SetBaseGraphic(PIGraphicGen* g, PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
{
  mGrC = g;
  xW0 = x0;  yW0 = y0; 
  xWd = dx;  yWd = dy; 
  SetUCS(xMin, xMax, yMin, yMax, kAxeDirSame, kAxeDirSame);
}

/* --Methode-- */
void PIGraphicUC::GetGrSpace(PIGrCoord& xmin, PIGrCoord& xmax, PIGrCoord& ymin, PIGrCoord& ymax)
{
xmin = xMin;  ymin = yMin;  
xmax = xMax;  ymax = yMax;
}

//++
// Titre	Systme de coordonnes utilisateur
//--
//++
// void       SetUCS(double xmin, double xmax, double ymin, double ymax, -
// int axrl=kAxeDirSame, int ayud=kAxeDirSame)
//	Dfinit le systme de coordonnes utilisateur. Axe X de "xmin"  "xmax" , 
//	Axe Y de "ymin"  "ymax". "axrl", "ayud" dfinit le sens des axes 
//	sur la fentre :
//|	kAxeDirSame : Pas de changement de sens
//|	kAxeDirLtoR : Axe X croissant de gauche  droite   
//|	kAxeDirRtoL : Axe X croissant de droite  gauche 
//|	kAxeDirDownUp : Axe Y croissant de Bas vers Haut
//|	kAxeDirUpDown : Axe Y croissant de Haut vers Bas
//|	kAxeDirAuto : Sens determine a partir de la valeur relative de xmin, xmax, ymin, ymax
//|	              xmin < xmax : kAxeDirLtoR   -  xmax < xmin : kAxeDirRtoL
//|	              ymin < ymax : kAxeDirDownUp -  ymax < ymin : kAxeDirUpDown
//	Par dfaut : "xmin=ymin=0." , "ymin=ymax=0." , "kAxeDirLtoR" , "kAxeDirDownUp"
// 
// void  UC2GrC(double x, double y, double& xpix, double& ypix)
//	Transformation de coordonnes utilisateur "(x,y)" en coordoonnes 
//	de l'espace graphic de dpart "(xpix, ypix)"
// void  GrC2UC(double xpix, double ypix, double& x, double& y)
//	Transformation de coordonnes de l'espace graphic de dpart. en coordonnes utilisateur.
// void  DUC2GrC(double dx, double dy, double& dxpix, double& dypix)
//	Transformation de distances selon X,Y en coordonnes utilisateur "dx, dy" en distance
//	dans l'espace graphique de dpart "dxpix , dypix".
// void  DGrC2UC(double dxpix, double dypix, double& dx, double& dy)
//	Transformation de distances selon X,Y dans l'espace graphique de dpart en distance
//	dans l'espace de  coordonnes utilisateur.
//--

/* --Methode-- */
void PIGraphicUC::SetUCS(double xmin, double xmax, double ymin, double ymax,
                         int axrl, int ayud)
{

  if (axrl == kAxeDirAuto) {
    double ff;
    if (xmax < xmin)  { axrl = kAxeDirRtoL;   ff=xmin;  xmin=xmax;  xmax=ff; }
    else  axrl = kAxeDirLtoR;
  }
  if (ayud == kAxeDirAuto) {
    double ff;
    if (ymax < ymin)  { ayud = kAxeDirUpDown;   ff=ymin;  ymin=ymax;  ymax=ff; }
    else  ayud = kAxeDirDownUp;
  }

  if (xmax <= xmin  || ymax <= ymin)
    return; // $CHECK$ exception ?
  if (axrl == kAxeDirLtoR)  aXdir = false;
  else if (axrl == kAxeDirRtoL)  aXdir = true;
  if (ayud == kAxeDirDownUp)  aYdir = false;
  else if (ayud == kAxeDirUpDown)  aYdir = true;
  xMin = xmin;
  xMax = xmax;
  yMin = ymin;
  yMax = ymax;

  xScale = xWd/(xMax-xMin);
  if (xScale < 1.e-19)  xScale = 1.e-19;  // Protection 
  if (aXdir)  { 
    xScale = -xScale;
    xOrg   = -xMax * xScale;
  }
  else xOrg   = -xMin * xScale;
  xOrg += xW0;

  yScale = yWd/(yMax-yMin);
  if (yScale < 1.e-19)  yScale = 1.e-19;  // Protection 
  if (!aYdir) {
    yScale = -yScale;
    yOrg   = -yMax  * yScale;
    }
  else yOrg   = -yMin * yScale;
  yOrg += yW0;
  return;
}


/* --Methode-- */
void PIGraphicUC::UC2GrC(double x, double y, double& xpix, double& ypix)
{
  xpix = (double)(xOrg + x*xScale);  
  ypix = (double)(yOrg + y*yScale);  
}

/* --Methode-- */
void PIGraphicUC::GrC2UC(double xpix, double ypix, double& x, double& y)
{
  x = (xpix-xOrg)/xScale;
  y = (ypix-yOrg)/yScale;
}

/* --Methode-- */
void PIGraphicUC::DUC2GrC(double dx, double dy, double& dxpix, double& dypix)
{
  dxpix = dx*xScale;
  if (dxpix < 0)  dxpix = -dxpix;
  dypix = dy*yScale;
  if (dypix < 0)  dypix = -dypix;
}

/* --Methode-- */
void PIGraphicUC::DGrC2UC(double dxpix, double dypix, double& dx, double& dy)
{
  dx = dxpix/xScale;
  dy = dypix/yScale;
  if (dx < 0.) dx = -dx;
  if (dy < 0.) dy = -dy;
}

/* --Methode-- */
void PIGraphicUC::Erase(PIGrCoord xc, PIGrCoord yc, PIGrCoord dxc, PIGrCoord dyc)
{
  if (!mGrC) return;
  double x, y, dx, dy;
  x = xc;  dx = dxc;
  y = yc;  dy = dyc;
  if (clip) {
    if (dx < 0) {
      x = x+dx;
      dx = -dx;
    }
    if (dy < 0) {
      y = y+dy;
      dy = -dy;
    }
    if (x > xMax || x+dx < xMin || y > yMax || y+dy < yMin) return;
    if (x < xMin) {
      dx -= (xMin-x);
      x = xMin;
    }
    if (y < yMin) {
      dy -= (yMin-y);
      y = yMin;
    }
    if (x+dx > xMax)
      dx = (xMax-x);
    if (y+dy > yMax)
      dy = (yMax-y);
  }

  double xf, yf, xf2, yf2;
  UC2GrC(x, y, xf, yf);
  UC2GrC(x+dx, y+dy, xf2, yf2);
  if (xf2 < xf)  dbl_swap(xf, xf2);
  if (yf2 < yf)  dbl_swap(yf, yf2);
  mGrC->Erase(xf, yf, xf2-xf, yf2-yf);
}

/* --Methode-- */
void PIGraphicUC::DrawString(PIGrCoord x, PIGrCoord y, const char* s, int pos)
{
  if (!mGrC) return;
  double xf, yf;
  UC2GrC(x, y, xf, yf);
  mGrC->DrawString(xf, yf, s, pos);  
}

/* --Methode-- */
void PIGraphicUC::DrawOpaqueString(PIGrCoord x, PIGrCoord y, const char* s, int pos)
{
  if (!mGrC) return;
  double xf, yf;
  UC2GrC(x, y, xf, yf);
  mGrC->DrawOpaqueString(xf, yf, s, pos);  
}

/* --Methode-- */
void PIGraphicUC::DrawLine(PIGrCoord xc1, PIGrCoord yc1, PIGrCoord xc2, PIGrCoord yc2)
{
  
  if (!mGrC) return;
  double x1, y1, x2, y2;
  x1 = xc1;  y1 = yc1;
  x2 = xc2;  y2 = yc2;
  if (clip) {
    if (x2 < xMin && x1 < xMin) return;
    if (x2 > xMax && x1 > xMax) return;
    if (y1 < yMin && y2 < yMin) return;
    if (y1 > yMax && y2 > yMax) return;
    if (x2 < x1) {
      dbl_swap(x1,x2);
      dbl_swap(y1,y2);
    }
    if (x2>xMax) {
      y2 = y1 + (y2-y1)/(x2-x1)*(xMax-x1);
      x2 = xMax;
    }
    if (x1<xMin) {
      y1 = y2 + (y1-y2)/(x1-x2)*(xMin-x2);
      x1 = xMin;
    }
    if (y2 < y1) {
      dbl_swap(x1,x2);
      dbl_swap(y1,y2);
    }
    if (y2>yMax) {
      x2 = x1 + (x2-x1)/(y2-y1)*(yMax-y1);
      y2 = yMax;
    }
    if (y1<yMin) {
      x1 = x2 + (x1-x2)/(y1-y2)*(yMin-y2);
      y1 = yMin;
    }
  }

  double xf1, yf1, xf2, yf2;
  UC2GrC(x1, y1, xf1, yf1);
  UC2GrC(x2, y2, xf2, yf2);
  mGrC->DrawLine(xf1, yf1, xf2, yf2);

}  


/* --Methode-- */
void PIGraphicUC::DrawBox(PIGrCoord xc, PIGrCoord yc, PIGrCoord dxc, PIGrCoord dyc)
{
  if (!mGrC) return;
  double x, y, dx, dy;
  x = xc;  dx = dxc;
  y = yc;  dy = dyc;
  if (clip) {
    if (dx < 0) {
      x = x+dx;
      dx = -dx;
    }
    if (dy < 0) {
      y = y+dy;
      dy = -dy;
    }
    if (x > xMax || x+dx < xMin || y > yMax || y+dy < yMin) return;
    if (x < xMin) {
      dx -= (xMin-x);
      x = xMin;
    }
    if (y < yMin) {
      dy -= (yMin-y);
      y = yMin;
    }
    if (x+dx > xMax)
      dx = (xMax-x);
    if (y+dy > yMax)
      dy = (yMax-y);
  }

  double xf, yf, xf2, yf2;
  UC2GrC(x, y, xf, yf);
  UC2GrC(x+dx, y+dy, xf2, yf2);
  if (xf2 < xf)  dbl_swap(xf, xf2);
  if (yf2 < yf)  dbl_swap(yf, yf2);
  mGrC->DrawBox(xf, yf, xf2-xf, yf2-yf);
}  

/* --Methode-- */
void PIGraphicUC::DrawFBox(PIGrCoord xc, PIGrCoord yc, PIGrCoord dxc, PIGrCoord dyc)
{
  if (!mGrC) return;
  double x, y, dx, dy;
  x = xc;  dx = dxc;
  y = yc;  dy = dyc;
  if (clip) {
    if (dx < 0) {
      x = x+dx;
      dx = -dx;
    }
    if (dy < 0) {
      y = y+dy;
      dy = -dy;
    }
    if (x > xMax || x+dx < xMin || y > yMax || y+dy < yMin) return;
    if (x < xMin) {
      dx -= (xMin-x);
      x = xMin;
    }
    if (y < yMin) {
      dy -= (yMin-y);
      y = yMin;
    }
    if (x+dx > xMax)
      dx = (xMax-x);
    if (y+dy > yMax)
      dy = (yMax-y);
  }

  double xf, yf, xf2, yf2;
  UC2GrC(x, y, xf, yf);
  UC2GrC(x+dx, y+dy, xf2, yf2);
  if (xf2 < xf)  dbl_swap(xf, xf2);
  if (yf2 < yf)  dbl_swap(yf, yf2);
  mGrC->DrawFBox(xf, yf, xf2-xf, yf2-yf);
}  

/* --Methode-- */
void PIGraphicUC::DrawCircle(PIGrCoord x, PIGrCoord y, PIGrCoord r)
{
  if (!mGrC) return;

  double xf, yf, rf,rfy;
  UC2GrC(x, y, xf, yf);
  DUC2GrC(r,r,rf,rfy);
  rf = 0.5*(rf+rfy);
  mGrC->DrawCircle(xf, yf, rf);
}  

/* --Methode-- */
void PIGraphicUC::DrawFCircle(PIGrCoord x, PIGrCoord y, PIGrCoord r)
{
  if (!mGrC) return;

  double xf, yf, rf, rfy;
  UC2GrC(x, y, xf, yf);
  DUC2GrC(r,r,rf,rfy);
  rf = 0.5*(rf+rfy);
  mGrC->DrawFCircle(xf, yf, rf);
}

/* --Methode-- */
void PIGraphicUC::DrawOval(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
{
  if (!mGrC) return;

  double xf, yf, dxf, dyf;
  UC2GrC(x0, y0, xf, yf);
  DUC2GrC(dx, dy, dxf, dyf);
  mGrC->DrawOval(xf, yf, dxf, dyf);
}

/* --Methode-- */
void PIGraphicUC::DrawFOval(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
{
  if (!mGrC) return;

  double xf, yf, dxf, dyf;
  UC2GrC(x0, y0, xf, yf);
  DUC2GrC(dx, dy, dxf, dyf);
  mGrC->DrawFOval(xf, yf, dxf, dyf);
}

#define NMXMULTP  30   // Pour multipoint sans new

/* --Methode-- */
void PIGraphicUC::DrawPolygon(PIGrCoord *x, PIGrCoord *y, int n, bool cinc)
{
  PIGrCoord xc[NMXMULTP], yc[NMXMULTP];
  PIGrCoord *pxc, *pyc;
  int i;
  double xf, yf;
 
  if (!mGrC) return;
  if (n <= 0)  return;

  if (n > NMXMULTP) { pxc = new PIGrCoord[n];  pyc = new PIGrCoord[n]; }
  else { pxc = xc;  pyc = yc; }
  for(i=0; i<n; i++) {
    UC2GrC(x[i], y[i], xf, yf);
    pxc[i] = xf;  pyc[i] = yf; 
  }
  mGrC->DrawPolygon(pxc, pyc, n, cinc);
  if (n > NMXMULTP) { delete[] pxc;  delete[] pyc; }
  return;
}
  
/* --Methode-- */
void PIGraphicUC::DrawFPolygon(PIGrCoord *x, PIGrCoord *y, int n, bool cinc)
{
  PIGrCoord xc[NMXMULTP], yc[NMXMULTP];
  PIGrCoord *pxc, *pyc;
  int i;
  double xf, yf;
 
  if (!mGrC) return;
  if (n <= 0)  return;

  if (n > NMXMULTP) { pxc = new PIGrCoord[n];  pyc = new PIGrCoord[n]; }
  else { pxc = xc;  pyc = yc; }
  for(i=0; i<n; i++) {
    UC2GrC(x[i], y[i], xf, yf);
    pxc[i] = xf;  pyc[i] = yf; 
  }
  mGrC->DrawFPolygon(pxc, pyc, n, cinc);
  if (n > NMXMULTP) { delete[] pxc;  delete[] pyc; }
  return;
}


/* --Methode-- */
void PIGraphicUC::DrawArc(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy,
                          double degdeb, double degfin)
{
  if (!mGrC) return;

  double xf, yf, dxf, dyf;
  UC2GrC(x0, y0, xf, yf);
  DUC2GrC(dx, dy, dxf, dyf);
  mGrC->DrawArc(xf, yf, dxf, dyf, degdeb, degfin);
}
/* --Methode-- */
void PIGraphicUC::DrawFArc(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy,
                           double degdeb, double degfin)
{
  if (!mGrC) return;

  double xf, yf, dxf, dyf;
  UC2GrC(x0, y0, xf, yf);
  DUC2GrC(dx, dy, dxf, dyf);
  mGrC->DrawFArc(xf, yf, dxf, dyf, degdeb, degfin);
}

/* --Methode-- */
void PIGraphicUC::DrawMarker(PIGrCoord x0, PIGrCoord y0)
{
  if (!mGrC) return;
  double xf, yf;
  UC2GrC(x0, y0, xf, yf);
  mGrC->DrawMarker(xf, yf);
  return;
}

/* --Methode-- */
void PIGraphicUC::DrawMarkers(PIGrCoord *x, PIGrCoord *y, int n)
{
  PIGrCoord xc[NMXMULTP], yc[NMXMULTP];
  PIGrCoord *pxc, *pyc;
  int i;
  double xf, yf;
 
  if (!mGrC) return;
  if (n <= 0)  return;

  if (n > NMXMULTP) { pxc = new PIGrCoord[n];  pyc = new PIGrCoord[n]; }
  else { pxc = xc;  pyc = yc; }
  for(i=0; i<n; i++) {
    UC2GrC(x[i], y[i], xf, yf);
    pxc[i] = xf;  pyc[i] = yf; 
  }
  mGrC->DrawMarkers(pxc, pyc, n);
  if (n > NMXMULTP) { delete[] pxc;  delete[] pyc; }
  return;
}

/* --Methode-- */
void PIGraphicUC::DrawPixmap(PIGrCoord x, PIGrCoord y, unsigned char *pix,
                        int sx, int sy, PIColorMap* cmap)
{
  if (!mGrC) return;
  if ( (sx < 1) || (sy < 1) )  return;
  if ((pix == NULL) || (cmap == NULL)) return; 
  double xf,yf;
  UC2GrC(x, y, xf, yf);
  mGrC->DrawPixmap(xf, yf, pix, sx, sy, cmap);
}



/* --Methode-- */
void PIGraphicUC::SelForeground(PIColors col)
{
  if (!mGrC) return;
  mGrC->SelForeground(col);
  return;  
}


/* --Methode-- */
void PIGraphicUC::SelBackground(PIColors col)
{
  if (!mGrC) return;
  mGrC->SelBackground(col);
  return;  
}

/* --Methode-- */
void PIGraphicUC::SelForeground(PIColorMap& cmap, int cid)
{
  if (!mGrC) return;
  mGrC->SelForeground(cmap, cid);
  return;  
}


/* --Methode-- */
void PIGraphicUC::SelBackground(PIColorMap& cmap, int cid)
{
  if (!mGrC) return;
  mGrC->SelBackground(cmap, cid);
  return;  
}

/* --Methode-- */
void PIGraphicUC::SelGOMode(PIGOMode mod)
{
  if (!mGrC) return;
  mGrC->SelGOMode(mod);
  return;  
}

/* --Methode-- */
void PIGraphicUC::SelFont(PIFont & fnt)
{
  if (!mGrC) return;
  mGrC->SelFont(fnt);
  myFont = mGrC->GetFont();
  return;  
}

/* --Methode-- */
void PIGraphicUC::SelFont(PIFontSize sz, PIFontAtt att)
{
  if (!mGrC) return;
  mGrC->SelFont(sz, att);
  myFont = mGrC->GetFont();
  return;  
}

/* --Methode-- */
void PIGraphicUC::SelFontSzPt(int npt, PIFontAtt att)
{
  if (!mGrC) return;
  mGrC->SelFontSzPt(npt, att);
  myFont = mGrC->GetFont();
  return;  
}


/* --Methode-- */
void PIGraphicUC::SelLine(PILineAtt att)
{
  if (!mGrC) return;
  mGrC->SelLine(att);
  return;  
}

/* --Methode-- */
void PIGraphicUC::SelMarker(int msz, PIMarker mrk)
{
  if (!mGrC) return;
  mGrC->SelMarker( msz, mrk);
}

/* --Methode-- */
void PIGraphicUC::SelFontSz(PIGrCoord sz, PIFontAtt att)
{
  if (!mGrC) return;
  double fx, fy;
  DUC2GrC((double)sz, (double)sz, fx, fy);
  int npt = (int)fy;
  if (npt < 6) npt = 6;
  if (npt > 127) npt = 127;
  mGrC->SelFontSzPt(npt, att);
  myFont = mGrC->GetFont();
}


/* --Methode-- */
void PIGraphicUC::SelMarkerSz(PIGrCoord sz, PIMarker mrk)
{
  if (!mGrC) return;
  double fx, fy;
  DUC2GrC((double)sz, (double)sz, fx, fy);
  int isz = int(0.5*(fx+fy+1));
  mGrC->SelMarker( isz, mrk);
}

/* --Methode-- */
void PIGraphicUC::SetClipRectangle(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)
{
  /* $CHECK$  A faire Reza 08/02/97  */
return;
}
/* --Methode-- */
void PIGraphicUC::ClearClipRectangle()
{
  /* $CHECK$  A faire Reza 08/02/97  */
return;
}

/* --Methode-- */
PIColors PIGraphicUC::GetForeground()
{
  if (!mGrC) return(PI_NotDefColor);
  return(mGrC->GetForeground());
}

/* --Methode-- */
PIColors PIGraphicUC::GetBackground()
{
  if (!mGrC) return(PI_NotDefColor);
  return(mGrC->GetBackground());
}

/* --Methode-- */
PIGOMode PIGraphicUC::GetGOMode()
{
  if (!mGrC) return(PI_GOCopy);
  return(mGrC->GetGOMode());
}


/* --Methode-- */
PILineAtt PIGraphicUC::GetLineAtt()
{
  if (!mGrC) return(PI_NotDefLineAtt);
  return(mGrC->GetLineAtt());
}

/* --Methode-- */
PIMarker PIGraphicUC::GetMarker()
{
  if (!mGrC) return(PI_NotDefMarker);
  return(mGrC->GetMarker());
}

/* --Methode-- */
int PIGraphicUC::GetMarkerSize()
{
  if (!mGrC) return(0);
  return(mGrC->GetMarkerSize());
}
/* --Methode-- */
PIGrCoord PIGraphicUC::GetFontHeight(PIGrCoord& asc, PIGrCoord& desc)
{
  asc = desc = 0;
  if (!mGrC) return(0);
  int h,a,d;
  double fh, fa, fd, x;
  h = myFont.GetFontHeight(a, d) ;
  DGrC2UC(0., h, x, fh);
  DGrC2UC(0., a, x, fa);
  DGrC2UC(0., d, x, fd);
  asc = a;  desc = d;
  return(fh);
}


/* --Methode-- */
PIGrCoord PIGraphicUC::CalcStringWidth(char const* s)
{
  if (!mGrC) return(0.);
  double dx = myFont.GetStringWidth(s);
  double dxx, dxy;
  DGrC2UC(dx, dx, dxx, dxy);
  return(dxx);
}


/* --Methode-- */
void PIGraphicUC::SaveGraphicAtt()
{
  if (!mGrC) return;
  mGrC->SaveGraphicAtt();
}

/* --Methode-- */
void PIGraphicUC::RestoreGraphicAtt()
{
  if (!mGrC) return;
  mGrC->RestoreGraphicAtt();
}

