#include <stdlib.h>
#include <string.h>

#include "branap.h"
#include "minifits.h"
#include "strutilxx.h"
#include "sopnamsp.h"

//--------------------------------------------------------------
// Projet BAORadio - (C) LAL/IRFU  2008-2010
// Classe de gestion des parametres programmes d'analyse 
//--------------------------------------------------------------

/* --Methode-- */
BRAnaParam::BRAnaParam(uint_4 nmean, uint_4 nzon, uint_4 npaqz)
{
  outpath_="./";
  nmean_=nmean;
  nbloc_=1;
  imin_=imax_=0;
  istep_=1;
  rdsamefc_=true;   // read paquets with same frame counter
  freqmin_=freqmax_=0;
  nbinfreq_=1;  
  paqsize_=16424;
  nzones_=nzon;
  npaqinzone_=npaqz;
  fgdatafft_=false;  fgsinglechannel_=false;
  prtlevel_=0;
  prtmodulo_=50000;
  nbcalgrp_=1;
  nthreads_=1;

  vmin_=0.;  vmax_=9e99;
  gainfile_="";
}

/* --Methode-- */
int BRAnaParam::DecodeArgs(int narg, char* arg[])
{
  if ((narg>1)&&(strcmp(arg[1],"-h")==0))  return Usage(false);
  if (narg<5) return Usage(true);

  bool okarg=false;
  int ka=1;
  while (ka<(narg-1)) {
    if (strcmp(arg[ka],"-act")==0) {
      action_=arg[ka+1];
      ka+=2;
    }
    else if (strcmp(arg[ka],"-out")==0) {
      outpath_=arg[ka+1];
      size_t lenp=outpath_.size();
      if ((lenp>0)&&(outpath_[lenp-1]!='/'))  outpath_+='/';
      ka+=2;
    }
    else if (strcmp(arg[ka],"-nmean")==0) {
      nmean_=atoi(arg[ka+1]);
      ka+=2;
    }
    else if (strcmp(arg[ka],"-nbloc")==0) {
      nbloc_=atoi(arg[ka+1]);
      ka+=2;
    }
    else if (strcmp(arg[ka],"-freq")==0) {
      sscanf(arg[ka+1],"%d,%d,%d",&freqmin_,&freqmax_,&nbinfreq_);  
      ka+=2;
    }
    else if (strcmp(arg[ka],"-zones")==0) {
      int nzon=4;
      int npaqz=128;
      sscanf(arg[ka+1],"%d,%d",&nzon,&npaqz);
      nzones_=nzon;  npaqinzone_=npaqz;
      ka+=2;
    }
    else if (strcmp(arg[ka],"-prt")==0) {
      sscanf(arg[ka+1],"%d,%ld",&prtlevel_,&prtmodulo_);
      ka+=2;
    }
    else if (strcmp(arg[ka],"-nthr")==0) {
      nthreads_=atoi(arg[ka+1]);
      ka+=2;
    }
    else if (strcmp(arg[ka],"-nvcal")==0) {
      nbcalgrp_=atoi(arg[ka+1]);
      ka+=2;
    }
    else if (strcmp(arg[ka],"-nosfc")==0) {
      rdsamefc_=false;
      ka++;
    }
    else if (strcmp(arg[ka],"-varcut")==0) {
      sscanf(arg[ka+1],"%lg,%lg",&vmin_,&vmax_);
      ka+=2;
    }
    else if (strcmp(arg[ka],"-gain")==0) {
      gainfile_=arg[ka+1];
      ka+=2;
    }
    else if (strcmp(arg[ka],"-in")==0) {
      if ((narg-ka)<4)  {
	cout << " BRAnaParam::DecodeArgs() / Argument error " << endl;
	return Usage(true);
      }
      sscanf(arg[ka+1],"%d,%d,%d",&imin_,&imax_,&istep_);  ka+=2;
      while(ka<(narg-1)) {
	string inpath = arg[ka]; 
	size_t lenp=inpath.size();
	if (lenp<1)  inpath="./";
	if ((lenp>0)&&(inpath[lenp-1]!='/'))  inpath+='/';
	vector<string> fiblist;
	string sa1 = arg[ka+1]; 
	FillVStringFrString(sa1, fiblist, ',');
	char dbuff[32];
	for(size_t i=0; i<fiblist.size(); i++) {
	  sprintf(dbuff,"Fiber%d/",(int)atoi(fiblist[i].c_str()));
	  dirlist_.push_back(inpath+dbuff);
	}
	ka += 2;
      }
      okarg=true;
    }
    else ka++;
  }

  if (!okarg) {
    cout << " BRAnaParam::DecodeArgs() / Argument error " << endl;
    return Usage(true);
  }
  return 0;
}

/* --Methode-- */
int BRAnaParam::Usage(bool fgshort)
{
  cout << " --- BRAnaParam : Reading/Processing BAORadio FITS files parameters " << endl;
  cout << " Usage:  prgname [-act ACT] [-out OutPath] [-nmean NMean] [-zones NZones,nPaqinZone] \n" 
       << "                 [-nbloc NBloc] [-freq NumFreqMin,NumFreqMax,NBinFreq] \n" 
       << "                 [-prt lev,modulo] [-nvcal n] [-nthr n] [-nosfc] \n" 
       << "                 [-gain filename] [-varcut min,max] \n" 
       << "         -in Imin,Imax,Istep InPath FiberList [InPath2 FiberList2 InPath3 FiberList3 ...] \n" << endl; 
  if (fgshort) {
    cout << " prgname -h for detailed instructions" << endl;
    return 5;
  }
  cout << " -act Action: cube3d or vis or viscktt or mspec \n"
       << "    cube3d: create 3D fits cubes, vis: compute visibilites,  \n" 
       << "    viscktt: compute visibilities and check TimeTag/FrameCounter\n "
       << "    mspec: compute and save mean spectra for each channel \n "
       << " -out OutPath: Output directory name \n"
       << " -nmean NMean: Number of packet used for spectra/visibility computation \n"
       << " -nbloc NBloc: Number of MemMgr blocs in output file\n"
       << " -zones NZones,NbPaqinZone : Number of Zones and number of paquets in one zone \n" 
       << " -freq NumFreqMin,NumFreqMax,NBinFreq : Frequency zone and number of bins \n" 
       << " -prt lev,modulo : Print level (0,1,2...) and print counter modulo (10000, 50000 ...) \n" 
       << " -nvcal n : number of BRVisibilityCalculator objects running in parallel in BRVisCalcGroup  (default=1) \n" 
       << " -nthr n : number of threads for parallel execution in BRVisibilityCalculator (default=1) \n" 
       << " -nosfc : Don't force reading with SAME FrameCounter \n" 
       << " -varcut min,max : min-max cut on variance \n" 
       << " -gain filename : spectral response fits file name \n" 
       << " -in : input files/directory definition : \n"
       << "       Imin,Imax,Istep: fits files signalII.fits Imin<=II<=Imax Istep=increment \n" 
       << "       InPath: Input directerory fits files in InPath/FiberJJ directory \n"
       << "       FiberList: List of fiber numbers (JJ - Ex: 2,1,4 )  \n" << endl;
  return 1;
}

/* --Fonction-- */
int BRAnaParam::PaqSizeFromFits()
{
  uint_4 paqsz, npaqf;
  char flnm[1024];
  sprintf(flnm,"%s/signal%d.fits",dirlist_[0].c_str(),imin_);
  return DecodeMiniFitsHeader(flnm,paqsize_, npaqinfile_,fgdatafft_, fgsinglechannel_); 
}

/* --Fonction-- */
ostream& BRAnaParam::Print(ostream& os)
{
  os << " BRAnaParam::Print() dirlist_.size()=" << dirlist_.size() << " Input directories: " << endl;
  for(size_t k=0; k< dirlist_.size(); k++) 
    cout << k+1 << " :  " << dirlist_[k] << endl;
  cout << " IMin= " << imin_ << " IMax= " << imax_ << " IStep= " << istep_ 
       << ((rdsamefc_)?" SameFrameCounter read mode":" AllOKPaquets read mode ") << endl;
  cout << " OutPath= " << outpath_ << endl;
  cout << " Action=" << action_ << "  NMean=" << nmean_ << " NBloc=" << nbloc_ << endl;
  cout << " FreqMin= " << freqmin_ << " FreqMax= " << freqmax_ << " NBinFreq= " << nbinfreq_ << endl;
  cout << " GainFileName=" << gainfile_ << " VarianceMin= " << vmin_ << " VarianceMax= " << vmax_ << endl;
  cout << " PaqSize=" << paqsize_ << "  -  NZones=" << nzones_ << " NPaqZone=" << npaqinzone_ 
       << " PrtLevel=" << prtlevel_ << " PrtCntModulo=" <<  prtmodulo_ << endl;
  cout << " AcqMode: " << ((fgdatafft_)?" Data_FFT " : " Data_Raw " ) 
       << ((fgsinglechannel_)?" SingleChannel " : " TwoChannels " ) << endl;
  cout << " NbVisibCalculator in Group: " <<  nbcalgrp_ << " with N//threads: " << nthreads_ << endl;


  return os;
}

/* --Fonction-- */
int BRAnaParam::DecodeMiniFitsHeader(const char* filename, uint_4& paqsz, uint_4& npaq, 
                                     bool& fgdatafft, bool& fgsinglechannel) 
{
  cout << " DecodeMiniFitsHeader - Opening file: " << filename << endl;
  MiniFITSFile mff(filename, MF_Read);
  string acqmode=mff.GetKeyValue("ACQMODE");
  cout << "DecodeMiniFitsHeader()... Type=" << mff.DataTypeToString() << " NAxis1=" << mff.NAxis1() 
       << " NAxis2=" << mff.NAxis2() << " AcqMode=" << acqmode << endl;
  paqsz = mff.NAxis1();
  npaq = mff.NAxis2();
  if (acqmode.substr(0,3)=="fft") fgdatafft=true;
  if (acqmode.find("1c") < acqmode.length()) fgsinglechannel=true;
  return 0;
}

