#ifndef BRPARAM_H
#define BRPARAM_H

//---------------------------------------------------------------
// Logiciels d'acquisition BAORadio 
//  LAL -   2009 - 2010  ,  R. Ansari, M.Taurigna   
//---------------------------------------------------------------

//---------------------------------------------------------------
//  Classes ADCBoardDesc , BRParList , BRConfList , BRAcqConfig 
//  + ADCBoardDesc : Description d'une carte ADC
//  + BRParList : Liste de parametres pour un run d'acquisition 
//  + BRConfList : Liste des parametres caracterisant une configuration 
//       (nombre de cartes, versions firwares ...) 
//  + BRAcqConfig : Classe donnant acces a un objet global BRParList et 
//         un objet global BRConfList
//---------------------------------------------------------------

#include "machdefs.h"
#include <string>
#include <vector>
#include <iostream>
#include "brpaqu.h"
#include "datacards.h"
#include "bracqvers.h"

using namespace std;
using namespace SOPHYA;

//-----------------------------------------------------------------------------
//---- Classe Caracterisant un board ADC 
class ADCBoardDesc {
public:
  ADCBoardDesc(int id, string sbid, string cyc_firmw, string str1_firmw, string str2_firmw);
  ADCBoardDesc(ADCBoardDesc const& bdes);

  inline int& getId() { return id_; } 
  inline string& getSId() { return sbid_; }
  inline string& CycloneFirmwareId() { return cyclone_firmware_; }
  inline string& Stratix1FirmwareId() { return stratix1_firmware_; }
  inline string& Stratix2FirmwareId() { return stratix2_firmware_; }
  ostream& Print(ostream& os);

  ADCBoardDesc& Set(ADCBoardDesc const & bdes);
  inline ADCBoardDesc& operator = (ADCBoardDesc const & bdes) { return Set(bdes); }
protected:
  int id_;
  string sbid_;
  string cyclone_firmware_;
  string stratix1_firmware_;
  string stratix2_firmware_;  
};

//-----------------------------------------------------------------------------
//---- Classe de liste des parametres de configuration BAORadio
class BRConfList {
public:
  BRConfList();
  BRConfList(string basedir);
  BRConfList(string basedir, vector<ADCBoardDesc> boards);
  BRConfList(BRConfList const & cf);

  void SetBaseDirectory(string basedir);
  void Set(BRConfList const & cf);
  ostream& Print(ostream& os) ;
  void ReadDCFile(string file);
  inline const string& BaseDirectory() const { return basedir_; }
//.......................................
  string basedir_;
  vector<ADCBoardDesc> boards_;
};


//-----------------------------------------------------------------------------
//---- Classe de liste des parametres d'acquisition BAORadio  
class BRParList {
public: 
  BRParList (string acqmode="fft2c", string fibres="1", unsigned int paqsz= 16424, 
	     unsigned int dmaszkb = 32, int nfiles=1, int nblocfile=10, string outpath="TstAcq", 
	     uint_4 nzon = 4, uint_4 npaqzon = 128);

  BRParList(BRParList const & p);

  void Set(BRParList const & p);
  ostream& Print(ostream& os) ;
  void ReadDCFile(string file);

  // Nombre de fibres 
  inline size_t NFibers() { return FiberNum.size(); } 
  inline size_t NbFibers() { return FiberNum.size(); } 
  // Taille des operations DMA (en octets) 
  inline uint_4 DMASizeBytes() { return dmasizekb*1024; }
  // Taille des operations DMA (en KB=1024 bytes) 
  inline uint_4 DMASizeKB() { return dmasizekb; }
  // Taille des paquets recus sur PCIExpress 
  inline uint_4 RecvPaquetSize() { return PaqSize; }
  // Taille des paquets apres reduction eventuelle (donc ds memory manager)
  inline uint_4 MMgrPaquetSize() { return (fgreducpsize)?redpqsize:PaqSize; }
  // Nombre de paquets dans chaque bloc de MMgr 
  inline uint_4 MMgrNbPaquet() { return nPaqZone; }
  // Operation de conversion/mise en ordre des donnees a la reception 
  inline BRDataFmtConv GetDataConvFg() { return swapall; };

  // Nombre maximum de fichiers enregistres 
  inline uint_4 MaxNbFiles() { return NbFiles; }
  // Nombre de blocs MMgr dans chaque fichiers 
  inline uint_4 BlocPerFile() { return NBlocPerFile; }
  // Nombre maximum de blocs MMgr traites 
  inline uint_8 MaxNbBlocs() { return (uint_8)NbFiles*(uint_8)NBlocPerFile; }
  // Nombre maximum paquets traites 
  inline uint_8 MaxNbPaquets() { return MaxNbBlocs()*(uint_8)nPaqZone; }

  // Taille des paquets recus sur PCIExpress 
  inline uint_4 PatternSize() { return ((PaqSize-(BRHDRSIZE+BRTRLSIZE))/4); }

  // Liste des machines cibles pour transfert sur ethernet
  inline void SetEthTargets(vector<string>& targs) { eths_targets=targs; return; }
  inline vector<string>& GetEthTargets() { return eths_targets; }
  // Nombre de liens ethernet comme source de donnees (paquets) 
  inline int NbEthLinks() { return ethr_nlink; }

  // Identification source ciel 
  inline string& SkySource() { return skysource; }

  //..........................................
  void Decode();

  string AcqMode;  // Mode d'acquisition (conditionne la mise en ordre/conversion des donnees lues par DMA 
  bool fgdatafft;  // true -> donnee FFT (provenant du firmware FFT)
  bool fgsinglechannel; // true -> un seul canal par fibre (par defaut=2 canaux/fibres)

  string FiberListS ;    // String_liste des fibres a utiliser 1,3,4 par exemple
  vector<int> FiberNum;  // liste des fibres a utiliser {1,3,4} par exemple
  string FiberIdsS ;     // String_liste des numeros 'absolu' des fibres 12,13,14,15 par exemple
  vector<int> FiberIds;  // liste des numero d'identification 'absolu' des fibres {12,13,14,15} par ex.

  string OutPathName;  // directory de base
  string ProcPathName; 
  uint_4 PaqSize ; // taille des paquets ou sizeFrame 

  // Controle/gestion  du DMA 
  uint_4 dmasizekb ; // taille du dma en multiple de 2 (2 a 56)
  unsigned int maxkwedma_;   // Nombre d'iterations de la boucle d'attente de fin de DMA (en unite de 1000)
  unsigned int nretrydma_;   // nombre maxi de tentatives pour terminer le DMA 
  unsigned int first_maxkwedma_;   // Nombre d'iterations de la boucle d'attente de fin du PREMIER DMA (u 1000)
  unsigned int first_nretrydma_;   // nombre maxi de tentatives pour terminer le PREMIER DMA 

  int NbFiles; // nombre de fichier produits
  int NBlocPerFile; // nombre de bloc par fichier
  // Gestionnaire zones memoire 
  uint_4 nZones;  // Nombre de zones memoires
  uint_4 nPaqZone;  // 128 Paquets / zone memoire - valeur par defaut

  bool fg_hard_ctrlc ; // autorise le CtrlC si true

  bool savesigfits ;   // si true on prduit de sfichier fits
  bool fgnulldev4fits ; 

  BRDataFmtConv swapall ; // type de l'acquisition voir brpaqu.h

  // ---- Pour gerer la reduction des tailles de paquets 
  bool fgreducpsize;   // true -> reduction taille des paquets 
  bool reducneedcopy;   // true -> besoin de copie intermediaire pour la reduction de taille de paquets
  BRPaqReducAction pqreducmode; 
  uint_4 redpqsize; 
  uint_4 reducoffset; 

  // ---- Variables specifiques em mode transfert DMA vers Ethernet 
  int tcpportid;     // No de port TCP/IP
  // Cote Send DMA -> Ethernet 
  bool pci2eth_fgdirect;   // true -> direct transfer DMA to Ethernet  
  vector< string > eths_targets;   // Liste des machines cibles pour les transferts DMA -> ethernet 
  // Cote reception 
  int ethr_nlink;    // Nombre total de source d'envoi (= nb total de fibre de fibre) 
  bool ethr_forcesamefc_;   // true -> on force receptions de paquets avec SAME FrameCounter sur tous les liens 
  uint_4 ethr_sfc_maxdpc_;   // valeur maximum de difference tolere entre compteurs de paquets de differentes fibres 
  uint_4 ethr_sfc_maxresync_;   // Nombre maximum de tentative de resynchronisation avant echec
  bool ethr_waitendmsg_;   // true -> EthernetReader attend le message END avant de s'arreter 

  // Identification source observee dans le ciel
  string skysource;

  //  Variables pour le traitement/monitoring 
  bool fgdoProc;    // false -> pas de thread de monitoring 
  uint_4 nmeanProc;  // nombre de paquets moyennes par le thread de traitement/monitoring 
  uint_4 stepProc;  // 1/stepProc zones traite - si =0 -> fraction de paquets traites depend de la puissance disponible
  uint_4 nmaxProc;  // Nombre de blocs traites par le thread de traitement/monitoring (0-> MaxNbBlocs())
 
  //  Variables pour le calcul de visibilites
  bool fgdoVisiC;           // true -> calcul des visibilites 
  uint_4 nmeanVisiC;        // nombre de paquets moyennes pour les calculs de visibilites 
  uint_4 nbcalgrpVisiC;     // Nb d'objets/threads dans BRVisCalcGroup 
  uint_4 nthrVisiC;         // Nb de threads pour l'execution parallele ds BRVisibilityCalculator
  uint_4 freqminVisiC, freqmaxVisiC, nbinfreqVisiC;  // si zone de frequence/rebinning pour visibilites

   
  // Pour des tests de performances ou sans carte ADC (pattern) 
  bool monothr ;           // pour les test mono thread 
  bool activate_pattern;  // true -> on active le pattern du firmware au lieu de la fibre

  double nopciLossRate;   // Utilise avec pour les tests sans carte pciexpress TestPCIWrapperNODMA

  // Niveau d'impression/debug ...
  int prtlevel_;
  long int prtmodulo_; 
};

//-----------------------------------------------------------------------------
//---- Classe de gestion des parametres d'acquisition - 
//---- Utilise l'instance global de BRParList
//-----------------------------------------------------------------------------

class BRAcqConfig {
 public:  
  
  BRAcqConfig();

  inline void ReadParamFile(string file) 
    { param_->ReadDCFile(file); }
  inline void Set(BRParList& par)
    { param_->Set(par); }
  inline void ReadConfigFile(string file) 
    { config_->ReadDCFile(file); }
  inline void Set(BRConfList& conf)
    { config_->Set(conf); }

  ostream& Print(ostream& os);

  int CreateOutputDirectories();

  static inline double AcqVersion()  { return BAOR_ACQ_VER ; }
  static inline const char* AcqVersionS() { return BAOR_ACQ_VER_STR ; } 

  inline BRParList&  GetParams() { return (*param_) ;};
  inline BRConfList&  GetConfig() { return (*config_) ;};
  
  inline BRDataFmtConv GetSwapall() { return param_->swapall; };
  inline bool GetPattern() { return param_->activate_pattern; };
  inline bool GetMonothr() { return param_->monothr; };
  inline bool GetSaveFits() { return param_->savesigfits; };
  inline bool GetFileDevNull() { return param_->fgnulldev4fits; };

  // Nombre de fibres 
  inline int  NFibers() { return param_->NbFibers(); } 
  inline int  NbFibers() { return param_->NbFibers(); } 

  // Taille des operations DMA (en octets) 
  inline uint_4 DMASizeBytes() { return param_->DMASizeBytes(); }
  // Taille des operations DMA (en KB=1024 bytes) 
  inline uint_4 DMASizeKB() { return param_->DMASizeKB(); }
  // Taille des paquets recus sur PCIExpress 
  inline uint_4 RecvPaquetSize() { return param_->RecvPaquetSize();  }
  // Taille des paquets apres reduction eventuelle (donc ds memory manager)
  inline uint_4 MMgrPaquetSize() { return param_->MMgrPaquetSize(); }
  // Nombre de paquets dans chaque bloc de MMgr 
  inline uint_4 MMgrNbPaquet() { return param_->MMgrNbPaquet(); }
  // Operation de conversion/mise en ordre des donnees a la reception 
  inline BRDataFmtConv GetDataConvFg() { return param_->GetDataConvFg(); };

  // Nombre maximum de fichiers enregistres 
  inline uint_4 MaxNbFiles() { return param_->MaxNbFiles(); }
  // Nombre de blocs MMgr dans chaque fichiers 
  inline uint_4 BlocPerFile() { return param_->BlocPerFile(); }
  // Nombre maximum de blocs MMgr traites 
  inline uint_8 MaxNbBlocs() { return param_->MaxNbBlocs(); }
  // Nombre maximum paquets traites 
  inline uint_8 MaxNbPaquets() { return param_->MaxNbPaquets(); }

  // Identification source ciel 
  inline string& SkySource() { return param_->SkySource(); }

  // repertoire d'acquisition 
  inline string OutputDirectory() { return (config_->BaseDirectory()+param_->OutPathName); }
  string OutputDirectoryFib(int fib);

 protected:
  static BRParList* param_;
  static BRConfList* config_;
};


#endif
