#ifndef  RACQUMEM_H_SEEN
#define  RACQUMEM_H_SEEN

//----------------------------------------------------------------
// ---- classes de gestion memoire pour acquisition BAORadio -----
// LAL -      R. Ansari - Juin/Juillet 2008 
//       Modifs Mai 2009 
//       - Introduction du numero d'ordre de remplissage 
//       - Introduction de mask de recyclage et sequence de traitement procA/B/C
//----------------------------------------------------------------


#include "zthread.h"
#include <string>
#include <vector>
#include <iostream>
#include "brtypes.h"
#include "timestamp.h"

using namespace std;

//-----------------------------------------------------------------------
// Projet BAORadio - (C) LAL/IRFU  2008-2010
//  Structures de gestion memoire et synchronisation inter-threads
//-----------------------------------------------------------------------

// Les flags (bits) correspondant aux operations effectuees sur une zone
// + Free : libre, 
// + Filled : rempli (par DMA ou lecture fits)
// + Saved : ecrit sur disque
// + Proc : traite (utilise principalement pour le monitoring durant l'acquisition)
// + ProcA/B/C/D/E/F : Un maximum de SIX traitements qui peuvent s'enchainer 
// + ProcSaved : ecrit sur disque apres traitement

enum MemZStatus { MemZS_Free=0, MemZS_Filled=1, MemZS_Saved=2, MemZS_Proc=4, 
                  MemZS_ProcA=8, MemZS_ProcB=16, MemZS_ProcC=32,
                  MemZS_ProcD=64, MemZS_ProcE=128, MemZS_ProcF=256,
                  MemZS_SavedProc=16384 };

// L'action en cours sur une zone (associe aux flags MemZStatus ci-dessus)
enum MemZaction { MemZA_None=0, MemZA_Fill=1, MemZA_Save=2, MemZA_Proc=4, 
                  MemZA_ProcA=8, MemZA_ProcB=16, MemZA_ProcC=32,
                  MemZA_ProcD=64, MemZA_ProcE=128, MemZA_ProcF=256,
                  MemZA_SaveProc=16384 };

// configuration du RAcqMemZoneMgr 
enum MemZconfig { MemZConf_SingleFiberDataOnly=0, MemZConf_MultiFiberDataOnly=1,
                  MemZConf_SingleFiberProcDataOnly=2, MemZConf_MultiFiberProcDataOnly=3,
                  MemZConf_SingleFiberDataAndProcData=4, MemZConf_MultiFiberDataAndProcData=5 };
// L'etat d'execution en cours : running, waiting, stopped
enum MemZRunState { MemZR_Running=0, MemZR_Waiting=1, MemZR_Stopped=2};


// --------------------------------------------------------------------
// Structure de gestion utilisee par la classe RAcqMemMgr
#define MXNACTMZM  10
typedef struct {
  uint_8 serial;    // Numero de serie permettant d'ordonner en temps les zones memoire rempli
  uint_4 act;       // Indique l'action en cours sur une zone Fill,Save ...
  uint_4 stat;      // Champ de flag (bits) indiquant les operations effectuees sur la zone
  uint_4 nbact[MXNACTMZM];  // compteurs nb de fois ou (fill,save,proc,procA/B/C) a ete effectuee sur la zone 
//  int tid;        Pas utilise pour le moment, permet l'identification du thread operant sur la zone
} St_MemZ;


// ---------------------------------------------------------
//  Classe pour donnees auxiliaires associees a chaque zone
// ---------------------------------------------------------
class RAcqMZAuxData {
public:
  RAcqMZAuxData() { }
  inline SOPHYA::TimeStamp& FillTime() { return filltime_; }

  SOPHYA::TimeStamp filltime_;
};

// -------------------------------------------------------------------
// Classe de gestion de zones memoire partagee (des buffers) entre
// threads - qui doivent operer successivement sur les vecteurs
// -------------------------------------------------------------------
class RAcqMemZoneMgr {
public:
// Constructeur, nz=Nb de zones, np=Nb de paquets/zone , psz=taille de paquet
//     Si procpsz>0 , on alloue aussi des zones memoire pour donnees traitees
//      dans ce cas, procpsz  la taille memoire des donnees traites pour chaque paquet 
// Constructeur 1 : l'objet ne contient que les donnees d'une fibre 
//                  garde pour maintenir la compatibilite avec version TAcq v <= 6.0 
  RAcqMemZoneMgr(uint_4 nz, uint_4 np, uint_4 psz);
// Constructeur 2 : l'objet peut contenir plusieurs fibres, et des donnees processes en plus
  RAcqMemZoneMgr(uint_4 nzon, uint_4 nfib, uint_4 npaq, uint_4 paqsz, uint_4 procsz=0);

  ~RAcqMemZoneMgr();
  
  // Retourne le nombre de zones memoire
  inline uint_4 NbZones() { return nzones; }
  // Retourne le nombre de fibres dans chaque zone
  inline uint_4 NbFibres() { return nfibres; }
  // Retourne la taille de paquet
  inline uint_4 PaqSize() { return paqsz; }
  // Retourne le nb de paquets dans une zone
  inline uint_4 NbPaquets() { return npaq; }
  // Retourne la taille totale d'une zone 
  inline uint_4 ZoneSize() { return nfibres*npaq*paqsz; }
  inline uint_4 OneFiberZoneSize() { return npaq*paqsz; }

  // Retourne la taille de la zone memoire correspondant aux donnees traitees d'un paquet
  inline uint_4 ProcPaqSize() { return procpaqsz; }
  // Retourne la taille totale d'une zone de donnees traitees
  inline uint_4 ProcZoneSize() { return nfibres*npaq*procpaqsz; }
  inline uint_4 OneFiberProcZoneSize() { return npaq*procpaqsz; }

  // Definit l'etat des operations obligatoires avant qu'une zone 
  // remplie ne puisse etre recyclee (re-remplie)
  inline void   SetFinalizedMask(uint_4 mask=((uint_4)MemZS_Saved))
    { mask_finalized_ = mask; }
  inline uint_4 GetFinalizedMask() { return mask_finalized_; }

  // Definit l'etat des operations obligatoires avant qu'une zone 
  // ne puissent etre sauvee apres traitement (subir l'action ProcSaved) 
  inline void   SetProcSavedReadyMask(uint_4 mask=((uint_4)MemZS_ProcA))
    { mask_saveproc_ready_ = mask; }
  inline uint_4 GetProcSavedReadyMask() { return mask_saveproc_ready_; }

  // Retourne l'identificateur d'une zone memoire disponible pour une action specifique
  int FindMemZoneId(MemZaction act);
  // Libere une zone memoire en marquant son etat
  int FreeMemZone(int id, MemZStatus st);

  // Retourne le pointeur de la zone numero k pour les datas pour une fibre donnee
  inline Byte* GetMemZone(int id, int fib)  {
    if ((id < 0) || (id >=(int) memzones.size()))  return NULL;
    if ((fib < 0) || (fib >=(int) nfibres))  return NULL;
    return memzones[id]+fib*OneFiberZoneSize();  
    }
  // Retourne le pointeur de la zone numero k pour les resultats de traitement pour une fibre donnee
  inline Byte* GetProcMemZone(int id, int fib)  {
    if ((id < 0) || (id >=(int) procmemzones.size()))  return NULL;
    return procmemzones[id]+fib*OneFiberProcZoneSize();  
    }
  // Retourne le pointeur de la zone numero k pour les datas
  inline Byte* GetMemZone(int id)  {
    if ((id < 0) || (id >=(int) memzones.size()))  return NULL;
    return memzones[id];  
    }
  // Retourne le pointeur de la zone numero k pour les resultats de traitement
  inline Byte* GetProcMemZone(int id)  {
    if ((id < 0) || (id >=(int) procmemzones.size()))  return NULL;
    return procmemzones[id];  
    }
  // Retourne le pointeur sur les donnees auxiliaire associee a une zone 
  inline RAcqMZAuxData* GetAuxData(int id)   {
    if ((id < 0) || (id >=(int) auxdatap.size()))  return NULL;
    return auxdatap[id];  
    }  
  // Retourne l'identificateur (numero absolu) de la fibre 
  inline int& FiberId(int fib) {
    if ((fib < 0) || (fib >=(int) nfibres))  return dummy_;
    return fiber_id[fib];
  }

  ostream& Print(ostream& os);
  inline ostream& Print() { return Print(cout); }

  // Pour controler l'etat d'execution global 
  void Stop(); 
  inline MemZRunState GetRunState() 
    { return runstate_; }
  inline MemZRunState ChangeRunState( MemZRunState st) 
    { runstate_ = st; return runstate_; }

  // Conversion Status -> action 
  static MemZaction Convert_Status2Action(MemZStatus st);

protected:
  void Init();  // Fait l'allocation des differents buffers - est appele par les constructeurs

  // ... variables membres
  SOPHYA::ZMutex mex;   
  uint_4  nzones, nfibres, npaq, paqsz, procpaqsz;
  vector< Byte* > memzones;            // pointeur de donnees de chaque zone 
  vector< St_MemZ > states;            // etat de chaque zone 
  vector< RAcqMZAuxData* > auxdatap;   // pointeur de donnees auxiliaires associees a chaque zone
  vector< Byte* > procmemzones;        // pointeur de donnees traitees de chaque zone  
// Identificateur (numero absolu) des fibres 
  vector< int > fiber_id;
  int dummy_;
  bool stop_;
  MemZRunState runstate_;
  uint_4 mask_finalized_;
  uint_4 mask_saveproc_ready_;
  uint_8 serial_;  
};




#endif
