#ifndef  BRVISCALC_H_SEEN
#define  BRVISCALC_H_SEEN

//----------------------------------------------------------------
// Projet BAORadio - (C) LAL/IRFU  2008-2010
// Classes de threads de calcul de visibilites pour BAORadio 
//  Principales classes utilisables 
//  - BRVisibilityCalculator (donnees FFT)
//  - BRVisCalcGroup  
//----------------------------------------------------------------

#include <string>
#include <vector>
#include <iostream>

// --- include SOPHYA
#include "machdefs.h"
#include "sopnamsp.h"
#include "tvector.h"
#include "parlex.h"
#include "datatable.h"
#include "ctimer.h"

//--- include BAORadio 
#include "racqumem.h"
#include "brpaqu.h"
#include "brbaseproc.h"

using namespace std;

//---------------------------------------------------------------------
// Classe de traitement - calcul de visibilite pour n fibres 
//---------------------------------------------------------------------
// Pour convertir une paire de numeros de canaux en un seul entier C1*CHANPAIRCONVFAC+C2
#define  CHANPAIRCONVFAC  1000 

class BRVisibilityCalculator : public BRBaseProcessor, ParallelTaskInterface {
public:	
  BRVisibilityCalculator(RAcqMemZoneMgr& memgr, string outpath, uint_4 nmean=1000, size_t nthr=1);
  virtual ~BRVisibilityCalculator(); // Fait le nettoyage final , sauvegarde des objets ...

  // Pour definir le rang de l'objet lors d'un traitement avec plusieurs BRVisibilityCalculator en //
  void DefineRank(uint_4 nbc, uint_4 cid, uint_4 pair1=0, uint_4 nbpairs=999999999, bool fgpimp=false);

  // Pour limiter le calcul des visibilites a certaines voies, determine par les numeros de sequences des paires 
  // Exemple pour 2 fibres et 4 voies 
  // kpair=numero sequentiel de la paire: 0->(0,0), 1->(0,1), 2->(0,2), 3->(0,3), 4->(1,1), 5->(1,2) ...
  // fgpimp=true : on calcule uniquement les visib avec un numero de voie paire, un numero de voie impair
  //               fgpimp=true est utile pour le calcul des visib entre deux cylindres (E-W) 
  uint_4 SelectPairs(uint_4 pair1=0, uint_4 nbpairs=999999999, bool fgpimp=false);
  // pour definir le binning en frequence pour la VisibDataTable 
  // par defaut, toutes les frequences avec un pas de 1 
  int SelectFreqBinning(uint_4 freq1=0, uint_4 freq2=0, uint_4 nbfreq=0);

  // Pour activer la verification des timetag
  // Creation / sauvegarde d'une matrice de TimeTag / FrameCounter avec NCols=maxnpaq (nb de paquets/trigger)  
  int ActivateTimeTagCheck(uint_8 maxnpaq);
  // Pour activer le remplissage du DataTable des visibilites 
  inline void ActivateVisDTable(bool fgfdt=false)  { fgvisdt_=fgfdt;  return; }

  // Methode d'execution standard du thread (on redefinit la methode de BRBaseProcessor)
  virtual void run();  
  // Methode de la classe ParallelTaskInterface 
  virtual int    execute(int tid);

  // Le nombre total d'operations effectuees
  inline r_8 TotNbFLOP() { return nb_flop_; }
  inline r_8 TotNbMegaFLOP() { return nb_flop_/(1024.e3); }

protected:
  virtual void UpdateChanIds(); 
  virtual int Process(); 
  virtual void UpdateVisMtxInfo();
  virtual int FillVisibTable(double fcm, double ttm); 
  virtual int CheckTimeTag(); 

  ParallelExecutor paralex_;  // Pour l'execution en parallele  
  size_t nparthr_;

  uint_4 nmean_;  // Nombre de spectres pour le calcul des moyennes 
  string outpath_;  // directory pour fichiers de sortie
  uint_4 nbcalc_, calcid_;  // Nombre total de threads/objets BRVisibilityCalculator parallles, l'identificateur de l'objet  

  uint_4 pairst_, nbpairs_;   // Numero de la premiere paire, nombre de paires ( 1 paire= 1 visibilites)
  bool fgpimp_;    //  true -> calculer uniquement visibilities avec no voie pair-impair  

  uint_4 jf1_,jf2_,djf_;  // binning en frequence pour la datatable

  vector<TwoByteComplex*> vpdata_;
  bool* fgallfibok;  

  double moyfc_, moytt_;  // moyenne des FrameCounter / TimeTag pour chaque matrice de visibilite
  uint_8 first_fc_,first_tt_;  // Valeurs premier FrameCounter / TimeTag pour chaque matrice de visibilite
  TimeStamp first_tmstamp_;   // Date-Heure pour premier FrameCounter / TimeTag pour chaque matrice de visibilite
  
  TMatrix< complex<r_4> > vismtx_; 
  TVector< uint_4 > chanids_;    // Numero de chaque canal=FiberId*2-1, FiberId*2
  TVector< uint_4 > chanpairnumall_;  // Liste de tous les numeros de paires de canaux (forme 1000*C1+C2=IIIJJJ)
  TMatrix< uint_4 > chanpairsall_ ;  // Liste de toutes paires de canaux (forme (C1,C2) )
  TVector< uint_4 > chanpairnum_;    // Liste des paires de canaux calcules par ce BRVisibilityCalculator
  TMatrix< uint_4 > chanpairs_ ;  // Liste des paires de canaux (forme (C1,C2) ) pour ce BRVisibilityCalculator

  //  FitsInOutFile dtfos_;
  DataTable visdt_;
  bool fgvisdt_;   // true -> fill datatable
  uint_4 numfile_;
  uint_8 totnbpaq_;

  r_8 nb_flop_;   // Nombre total d'operations effectuees

  // Variables/objets pour la verifications des timetag
  bool fgcktt_;  // true -> checktimetag 
  int_8 mindeltatt_;       // difference minimum de deux timetag successifs 
  TMatrix< uint_8 > fcmtx_; // matrices des framecounters
  TMatrix< uint_8 > ttmtx_; // matrices des timetag 
  vector< uint_8 > vlasttt_;  // dernieres valeurs de TimeTag / fibre
  vector< uint_8 > vbadtt_;  // Number of bad TimeTag paquets per fiber 
  vector< uint_8 > vnsamett_;  // Number of paquets having TimeTag <> TT_Fiber1 per fiber 
  vector< uint_8 > vndiff5tt_;  // Number of paquets having |TimeTag-TT_Fiber1|>5 per fiber 

};


//-------------------------------------------------------------------------------
// Classe Groupe (ensemble) de Calculateur de Visibilites, tournant en parallele
//-------------------------------------------------------------------------------

class BRVisCalcGroup {
public:	
  BRVisCalcGroup(size_t nbcalc, RAcqMemZoneMgr& memgr, string outpath, uint_4 nmean=1000,
		 uint_4 pair1=0, uint_4 nbpairs=999999999, bool fgpimp=false, size_t nthr=1);
  ~BRVisCalcGroup();
  int SelectFreqBinning(uint_4 freq1=0, uint_4 freq2=0, uint_4 nbfreq=1);
  inline int ActivateTimeTagCheck(uint_8 maxnpaq)
  { return viscalcp_[0]->ActivateTimeTagCheck(maxnpaq); }
  void ActivateVisDTable(bool fgfdt=false);

  void start();
  void join();
protected:
  vector<BRVisibilityCalculator *> viscalcp_; 
  Timer tm_;
};


#endif
