// lecture des fichiers acq de Pittsburgh Dec 2010
// et ecriture des matrices de visi temps-frequence
// > make OBJ=$CMVPROG/obj/ EXE=$CMVPROG/exe/ chanum_1210 svv2mtx2_1210

#include "sopnamsp.h"
#include "machdefs.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <iostream>
#include <string>

#include "pexceptions.h"
#include "tvector.h"
#include "fioarr.h"

/*
export TACQEXE=~/Reza/TAcq/Objs
./CasA02Dec.csh > CasA02Dec.log 2>&1
./Crab03Dec.csh > Crab03Dec.log 2>&1
./Sun03Dec.csh  > Sun03Dec.log  2>&1
./CasA03Dec.csh > CasA03Dec.log 2>&1
*/

//--------------------------- Fonctions de ce fichier   ------------------- 
void usage(void);
void usage(void)
{
cout<<"svv2mtx2_1012 [options] dir : lecture des fichiers acq de Pittsburgh Dec 2010"<<endl
    <<" dir : repertoire ou se trouvent les fichiers d'acq"<<endl
    <<" -C : compute and store the complexe conjugated visi"<<endl
    <<" -D : visi is a duplicated one"<<endl
    <<" -o visi.ppf : nom du ficher ppf pour ecrire la visi temps-frequence"<<endl
    <<" -t thr : numero de la thread ayant traite cette visi [0-N]"<<endl
    <<" -r row : numero de la ligne de la matrice acq pour cette visi [0-Nrow["<<endl
    <<" -f freq0 : 1ere frequence en MHz"<<endl
    <<" -T it1,it2 : numero (temps) des fichiers a traiter [it1,it2]"<<endl
    <<" -F if1,if2,ngrpfreq : numero des frequences [if1,if2] de [0,N[ a traiter et regroupement par ngrpfreq"<<endl
    <<"ATTENTION: lancer ce prog depuis le repertoire ou doivent etre ecrits les fichiers de visi"<<endl;
}

//----------------------------------------------------
int main(int narg, char* arg[])
{
  // --- Decodage des arguments et traitement 
  string outname = "";
  int numthread = -1, numrow = -1;
  bool dupli = false, doconj = false;
  double freq0 = 0.;
  int ifilmin=0, ifilmax=99999;
  int jfr1=0, jfr2=-1, ngrpfreq=1;
  char str[2048];

  char c;
  while((c = getopt(narg,arg,"hDCo:t:r:f:T:F:")) != -1) {
    switch (c) {
    case 'D' :
      dupli = true;
      break;
    case 'C' :
      doconj = true;
      break;
    case 'o' :
      outname = optarg;
      break;
    case 't' :
      numthread = atoi(optarg);
      break;
    case 'r' :
      numrow = atoi(optarg);
      break;
    case 'f' :
      freq0 = atof(optarg);
      break;
    case 'T' :
      sscanf(optarg,"%d,%d",&ifilmin,&ifilmax);
      if(ifilmin<0) ifilmin=0;
      break;
    case 'F' :
      sscanf(optarg,"%d,%d,%d",&jfr1,&jfr2,&ngrpfreq);
      if(ngrpfreq<=0) ngrpfreq=1;
      break;
    case 'h' :
    default :
      usage(); return -1;
    }
  }
  if(optind>=narg) {usage(); return -1;}
  string indir = arg[optind];

  cout<<"thread="<<numthread<<endl; if(numthread<0) return -2;
  cout<<"numrow="<<numrow<<endl;  if(numrow<0) return -2;
  cout<<"dupli="<<(int)dupli<<endl;
  cout<<"doconj="<<(int)doconj<<endl;
  cout<<"outname="<<outname<<endl; if(outname.size()<=0) return -2;
  cout<<"indir="<<indir<<endl;
  cout<<"freq0="<<freq0<<" MHz"<<endl;
  cout<<"request: file "<<ifilmin<<" to "<<ifilmax<<endl; if(ifilmax<ifilmin) return -3;
  cout<<"request: freq "<<jfr1<<" to "<<jfr2<<" grouped by "<<ngrpfreq<<endl;

  // --- recherche et comptage des fichiers de visibilites
  // ATTENTION: il peut manquer des fichiers au debut ou dans la sequence
  struct stat buffer;
  int nfile = 0;
  {
  int ifmin2 = ifilmin, ifmax2 = ifilmax; ifilmax = -1;
  bool foundfirst = false;
  for(int ifile=ifmin2; ifile<=ifmax2; ifile++) {
    sprintf(str, "%s/vismtx_%d_%d.ppf",indir.c_str(),numthread,ifile);
    int status = stat(str,&buffer);
    if(!foundfirst && status==0) {
      cout<<"first file found: "<<str<<endl;
      foundfirst = true;
      if(ifile>ifilmin) ifilmin = ifile;
    }
    if(!foundfirst) continue;
    if(ifile<ifilmin) continue;
    if(foundfirst && status!=0) continue;
    nfile++;
    ifilmax = ifile;
  }
  cout<<"Found "<<nfile<<" files from "<<ifilmin<<" to "<<ifilmax<<endl;
  if(nfile==0 || ifilmax<ifilmin) return -4;
  }


  //--------------------------------------------------------------------
  int rc = 0;
  try {

  // --- read visibility files
  TMatrix< complex<r_4> > MVisi;
  TVector<r_8> MeanTT(nfile);
  TVector<r_4> Freq;
  TVector<int_4> Npaqsum(nfile);
  string tudeb, tufin;
  double tudeb_day, tufin_day;
  int nfreq = 0;
  int lpmod = nfile/10; if(lpmod<=0) lpmod=1;

  int_4 ntimefill = 0, ntimebad = 0;
  for(int ifile=ifilmin; ifile<=ifilmax; ifile++) {

    // --- Lecture d'une visi elementaire (fichier acq)
    sprintf(str, "%s/vismtx_%d_%d.ppf",indir.c_str(),numthread,ifile);
    int status = stat(str,&buffer);
    if(status) continue; // fichier inexistant
    if(ifile==ifilmin || ifile==ifilmax || (ifile-ifilmin)%lpmod==0)
        cout<<ntimefill<<" "<<ifile<<" opening: "<<str<<endl;
    TMatrix< complex<r_4> > vismtx;
    try {
      PInPersist pin(str);
      pin >> vismtx;
    } catch(...) {
      cout<<"ERROR: bad file "<<str<<endl;
      ntimebad++;
      continue;
    }
    tufin = (string)vismtx.Info()["DATEOBS"];

    // --- Time keeping and number of summed elementary visibilities
    MeanTT(ntimefill) = (double)vismtx.Info()["MeanTT"]/125.e6;
    uint_4 npaqsum = vismtx.Info()["NPAQSUM"];

    // --- Initialisation purposes for the first read file
    if (ifile==ifilmin) {

      tudeb = tufin;
      printf("Reference Time: tt = %.5f sec, TU = %s\n",MeanTT(0),tudeb.c_str());
      if(numrow>=vismtx.NRows()) {
        cout<<"ERROR: requested numrow="<<numrow<<" => "<<vismtx.NRows()<<endl;
        return -5;
      }

      // frequency grouping
      int nfreq0 = vismtx.NCols();
      cout<<"vismtx: number of frequencies = "<<nfreq0<<endl;
      if(jfr1<=0) jfr1=0; if(jfr1>=nfreq0) jfr1=nfreq0-1;
      if(jfr2<jfr1 || jfr2>=nfreq0) jfr2=nfreq0-1;
      cout<<"frequency from jfr1="<<jfr1<<" to jfr2="<<jfr2<<" grouped by "<<ngrpfreq<<endl;
      nfreq = 0; for(int i=jfr1;i<=jfr2;i+=ngrpfreq) nfreq++;
      cout<<"frequency from [jfr1="<<jfr1<<" , jfr2="<<jfr2<<"] grouped by "<<ngrpfreq
          <<": total of "<<nfreq<<" freq"<<endl;
      Freq.ReSize(nfreq); Freq = 0.;
      for(int i=0;i<nfreq;i++) {
        int nf=0;
        for(int j=0; j<ngrpfreq; j++) {
          int f = jfr1 + i*ngrpfreq + j;
          if(f>jfr2) break;
          Freq(i) += f;
	  nf++;
	}
        if(nf>0) Freq(i) /= (double)nf;
	else {cout<<"ERROR: last freq bin with 0 freq in it"<<endl; return -6;}
        if(i<4 || i>nfreq-4) cout<<"  F("<<i<<") = "<<Freq(i)<<" nf="<<nf<<endl;
      }

      // allocate visib matrice <f> vs t
      cout<<"allocating visibility matrice ("<<nfile<<","<<nfreq<<") "
          <<nfile*nfreq*sizeof(complex<r_4>)/1.e6<<" Mo"<<endl;
      MVisi.ReSize(nfile,nfreq);
      MVisi = complex<r_4>(0.,0.);
    }

    // Fill time-freq visibility matrix
    for(int i=0;i<nfreq;i++) {
      int nf=0;
      for(int j=0; j<ngrpfreq; j++) {
        int f = jfr1 + i*ngrpfreq + j;
        if(f>jfr2) break;
        MVisi(ntimefill,i) += vismtx(numrow,f);
        nf++;
      }
      Npaqsum(ntimefill) = nf * npaqsum;
      MVisi(ntimefill,i) /= double(Npaqsum(ntimefill));
      if(doconj) MVisi(ntimefill,i) = conj(MVisi(ntimefill,i));
    }
    ntimefill++;

  }  // fin boucle sur le fichiers d'acq
  cout<<"ntimefill = "<<ntimefill<<" / "<<nfile<<" , ntimebad="<<ntimebad<<" bad files"<<endl;

  // --- keeping info with visi
  {
  // bricolo en unite de jours depuis le 0/12/2010 a 0h
  int y,m,d,h,mn; double s;
  sscanf(tudeb.c_str(),"%d-%d-%dT%d:%d:%lf",&y,&m,&d,&h,&mn,&s);
  tudeb_day = (double)d + ((double)h + mn/60. + s/3600.)/24.;
  sscanf(tufin.c_str(),"%d-%d-%dT%d:%d:%lf",&y,&m,&d,&h,&mn,&s);
  tufin_day = (double)d + ((double)h + mn/60. + s/3600.)/24.;
  }
  MVisi.Info()["nth"] = numthread;
  MVisi.Info()["row"] = numrow;
  MVisi.Info()["dir"] = indir;
  MVisi.Info()["dupli"] = (dupli) ? 1: 0;
  MVisi.Info()["isconj"] = (doconj) ? 1: 0;
  MVisi.Info()["TUobs_0"] = tudeb;
  MVisi.Info()["TUobs_N"] = tufin;
  MVisi.Info()["TUday_0"] = tudeb_day;
  MVisi.Info()["TUday_N"] = tufin_day;
  MVisi.Info()["TTag_0"] = MeanTT(0);
  if(ntimefill>0) MVisi.Info()["TTag_N"] = MeanTT(ntimefill-1);
  MVisi.Info()["freq0"] = freq0;
  MVisi.Info()["dfreq0"] = 500./8192.;
  MVisi.Info()["jfr1"] = jfr1;
  MVisi.Info()["jfr2"] = jfr2;
  MVisi.Info()["ngrpfreq"] = ngrpfreq;
  MVisi.Info()["ifilmin"] = ifilmin;
  MVisi.Info()["ifilmax"] = ifilmax;
  MVisi.Info()["ntimefill"] = ntimefill;
  MVisi.Info()["ntimebad"] = ntimebad;

  // --- writing visibility matrix to file
  sprintf(str,"%s",outname.c_str());
  cout<<"writing visibility matrix to file "<<str<<endl;
  POutPersist pos(str);
  pos.PutObject(MeanTT,"meantt");
  pos.PutObject(Npaqsum,"npaqsum");
  pos.PutObject(Freq,"ifreq");
  pos.PutObject(MVisi,"visi");
  }

  //--------------------------------------------------------------------
  catch(PException& exc) {
    cerr<<" svv2mtx2_1210.cc catched PException "<<exc.Msg()<<endl;
    rc = 77;
  }
  catch (std::exception& sex) {
    cerr<<"\n svv2mtx2_1210.cc std::exception :" 
        <<(string)typeid(sex).name() << "\n msg= "<<sex.what()<<endl;
    rc = 78;
  }
  catch(...) {
    cerr<<" svv2mtx2_1210.cc catched unknown (...) exception  "<<endl; 
    rc = 79; 
  }
  cout<<">>>> svv2mtx2_1210.cc ------- END ----------- RC="<<rc<<endl;
  return rc;
}


/*
openppf visi1_01_05.ppf

print visi 5

n/plot ifreq.val%n ! ! "cpts"

n/plot npaqsum.val%n ! ! "cpts"

n/plot meantt.val%n ! ! "cpts"
cp meantt dmeantt
c++exec dmeantt=0.; for(int i=1;i<dmeantt.Size();i++) dmeantt(i)=meantt(i)-meantt(i-1);
n/plot dmeantt.val%n n>0 ! "cpts nsta"

imag visi "cdmod"
imag visi "cdreal"
imag visi "cdimag"
imag visi "cdphas"

 */
