#ifndef  BRPROC_H_SEEN
#define  BRPROC_H_SEEN

//----------------------------------------------------------------
// Projet BAORadio - (C) LAL/IRFU  2008-2010
// Classes de threads de traitememt donnees BAORadio 
//  Principales classes utilisables 
//  - BRMeanSpecCalculator (donnees FFT ou RAW apres BRFFTCalculator)
//  - BRFFTCalculator (donnees RAW) 
//----------------------------------------------------------------


#include <string>
#include <vector>
#include <iostream>

#include "racqumem.h"
#include "brbaseproc.h"
#include "parlex.h"
#include "sopnamsp.h"
#include "tvector.h"
#include "ntuple.h"
#include "datatable.h"
#include "swfitsdtable.h"
#include "FFTW/fftw3.h"

using namespace std;


 
//---------------------------------------------------------------------
// Classe de traitement simple - calcul de spectres moyennes / voie
// a partir de donnees de type FFT (firmware FFT)
//---------------------------------------------------------------------
class BRMeanSpecCalculator : public BRBaseProcessor
{
public:
  BRMeanSpecCalculator(RAcqMemZoneMgr& memgr, string outpath, uint_4 nmean=1000, 
		       bool fgdatafft=true, bool fgsinglechan=false);
  virtual ~BRMeanSpecCalculator(); // Fait le nettoyage final , sauvegarde des objets ...

// Definit la taille de la fenetre temporelle (winsz = nb de paquets) des spectres a traiter
// Cette zone est etendu de wszext paquet en debut et en fin de paquet 
  void SetSpectraWindowSize(uint_4 winsz=15, uint_4 wszext=3);
// Taille de la fenetre en temps (nb de paquets a traiter en bloc
  inline sa_size_t GetSpectraWindowSize() { return spec_window_.SizeZ()-2*spwin_ext_sz_; }
// Taille de la zone d'extension de la fenetre de chaque cote 
  inline sa_size_t GetSpecWinExtensionSize() { return spwin_ext_sz_; }
// La taille de recouvrement est le double de la taille de la zone d'extension 
  inline sa_size_t GetSpecWinOverlapSize() { return 2*spwin_ext_sz_; }
// Retourne l'index (selon Z) dans le tableau des spectres en fonction du numero de paquet
  inline sa_size_t PaqNumToArrayIndex(uint_8 numpaq)  { return numpaq%spec_window_.SizeZ() ; }

// Pour definir le nombre maximum de fichiers de fenetres de spectres a sauver sur fichiers
  inline void SetMaxNbSepcWinFiles(uint_4 nmax=0)  {  nmaxfiles_specw_=nmax; }

// Pour definir le datatable a remplir - pas de DataTable rempli si pas appele 
  virtual void DefineDataTable();

  inline void SetVarianceLimits(double vmin=0., double vmax=9.e99)
    { varmin_=vmin;  varmax_=vmax; }

// lecture des gains en fonction de la frequence depuis un fichier FITS
// flag fgapp: Active ou desactive la renormalisation de coefficients de Fourier par les gains=f(frequence)  
  void ReadGainFitsFile(string filename, bool fgapp=true);

protected:
  virtual int Process();   // appele paquet par paquet 
// appele par Process() pour traiter un groupe de paquets correspondant a une fenetre en temps
  virtual void ProcSpecWin(uint_8 numpaqstart, uint_8 numpaqend);  // appele fenetre par fenetre  
  virtual void FlagBadPackets(uint_8 numpaqstart, uint_8 numpaqend);   
  virtual void SaveMeanSpectra();  // Pour sauver les spectres moyennes ( + sigma ) dans un fichier 
  virtual void SaveSpectraWindow();  // Pour sauver les spectres de la fenetre temporel ds un fichier 


  uint_4 nmean_;  // Nombre de spectres pour le calcul des moyennes 
  string outpath_;  // directory pour fichiers de sortie
  bool fgdatafft_;
  bool fgsinglechannel_;
  uint_4 numfile_;
  vector<uint_8> nbpaq4mean_;
  uint_8 totnbpaq_;

  TMatrix< r_4 > mspecmtx_;    // Matrice des spectres moyennees 
  TMatrix< r_4 > sigspecmtx_;    // Matrice des sigmas des spectres  

  TArray< r_4 > spec_window_;    // fenetre en temps sur les spectres  
  uint_4 spwin_ext_sz_;          // Taille de la zone d'extension en debut et en fin 
  TArray< uint_2 > clnflg_;      // flag de nettoyage resultant de la routine   FlagBadPackets() 
  uint_8 paqnum_w_start;     // numero de paquet du debut de la fenetre en temps (inclusif) 
  uint_8 paqnum_w_end;       // numero de paquet final de la fenetre en temps (non inclusif) 

  uint_4 nbtot_specwin_;    // nombre total de fenetre de spectres traites 
  uint_4 nmaxfiles_specw_;   // nombre maximum de fichier de fenetre_spectre 
  
  TMatrix< r_4 > sgain_;   // gain en fonction de la frequence  
  bool fg_apply_gains_; 

  double varmin_, varmax_;  // limites pour la coupure sur la variance
  vector<uint_8> nbadpaq_;

  FitsInOutFile* ofsdtp_;   // Output fits stream for datatable 
  SwFitsDataTable* dtp_;    // DataTable 
  double* xnt_; 
}; 

//-------------------------------------------------------------------------
// Classe WBRFFT : Calcul de TF sur donnees brutes (firmware RAW)
//   en utilisant FFTW et en effectuant une copie ds tableau local
//-------------------------------------------------------------------------
typedef unsigned char IDT ;   // definition type de donnees en entree de WBRFFT
typedef int_2  IIDT ;      // definition type de donnees intermediaire 
typedef float  ODT;  // type de donnees en sortie (complex < ODT > )
//----Classe WBRFFT----
class WBRFFT {
public:
  WBRFFT(uint_4 sz=0);
  virtual ~WBRFFT();

  void SetInDataSize(uint_4 sz);
  void DoFFT(IDT *indata, complex<ODT> * ofc);
  static void PrintData(IDT *indata,  complex<ODT> * ofc, uint_4 sz);

  uint_4 sz_;   // Taille tableau d'entree 
  TVector< ODT > inp;
  TVector< complex<ODT> > outfc;
  fftwf_plan myplan_;
  static ZMutex* mtx_fftwp_;  // Mutex de protection pour la creation du plan fftw 
};
//--------------------

//---------------------------------------------------------------------
// Classe de thread de calcul de FFT sur donnees RAW 
//---------------------------------------------------------------------
class BRFFTCalculator : public BRBaseProcessor
{
public:
  BRFFTCalculator(RAcqMemZoneMgr& memgr, bool fgsinglechannel=false);
  virtual ~BRFFTCalculator(); // Fait le nettoyage final , sauvegarde des objets ...

protected:
  virtual int Process(); 
  
  bool fgsinglechannel_;
  uint_8 totnbfftpaq_;
  WBRFFT ffts_;
};


//-------------------------------------------------------------------------
//    **** OBSOLETE : BRProcA2C et BRProcB4C
//   Anciennes classes de traitement (Avant Nov2009) utilises par mcrd.cc
//-------------------------------------------------------------------------
//---------------------------------------------------------------------
// Classe thread de traitement avec 2 voies par frame (donnees raw/FFT
//---------------------------------------------------------------------

/*-- Arguments du constructeur : 
  o mem : gestionnaire zones memoires avec contenant les "frames" avec 2 voies/frame
  o path : Chemin et nom des fichiers PPF produits (on ajoute _numero.ppf )
  o nmean : Nombre de frame (paquets) moyennes
  o nmax : Nombre maxi de frames traites 
  o fghist = true -> fait aussi l'histo des valeurs des echantillons temps
  o nfsmap>0 : fabrication de cartes (image 2D) des spectres, 
     avec nfsmap=nombre de frequences sommes  
  o fgnotrl = true -> fichiers fits / paquets sans trailer 
  o card : numero de carte , juste pour les impressions 
*/
class BRProcA2C : public ZThread {   // !!!! OBSOLETE  !!!!
public:	
  BRProcA2C(RAcqMemZoneMgr& mem, string& path, bool fgraw, 
	    uint_4 nmean=10, uint_4 nmax=100, bool fgtshist=false, 
	    uint_4 nfsmap=0, bool fgnotrl=false, int card=1);
  virtual void run(); 
  void Stop();
  inline void STOP() { stop_ = true; }	
protected:
  RAcqMemZoneMgr& memgr;
  bool stop_;  
  bool fgraw_;   // true -> raw data 
  uint_4 nmax_;  // Nombre maxi de blocs MemZone traites
  uint_4 nmean_;  // Nombre de spectres pour le calcul des moyennes 
  uint_4 nfsmap_;  // Binning en frequence pour les cartes 2D temps-freq
  string path_;  
  bool fghist_;   // if true, fill and save histogram with time sample values
  bool fgnotrl_;   // if true, don't check packet trailer - when using pre-june fits files
  int card_;   // Numeros de voies 2*card_-1, 2*card_
};

//---------------------------------------------------------------------
// Classe thread de traitement 2 x 2 voies/paquet (Apres BRProcA2C)
//         !!!! OBSOLETE  !!!!   
//---------------------------------------------------------------------
/*-- Arguments du constructeur : 
  o mem1,mem2 : gestionnairea zones memoires avec contenant les donnees des 4 voies (2 voies/frame)
  o path : Chemin et nom des fichiers PPF produits (on ajoute _numero.ppf )
  o nmean : Nombre de frame (paquets) moyennes
  o nmax : Nombre maxi de frames traites 
  o fgnotrl = true -> fichiers fits / paquets sans trailer 
  o card : numero de carte , juste pour les impressions 
*/

class BRProcB4C : public ZThread {  //   !!!! OBSOLETE  !!!!   

public:	
  BRProcB4C(RAcqMemZoneMgr& mem1, RAcqMemZoneMgr& mem2, string& path, 
	    bool fgraw, uint_4 nmean=10, uint_4 nmax=100, bool fgnotrl=false);
  virtual void run(); 
  void Stop();
  inline void STOP() { stop_ = true; }	
protected:
  RAcqMemZoneMgr& memgr1;
  RAcqMemZoneMgr& memgr2;
  bool stop_;  
  bool fgraw_;   // true -> raw data 
  uint_4 nmax_;  // Nombre maxi de blocs MemZone traites
  uint_4 nmean_;  // Nombre de blocs pour le calcul des moyennes 
  string path_;  
  bool fgnotrl_;   // if true, don't check packet trailer - when using pre-june fits files
};



#endif
