// archtoi.cc
// Eric Aubourg         CEA/DAPNIA/SPP   juillet 1999


#include <string>
#include <iostream.h>
#include <fstream.h>
#include <iomanip.h>

#include "archeopsfile.h"
#include "toisvr.h"
#include "archtoi.h"
#include "archparam.h"
#include "asigps.h"

using namespace std; 


ArchTOI::ArchTOI(istream& str)
{
  init();
  readReq(str);
}


ArchTOI::ArchTOI(string const& filename)
{
  init();
  ifstream str(filename.c_str());
  readReq(str);
}

void ArchTOI::init()
{
  format = ascii_fmt;
  undef  = "#";
  allBolos = false;
  fptr = NULL;
  ostr = NULL;
}

void ArchTOI::readReq(istream& str)
{
   string line;
   while (str) {
     getline(str,line);
     if (!str) break;
     if (line[0] == '@') processTOIReq(line);
     else if (line[0] == '#') 
       if (!processOption(line)) break;
   }
}

#define tsttoi(toi)   if (key == "@"#toi) kind = toi;

void ArchTOI::processTOIReq(string line)
{
  // find TOI kind, index and options
  TOIKind kind= (TOIKind)-1;
  int index=-1;
  bool interp=false;
  bool repet =false;
  bool flag  =false;
  bool notrig=false;
  int x = line.find(' ');
  string key = line.substr(0, x);
  string opts = (x>0) ? line.substr(x) : string("");
  tsttoi(sampleNum)
  else tsttoi(internalTime)
  else tsttoi(mjd)
  else tsttoi(boloTens)
  else tsttoi(boloRaw)
  else tsttoi(sstDiode)
  else tsttoi(sstChannel)
  else tsttoi(sstStarZ)
  else tsttoi(sstStarF)
  else tsttoi(sstStarT)
  else tsttoi(gyroRaw)
  else tsttoi(gpsTime)
  else tsttoi(longitude)
  else tsttoi(latitude)
  else tsttoi(altitude)
  else tsttoi(tsid)
  else tsttoi(azimut)
  else tsttoi(alphaAxis)
  else tsttoi(deltaAxis)
  else tsttoi(alphaBolo)
  else tsttoi(deltaBolo)
  else {
    cerr << "*Warning, unrecognized TOI " << line << endl;
    return;
  }
  if (kind  == sampleNum) notrig = true;
  while (opts != "") {
    if (opts[0] == ' ') {
      opts = opts.substr(opts.find_first_not_of(' '));
      if (opts == "") break;
    }
    x = opts.find(' ');
    string opt = opts.substr(0, x);
    opts = (x>0) ? opts.substr(x) : string("");
    if (opt[0]>='0' && opt[0]<='9') {
      index = atoi(opt.c_str());
    } else if (opt == "notrig") {
      notrig = true;
    } else if (opt == "repet") {
      repet = true; interp = false;
    } else if (opt == "interp") {
      interp = true; repet = false;
    } else if (opt == "flag") {
      flag = true;
    }
  }
  headertoi.push_back(line);
  string toiname = key.substr(1); 
  if (index>=0) {
    char idx[10]; 
    sprintf(idx,"_%d",index);
    toiname += idx;
  }
  if (flag) {
    toinames.push_back("flg_"+toiname);
  }
  toinames.push_back(toiname);
  toiflags.push_back(flg((flag?hasflag:0)+((!repet&&!interp)?useNA:0)));
  if (index<0) index=0;
  svr.AddInfo(kind, index, !notrig, interp);
} 

bool ArchTOI::processOption(string line)
{
  int x = line.find(' ');
  string key = line.substr(0, x);
  string arg = (x>0) ? line.substr(x) : string("");
  if (arg.length()>0 && arg[0] == ' ') {
    arg = arg.substr(arg.find_first_not_of(' '));
  }
  if (key == "#ASCII") {
    format = ascii_fmt;
  } else if (key == "#FITS") {
    format = fits_fmt;
  } else if (key == "#TRANGE") {
    double tmin, tmax;
    sscanf(arg.c_str(), "%lg %lg", &tmin, &tmax);
    svr.SetTimeInterval(tmin, tmax);
  } else if (key == "#PATH") {
    svr.SetDirectory(arg);
  } else if (key == "#FILE") {
    svr.AddFile(arg);
  } else if (key == "#UNDEF") {
    undef=arg;
  } else if (key == "#ALLBOLOS") {
    allBolos=true;
  } else if (key == "#RECORDER") {
    svr.OnBoardRecorderFiles(true);
  } else if (key == "#MJD0") {
    double t0;
    sscanf(arg.c_str(), "%lg", &t0);
    archParam.acq.tBlock0 = t0;
  } else if (key == "#PERECH") {
    double t0;
    sscanf(arg.c_str(), "%lg", &t0);
    archParam.acq.perEch = t0;
  } else if (key == "#ASIGPS") {
    ASIGPS* gps = new ASIGPS(arg);
    gps->FitsDump("GPSDump.fits");
    svr.UseAuxGPS(gps);
  } else if (key == "#END") {
    return false;
  } else {
    cerr << "*Warning, unrecognized option " << line << endl;
    return true;
  }
  
  headeropt.push_back(line);
  return true;
}


void ArchTOI::run(string const& outfilename)
{
  if (format == ascii_fmt) {
    openFile   = &openFile_A;
    outHeader  = &outHeader_A;
    outValue   = &outValue_A;
    endLine    = &endLine_A;
    closeFile  = &closeFile_A;
  } else { // fits_fmt 
    openFile   = &openFile_F;
    outHeader  = &outHeader_F;
    outValue   = &outValue_F;
    endLine    = &endLine_F;
    closeFile  = &closeFile_F;
  }
  
  cout << "starting query" << endl;
  TOIIter iter = svr.DoQuery();
  (this->*openFile)(outfilename);
  (this->*outHeader)(iter);
  
  cout << "processing" << endl;
  while (iter.Next()) {
    int nn = iter.getSampleIndex();
     if (nn%200 == 0) {
       cout << "."; cout.flush();
     }
     if (nn%(200*80) == 0) cout << endl;
    // Si rien de dispo parmi les triggering, alors on passe au suivant
    bool hasValue = false;
    for (int i=0; i<toiflags.size(); i++) {
       if (!iter.isTrig(i)) continue;
       if (iter.canGetValue(i)) {hasValue=true; break;}
    }
    if (!hasValue) continue;
    int icol=0;
    for (int i=0; i<toiflags.size(); i++) {
      double value = iter.getValue(i);
      bool   ok    = iter.canGetValue(i);
      bool   isnew = iter.newValue(i);
      flg    flag  = toiflags[i];
      if (flag & hasflag) {
        (this->*outValue)(icol,  (ok && isnew ? 1 : 0));
        icol++;
      }
      if (((flag & useNA)!=0 && !isnew) || !ok) 
        (this->*outValue)(icol,  0, true);
      else
        (this->*outValue)(icol, value);
      icol++;
    }
    (this->*endLine)();
  }
  (this->*closeFile)();
  cout << "\nDone." << endl;
}


void ArchTOI::openFile_A(string const& filename) {
  ostr = new ofstream(filename.c_str());
}

void ArchTOI::outHeader_A(TOIIter& iter) {
  if (!ostr) return;
  for (list<string>::iterator i = headertoi.begin(); i != headertoi.end(); i++)
    *ostr << (*i) << '\n';
  for (list<string>::iterator i = headeropt.begin(); i != headeropt.end(); i++)
    *ostr << (*i) << '\n';;
  block_type_param* blk = iter.lastParam();
  if (blk) {
    int nb = blk->param.nb_bolo;
    for (int i=0; i<nb; i++) {
#if version_num > 25
      if (allBolos) {
        *ostr << "$BOLO " << i << " " 
            << blk->param.nom_coef[blk->param.bolo[i].numero_nom_coef].bolo_nom << " " 
            << blk->param.bolo[i].bolo_code_util << '\n';
      } else if (blk->param.bolo[i].bolo_code_util != bolo_hors_service &&
                 blk->param.bolo[i].bolo_code_util != bolo_normal_non_transmis) {
        *ostr << "$BOLO " << i << " " 
            << blk->param.nom_coef[blk->param.bolo[i].numero_nom_coef].bolo_nom << "\n";
      }
#else
      *ostr << "$BOLO " << i << " " << 
        blk->param.bolo[i].bolo_nom << '\n';
#endif
    }
  }
  *ostr << "#END" << endl;
}

void ArchTOI::outValue_A(int icolumn, double value, bool notdef) {
  if (!ostr) return;
  if (icolumn > 0) *ostr << '\t';
  if (notdef) {
     *ostr << undef ;
  } else {
     *ostr << setprecision(11) << value ;
  }
}

void ArchTOI::endLine_A() {
  if (!ostr) return;
  *ostr << '\n';
}

void ArchTOI::closeFile_A() {
  delete ostr;
  ostr = NULL;
}


void ArchTOI::openFile_F(string const& filename) {
  fitsStatus=0;
  remove(filename.c_str());
  if (fits_create_file(&fptr, filename.c_str(), &fitsStatus)) {
    fits_report_error(stderr, fitsStatus);
    exit(-1);
  }
}

void ArchTOI::outHeader_F(TOIIter& iter) {  
  int ncols=toinames.size();
  
  char** colnames = new (char*[ncols]);
  char** coltypes = new (char*[ncols]);
  char** colunits = new (char*[ncols]);
  int j=0;
  for (list<string>::iterator i = toinames.begin(); i != toinames.end(); i++,j++) {
    colnames[j] = const_cast<char*>((*i).c_str()); // should work for most STL implementations... Check...
    coltypes[j] = "1D";
    colunits[j] = " ";
  }
  
  fits_create_tbl(fptr, BINARY_TBL, 0, ncols, colnames, coltypes, colunits, NULL, &fitsStatus);
  fits_write_date(fptr, &fitsStatus);

  delete[] colunits;
  delete[] coltypes;
  delete[] colnames;

  j=1;
  // Rappel dans le header des requetes...
  int ntoireq = headertoi.size();
  fits_write_key(fptr, TINT, "TOIREQ", &ntoireq, NULL, &fitsStatus);
  for (list<string>::iterator i = headertoi.begin(); i != headertoi.end(); i++,j++) {
    char line[80];
    strcpy(line, (*i).c_str());
    char* pline = line;
    fits_write_keys_str(fptr, "TOIREQ", j, 1, &pline, (char**) NULL, &fitsStatus);
  }
  j=1;
  int noptreq = headeropt.size();
  fits_write_key(fptr, TINT, "OPTREQ", &noptreq, NULL, &fitsStatus);
  for (list<string>::iterator i = headeropt.begin(); i != headeropt.end(); i++,j++) {
    char line[80];
    strcpy(line, (*i).c_str());
    char* pline = line;
    fits_write_keys_str(fptr, "OPTREQ", j, 1, &pline, (char**) NULL, &fitsStatus);
  }

  // Noms des bolos
  block_type_param* blk = iter.lastParam();
  if (blk) {
    int nb = blk->param.nb_bolo;
    j=0;
    for (int i=0; i<nb; i++) {
#if version_num > 25
      if (allBolos || 
           (blk->param.bolo[i].bolo_code_util != bolo_hors_service &&
            blk->param.bolo[i].bolo_code_util != bolo_normal_non_transmis)) {
        //j++;
        char line[80];
        strcpy(line, blk->param.nom_coef[blk->param.bolo[i].numero_nom_coef].bolo_nom);
        char* pline = line;
        fits_write_keys_str(fptr, "BOLO", i, 1, &pline, (char**) NULL, &fitsStatus);
      }
#else
        //j++;
        char line[80];
        strcpy(line, blk->param.bolo[i].bolo_nom);
        char* pline = line;
        fits_write_keys_str(fptr, "BOLO", i, 1, &pline, (char**) NULL, &fitsStatus);
#endif
    }
    //fits_write_key(fptr, TINT, "BOLO", &j, NULL, &fitsStatus);
  }
  fits_write_comment(fptr, "Generated with archtoi " ARCHTOI_VERS, &fitsStatus);
  char line[80];
  sprintf(line, "using archeops.h %d", version_num);
  fits_write_comment(fptr, line, &fitsStatus);
  
  fitsLine = 1;
}

void ArchTOI::outValue_F(int icolumn, double value, bool notdef) {
  if (notdef) {
    fits_write_col_null(fptr, icolumn+1, fitsLine, 1, 1, &fitsStatus); 
  } else {
    fits_write_col_dbl(fptr, icolumn+1, fitsLine, 1, 1, &value, &fitsStatus);
  }
}

void ArchTOI::endLine_F() {
  fitsLine++;
}

void ArchTOI::closeFile_F() {
  fits_close_file(fptr, &fitsStatus);
  fits_report_error(stderr, fitsStatus);  /* print out any error messages */
}