#include "sopnamsp.h"
#include "machdefs.h"
#include <iostream>

#include "bitvector.h"
#include "pexceptions.h"
#include "fiondblock.h"

namespace SOPHYA {

//! create a vecsize element vector than can handle AT LEAST nbits bits (set to 0)
BitVector::BitVector(uint_8 nbits,uint_4 vecsize)
: nbits_(nbits), vecsize_(vecsize)
{
  if(vecsize_==0) vecsize_ = 128;
  bitpervec_ = vecsize_ * BitVec_BITPERWORD_;
  nvector_ = nbits_ / bitpervec_ + 1;
  for(uint_8 i=0;i<nvector_;i++) {
    NDataBlock<BitVec_Type> V(vecsize_);
    V = (BitVec_Type)0;
    vv_.push_back(V);
  }
  nbits_ = nvector_ * bitpervec_;
}

//! copy constructor: data are shared
BitVector::BitVector(BitVector const& bv)
: nbits_(bv.nbits_), vecsize_(bv.vecsize_), bitpervec_(bv.bitpervec_), nvector_(bv.nvector_)
{
  if(nvector_==0) return;
  // attention: partage de reference
  for(uint_8 i=0;i<nvector_;i++) vv_.push_back(bv.vv_[i]);
}

//! default constructor
BitVector::BitVector(void)
: nbits_(0), vecsize_(0), bitpervec_(0), nvector_(0)
{
}

//! destructor
BitVector::~BitVector(void)
{
  delete_();
}

//! delete
void BitVector::delete_(void)
{
  nbits_ = 0;
  vecsize_ = 0;
  bitpervec_ = 0;
  nvector_ = 0;
  vv_.resize(0);
}

//! extend bit vector to contain nbitsnew bits
void BitVector::extend_(uint_8 nbitsnew)
{
  if(nbitsnew<=nbits_) return;
  uint_8 nvectornew = nbitsnew / bitpervec_ + 1;
  if(nvectornew<=nvector_) return;
  for(uint_8 i=nvector_;i<nvectornew;i++) {
    NDataBlock<BitVec_Type> V(vecsize_);
    V = (BitVec_Type)0;
    vv_.push_back(V);
  }
  nvector_ = vv_.size();
  nbits_ = nvector_ * bitpervec_;
}

//! operator = : data are duplcated
BitVector& BitVector::operator = (const BitVector& bv)
{
 if(this == &bv) return *this;

 nbits_ = bv.nbits_;
 vecsize_ = bv.vecsize_;
 bitpervec_ = vecsize_ * BitVec_BITPERWORD_;
 nvector_ = bv.nvector_;
 vv_.resize(0);

 if(nvector_!=0) {
  for(uint_8 i=0;i<nvector_;i++) {
    NDataBlock<BitVec_Type> V(bv.vv_[i],false); // copie des donnees
    vv_.push_back(V);
  }
 }

 return *this;
}

// set all bit vector to v
BitVector& BitVector::operator = (bool v)
{
  if(nvector_==0) return *this;

  BitVec_Type veltv = 0; if(v) veltv = ~veltv;
  for(uint_8 i=0;i<nvector_;i++) vv_[i] = veltv;

  return *this;
}

/*! make AND between 2 vectors
\verbatim
If (*this) has extra length:
   putextra = -1 : put extra bits to false
            =  0 : let extra bits to their value
            = +1 : put extra bits to true
\endverbatim
*/
BitVector& BitVector::AND(const BitVector& bv,int putextra)
{
  uint_8 nbits = (nbits_<=bv.nbits_) ? nbits_: bv.nbits_;
  if(nbits==0) return *this;
  for(uint_8 i=0;i<nbits;i++) {
    BitVec_Type v = (*this)(i) & bv(i);
    this->Set(i,(bool)v);
  }
  if(putextra!=0 && nbits_>nbits) {
    bool v = (putextra>0) ? true: false;
    for(uint_8 i=nbits;i<nbits_;i++) this->Set(i,v);
  }
  return *this;
}

/*! make OR between 2 vectors
\verbatim
If (*this) has extra length:
   putextra = -1 : put extra bits to false
            =  0 : let extra bits to their value
            = +1 : put extra bits to true
\endverbatim
*/
BitVector& BitVector::OR(const BitVector& bv,int putextra)
{
  uint_8 nbits = (nbits_<=bv.nbits_) ? nbits_: bv.nbits_;
  if(nbits==0) return *this;
  for(uint_8 i=0;i<nbits;i++) {
    BitVec_Type v = (*this)(i) | bv(i);
    this->Set(i,(bool)v);
  }
  if(putextra!=0 && nbits_>nbits) {
    bool v = (putextra>0) ? true: false;
    for(uint_8 i=nbits;i<nbits_;i++) this->Set(i,v);
  }
  return *this;
}

// negate all bits
BitVector& BitVector::Negate(void)
{
 if(nvector_==0) return *this;
 for(uint_8 i=0;i<nvector_;i++)
   for(uint_8 j=0;j<vecsize_;j++) vv_[i](j) = ~(vv_[i](j));
 return *this;
}

//! return bit k
bool BitVector::operator() (uint_8 k) const
{
  if(k>=nbits_) throw(RangeCheckError("bool BitVector::operator(): indices out of range \n"));

  uint_8 nv = k / bitpervec_; // numero du vecteur
  k %= bitpervec_;
  uint_8 iv = k / BitVec_BITPERWORD_; // numero de l'element du vecteur
  k %= BitVec_BITPERWORD_; // numero de bit dans l'element du vecteur

  return ( vv_[nv](iv) & (1ULL<<k) );
}

//! set bit k to v (with automatic extend)
void BitVector::Set(uint_8 k,bool v)
{
  if(k>nbits_) extend_(k);

  uint_8 nv = k / bitpervec_; // numero du vecteur
  k %= bitpervec_;
  uint_8 iv = k / BitVec_BITPERWORD_; // numero de l'element du vecteur
  k %= BitVec_BITPERWORD_;  // numero de bit dans l'element du vecteur

  if(v) vv_[nv](iv) |= (1ULL<<k); else vv_[nv](iv) &= ~(1ULL<<k);
}

//! set bits [k1,k2] to v (no automatic extend)
void BitVector::Set(uint_8 k1,uint_8 k2,bool v)
{
  if(k1>=nbits_) return;
  if(k2<k1 || k2>=nbits_) k2 = nbits_ - 1;
  for(uint_8 k=k1;k<=k2;k++) Set(k,v);
}

//! Return the number of bits with value v (true/false)
uint_8 BitVector::NValues(bool v)
{
  uint_8 n = 0;
  if(nbits_==0) return n;
  for(uint_8 i=0;i<nbits_;i++) if((*this)(i)==v) n++;
  return n;
}

//! print bit vector between [k1,k1+dk[, if dk=0 don't print elements, dk<0 print [k1,nbits[
void BitVector::Print(uint_8 k1,int_8 dk)
{
  cout<<"BitVector::Print: nbits="<<nbits_<<" nvector="<<nvector_
      <<" vecsize="<<vecsize_<<" bitpervec="<<bitpervec_<<" ("<<nbits_/8000000.<<" MB)"<<endl;
  if(k1>=nbits_ || dk==0) return;
  uint_8 k2 = k1 + dk;
  if(k2>nbits_ || k2<=k1) k2 = nbits_;
  if(k1!=0) cout<<"print ["<<k1<<","<<k2<<"["<<endl;
  for(uint_8 k=k1;k<k2;k++) cout<<(*this)(k);
  cout<<endl;
}

//-----------------------------------------------------
DECL_TEMP_SPEC  /* equivalent a template <> , pour SGI-CC en particulier */
void   ObjFileIO<BitVector>::WriteSelf(POutPersist& s)  const
{
  if (dobj == NULL)   return;
  s.Put(dobj->nbits_);
  s.Put(dobj->vecsize_);
  s.Put(dobj->bitpervec_);
  s.Put(dobj->nvector_);
  if(dobj->nvector_==0) return;
  /* A REMPLACER par l'ecriture directe de vector< NDataBlock<uint_8> > */
  for(uint_8 i=0;i<dobj->nvector_;i++) s << dobj->vv_[i];
  return;
}

DECL_TEMP_SPEC  /* equivalent a template <> , pour SGI-CC en particulier */
void  ObjFileIO<BitVector>::ReadSelf(PInPersist& s)
{
  if (dobj == NULL) dobj = new BitVector;
  else dobj->delete_();
  s.Get(dobj->nbits_);
  s.Get(dobj->vecsize_);
  s.Get(dobj->bitpervec_);
  s.Get(dobj->nvector_);
  if(dobj->nvector_==0) return;
  /* A REMPLACER par la lecture directe de vector< NDataBlock<uint_8> > */
  for(uint_8 i=0;i<dobj->nvector_;i++) {
    NDataBlock<BitVec_Type> V;
    s >> V;
    dobj->vv_.push_back(V);
  }

  return;
}

#ifdef __CXX_PRAGMA_TEMPLATES__
#pragma define_template ObjFileIO<BitVector>
#endif

#if defined(ANSI_TEMPLATES) || defined(GNU_TEMPLATES)
template class ObjFileIO<BitVector>;
#endif

} // FIN namespace SOPHYA 
