// archeopsfile.h
// Eric Aubourg         CEA/DAPNIA/SPP   juillet 1999

#ifndef ARCHEOPSFILE_H
#define ARCHEOPSFILE_H

#include <stdio.h>
#include <string>
#include <stack>


#ifndef nb_max_bolo
#define _archeops		//  Acquisition Archeops (avec transputer) 
#define programme
extern "C" {
#include "archeops.h"
}
#endif

#include "ssthandler.h"
#include "gpsparser.h"

enum {
  block_param_mask = 1 << block_param,
  block_journal_mask = 1 << block_journal,
  block_reglage_mask = 1 << block_reglage,
  block_dilution_mask = 1 << block_dilution,
  block_gps_mask = 1 << block_gps,
  block_une_periode_mask = 1 << block_une_periode,
  block_synchro_sol_mask = 1 << block_synchro_sol,
  block_pointage_sol_mask = 1 << block_pointage_sol,
  block_bolo_nc_mask = 1 << block_bolo,
  block_gyro_nc_mask = 1 << block_gyro,
  block_sst_nc_mask = 1 << block_sst,
  block_11_mask = 1 << block_11,
  block_bolo_comprime_mask = 1 << block_bolo_comprime,
  block_gyro_comprime_mask = 1 << block_gyro_comprime,
  block_sst_comprime_mask = 1 << block_sst_comprime,
 // block_catalog_flash_mask = 1 << block_catalog_flash,
  block_cmd_flash_mask = 1 << block_cmd_flash,
  block_data_brute_mask = 1 << block_data_brute,
  block_18_mask = 1 << block_18,
  block_19_mask = 1 << block_19,

  block_bolo_mask = block_bolo_nc_mask | block_bolo_comprime_mask,
  block_gyro_mask = block_gyro_nc_mask | block_gyro_comprime_mask,
  block_sst_mask = block_sst_nc_mask | block_sst_comprime_mask
};

class BlockSet;

class ArchExc {
public:
  ArchExc(string s) : msg(s) {}
  string Msg() {return msg;}
protected:
  string msg;
};

class ArcheopsFile {
public:
  ArcheopsFile(string const& fname);  // starting before first block
  ArcheopsFile(ArcheopsFile const&);
  virtual ~ArcheopsFile();
  
  void setUTCOffset(int UTCOffset=2); // StartMJD : decodage a partir du nom du bloc...
  double getStartMJD();
  
  bool nextBlock();   // go to next block, true if block exists (ie not EOF)
  bool nextBlock(long mask); // go to next block matching mask
  bool gotValidBlock(); // current block is good
  int  blockKind();   // kind of the current bloc, enum in archeops.h
  int  blockRawKind();   // kind of the current raw bloc, enum in archeops.h
  int  blockNum();   // number (timestamp) of the current bloc
  string blockKdName();
  string blockRawKdName();
  string filename() {return fn;}
  
  double perEchant(); // periode d'echantillonage pour le dernier bloc reglage
  double perBlock();  // duree (en secondes) correspondant a un bloc bolo
                      // utilise infos dernier bloc param et dernier bloc reglage.
  int nEchBlock(); // Nb d'echantillons dans un bloc bolo
  
  
  void pushMark();    // push current file position, and "last" blocks
  void popMark();     // pops last file position and "last" blocks
    
  void grabLastBlocs(ArcheopsFile const& oldFile);
  
  block_type_modele*  aheadBlock();   // internal buffer
  block_type_modele*  currentBlock(); // internal buffer
  bool sameBlockNumAhead();
  
  block_type_param*         lastParam();  // saved blocks, including current
  block_type_journal*       lastJournal();
  block_type_reglage*       lastReglage();
  block_type_dilution*      lastDilution();
  block_type_gps*           lastGPS();
  block_type_une_periode*   lastUnePeriode();
  block_type_synchro_sol*   lastSynchroSol();
  block_type_pointage_sol*  lastPointageSol();
  block_type_bolo*          lastBolo();   // can be uncompressed bolo_comp
  block_type_bolo*          llastBolo();   
  block_type_gyro*          lastGyro();
  block_type_sst*           lastSST();
  block_type_bolo_comprime* lastBoloComp();   
  block_type_gyro_comprime* lastGyroComp();
  block_type_sst_comprime*  lastSSTComp();
  
  void forceBlock(block_type_modele* blk);
  
  // Decodage des derniers blocs..
  
  // GPS
  // On travaille en MJD = JD - 2450000.0, = 9 octobre 1995 midi UTC
  // 1er juillet 1999 minuit UTC = JD 2451360.5,   MJD 1360.5 
  // Ces fonctions travaillent sur le dernier bloc GPS 
  int getGPSBlockNum();
  bool   hasGPSTime();
  bool   hasGPSPos();
  bool   hasGPSAlt();
  double getGPSUTC();  // en secondes depuis minuit UTC
  double getGPSMJD();  // modified julian day du dernier bloc GPS, JD - 2450000
  double getGPSLat();  // degres, +  = NORD
  double getGPSLong(); // degres, +  = EST
  double getGPSAlt();  // meters from sea level
    
  // Bolo
  int getReglageBlockNum();
  int getBoloBlockNum();
  int getBoloCompBlockNum();
  long getRawBolo(int ibolo, int imesure);   // donnee brute, avec seulement soustraction offset
  double getRawBoloCN(int ibolo, int imesure);  // bruit de compression, sur donnee brute
  double getMuVBoloCN(int ibolo, int imesure);  // bruit de compression, en microVolts
  double getMuVBolo(int ibolo, int imesure); // microvolts, filtre avec filtre carre
  double getMuVBolo2(int ibolo, int imesure); // microvolts, filtre +sioux
  double getGainAmpli(int ibolo);
  double getDACV(int ibolo); // offset en microvolt
  double getDACI(int ibolo); // courant en microAmps
  double getMuVBolo2T(int ibolo, int imesure); // avec soustraction dacV
  double getBoloRes(int ibolo, int imesure); // Ohms
  
  // Dilution
  int getDilutionBlockNum();
  int4 getADCDil(int iADC);
  int4 getSwitchDil();
  
  // SST, gyros, etc
  int getSSTBlockNum();
  int getSSTCompBlockNum();
  void needSSTProcessMask(int mask);
  long getSSTRawSignal(int iDiode, int imesure);
  long getSSTSignal(int iDiode, int imesure);
  long getSSTRawSignalCN(int iDiode, int imesure);
  long getSSTSignalCN(int iDiode, int imesure);
  double getSSTStarZ(int iStar, int imesure);
  double getSSTStarF(int iStar, int imesure);
  double getSSTStarT(int iStar, int imesure);
  double getNumbStar(int imesure);
  
  int getGyroBlockNum();
  long getGyroRaw(int iGyro, int imesure);
  double getGyroTens(int iGyro, int imesure);
  double getGyroSpeed(int iGyro, int imesure);
  
  double getAzimut(int imesure);
  double getPendDirect(int imesure);
  double getPendOrth(int imesure);
  
  double getAlpha(int imesure);
  double getDelta(int imesure);
  
  static double decodeMJD(string const& dateString); // sans tenir compte de TU-TLeg
    
protected:
  BlockSet*                 blockSet;
  stack<BlockSet*>          blockStack;
  stack<size_t>             fposStack;
  
  void saveCurBlock();
  void postProcessBlock();

  void swapEntete(block_type_modele* blk);
  void swapContent(block_type_modele* blk);
  
  void computeMJD(string const& fname);
  
  bool fixBlock(block_type_modele* blk);
  long searchNextBlock(long pos);
    
  int                       curKind;
  int                       curRawKind;
  long                      curPos;
  long                      peekPos;
  size_t                    fLen;
  FILE* f;
  string fn;
  
  double rawMJD;   // decodage a partir du nom de fichier, sans application d'offset heure locale
  double startMJD; // MJD du debut du fichier.
  double utcOffset;
private:
};

#endif
