// toiiter.h
// Eric Aubourg         CEA/DAPNIA/SPP   juillet 1999

#ifndef TOIITER_H
#define TOIITER_H


#include <vector>
#include <set>
#include <map>
#include <string>
#include "archeopsfile.h"
#include "auxgps.h"
#include "tsid.h"

enum TOIKind {
  sampleNum,     // Numero d''echantillon (timer interne transputer)
  internalTime,  // temps transputer depuis debut manip, en secondes.
  mjd,           // UTC (en jours, MJD = JD - 2450000).
  mutc,          // UTC en heure, avec soustraction d'origine
  boloMuV,       // tension en microVolts, filtree avec filtre carre indexe par bolometre
  boloMuV2,      // tension en microVolts, avec soustraction offset filtre
  boloRawMuV,    // tension brute en microVolts, non filtree indexe par bolometre, sans chgt signe
  boloTemp,      // Kelvins, need R(T) -> N/A
  boloDACV,      // tension de reference du DAC (microVolts)
  boloDACI,      // courant (microAmps)
  boloRawMuVCN,  // compression noise on bolo (microVolts), no filter
  boloMuV2T,     // total, microVolts, avec consigne DAC
  boloRes,       // Resistance, ohms
  boloGainAmpli, //
  dilDAC,        // dilution DAC, index = 0-47
  dilSwitch,     // dilution, mot de 32 bits
  serviceTemp,   // service temperature, K
  sstDiode,      // signal brut SST, indice = diode, 0-45
  sstChannel,    // signal brut SST, indice = canal, 0-47
  sstDiodeCN,    // compression noise
  sstChannelCN,  // compression noise
  sstStarCnt,    // nombre d''etoiles detectees durant ce sample
  sstStarZ,      // numero de diode d''une eventuelle etoile,   i=etoile 0..n
  sstStarF,      // flux d''une eventuelle etoile,             <0 si moins de i+1 etoiles
  sstStarT,      // temps d''une eventuelle etoile,            <0 si moins de i+1 etoiles
  gyroRaw,       // valeur brute du gyro, index = 0,1,2
  gyroV,         // tension gyro, V, index = 0,1,2
  gyroSpeed,     // vitesse gyro, deg/s, index = 0,1,2, calibree...
  gpsTime,       // temps fourni par GPS
  longitude,     // position ballon, degrees  +=EST
  latitude,      // position ballon, degrees  +=NORD
  altitude,      // position ballon, m
  tsid,          // temps sideral en secondes
  azimuthBolo,   // azimut 
  alphaRotAxis,  // 
  deltaRotAxis,  //
  alphaZenith,   // 
  deltaZenith,   //
  alphaSst,      // 
  deltaSst,      //
  alphaFPAxis,   //
  deltaFPAxis,   //
  alphaBolo,     // sur le ciel, indexe par bolometre
  deltaBolo,     // sur le ciel, indexe par bolometre
  voyantEVO,
  voyantEVF,
  commandeEV0,
  commandeEVF,
  commandeEVB,
  commandeEVV,
  pressEnt3He,
  debit3He,
  pressSor3He,
  pressEnt4He,
  debit4He,
  pressSor4He,
  pressAirVanne,
  pressPompChar,
  pressMembrane,
  pressExterne,
  tensPile10T,
  tensPileP18D,
  tensPileM18D,
  tensPile10B,
  tensPileP18B,
  tensPileM18B,
  tensPileCh,
  swPile5,
  swPile15,
  tempCaissH1,
  tempCaissH2,
  tempCaissB1,
  tempCaissB2,
  tempCaissTHe,
  tempCaissPiles,
  tempCaissDrv,
  pressHeBain,
  pressPirani
}; 

class TOIInterpolator;

class TOIIter {
public:
   TOIIter(TOIIter const&);   // $CHECK$ prevoir operator = ?
   ~TOIIter();
   bool Next();
   
   bool    canGetValue(int column);  // data is available for that...
   double  getValue(int column);
   bool    newValue(int column);     // a juste change ?
   bool    extendValue(int column);  // une valeur plus ancienne, etendue ?
   bool    interpValue(int column);  // une valeur interpolee avec valeur future ?
   bool    isTrig(int column);  
   TOIKind getKind(int column);
   int     getIndex(int column);     // si plusieurs, exemple, tension bolo 2
   
   bool    canGetTOI(TOIKind kind, int index=0);
   double  getTOI(TOIKind kind, int index=0);
   
   
   int    getBlockSampleIndex();        // numero d''echantillon dans dernier bloc bolo
   int    getSampleIndex();             // numero d''echantillon 
   int    getUnderSampling();           
   double getMJD();            

   block_type_param*         lastParam();  
   block_type_journal*       lastJournal();
   block_type_reglage*       lastReglage();
   block_type_dilution*      lastDilution();
   block_type_gps*           lastGPS();
   block_type_une_periode*   lastUnePeriode();
   block_type_synchro_sol*   lastSynchroSol();
   block_type_pointage_sol*  lastPointageSol();
   block_type_bolo*          lastBolo();   
   block_type_gyro*          lastGyro();
   block_type_sst*           lastSST();
   block_type_bolo_comprime* lastBoloComp();   
   block_type_gyro_comprime* lastGyroComp();
   block_type_sst_comprime*  lastSSTComp();
   
   ArcheopsFile* currentFile() {if (!initDone) Init(); return file;}

protected:
   TOIIter();
   void Init(); // After setting the options, opens the first file.
   void ScanFiles(); // Parcourt tous les fichiers, cherche le premier echantillon...
   
   bool Next1();
   bool NextSample(); // no test of trange
   bool NextFile();
   
   ArcheopsFile* file;
   string directory;   // If directory scanning
   set<string> files;  // list of files to be read...
   map<string, long> file1stSamp; // premier echantillon de chaque fichier
   set<string>::iterator curFile;
   bool  isOnBoardRecorder;
   
   int imes;
   
   double tStart, tEnd;      // MJD
   double utcStart, utcEnd;  // UTC, will be converted towards tStart tEnd at init
   long   sStart, sEnd;      // samplenum
   
   //double tBlock0;  // MJD du bloc numero zero...
   //double perEch;   // periode d''echantillonage en secondes
   
   struct info {
     TOIKind kind;
     int index;
     bool triggering;
     bool interpolated;
   };
   
   int getColTOI(TOIKind kind, int index=0);
   
   long trigMask;
   
   vector<info>   infos;
   
   friend class TOISvr;
   
   // Gestion de l''interpolation, par delegation a un autre iterateur
   TOIIter* rawIter; // iterateur sans interpolation, qui avance avec retard
                     // en bufferisant les valeurs. S''il existe, on lui delegue
                     // toutes les operations agissant sur le fichier.
   TOIInterpolator* interp; // interpolateurs pour stocker les valeurs
   long lastSample;  // last interpolated sample
   long maxLookAhead;
   bool fetchAhead(); // avance d''une TOI en nourrissant les interpolateurs
   
   AuxGPS* auxGPS;
   TSid tSid;
   
   int underSample;
   
   private:
   bool initDone;
};

#endif
